/* evaluate
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "stacksg.h"
#include "jalcomp.h"
#include "treerep.h"
#include "treetools.h"
#include "scanner.h"
#include "compdriver.h"
#include "cstringf.h"

void check_zero(tree p, int x)
{
    if (x == 0) {
        fatal(p->loc, (m, "second operand should not be zero"));
    }
}

/* evaluate a tree */
void evaluate(tree p)
{
    int a, b, x;
    stack_guard;
    assert_pointer(NULL, p);

    a = b = x = 0;
    switch (p->kind) {

    case node_ref:{

            /* evaluate lower, copy value */
            assert_pointer(p->loc, p->first);
            evaluate(p->first);
            p->value = p->first->value;
            return;
        }

    case node_const:{

            /* should already have a value */
            /* interferes with assembly constants */
#ifdef __DEBUG__
            assert_kind(p->loc, p->value, node_value);
#endif
            return;
        }

    case node_op:{

            /* continue below */
            break;
        }

    default:{
            return;
        }
    }

    /* discard any previous value */
    p->value = NULL;

    /* can we evaluate? */
    if (!is_buildin_type(p->type))
        return;

    /* evaluate left and (for duop) right */
    assert_pointer(NULL, p->first);
    evaluate(p->first);
    if (p->first->value == NULL)
        return;
    a = p->first->value->x;

    if (!is_monop(p->op)) {
        assert_pointer(NULL, p->next);
        evaluate(p->next);
        if (p->next->value == NULL)
            return;

        b = p->next->value->x;
    }

    switch (p->op) {
    case op_and:
        x = a & b;
        break;
    case op_or:
        x = a | b;
        break;
    case op_xor:
        x = a ^ b;
        break;
    case op_smaller:
        x = a < b;
        break;
    case op_larger:
        x = a > b;
        break;
    case op_smaller_or_equal:
        x = a <= b;
        break;
    case op_larger_or_equal:
        x = a >= b;
        break;
    case op_equal:
        x = a == b;
        break;
    case op_not_equal:
        x = a != b;
        break;
    case op_minus:
        x = b - a;
        break;                  /* !!!!! */
    case op_plus:
        x = a + b;
        break;
    case op_times:
        x = a * b;
        break;
    case op_shift_left:
        x = a << b;
        break;
    case op_shift_right:
        x = a >> b;
        break;
    case op_modulo:
        check_zero(p, b);
        x = a % b;
        break;
    case op_divide:
        check_zero(p, b);
        x = a / b;
        break;

    case op_mnot:
        x = (-1) ^ a;
        break;
    case op_mminus:
        x = -a;
        break;
    case op_mplus:
        x = a;
        break;
    case op_check: /* Added by Javi. Needs optimization to avoid a call */ 
        if (a == b) {
            x = 1;
        } else {
            x = 0;
        }
        break;

    default:
        snark_node(p->loc, p);
    }


    p->value = new_value(p->type, x);
    chop(p->value);
#ifdef __DEBUG__
    show_subtree(p);
    log((m, "a=%d b=%d x=%d y=%d", a, b, x, p->value->x));
#endif
}

/* determine and set the type of a node */
void set_type(tree p)
{
    stack_guard;
    assert_pointer(NULL, p);
    switch (p->kind) {
    case node_assign:
    case node_op:{
            if (is_monop(p->op)) {
                p->type = p->first->type;
            } else {

                if ((p->first->type == type_universal)
                    && (is_buildin_type(p->next->type))
                    ) {
                    p->type = p->next->type;

                } else if ((p->next->type == type_universal)
                           && (is_buildin_type(p->first->type))
                    ) {
                    p->type = p->first->type;

                } else if (p->first->type == p->next->type) {
                    p->type = p->first->type;

                } else {
                    fatal(p->loc,
                          (m, "not possible with types %s and %s", p->first->type->name,
                           p->next->type->name));
                }
            }
            break;
        }
    default:
        snark_node(p->loc, p);
    }
}


/********** debug print **********/

char *mode_string(int m)
{
    stack_guard;
    switch (m) {
    case mode_in:
        return "in";
    case mode_out:
        return "out";
    case mode_in | mode_out:
        return "in-out";
    default:
        return "??";
    }
}

/* show this node */
void show_tree_node(tree p, int prefix, char *name)
{
    string s, v;
    if (p->kind == node_value) {
        sprintf(v, "%d", p->x);
    } else {
        sprintf(v, "");
    }
    sprintf(s, "%s%04d@%04d %s %s %s %s %s '%s' %s %s %s %s [%d] %s %s %s", bar(prefix, ' '), p->nr,
            ((p->loc == NULL) ? 0 : p->loc->line_nr), name, node_name[p->kind],
            ((p->kind == node_op) ? op_name[p->op] : ""),
            ((p->kind == node_asm) ? opcode_name[p->opcode] : ""), p->name,
            ((p->type != NULL) ? p->type->name : "<untyped>"), v, (p->indirect ? "i" : " "),
            (p->is_volatile ? "v" : " "), (((p->kind == node_ref) || (p->kind == node_var))
                                           ? ref_name[p->ref]
                                           : ""), p->used, ((p->kind != node_ref)
                                                            ? "" : ref_name[p->ref]
            ), ((p->is_chained) ? "ch" : ""), ((p->pass_in_w) ? "W" : "")
        );
    printf("%s\n", s);
}

/* print a tree (for debugging) */
void show_tree2(tree p, int prefix, char *name, int level)
{
    int prefix2 = prefix + 2;
    stack_guard;
    level--;
    if (p == NULL)
        return;
    cassert((p->nr >= 0) && (p->nr <= node_nr));
#ifdef __DEBUG__
    log((m, "n=%d", p->nr));
#endif
    show_tree_node(p, prefix, name);
    if (level > 0) {
        show_tree2(p->condition, prefix2, "C:", level);
        show_tree2(p->impl, prefix2, "M:", 2);
        show_tree2(p->ret, prefix2, "R:", 2);
        show_tree2(p->var, prefix2, "V:", level);
        show_tree2(p->start, prefix2, "S:", level);
        show_tree2(p->end, prefix2, "E:", level);
        show_tree2(p->step, prefix2, "I:", level);
        show_tree2(p->address, prefix2, "A:", level);
        show_tree2(p->value, prefix2, "X:", level);
        show_tree2(p->put, prefix2, "P:", 2);
        show_tree2(p->get, prefix2, "G:", 2);
        show_tree2(p->uncle, prefix2, "U:", 2);
        show_tree2(p->master1, prefix2, "M1:", 2);
        show_tree2(p->master2, prefix2, "M2:", 2);
        show_tree2(p->result, prefix2, "R:", level);
        switch (p->kind) {
        case node_ref:
        case node_return:
        case node_asm:
        case node_precall:
        case node_call:{
                show_tree2(p->first, prefix2, "F:", 2);
                break;
            }
        default:{
                show_tree2(p->first, prefix2, "F:", level);
                break;
            }
        }
        show_tree2(p->next, (p->kind == node_chain) ? prefix : prefix2, "N:", level);
    }
}
void show_subtree(tree p)
{
    show_tree2(p, 0, "", 100000);
}

void show_node(tree p)
{
    show_tree2(p, 0, "", 2);
}
void show_tree(void)
{
    show_subtree(root_block);
}
