/* recursive-decent parser
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "cstringf.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "regalloc.h"
#include "reswords.h"
#include "compdriver.h"
#include "parser.h"
#include "treetools.h"
#include "eval.h"


block_options_t block_options;

tree current_subprogram = NULL;


void force_type(tree p, tree t)
{
    stack_guard;
    assert_kind(NULL, t, node_type);
    assert_pointer(NULL, p);
    if (p->type == type_universal) {
        p->type = t;
    }
}

tree parse_op(loc_t loc, tree left, int op, tree f(void))
{
    tree p;
    stack_guard;
    log_parse("parse generic op");
    p = new_node(loc, node_op);
#ifdef __DEBUG__
    trace_subtree(p);
    trace_subtree(left);
#endif
    p->first = left;
    p->op = op;
    p->type = type_universal;
    scanner_next_token();
    cassert(is_monop(p->op) == (left == NULL));
    if (is_monop(op)) {
        p->first = f();
    } else {
        p->next = f();
#ifdef __DEBUG__
        trace_subtree(p->next);
#endif
    }
///*  Erased by Javi 2003-03-31. Done in codegen/code_byte_op
   if (op == op_minus) {
        swapp(p->first, p->next);
    }
///*/
#ifdef __DEBUG__
    trace_subtree(p->next);
    trace_subtree(p->first);
    trace_subtree(p);
#endif

    if ((p->first->type == type_universal)
        && ((p->next == NULL)
            || (p->next->type == type_universal)
        )
        ) {
        /* universal <op> universal is always OK */
        p->type = universal_result(p->op);
        p->impl = NULL;
    } else {
#ifdef __DEBUG__
        trace_subtree(p)
#endif
            p->impl = find_operator(p->op, p->first, p->next);
        if (p->impl == NULL) {
            fatal(p->loc, (m, "operator not known for these operands"));
        }
        p->type = p->impl->type;
    }
    return p;
}

tree parse_constant(void)
{
    tree p = NULL;
    int x;
    stack_guard;

    if (scanner_is_like(s_true)) {
        p = new_value(type_bit, 1);

    } else if (scanner_is_like(s_false)) {
        p = new_value(type_bit, 0);

    } else if (string_int_value(scanner_context->token, &x)) {
        p = new_value(type_universal, x);

    } else if (scanner_context->token[0] == '"') {
        if (scanner_context->token[string_length(scanner_context->token) - 1] != '"') {
            fatal(scanner_context->loc, (m, "unterminated character constant"));
        }
        if (string_length(scanner_context->token) != 3) {
            fatal(scanner_context->loc, (m, "illegal character constant"));
        }
        p = new_value(type_byte, scanner_context->token[1]);

    } else {
        p = find(scanner_context->token);
        if (p != NULL) {
            if (p->kind == node_const) {
                p = p->value;
            } else {
                p = NULL;
            }
        }
    }

    if (p != NULL) {
        scanner_next_token();
        p = new_const(p);
    }

    return p;
}

tree parse_expect_constant(void)
{
    loc_t loc = scanner_here();
    tree p;
    stack_guard;
    log_parse("parse_expect_constant");

    p = parse_expression();
    evaluate(p);

    if ((p == NULL) || (p->value == NULL)) {
#ifdef __DEBUG__
        show_subtree(p);
#endif
        fatal(loc, (m, "constant expected", loc));
    }
    return p;
}

void parse_procedure_call(loc_t loc, tree d);
tree f_parse_procedure_call(loc_t loc, tree d);
boolean procedure_is_constant = false;
boolean is_now_actual = false;
tree parse_primitive(void)
{
    tree p, call;
    loc_t loc = scanner_here();
    stack_guard;
    log_parse("parse_primitive");

    p = parse_constant();
    if (p != NULL) {
#ifdef __DEBUG__
        trace_subtree(p);
#endif
        return new_ref(loc, ref_const, p, NULL);
    }

    log_parse("parse_primitive 2");
    if (is_id_start(scanner_context->token[0])) {
        p = find(scanner_context->token);
#ifdef __DEBUG__
        trace_subtree(p)
#endif
            if (p == NULL) {
            fatal(loc, (m, "expression expected ", scanner_context->token));
        } else if (p->kind == node_const) {
            /* OK */
        } else if (p->kind == node_var) {
            /* OK */
            if (!p->read_allowed) {
                if (!is_now_actual) {
                    fatal(loc, (m, "can not be read"));
                }
            }
        } else if (p->kind == node_procedure) {
            if (procedure_is_constant) {
                /* OK */
            } else if (p->result == 0) {        /* procedure */
                fatal(loc, (m, "expression expected, procedure found ", scanner_context->token)
                    );
            } else {            /* function */
                scanner_next_token();
                call = f_parse_procedure_call(loc, p);
                return new_chain2(call, new_ref(loc, ref_var, p->result->first, current_subprogram)
                    );
            }
        } else if (p->kind == node_type) {
            fatal(loc, (m, "expression expected, type found ", scanner_context->token));
        } else {
            trace_subtree(p);
            snark_node(loc, p);
        }
        scanner_next_token();

        return new_ref(loc, p->ref, p, current_subprogram);
    }

    fatal(loc, (m, "expression expected ", scanner_context->token));
    return NULL;
}

tree parse_primary(void)
{
    stack_guard;
    log_parse("parse_primary 1");
    if (scanner_is_like_advance(s_brace_open)) {
        tree p;
        log_parse("parse_primary 2");
        p = parse_expression();
        log_parse("parse_primary 2");
        scanner_check_advance(s_brace_close);
        log_parse("parse_primary 3");
        return p;
    }
    log_parse("parse_primary 4");
    return parse_primitive();
}

#define parse_try_op( loc, left, op, lower ){ \
   loc_t loc = scanner_here(); \
   if( scanner_is_like( op_name[ op ] ) ){ \
     tree p; \
     log_parse( "parse_try_op 1" ) \
     p = parse_op( loc, left, op, lower ); \
     log_parse( "parse_try_op 2" ) \
     return p; \
   } \
}

tree parse_factor(void)
{
/*    loc_t loc = scanner_here(); */
    stack_guard;
    log_parse("parse_factor 1");
    parse_try_op(loc, NULL, op_mnot, parse_factor);
    log_parse("parse_factor 2");
    return parse_primary();
}

tree parse_term(void)
{
/*    loc_t loc = scanner_here(); */
    tree left;
    stack_guard;
    log_parse("parse_term 1");
    left = parse_factor();
    log_parse("parse_term 2");
    parse_try_op(loc, left, op_times, parse_term);
    parse_try_op(loc, left, op_divide, parse_term);
    parse_try_op(loc, left, op_modulo, parse_term);
    parse_try_op(loc, left, op_check, parse_term);
    log_parse("parse_term 3");
    return left;
}

tree parse_simple(void)
{
  /*  loc_t loc = scanner_here(); */
    tree left;
    stack_guard;
    log_parse("parse_simple 1");
    parse_try_op(loc, NULL, op_mplus, parse_simple);
    parse_try_op(loc, NULL, op_mminus, parse_simple);
    log_parse("parse_simple 2");
    left = parse_term();
    log_parse("parse_simple 3");
    parse_try_op(loc, left, op_plus, parse_simple);
    parse_try_op(loc, left, op_minus, parse_simple);
    log_parse("parse_simple 4");
    return left;
}

tree parse_relation(void)
{
/*    loc_t loc = scanner_here(); */
    tree left;
    stack_guard;
    log_parse("parse_relation 1");
    left = parse_simple();
    log_parse("parse_relation 2");
    parse_try_op(loc, left, op_larger, parse_simple);
    parse_try_op(loc, left, op_smaller, parse_simple);
    parse_try_op(loc, left, op_larger_or_equal, parse_simple);
    parse_try_op(loc, left, op_smaller_or_equal, parse_simple);
    parse_try_op(loc, left, op_equal, parse_simple);
    parse_try_op(loc, left, op_not_equal, parse_simple);
    parse_try_op(loc, left, op_shift_right, parse_simple);
    parse_try_op(loc, left, op_shift_left, parse_simple);
    log_parse("parse_relation 3");
    return left;
}

tree parse_expression(void)
{
/*    loc_t loc = scanner_here(); */
    tree left;
    stack_guard;
    log_parse("parse_expression 1");
    left = parse_relation();
    log_parse("parse_expression 2");
    parse_try_op(loc, left, op_and, parse_expression);
    parse_try_op(loc, left, op_or, parse_expression);
    parse_try_op(loc, left, op_xor, parse_expression);
    log_parse("parse_expression 3");
    if (!((left->kind == node_ref)
          && (left->first->kind == node_const)
        )) {
        log_parse("parse_expression 4");
        evaluate( left ); /* why? */
        log_parse("parse_expression 5");
        if (left->value != NULL) {
            left = new_ref(left->loc, ref_const, new_const(left->value), NULL);
        }
        log_parse("parse_expression 6");
    }
    log_parse("parse_expression 7");
    return left;
}

void parse_assign(tree d)
{
    loc_t loc;
    stack_guard;
    assert_kind(d->loc, d, node_ref);
    log_parse("parse_assign");

    /* give the statement the location of the assignment operator */
    loc = scanner_here();
    scanner_check_advance(s_assign);

    /* parse right side and add to the current block */
    add(new_assign(loc, d, parse_expression()));
}

tree parse_type(boolean universal_allowed)
{
    tree t;
    stack_guard;
    log_parse("parse_type");
    t = find(scanner_context->token);
    if (((t != NULL) && (t->kind != node_type))
        || ((t == NULL) && (!universal_allowed))
        ) {
        cfatal((m, "type not known"));
    }
    if (t != NULL)
        scanner_check_identifier_advance();
    if (t == NULL)
        t = type_universal;
    return t;
}

void parse_optional_address(tree p)
{
    if (scanner_is_like_advance(s_at)) {
        tree x = NULL;
        tree y = new_value(type_universal, 0);
        tree m;

        m = find(scanner_context->token);
        if ((m != NULL) && (m->kind == node_var)) {
            scanner_check_identifier_advance();
            p->master1 = m;
            p->master2 = m;
        } else {
            x = parse_expect_constant()->value;
        }

        if (scanner_is_like_advance(s_colon)) {
            y = parse_expect_constant()->value;
            p->master2 = 0;
            if (p->type != type_bit) {
                fatal(p->loc, (m, "a bit can not be specified for a byte variable")
                    );
            }
        } else {
            if (p->type == type_bit) {
                fatal(p->loc, (m, "a bit must be specified for a bit variable")
                    );
            }
        }

        p->fixed = true;
        p->address = new_chain2(x, y);

    } else if (scanner_is_like_advance(s_is)) {

        tree m = find(scanner_context->token);
        if ((m != NULL) && (m->kind == node_var)) {
            scanner_check_identifier_advance();
            p->uncle = m;
            if (p->type != m->type) {
                cfatal((m, "must be of the same type"));
            }

        } else {
            cfatal((m, "variable expected"));

        }
    }
}

void parse_var(void)
{
    tree t;
    boolean vol = false;
    loc_t loc = scanner_here();
    stack_guard;
    log_parse("parse_var");
    if (scanner_is_like_advance(s_volatile)) {
        vol = true;
    }

    t = parse_type(false);
    for (;;) {
        tree p = new_var(scanner_here(),
                         scanner_context->token,
                         t);
        p->is_volatile = vol;
        scanner_check_identifier_advance();
        parse_optional_address(p);
        check_and_add(new_decl(loc, p));
        loc = scanner_here();
        if (scanner_is_like_advance(s_assign)) {
            tree v = parse_expression();
            add(new_assign(loc, new_ref(loc, ref_var, p, NULL), v));
        }
        if (!scanner_is_like(s_comma)) {
            log_parse("parse_var end");
            return;
        }
        scanner_check_advance(s_comma);
    }
}

void parse_const(void)
{
    tree t, q;
    loc_t loc = scanner_here();
    stack_guard;
    log_parse("parse_const");
    t = parse_type(true);
    for (;;) {
        tree p = new_node(loc, node_const);
        p->type = t;
        p->name = new_string(scanner_context->token);
        scanner_check_identifier_advance();
        check_and_add(new_decl(loc, p));
        scanner_check_advance(s_assign);
        q = parse_expect_constant();
        p->value = q->value;
        if (!scanner_is_like(s_comma)) {
            log_parse("parse_const end");
            return;
        }
        scanner_check_advance(s_comma);
    }
}

/* if statement */
tree parse_if(loc_t loc)
{
    tree p = new_node(loc, node_if);
    stack_guard;

    p->condition = parse_expression();
#ifdef __DEBUG__
    trace_subtree(p)
#endif
        if (follow(p->condition)->type != type_bit) {
#ifdef __DEBUG__
        trace_subtree(p->condition)
#endif
            fatal(p->condition->loc, (m, "must be type bit"));
    }

    scanner_check_advance(s_then);
    p->first = parse_statement_list();

    if (scanner_is_like_advance(s_elsif)) {
        p->next = parse_if(scanner_here());
    } else {
        if (scanner_is_like_advance(s_else)) {
            p->next = parse_statement_list();
        }
        scanner_check_advance(s_end);
        scanner_check_advance(s_if);
    }

    /* optimize dead parts away (for pragma error!) */
    evaluate(p->condition);
#ifdef __DEBUG__
    trace_subtree(p->condition);
#endif
    if (p->condition->value != NULL) {
#ifdef __DEBUG__
        trace
#endif
            if (p->condition->value->x == 0) {
            p->first = NULL;
        } else {
            p->next = NULL;
        }
    }

    return p;
}

void old_parse_if(loc_t loc)
{
    add(parse_if(loc));
}

tree create_while(loc_t loc, tree condition, tree body)
{
    tree p = new_node(loc, node_while);
    stack_guard;
    p->condition = condition;
    p->first = body;
    return p;
}

/* while statement */
void parse_while(loc_t loc, tree condition)
{
    tree body;
    stack_guard;

    if (condition == NULL) {
        condition = parse_expression();
    }
    if (follow(condition)->type != type_bit) {
#ifdef __DEBUG__
        trace_subtree(condition)
#endif
            fatal(condition->loc, (m, "must be type bit"));
    }

    scanner_check_advance(s_loop);
    body = parse_statement_list();

    scanner_check_advance(s_end);
    scanner_check_advance(s_loop);
    {
        tree q = create_while(loc, condition, body);
#ifdef __DEBUG__
        trace_subtree(condition);
#endif
        add(q);
    }
#ifdef __DEBUG__
    add(create_while(loc, condition, body));
#endif
}

/* for statement */
void parse_for(loc_t loc)
{
    tree q;
    tree p = new_node(loc, node_for);
    stack_guard;
    add(p);

    p->var = NULL;
    p->first = NULL;
    p->step = NULL;
    p->end = parse_expression();

    if (p->end == NULL) {
        trace_subtree(p);
    }
    q = follow(p->end);
    assert_kind(loc, q->type, node_type);
    if (q->type == type_universal) {
        q->type = type_byte;
    }

    scanner_check_advance(s_loop);
    p->first = parse_statement_list();
    scanner_check_advance(s_end);
    scanner_check_advance(s_loop);
}

/* procedure call */
tree f_parse_procedure_call(loc_t loc, tree d)
{
    int i;
    tree q, pre_args, post_args;
    stack_guard;
#ifdef __DEBUG__
    trace_subtree(d);
#endif
    /* clear actual argument list */
    for (i = 1; NULL != (q = arg(d, i, false)); i++) {
#ifdef __DEBUG__
        log((m, "arg node %d", q->nr));
#endif
        assert_kind(q->loc, q, node_var);
        q->actual = NULL;
    }

    /* process actual arguments */
    if (scanner_is_like_advance(s_brace_open)) {
        i = 1;
        q = arg(d, i, false);
        while (!scanner_is_like_advance(s_brace_close)) {
            if (i > 1) {
                scanner_check_advance(s_comma);
            }
#ifdef __DEBUG__
            log((m, "arg node %d", q->nr));
#endif
            if (q == NULL) {
                cfatal((m, "too many arguments"));
            }
            if ((q->is_volatile)
                && (current_subprogram != NULL)
                && (current_subprogram->is_virtual)
                ) {
                cfatal((m, "passing a volatile from a virtual is not allowed"));
            }
            is_now_actual = true;
            q->actual = parse_expression();
            is_now_actual = false;
#ifdef __DEBUG__
            trace_subtree(q->actual);
#endif
            q = arg(d, ++i, false);
        }
    }

    /* pass mode_in parameters */
    pre_args = NULL;
    for (i = 1; NULL != (q = arg(d, i, false)); i++) {
        tree par;
        assert_kind(NULL, q, node_var);
        assert_pointer(NULL, q->name);
        if (mode_has(q->mode, mode_in) || (q->is_volatile)) {
            tree p = q->actual;
            if (p == NULL)
                p = q->first;
            if (p == NULL) {
                /* t0023 */
                fatal(loc, (m, "no actual input for %s", q->name));
            }
            if (q->pass_in_w) {
#ifdef __DEBUG__
                show_subtree(par);
#endif
                par = new_ref(loc, ref_actual, new_w(NULL, q->mode), NULL);
                par->first->is_volatile = q->is_volatile;
            } else {
                par = new_ref(loc, ref_actual, q, NULL);
            }
            pre_args = new_chain2(pre_args, new_assign_actual(loc, par, p)
                );
        }
    }

    /* assign mode_out parameters */
    post_args = NULL;
    for (i = 1; NULL != (q = arg(d, i, false)); i++) {
        assert_kind(q->loc, q, node_var);
        if (mode_has(q->mode, mode_out) && (!q->is_volatile)) {
            tree p = q->actual;
            if (p == NULL)
                p = q->first;
            if (p == NULL) {
                /* t0024 */
                fatal(loc, (m, "no actual output for %s", q->name));
            }
			if( p->kind != node_ref) {
                fatal(loc, (m, "invalid actual for %s", q->name));
            }
/*            assert_kind(NULL, p, node_ref); */
            assert_pointer(NULL, p->first);
            if (p->first->kind != node_var) {
                /* t0022 */
                fatal(p->loc, (m, "invalid actual for %s", q->name));
            }
            post_args = new_chain2(post_args, new_assign(loc, p, new_ref(loc, ref_actual, q, NULL))
                );
        }
    }
    return new_chain4(new_precall(loc, d), pre_args, new_call(loc, d), post_args);
}

void parse_procedure_call(loc_t loc, tree d)
{
    add(f_parse_procedure_call(loc, d));
}

tree parse_argument(void)
{
    tree p = new_var(scanner_here(), "", type_universal);
    stack_guard;
    p->mode = 0;
    p->is_argument = true;

    p->type = parse_type(false);
    p->is_volatile = scanner_is_like_advance(s_volatile);

    if (scanner_is_like_advance(s_mode_in)) {
        p->mode |= mode_in;
    }
    if (scanner_is_like_advance(s_mode_out)) {
        p->mode |= mode_out;
    }
    if (p->mode == 0) {
        p->mode |= mode_in;
        p->mode |= mode_out;
    }

    scanner_check_identifier();
    p->name = new_string(scanner_context->token);
#ifdef __DEBUG__
    trace_subtree(current_subprogram);
#endif
    if (NULL != find_local(p->name)) {
#ifdef __DEBUG__
        trace_subtree(find_local(p->name));
#endif
        cfatal((m, "identifier already declared in current argument list"));
    }
    scanner_next_token();

    parse_optional_address(p);
#ifdef __DEBUG__
    show_subtree(p);
#endif
    if (scanner_is_like_advance(s_assign)) {
        loc_t loc = scanner_context->loc;
        p->first = parse_expression();
#ifdef __DEBUG__
        trace_subtree(p->first);
#endif
        if (mode_has(p->mode, mode_out)) {
            assert_pointer(p->loc, p->first);
            if ((p->first->kind != node_ref)
                || (p->first->first == NULL)
                || (p->first->first->kind != node_var)
                ) {
                fatal(loc, (m, "variable expected"));
            }
        }
    }
    return p;
}

void parse_formal_parameters(boolean * has_out)
{
    stack_guard;
    if (scanner_is_like_advance(s_brace_open)) {
        tree q = parse_argument();
        if (mode_has(q->mode, mode_out)) {
            *has_out = true;
        }
        q->ref = ref_formal;
        check_and_add(new_decl(q->loc, q));
        while (!scanner_is_like(s_brace_close)) {
            scanner_check_advance(s_comma);
            q = parse_argument();
            if (mode_has(q->mode, mode_out)) {
                *has_out = true;
            }
            q->ref = ref_formal;
#ifdef __DEBUG__
            trace_subtree(q)
#endif
                check_and_add(new_decl(q->loc, q));
        }
        scanner_check_advance(s_brace_close);
        if ((q->type == type_byte)
            && ((target_cpu == pic_14) | (target_cpu == pic_16))
            ) {
#ifdef __DEBUG__
            trace_subtree(q);
#endif
            q->pass_in_w = block_options.pass_in_w;
        }
    }
}

/* subprogram declaration */
tree last_subprogram;
void parse_subprogram_declaration2(tree p, boolean is_function, boolean is_operator,
                                   tree surrounding_subprogram)
{
    boolean is_get = false, is_put = false;
    boolean has_out = false;
    char *label = p->name;
    tree target = NULL;
    stack_guard;
    log_parse("parse_subprogram");

    /* handle 'get and 'put */
    if (!(is_function && is_operator)) {
        scanner_check_identifier_advance();
        if (scanner_is_like_advance(s_tick)) {
            string s;
            p->is_virtual = true;

            /* not allowed inside another procedure or function */
            if (surrounding_subprogram != NULL) {
                fatal(p->loc, (m, "a virtual can not be nested"));
            }

            /* if the base name exists it should be an indirect var */
            target = find_local(p->name);
            if ((target != NULL) && (target->kind != node_var)) {
                fatal(p->loc, (m, "identifier already declared differently"));
            }
            if (target == NULL) {
                target = new_var(p->loc, p->name, type_universal);
                check_and_add(new_decl(p->loc, target));
                target->indirect = true;
                target->read_allowed = false;
                target->write_allowed = false;
            }

            sprintf(s, "_%s__%s", p->name, scanner_context->token);
            p->name = new_string(s);

            if (is_function && scanner_is_like_advance(s_get)) {
                if (target->get != NULL) {
                    fatal(p->loc, (m, "duplicate get"));
                }
                target->get = p;
                is_get = true;
                label = "get";
                target->read_allowed = true;
            } else if ((!is_function)
                       && scanner_is_like_advance(s_put)) {
                if (target->put != NULL) {
                    fatal(p->loc, (m, "duplicate put"));
                }
                target->put = p;
                is_put = true;
                label = "put";
                target->write_allowed = true;
            } else {
                fatal(scanner_here(), (m, "invalid attribute"));
            }
        }
    } else {
        scanner_next_token();
    }

    {
        open_list(temp);
        p->first = temp;
        parse_formal_parameters(&has_out);
        if (is_function && is_operator) {
            tree arg1 = arg(p, 1, false);
            tree arg2 = arg(p, 2, false);
            tree arg3 = arg(p, 3, false);

            /* check argument requirements */
            if ((arg1 == NULL) || (arg3 != NULL)) {
                fatal(p->loc, (m, "an operator must have one or two arguments"));
            }
            if (arg1->mode != mode_in) {
                fatal(arg1->loc, (m, "operator arguments must be of mode in"));
            }
            if ((arg2 != NULL) && (arg2->mode != mode_in)) {
                fatal(arg2->loc, (m, "operator arguments must be of mode in"));
            }

            p->op = operator_index(p->name, (arg2 == NULL));
            if (p->op == 0) {
                fatal(p->loc, (m, "invalid operator symbol"));
            }
            label = op_label[p->op];
        }

        if (is_function) {
            scanner_check_advance(s_return);
            p->type = parse_type(false);
        }
        scanner_check_advance(s_is);

        /* the label separates the parameters from the body */
        {
            string full_label;
            sprintf(full_label, "p_%d_%s", p->nr, label);
            p->label = new_string(full_label);
            string_to_lowercase(p->label);
            add(p->address = new_label(NULL, full_label, 0, 0));
        }
#ifdef __DEBUG__
        log((m, "if=%d ho=%d n=%s", (int) is_function, (int) has_out, p->name));
#endif
        if (is_function) {
            if ((!has_out)
                && (p->type == type_byte)
                && (!p->is_virtual)
                && (target_chip != t_12c509a)
                && (target_chip != t_12c508)
                && (target_chip != t_none)
                ) {
                p->result = new_ref(NULL, ref_var, new_w(NULL, mode_out), NULL);
            } else {
                p->result = new_decl(p->loc, new_var(p->loc, " return value", p->type)
                    );
            }
            add(p->result);
        }

        if (is_put) {
            tree arg1 = arg(p, 1, false);
            tree arg2 = arg(p, 2, false);

            if ((arg1 == NULL) || (arg2 != NULL)) {
                fatal(p->loc, (m, "a 'put must have one argument"));
            }
            assert_kind(p->loc, arg1->type, node_type);
            target->type = arg1->type;
        }

        if (is_get) {
            tree arg1 = arg(p, 1, false);

            if (arg1 != NULL) {
                fatal(p->loc, (m, "a 'get can not have an argument"));
            }
            assert_kind(p->loc, p->result->first->type, node_type);
            target->type = p->result->first->type;
        }

        log_parse("parse_subprogram start of block");
        if (is_put) {
            p->tlabel = new_label(p->loc, "p_%d_%s_t", p->nr, p->name);
            p->transfer1 =
                new_assign(p->loc, new_ref(p->loc, ref_var, arg(p, 1, true), NULL),
                           new_ref(p->loc, ref_var, transfer_variable(arg(p, 1, true)->type), NULL)
                );
        }
        {
            int n = 1;
            tree a = arg(p, 1, false);
            while (a != NULL) {
                if ((a != NULL) && (a->pass_in_w)) {
                    add(new_assign
                        (NULL, new_ref(NULL, ref_var, a, NULL),
                         new_ref(NULL, ref_var, new_w(NULL, mode_out), NULL)
                        )
                        );
                }
                n++;
                a = arg(p, n, false);
            }
        }
        add(parse_statement_list());
        log_parse("parse_subprogram end of block");
        scanner_check_advance(s_end);
        add(p->ret = new_label(NULL, "e_%d_%s", p->nr, label));
        if (is_get) {
            p->transfer2 =
                new_assign(p->loc,
                           new_ref(p->loc, ref_var, transfer_variable(p->result->first->type),
                                   NULL), new_ref(p->loc, ref_var, p->result->first, NULL)
                );
        }
    close_list}
}
void parse_subprogram_declaration(boolean is_function, boolean is_operator)
{
    tree save_subprogram = current_subprogram;
    tree p = new_node(scanner_here(), node_procedure);
    current_subprogram = p;
    p->name = new_string(scanner_context->token);
    p->call_label = new_label(NULL, "_%d_%s_vector", p->nr, p->label);
    stack_guard;
    parse_subprogram_declaration2(p, is_function, is_operator, save_subprogram);
    check_and_add(new_decl(p->loc, p));
    current_subprogram = save_subprogram;
}

void parse_return(void)
{
    loc_t loc = scanner_here();
    stack_guard;
    assert_kind(NULL, current_subprogram, node_procedure);
    current_subprogram->has_return = true;

    if (current_subprogram->type != NULL) {
        loc_t loc = scanner_here();

        /* function or operator return value */
        tree p = parse_expression();
#ifdef __DEBUG__
        trace_subtree(current_subprogram);
#endif
        add(new_assign(loc, new_ref(loc, ref_var, current_subprogram->result->first, NULL), p)
            );
    }

    /* procedure return */
    add(new_return(loc, current_subprogram));
}

/* function declaration */
void parse_function_declaration(void)
{
    stack_guard;
    parse_subprogram_declaration(true, false);
    scanner_check_advance(s_function);
}

/* operator declaration */
void parse_operator_declaration(void)
{
    stack_guard;
    parse_subprogram_declaration(true, true);
    scanner_check_advance(s_operator);
}

/* procedure declaration */
void parse_procedure_declaration(void)
{
    stack_guard;
    parse_subprogram_declaration(false, false);
    scanner_check_advance(s_procedure);
}

void parse_test_assert(loc_t loc)
{
    tree c;
    tree p = new_node(loc, node_test);
    tree v = find(scanner_context->token);
    stack_guard;
    p->op = op_test_assert;
    if (v == NULL) {
        cfatal((m, "unknown identifier %s", scanner_context->token));
    }
    scanner_next_token();
    scanner_check_advance(op_name[op_equal]);
    p->first = v;
    c = parse_expect_constant();
#ifdef __DEBUG__
    trace_subtree(c);
#endif
    if (c->kind == node_ref)
        c = c->first;
    assert_kind(NULL, c, node_const);
    assert_kind(NULL, c->value, node_value);
    p->x = c->value->x;
    add(new_chain2(p, new_asm(loc, opcode_nop, NULL, 0)));
}

void parse_pragma_test(loc_t loc)
{
    tree c;
    stack_guard;
    if (scanner_is_like_advance(s_done)) {
        tree p = new_node(loc, node_test);
        tree n = new_asm(loc, opcode_nop, NULL, 0);
        p->op = op_test_end;
        add(p);
        add(n);
    } else if (scanner_is_like_advance(s_catch)) {
        catch_line = scanner_here()->line_nr + 1;
        c = parse_expect_constant();
        if (c->kind == node_ref)
            c = c->first;
        assert_kind(NULL, c, node_const);
        assert_kind(NULL, c->value, node_value);
        catch_pos = c->value->x;
    } else if (scanner_is_like_advance(s_assert)) {
        parse_test_assert(loc);
    } else {
        cfatal((m, "test pragma not recognized"));
    }
}

void parse_pragma_name(loc_t loc)
{
    stack_guard;
    scanner_check_advance(scanner_context->file_name);
}

boolean parse_on_off(void)
{
    stack_guard;
    if (scanner_is_like_advance(s_on)) {
        return true;
    } else if (scanner_is_like_advance(s_off)) {
        return false;
    } else {
        cfatal((m, "on or off expected"));
    }
    return true;                /* dummy */
}

/* old int eeprom_start = 0x2100; */
#define eeprom_start ( ( target_cpu != pic_16 ) ? 0x2100 : 0xF00000 )
int eeprom_index = -1;
int eeprom_data[2048];
int eeprom_last = 0;
boolean parse_pragma_eedata(void)
{
    stack_guard;
    for (;;) {
        tree p = parse_expect_constant();
        eeprom_index++;
        if (eeprom_index > eeprom_last) {
            cfatal((m, "more than %d eeprom data bytes", eeprom_last));
        }
        eeprom_data[eeprom_index] = p->value->x;
        if (!scanner_is_like_advance(s_comma)) {
            return true;
        }
    }
    return true;                /* dummy */
}

void parse_pragma_target(loc_t loc)
{
    stack_guard;
    if (scanner_is_like_advance(s_chip)) {
        tree chip = find("target_chip");
        tree cpu = find("target_cpu");
        if (target_chip != t_none) {
            cfatal((m, "target already specified"));
        }
        if (scanner_is_like_advance("12ce674")) {
            target_chip = t_12ce674;
            target_name = "12ce674";
            target_cpu = pic_14;
            target_first_ram = 0x20;
            target_last_ram = 0x7F;
            target_last_rom = 0x3FF;
            eeprom_last = 0;
        } else if (scanner_is_like_advance("16c84")) {
            target_chip = t_16c84;
            target_name = "16c84";
            target_cpu = pic_14;
            target_first_ram = 0x0C;
            target_last_ram = 0x0C - 1 + 36;
            target_last_rom = 0x3FF;
            eeprom_last = 64;
        } else if (scanner_is_like_advance("16f84")
                   || scanner_is_like_advance("16f84a")
            ) {
            target_chip = t_16f84;
            target_name = "16f84";
            target_cpu = pic_14;
            target_first_ram = 0x0C;
            target_last_ram = 0x0C - 1 + 68;
            target_last_rom = 0x3FF;
            eeprom_last = 64;
        } else if (scanner_is_like_advance("16f877")) {
            target_chip = t_16f877;
            target_name = "16f877";
            target_cpu = pic_14;
            target_first_ram = 0x20;
            target_last_ram = 0x7F;
            target_last_rom = 0x1FFF;
            eeprom_last = 256;
        } else if (scanner_is_like_advance("16f876")) { /* Added 16f876 & 16f873 by Javi 2003-03-01 */
            target_chip = t_16f876;
            target_name = "16f876";
            target_cpu = pic_14;
            target_first_ram = 0x20;
            target_last_ram = 0x7F;
            target_last_rom = 0x1FFF;
            eeprom_last = 256;
        } else if (scanner_is_like_advance("16f873")) { /* Added 16f876 & 16f873 by Javi 2003-03-01 */
            target_chip = t_16f873;
            target_name = "16f873";
            target_cpu = pic_14;
            target_first_ram = 0x20;
            target_last_ram = 0x7F;
            target_last_rom = 0x0FFF;
            eeprom_last = 128;
        } else if (scanner_is_like_advance("16f628")) {
            target_chip = t_16f628;
            target_name = "16f628";
            target_cpu = pic_14;
            target_first_ram = 0x20;
            target_last_ram = 0x7F;
            target_last_rom = 0x7FF;
            eeprom_last = 128;
        } else if (scanner_is_like_advance("sx18")) {
            target_chip = t_sx18;
            target_name = "SX18";
            target_cpu = sx_12;
            target_first_ram = 0x07;
            target_last_ram = 0x10 - 1 + 136;
            target_last_rom = 0x7FF;
            eeprom_last = 0;
        } else if (scanner_is_like_advance("sx28")) {
            target_chip = t_sx28;
            target_name = "SX28";
            target_cpu = sx_12;
            target_first_ram = 0x08;
            target_last_ram = 0x10 - 1 + 136;
            target_last_rom = 0x7FF;
            eeprom_last = 0;
        } else if (scanner_is_like_advance("12c509a")
                   || scanner_is_like_advance("12c509")
            ) {
            target_chip = t_12c509a;
            target_name = "12c509a";
            target_cpu = pic_12;
            target_first_ram = 0x07;
            target_last_ram = 0x10 - 1 + 41;
            target_last_rom = 0x3FF;
            eeprom_last = 0;
        } else if (scanner_is_like_advance("12c508")) {
            target_chip = t_12c508;
            target_name = "12c508";
            target_cpu = pic_12;
            target_first_ram = 0x07;
            target_last_ram = 0x10 - 1 + 25;
            target_last_rom = 0x1FF;
            eeprom_last = 0;
        } else if (scanner_is_like_advance("18f242")) {
            target_chip = t_18f242;
            target_name = "18f242";
            target_cpu = pic_16;
            target_first_ram = 0x0000;
            target_last_ram = 0x02FF;
            target_last_rom = 0x3FFF;
            eeprom_last = 255;
        } else if (scanner_is_like_advance("18f252")) {
            target_chip = t_18f252;
            target_name = "18f252";
            target_cpu = pic_16;
            target_first_ram = 0x0000;
            target_last_ram = 0x05FF;
            target_last_rom = 0x7FFF;
            eeprom_last = 255;
        } else if (scanner_is_like_advance("18f442")) {
            target_chip = t_18f442;
            target_name = "18f442";
            target_cpu = pic_16;
            target_first_ram = 0x0000;
            target_last_ram = 0x02FF;
            target_last_rom = 0x3FFF;
            eeprom_last = 255;
        } else if (scanner_is_like_advance("18f452")) {
            target_chip = t_18f452;
            target_name = "18f452";
            target_cpu = pic_16;
            target_first_ram = 0x0000;
            target_last_ram = 0x05FF;
            target_last_rom = 0x7FFF;
            eeprom_last = 255;
        } else if (scanner_is_like_advance("12f629")) {
            target_chip = t_12f629;
            target_name = "12f629";
            target_cpu = pic_14;
            target_first_ram = 0x0020;
            target_last_ram = 0x005F;
            target_last_rom = 0x03FE;
            eeprom_last = 127;
        } else if (scanner_is_like_advance("12f675")) {
            target_chip = t_12f675;
            target_name = "12f675";
            target_cpu = pic_14;
            target_first_ram = 0x0020;
            target_last_ram = 0x005F;
            target_last_rom = 0x03FE;
            eeprom_last = 127;
				 /* jallist #12413 bug found, corrected 2004-01-20 */
        } else if (scanner_is_like_advance("16f88")) {
            target_chip = t_16f88;
            target_name = "16f88";
            target_cpu = pic_14;
            target_first_ram = 0x0020;
            target_last_ram = 0x016F;  // was 0x007F, corrected HdR 08-10-04
            target_last_rom = 0x1BFF;  // was 0x07FF, corrected HdR 08-10-04
            eeprom_last = 255;
        } else {
            cfatal((m, "target not recognized"));
        }
        chip->value->x = target_chip;
        cpu->value->x = target_cpu;
    } else if (scanner_is_like_advance(s_clock)) {
        tree c = find("target_clock");
        tree p = parse_expect_constant();
        target_clock = p->value->x;
        assert_kind(loc, c, node_const);
        assert_kind(loc, c->value, node_value);
        if (c->value->x != 0) {
            cfatal((m, "clock already specified"));
        }
        c->value->x = target_clock;
    } else if (scanner_is_like_advance(s_osc)) {
        if (scanner_is_like_advance("hs")) {
            config_osc = osc_hs;
        } else if (scanner_is_like_advance("xt")) {
            config_osc = osc_xt;
        } else if (scanner_is_like_advance("rc")) {
            config_osc = osc_rc;
        } else if (scanner_is_like_advance("lp")) {
            config_osc = osc_lp;
        } else if (scanner_is_like_advance("internal")) {
            config_osc = osc_int;
        } else {
            cfatal((m, "osc configuration not recognized"));
        }
    } else if (scanner_is_like_advance(s_watchdog)) {
        config_watchdog = parse_on_off();
    } else if (scanner_is_like_advance(s_protection)) {
        config_protection = parse_on_off();
    } else if (scanner_is_like_advance(s_powerup)) {
        config_powerup = parse_on_off();
    } else if (scanner_is_like_advance(s_int_mclr)) {
        config_int_mclr = parse_on_off();
    } else if (scanner_is_like_advance("fuses")) {
/* Wouter's Dwarf Patch (config words in PIC16) */
     	 loc_t loc = scanner_context->loc;
       tree p = parse_expect_constant();
       /* cfatal((m, "xxxxinvalid fuses index")); */
       if (scanner_is_like_advance(",")){
          int i = p->value->x;
          tree vp = parse_expect_constant();   
          int v = vp->value-> x;
          if(( i < 0 ) || ( i >= target_fuses_array_size )){
             fatal( loc, (m, "invalid fuses index"));	
          }
          target_fuses_array[ i ] = v;
          target_fuses_count++;
       } else {
          tree c = find("target_fuses");
          target_fuses = p->value->x;
          assert_kind(loc, c, node_const);
          assert_kind(loc, c->value, node_value);
          if (c->value->x != 0) {
             cfatal((m, "fuses already specified"));
          }
          c->value->x = target_fuses;
/*        tree c = find("target_fuses");     */
/*        tree p = parse_expect_constant();  */
/*                                           */ 
/*        target_fuses = p->value->x;        */
/*        assert_kind(loc, c, node_const);   */
/*        assert_kind(loc, c->value, node_value);      */
/*        if (c->value->x != 0) {                      */
/*            cfatal((m, "fuses already specified"));  */
        }
/*        c->value->x = target_fuses;                  */
    } else if (scanner_is_like_advance("origin")) {
		tree p = parse_expect_constant();
		target_origin = p->value->x;
    } else {
        cfatal((m, "pragma target not recognized"));
    }
}

boolean parse_pragma_keep_var(void)
{
    stack_guard;
    for (;;) {
        tree p = find(scanner_context->token);
        if (p == NULL) {
            cfatal((m, "unknown identifier %s", scanner_context->token));
        }
        if (p->kind != node_var) {
            cfatal((m, "%s is not a variable", scanner_context->token));
        }
        p->sacred = true;
        scanner_next_token();
        if (!scanner_is_like_advance(s_comma)) {
            return true;
        }
    }
    return true;                /* dummy */
}

boolean parse_pragma_keep(void)
{
    stack_guard;
    for (;;) {
        if (scanner_is_like_advance(s_page)) {
            block_options.keep_page = true;
        } else if (scanner_is_like_advance(s_bank)) {
            block_options.keep_bank = true;
        } else if (scanner_is_like_advance(s_var)) {
            parse_pragma_keep_var();
        } else {
            cfatal((m, "page, bank or var expected"));
        }
        if (!scanner_is_like_advance(s_comma)) {
            return true;
        }
    }
    return true;                /* dummy */
}

boolean parse_pragma_optimize(boolean polarity)
{
    stack_guard;
    for (;;) {
        if (scanner_is_like_advance(s_pass_in_w)) {
            block_options.pass_in_w = polarity;
        } else {
            cfatal((m, "optimization option expected"));
        }
        if (!scanner_is_like_advance(s_comma)) {
            return true;
        }
    }
    return true;                /* dummy */
}

/* a possible interrupt routine */
tree interrupt_service_routine = NULL;

void parse_pragma(loc_t loc)
{
    boolean polarity = true;
    stack_guard;

    if (scanner_is_like_advance(s_no)) {
        polarity = false;
    }

    if (scanner_is_like_advance(s_test)) {
        parse_pragma_test(loc);
    } else if (scanner_is_like_advance(s_name)) {
        parse_pragma_name(loc);
    } else if (scanner_is_like_advance(s_eedata)) {
        parse_pragma_eedata();
    } else if (scanner_is_like_advance(s_keep)) {
        parse_pragma_keep();
    } else if (scanner_is_like_advance(s_target)) {
        parse_pragma_target(loc);
    } else if (scanner_is_like_advance(s_optimize)) {
        parse_pragma_optimize(polarity);
    } else if (scanner_is_like_advance(s_jump_table)) {
        if (current_subprogram != NULL) {
            current_subprogram->last_page = true;
        } else {
            fatal(loc, (m, "pragma not allowed outside procedure or function"));
        }
    } else if (scanner_is_like_advance(s_interrupt)) {
        if ((target_chip == t_12c508) || (target_chip == t_12c509a)) {  /* Added by Javi 20030305 */
            fatal(loc, (m, "interrupts not allowed in this chip"));
        }
        if ((current_subprogram != NULL)
            && (current_subprogram->result == NULL)
            ) {
            if (interrupt_service_routine != NULL) {
                fatal(loc, (m, "an interrupt procedure has already been identified"));
            }
            current_subprogram->is_interrupt = true;
            interrupt_service_routine = current_subprogram;

		} else {
            fatal(loc, (m, "pragma not allowed outside a procedure"));
        }
    } else if( scanner_is_like_advance( s_raw_interrupt ) ){
		if( ( current_subprogram != NULL ) && ( current_subprogram->result == NULL )){
			if( interrupt_service_routine != NULL ){
				fatal( loc, (m, "an interrupt procedure has already been identified" ));
			}
			current_subprogram->is_interrupt = true;
			current_subprogram->is_raw = true;
			interrupt_service_routine = current_subprogram;
		} else {
			fatal( loc, (m, "pragma not allowed outside a procedure" ));
		}
	} else if (scanner_is_like_advance(s_label)) {
        add(new_label(loc, scanner_context->token, 0, ""));
        scanner_check_identifier_advance();
    } else if (scanner_is_like_advance(s_error)) {
        add(new_error(loc));
    } else {
        fatal(loc, (m, "pragma not recognized"));
    }
}

void parse_include(void)
{
    string s, t;
    tree p = new_node(scanner_context->loc, node_const);
    stack_guard;

    if (!find_name(s, scanner_context->token)) {
        cfatal((m, "can not open the file '%s'", scanner_context->token));
    }

    /* have we aready parsed this file? */
    sprintf(t, " include file %s", s);
    p->name = new_string(t);
    if (find(p->name) != NULL) {
        /* already done, don't read the file twice */
        scanner_next_token();
        return;
    }
    check_and_add(new_decl(p->loc, p));

    scanner_open(NULL, s, false);
}

int get_opcode_index(char *s)
{
    int i;
    for (i = 1; opcode_name[i][0] != '\0'; i++) {
        if (string_match(s, opcode_name[i])) {
            return i;
        }
    }
    return -1;
}

void set_sacred(tree p)
{
    if (p->kind != node_asm)
        return;
/*   trace_subtree(p); */
    assert_kind(NULL, p, node_asm);
    if (p->opcode == opcode_page) {
        p->sacred = block_options.keep_page;
    } else if (p->opcode == opcode_bank) {
        p->sacred = block_options.keep_bank;
    }
}

tree parse_expression_from_asm(void)
{
    tree p;
    loc_t loc = scanner_here();
    stack_guard;

    procedure_is_constant = true;
    p = parse_expression();
    procedure_is_constant = false;
    evaluate(p);

    if (p != NULL) {
        /*if( p->first->kind == node_label ){
           return p;
           } */
        if (p->value != NULL) {
            return p;
        }
        if ((p->kind == node_ref) && (p->first != NULL)) {
            switch (p->first->kind) {
            case node_procedure:
            case node_const:
            case node_var:{
                    return p;
                    break;
                }
            default:{
                    break;
                }
            }
        }
    }
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    fatal(loc, (m, "assembly constant expected"));
    return p;                   /* dummy */
}

boolean parse_asm(void)
{
    tree q, p = NULL;
    loc_t loc = scanner_here();
    string s;
    int i;
    boolean paged = false;
    boolean banked = false;
    tree origin;
    stack_guard;

    if (scanner_is_like_advance(s_end)) {
        return false;
    }
    if_advance_return(s_pragma, parse_pragma(loc));
    string_copy(s, scanner_context->token);

    /* 18F's core TBLRDs & TBLWTs mnemonics update */
    if (target_cpu == pic_16) {
       if ( (string_match (s, "tblrd")) || (string_match (s, "tblwt")) ) {
          if ( char_match ( scanner_next_char() , '*') ) {
             scanner_advance_char();
             string_copy(s, scanner_context->token);
             if ( char_match ( scanner_next_char() , '+') ) {
                scanner_advance_char();
                string_copy(s, scanner_context->token);
             } else if ( char_match ( scanner_next_char() , '-') ) {
                scanner_advance_char();
                string_copy(s, scanner_context->token);
             }
          } else if ( char_match ( scanner_next_char() , '+') ) {
             scanner_advance_char();
             string_copy(s, scanner_context->token);
             if ( char_match ( scanner_next_char() , '*') ) {
                scanner_advance_char();
                string_copy(s, scanner_context->token);
             }
          }
       }
    }

    /* org directive? */
    if (scanner_is_like_advance("org")) {
        origin = parse_expect_constant();
        add(new_org(constant_value(origin)));
        return true;
    }

    /* (list of) locals? */
    if (scanner_is_like_advance(s_local)) {
        for (;;) {
            p = new_node(scanner_here(), node_const);
            p->type = type_universal;
            p->name = new_string(scanner_context->token);
            check_and_add(new_decl(p->loc, p));
            scanner_check_identifier_advance();
            if (!scanner_is_like(s_comma)) {
                return true;
            }
            scanner_check_advance(s_comma);
        }
    }

    scanner_check_identifier_advance();

    q = find(s);
    if (scanner_is_like_advance(s_colon)) {
        if (q == NULL) {
            cfatal((m, "label %s not known", s));
        }
        if (q->kind != node_const) {
            cfatal((m, "%s is not a local label", s));
        }
        if (q->value != NULL) {
            cfatal((m, "label %s already fixed", s));
        }
        p = new_label(loc, "as_%d_%s", q->nr, q->name);
        q->value = p;
        add(p);
        return true;
    }

    i = get_opcode_index(s);
    if (i < 0) {
        /* t0029 */
        fatal(loc, (m, "assembler mnemonic expected, found: %s ", s ));
        return false;
    }
    banked = (i == opcode_bank);
    paged = (i == opcode_page);
    if ((paged | banked)
        && (get_opcode_index(scanner_context->token) > 0)
        ) {
        string_copy(s, scanner_context->token);
        scanner_check_identifier_advance();
        i = get_opcode_index(s);
    } else {
        paged = banked = false;
    }
    jal_assert(NULL, i >= 0);

    p = new_node(loc, node_asm);
    p->opcode = i;

    if (paged & (!code_has(p->opcode, field_label))) {
        fatal(loc, (m, "page prefix not appropriate"));
    }
    if (banked & (!code_has(p->opcode, field_file))) {
        fatal(loc, (m, "bank prefix not appropriate"));
    }

    if (code_has(p->opcode, field_const)
        || code_has(p->opcode, field_file)
        || code_has(p->opcode, field_label)
        || code_has(p->opcode, field_tris)
	|| code_has(p->opcode, field_fsr)
        ) {
        p->first = parse_expression_from_asm();
    }

    if (code_has(p->opcode, field_flabel)){
        if (scanner_is_like_advance(s_comma)) {
            p->next = parse_expect_constant();
        }
    }

    if (code_has(p->opcode, field_dest)) {
        scanner_check_advance(s_comma);
        if (scanner_is_like_advance(s_dest_f)) {
            p->dest = dest_f;
        } else if (scanner_is_like_advance(s_dest_w)) {
            p->dest = dest_w;
        } else {
            cfatal((m, "either f or w expected"));
        }
    }

    if (code_has(p->opcode, field_bit)) {
        if (scanner_is_like_advance(s_comma)) {
            p->next = parse_expression_from_asm();
        } else {
            if ((p->first->kind == node_ref)
                && (p->first->first->kind == node_var)
                ) {
                p->next = p->first;
            } else {
                scanner_check_advance(s_comma);
            }
        }
    }

    set_sacred(p);
    if (paged) {
        p = code_code_page(p->first, p);
        set_sacred(p->first);
    } else if (banked) {
#ifdef __DEBUG__
        trace_subtree(p->first);
        trace_subtree(p);
#endif
        p = code_register_bank(p->first, p);
#ifdef __DEBUG__
        trace_subtree(p);
#endif
        set_sacred(p->first);
    }

    if (target_cpu == pic_16) {
        if ((p->opcode == opcode_goto)
            || (p->opcode == opcode_call)) {
            p = new_chain2(p, new_asm(loc, opcode_a2nd, p->first, 0));
        } else if (p->opcode == opcode_lfsr) {
            p = new_chain2(p, new_asm(loc, opcode_f2nd, p->first, p->next->value->x));
          }
    }

    add(p);

    return true;
}

/* parse one statement, return false iff no more appropriate input */
boolean parse_statement(void)
{
    tree p;
    loc_t loc = scanner_here();
    stack_guard;
    log_parse("parse_statement");

    if_advance_return(s_var, parse_var());
    if_advance_return(s_const, parse_const());
    if_advance_return(s_function, parse_function_declaration());
    if_advance_return(s_operator, parse_operator_declaration());
    if_advance_return(s_procedure, parse_procedure_declaration());
    if_advance_return(s_if, old_parse_if(loc));
    if_advance_return(s_while, parse_while(loc, NULL));
    if_advance_return(s_for, parse_for(loc));
    if_advance_return(s_forever, parse_while(loc, new_const(new_value(type_bit, 1))));
    if_advance_return(s_assembler, parse_assembler_list());
    if_advance_return(s_begin, parse_statement());
    if_advance_return(s_pragma, parse_pragma(loc));
    if_advance_return(s_asm, parse_asm());
    if_advance_return(s_include, parse_include());
    if (current_subprogram != NULL) {
        if_advance_return(s_return, parse_return());
    }
    if (scanner_is_like(s_else)) {
        return false;
    }
    if (scanner_is_like(s_elsif)) {
        return false;
    }
    if (scanner_is_like(s_end)) {
        return false;
    }
    if (scanner_context->eoi) {
        return false;
    }

    scanner_check_identifier();
    p = find(scanner_context->token);
    if (p == NULL) {
        cfatal((m, "unknown identifier %s", scanner_context->token));
    }
    scanner_check_identifier_advance();
    switch (p->kind) {
    case node_var:
        parse_assign(new_ref(loc, p->ref, p, current_subprogram));
        break;
    case node_procedure:
        parse_procedure_call(loc, p);
        break;
    case node_type:{
            fatal(loc, (m, "type not allowed here"));
            break;
        }
    case node_const:{
            fatal(loc, (m, "constant not allowed here"));
            break;
        }
    default:
        snark_node(p->loc, p);
    }

    return true;
}

/* check whether all labels in a block have a value */
void check_labels(tree p)
{
    stack_guard;
    if (p == NULL)
        return;
    check_labels(p->first);
    check_labels(p->next);
    if (p->kind != node_decl)
        return;
    assert_pointer(NULL, p->first);
    if (p->first->kind != node_const)
        return;
    if (p->first->value == NULL) {
        /* t0028 */
        fatal(p->loc, (m, "label %s has no value", p->name));
    }
}

/* parse a list of assembler statements */
void parse_assembler_list(void)
{
    open_list(p)
        stack_guard;
    while (parse_asm());
    scanner_check_advance(s_assembler);
#ifdef __DEBUG__
    trace_subtree(p);
#endif
    check_labels(p);
    close_list add(p);
}

/* parse list of statements without block context */
void parse_statement_sequence(void)
{
    boolean go = true;
    stack_guard;
    log_parse("parse_statement_sequence start");

    while (go) {
        log_parse("parse_statement_sequence next");
        go = parse_statement();
    }

    log_parse("parse_statement_sequence end");
}

/* parse a list of normal statements */
tree parse_statement_list(void)
{
    block_options_t save_block_options = block_options;
    open_list(p)
        stack_guard;
    parse_statement_sequence();
    block_options = save_block_options;
    close_list return p;
}

/* parse all input */
void parse(tree * dummy)
{
    stack_guard;
    block_options.keep_page = false;
    block_options.keep_bank = false;
    block_options.pass_in_w = optimize_pass_in_w;
    parse_statement_sequence();
}

/* check whether the sources are OK */
void parse_done(void)
{
    stack_guard;
    if (target_chip == t_none) {
        cfatal((m, "no target chip specified"));
    }
    if (target_clock == 0) {
        cfatal((m, "no target clock frequency specified"));
    }
/* Wouter's Dwarf patch (config words in PIC16 */    
    if( target_fuses_count == 0 ){
       if (config_osc == osc_none) {
          cfatal((m, "no target oscillator setting specified"));
       }
/*    if (config_osc == osc_none) {                                */
/*        cfatal((m, "no target oscillator setting specified"));   */
    }                                                             
}
