// Unit Tests for PortageFiles.cpp
// SPDX-FileCopyrightText: 2024 <A Schenck> <galiven@users.sourceforge.net>
// SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only OR LicenseRef-KDE-Accepted-GPL

#undef QT_NO_CAST_FROM_ASCII

#include "testhelpers.hpp"
#include "portagefilestests.h"

#include <QDir>
#include <QTest>
#include <QIODevice>

#include "src/core/portagefiles.h"
#include "src/settings.h"


QTEST_APPLESS_MAIN(PortageFilesTests);

void PortageFilesTests::initTestCase() {
	// Called before the first testfunction is executed
	//Make sure we're running under sandbox
	QCOMPARE(qEnvironmentVariable("LD_PRELOAD"), "libsandbox.so");
	//And make sure we get 'false' when trying to open /etc/portage/package.use/test for write
	QVERIFY2(!QFile("/etc/portage/package.use/test").open(QIODevice::ReadWrite), "For good measure, these tests run under sandbox and verify that they _can't_ write /etc/portage/package.use/test. Add SANDBOX_PREDICT=\"/etc/portage/package.use\" to the command line to make the attempted open-write fail but not abort the process");
	
	TestHelpers::initTestDir();
}

void PortageFilesTests::cleanupTestCase() {
	// Called after the last testfunction was executed
	TestHelpers::cleanupTestDir();
}

void PortageFilesTests::init() {
	// Called before each testfunction is executed
}

void PortageFilesTests::cleanup() {
	// Called after every testfunction
}

void PortageFilesTests::testRecursiveReadShouldReadFlat() {
	//Arrange
	QString testFilePath = "testRecursiveReadFlat";
	TestHelpers::writeFilesInTestDir({{ testFilePath, "test1\ntest2\ntest3" }});
	//Act/Assert
	QCOMPARE(recursiveRead(TestHelpers::baseTestDir->path() + u'/' + testFilePath), QStringList({ "test1", "test2", "test3" }));
}

/**
 * This sets up a directory structure as follows
 * /tmp/kuroo_test/testRecursiveReadNested/
 * 					|-file1
 * 					|-file2
 * 					|-dir1/
 * 						|-file1
 * 						|-file2
 */
void PortageFilesTests::setupRecursiveRead() {
	QString nestedTestDirName = QLatin1String("testRecursiveReadNested");
	nestedTestDirPath = TestHelpers::baseTestDir->path() + u'/' + nestedTestDirName;
	if (!TestHelpers::baseTestDir->exists(nestedTestDirName)) {
		TestHelpers::writeFilesInTestDir({{ nestedTestDirName + u"/file1", "test1\ntest2" },
										 { nestedTestDirName + u"/file2", "test3" },
										 { nestedTestDirName + u"/dir1/file1", "test4\ntest5" },
										 { nestedTestDirName + u"/dir1/file2", "test6\ntest7" }});
	}
}

void PortageFilesTests::testRecursiveReadShouldRaedNested() {
	//Arrange
	setupRecursiveRead();
	//Act
	auto result = recursiveRead(nestedTestDirPath);
	result.sort();
	//Assert
	QCOMPARE(result, nestedResults);
}

void PortageFilesTests::testRecursiveReadShouldExcludeTopLevelFile() {
	//Arrange
	setupRecursiveRead();
	//Act
	auto result = recursiveRead(nestedTestDirPath, { "file2" });
	result.sort();
	//Assert
	QCOMPARE(result, nestedResults.sliced(0, 2) + nestedResults.sliced(3));
}

void PortageFilesTests::testRecursiveReadShouldExcludeTopLevelDir() {
	//Arrange
	setupRecursiveRead();
	//Act
	auto result1 = recursiveRead(nestedTestDirPath, { "dir1" });
	//Assert
	QCOMPARE(result1, nestedResults.sliced(0, 3));
}

void PortageFilesTests::testRecursiveReaShouldExcludeNestedFile() {
	//Arrange
	setupRecursiveRead();
	//Act
	auto result = recursiveRead(nestedTestDirPath, { "dir1/file1" });
	result.sort();
	//Assert
	QCOMPARE(result, nestedResults.sliced(0, 3) + nestedResults.sliced(5));
}
