/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  writerom.c

  Games and Roms generation on disk
*/

#include "common.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <time.h>

#include <mamory/readinfo.h>
#include <mamory/rommngr.h>
#include <mamory/chainlst.h>
#include <mamory/writerom.h>
#include <mamory/zip.h>
#include <sortbox/sortbox.h>

int GetSyncDiskSpaceNeed(s_Diff *Diff,unsigned char *BackUpPath,unsigned int Test)
{
  unsigned int tempsize=0;
  int mainsize=0;
  unsigned int backupsize=0,i;
  struct stat stat_buf;
  s_DiffGame *ThisDiffGame;
  s_DiffRom *ThisDiffRom;
  s_Zip *Zip=NULL,*TempZip=NULL;
  
  unsigned char *rompath,*path;
  
  for(i=0;i<Diff->GamesList1->NbrSources;i++)
    if(Diff->GamesList1->Sources[i]->Type!=TYPE_DIR)
      {
	LPRINTF(ERROR,"At least one of the rom path you want to synchronize");
	LPRINTF(ERROR,"is not a directory");
      return FALSE;
      };

  if(Test==TRUE)
    BackUpPath=NULL;

  path=XSTRDUP(Diff->GamesList1->Sources[0]->Source);
  path=XREALLOC(path,unsigned char,strlen(path)+1+1);

  strcat(path,"/");
  
  if(BackUpPath!=NULL)
    {
      if(stat(BackUpPath,&stat_buf)!=0)
	  {
	    LPRINTF(ERROR,"'%s' directory is not available",
		    BackUpPath);
	    BackUpPath=NULL;
	  };
    };

  if(Diff->NbrGames!=0)
    {
      Diff->Games=CLFirst(Diff->Games);
      do  
	{
	  ThisDiffGame=DIFF_GAME(CLGetData(Diff->Games));
	  
	  if (ThisDiffGame->NbrRoms==(ThisDiffGame->NbrSAMERoms+
				      ThisDiffGame->NbrMISSINGRoms))
	    {
	      if(Diff->Games->Next!=NULL)
		Diff->Games=Diff->Games->Next;
	      else
		break;
	      continue;
	    };

	  rompath=XSTRDUP(path);
	  rompath=XREALLOC(rompath,
			   unsigned char,
			   strlen(rompath)+
			   strlen(ThisDiffGame->Game->Name)+
			   strlen(".zip")+
			   1);

	  strcat(rompath,ThisDiffGame->Game->Name);
	  strcat(rompath,".zip");
	  
	  memset(&stat_buf,0,sizeof(struct stat));
	  if(stat(rompath,&stat_buf)==0)
	    {
	      if(BackUpPath!=NULL)
		backupsize+=stat_buf.st_size;
	      else
		{
		  if(tempsize<stat_buf.st_size)
		    tempsize=stat_buf.st_size;
		};
	      if(Test==FALSE)
		mainsize-=stat_buf.st_size;
	    };

	  if(Test==FALSE)
	    Zip=InitVirtualZip();

	  ThisDiffGame->Roms=CLFirst(ThisDiffGame->Roms);
	  do
	    {
	      ThisDiffRom=DIFF_ROM(CLGetData(ThisDiffGame->Roms));
	      switch(ThisDiffRom->Type)
		{
		case ADDEDTRANSFERT:
		case CHANGEDNAME:
		case SAME:
		  if(Test==FALSE)
		    {
		      if(!ThisDiffRom->SourceRom)
			break;
		      if(strcmp(ThisDiffRom->SourceRom->Name,ThisDiffRom->Rom->Name)==0)
			AddToVirtualZip(Zip,
					ThisDiffRom->Rom->Name,
					ThisDiffRom->Rom->Content->CompSize);
		      else
			AddToVirtualZip(Zip,
					ThisDiffRom->SourceRom->Name,
					ThisDiffRom->Rom->Content->CompSize);
		    };
		  break;
		case SUPPRESSED:
		case SUPPRTRANSFERT:
		  if (TempZip==NULL)
		    TempZip=InitVirtualZip();

		  AddToVirtualZip(TempZip,
				  ThisDiffRom->Rom->Name,
				  ThisDiffRom->Rom->Content->CompSize);
		  break;			    
		};
	      	      
	      if(ThisDiffGame->Roms->Next!=NULL)
		ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
	      else
		break;
	    }while(ThisDiffGame->Roms!=NULL);	  
	  
	  if(TempZip!=NULL)
	    {	      
	      mainsize+=GetVirtualZipSize(TempZip);
	      FreeZip(TempZip);
	      TempZip=NULL;
	    };

	  if(Test==FALSE)
	    mainsize+=GetVirtualZipSize(Zip);

	  if(Test==FALSE)
	    FreeZip(Zip);
	  Zip=NULL;
	  
	  XFREE(rompath);

	  if(Diff->Games!=NULL && Diff->Games->Next!=NULL)
	    Diff->Games=Diff->Games->Next;
	  else
	    break;
	}while(Diff->Games!=NULL);  
    };

  XFREE(path);
  Diff->TempNeededSize=tempsize;
  Diff->MainNeededSize=mainsize;
  Diff->BackUpNeededSize=backupsize;

/*
  printf("/tmp max used space: %i - /tmp Free space; %i\n",
	 tempsize,
	 GetFreeDiskSpace("/tmp"));
  printf("main used space: %i - main Free space: %i\n",
	 mainsize,
	 GetFreeDiskSpace(path));
  if (BackUpPath!=NULL)
    printf("backup used space: %i - backup Free space; %i\n",
	   backupsize,
	   GetFreeDiskSpace(BackUpPath));
*/
  return TRUE;
};
  

int synchronize(s_Diff *Diff,unsigned char *BackUpPath_p,unsigned int Test)
{
  s_DiffGame *ThisDiffGame;
  s_DiffRom *ThisDiffRom;
  FILE *fp_backup;
  s_Zip *DestZip, *SourceZip,*TempZip=NULL;
  unsigned char *path;
  unsigned char *rompath=NULL,*gamebackuppath=NULL,*sourcepath=NULL;
  unsigned char *temppath=NULL;
  unsigned char *SourceRomName=NULL;
  unsigned char *BackUpPath=NULL;
  unsigned char session[35];

  unsigned int nb_read, nb_write,i;
  unsigned char buffer[BUFSIZ];
  struct tm *timesession;
  DIR *dirp=NULL;
  struct dirent *DirEntry=NULL;
  time_t t;

  

  for(i=0;i<Diff->GamesList1->NbrSources;i++)
    if(Diff->GamesList1->Sources[i]->Type!=TYPE_DIR)
      {
	LPRINTF(ERROR,"At least one of the rom path you want to synchronize");
	LPRINTF(ERROR,"is not a directory");
      return FALSE;
      };

  if (Test==TRUE)
    BackUpPath=NULL;
  else
    BackUpPath=XSTRDUP(BackUpPath_p);

  time(&t);
  timesession=localtime(&t);
  sprintf(session,"mamory_roms_backup-%04i%02i%02i%02i%02i%02i/",
	  timesession->tm_year+1900,
	  timesession->tm_mon+1,
	  timesession->tm_mday,
	  timesession->tm_hour,
	  timesession->tm_min,
	  timesession->tm_sec);

  path=XSTRDUP(Diff->GamesList1->Sources[0]->Source);
  path=XREALLOC(path,unsigned char,strlen(path)+1+1);

  strcat(path,"/");

  if(BackUpPath!=NULL)
    {
      BackUpPath=XREALLOC(BackUpPath,
			  unsigned char,
			  strlen(BackUpPath)+1+
			  strlen(session)+1);

      strcat(BackUpPath,"/");
      strcat(BackUpPath,session);
      if(Test==FALSE)
	if(mkdir(BackUpPath,S_IRWXU|S_IRWXG)!=0)
	  {
	    LPRINTF(ERROR,"'%s' directory cannot be created",
		    BackUpPath);
	    BackUpPath=NULL;
	  };
    };

  if(Diff->NbrGames!=0)
    {
      Diff->Games=CLFirst(Diff->Games);
      do  
	{
	  ThisDiffGame=DIFF_GAME(CLGetData(Diff->Games));
	  
	  if(ThisDiffGame->NbrMISSINGRoms!=0 &&
	     (Diff->OptionFlags & OPTION_DELETE_INCOMPLETE_GAMES))
	    /* Skip games that have unavailable roms
	       == Modify complete romsets only
	       WARNING: this means that for merge mode, all roms from clones
	                and originals have to be available */
	    {
	      rompath=XSTRDUP(path);
	      rompath=XREALLOC(rompath,
			       unsigned char,
			       strlen(rompath)+
			       strlen(ThisDiffGame->Game->Name)+
			       strlen(".zip")+
			       1);
	      
	      strcat(rompath,ThisDiffGame->Game->Name);
	      strcat(rompath,".zip");

	      DestZip=Zipfopen(rompath,"r");
	      if(DestZip==NULL)
		LPRINTF(OUTPUT,
			"Skipping game %s because romset is incomplete.",
			ThisDiffGame->Game->Name);
	      else
		{
		  LPRINTFW(OUTPUT,
			   "Deleting game %s because romset is incomplete ...",
			   ThisDiffGame->Game->Name);
		  Zipfclose(DestZip);
		  DestZip=NULL;
		  if(Test==FALSE)
		    unlink(rompath);
		  if(Test==TRUE)
		    LPRINTFW(OUTPUT," test");
		  LPRINTF(OUTPUT," done");
		};
	      
	      XFREE(rompath);

	      if(Diff->Games->Next!=NULL)
		Diff->Games=Diff->Games->Next;
	      else
		break;
	      continue;
	    }

	  if (ThisDiffGame->NbrRoms==(ThisDiffGame->NbrSAMERoms+
				      ThisDiffGame->NbrMISSINGRoms))
	    // Skip games for which all required roms are unavailable
	    {
	      LPRINTF(DEBUG,
		      "Skipping game %s: all required roms are unavailable",
		      ThisDiffGame->Game->Name );
	      if(Diff->Games->Next!=NULL)
		Diff->Games=Diff->Games->Next;
	      else
		break;
	      continue;
	    };

	  // Set the complete path of the game to create/alter
	  rompath=XSTRDUP(path);
	  rompath=XREALLOC(rompath,
			   unsigned char,
			   strlen(rompath)+
			   strlen(ThisDiffGame->Game->Name)+
			   strlen(".zip")+
			   1);

	  strcat(rompath,ThisDiffGame->Game->Name);
	  strcat(rompath,".zip");

	  DestZip=Zipfopen(rompath,"r+");
	  if(DestZip==NULL && myerrno==MENOENT)
	    {
	      LPRINTFW(OUTPUT,"Creating game %s ...",
		       ThisDiffGame->Game->Name);
	      DestZip=Zipfopen(rompath,"w+");
	      if(DestZip==NULL)
		{
		  LPRINTF(OUTPUT," failed.");
		  if(myerrno==MEWACCESS)
		    {
		      if(Test==FALSE)
			LPRINTF(ERROR,"Write permission denied to create file %s.",
				rompath);
		      else
			LPRINTF(ERROR,"Won't have write permission to create file %s.",
				rompath);
		    }
		  else
		    LPRINTF(ERROR,"Unable to create file %s",
			    rompath);

		  if(Test==FALSE)
		    LPRINTF(OUTPUT,"Skipping creation of game %s.",
			    ThisDiffGame->Game->Name);
		  else
		    LPRINTF(OUTPUT,"Will skip creation of game %s.",
			    ThisDiffGame->Game->Name);

		  XFREE(rompath);
		  
		  if(Diff->Games->Next!=NULL)
		    Diff->Games=Diff->Games->Next;
		  else
		    break;
		  continue;
		};
       
	      if(Test==TRUE)
		{
		  Zipfclose(DestZip);
		  unlink(rompath);
		};
	       
	      LPRINTF(OUTPUT, " ");
	       
	    }
	  else
	    {
	      LPRINTFW(OUTPUT,"Altering game %s ...",
		      ThisDiffGame->Game->Name);

	      if(DestZip==NULL)
		{
		  LPRINTF(OUTPUT," failed.");
		  if(myerrno==MEACCESS)
		    {
		      if(Test==FALSE)
			LPRINTF(ERROR,"Write and/or read permission denied to alter file %s.",
				rompath);
		      else
			LPRINTF(ERROR,"Won't have write and/or read permission to alter file %s.",
				rompath);
		    }
		  else
		    LPRINTF(ERROR,"Unable to alter file %s",
			    rompath);

		  if(Test==FALSE)
		    LPRINTF(OUTPUT,"Skipping alteration of game %s.",
			    ThisDiffGame->Game->Name);
		  else
		    LPRINTF(OUTPUT,"Will skip alteration of game %s.",
			    ThisDiffGame->Game->Name);

		  XFREE(rompath);
		  
		  if(Diff->Games->Next!=NULL)
		    Diff->Games=Diff->Games->Next;
		  else
		    break;
		  continue;
		};

	      LPRINTF(OUTPUT, " ");

	      // Set the backup path from the appropriate directory
	      if (BackUpPath!=NULL)
		{
		  gamebackuppath=XCALLOC(unsigned char,
					 strlen(BackUpPath)+
					 strlen(ThisDiffGame->Game->Name)+
					 strlen(".zip")+1);

		  strcat(gamebackuppath,BackUpPath);
		  strcat(gamebackuppath,ThisDiffGame->Game->Name);
		  strcat(gamebackuppath,".zip");
		}
	      else
		{
		  gamebackuppath=XCALLOC(unsigned char,
					 strlen("/tmp/")+
					 strlen("mamory_")+
					 strlen(ThisDiffGame->Game->Name)+
					 strlen(".zip")+1);

		  strcat(gamebackuppath,"/tmp/");
		  strcat(gamebackuppath,"mamory_");
		  strcat(gamebackuppath,ThisDiffGame->Game->Name);
		  strcat(gamebackuppath,".zip");
		};

	      // Copy the game to backup path
	      fp_backup=fopen(gamebackuppath,"w"); /* this doesn't need to be tested:
						      whether it is /tmp, and then it
						      already have been tested, whether 
						      it's a directory created by mamory 
						   */
	      while((nb_read=fread(buffer,1,BUFSIZ,DestZip->fp))!=0)
		{
		  nb_write=fwrite(buffer,1,nb_read,fp_backup);
		  if (nb_write!=nb_read)
		    {
		      // TODO : Handle problems during Backup (not enough space, etc...)
		    };
		};
	      fclose(fp_backup);

	      Zipfclose(DestZip);
	      DestZip=NULL;
	      if(Test==FALSE)
		// Remove the old version game
		unlink(rompath);

	      if(Test==FALSE)
		// Create the new version game
		DestZip=Zipfopen(rompath,"w+");

	      SourceZip=Zipfopen(gamebackuppath,"r");

	      // Import the unmodified roms from the backup game , and rename 
	      // them if required
	      // Move the suppressed roms to temporary file, in order to
	      // be able to retrieve them later in the case where they have
	      // been moved to another game
	      ThisDiffGame->Roms=CLFirst(ThisDiffGame->Roms);
	      do
		{
		  ThisDiffRom=DIFF_ROM(CLGetData(ThisDiffGame->Roms));
		  switch(ThisDiffRom->Type)
		    {
		    case SAME:
		      if(Test==FALSE)
			CopyZipLocalFile(DestZip,SourceZip,
					 ThisDiffRom->Rom->Name,NULL);
		      break;
		    case CHANGEDNAME:
		      LPRINTFW(OUTPUT,"\tChanging rom name from %s to %s ...",
			      ThisDiffRom->SourceRom->Name,
			      ThisDiffRom->Rom->Name);
		      if(Test==FALSE)
			CopyZipLocalFile(DestZip,SourceZip,
					 ThisDiffRom->SourceRom->Name,
					 ThisDiffRom->Rom->Name);
		      if(Test==TRUE)
			LPRINTFW(OUTPUT," test");
		      LPRINTF(OUTPUT," done");
		      ThisDiffRom->Type=SAME;
		      ThisDiffGame->NbrSAMERoms++;
//		      XFREE(ThisDiffRom->OldName);
		      break;		   		       
		    case SUPPRESSED:
		    case SUPPRTRANSFERT:
		      if (TempZip==NULL)
			{
			  temppath=XCALLOC(unsigned char,
					   strlen(path)+
					   strlen(".mamory_")+
					   strlen(ThisDiffGame->Game->Name)+
					   strlen(".zip")+1);

			  strcat(temppath,path);
			  strcat(temppath,".mamory_");
			  strcat(temppath,ThisDiffGame->Game->Name);
			  strcat(temppath,".zip");

			  TempZip=Zipfopen(temppath,"w+");
			};
		      CopyZipLocalFile(TempZip,SourceZip,
				       ThisDiffRom->Rom->Name,NULL);
		      break;			    
		    };


		  if(ThisDiffGame->Roms->Next!=NULL)
		    ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
		  else
		    break;
		}while(ThisDiffGame->Roms!=NULL);

	      // Close the temp file containing the suppressed roms
	      if(TempZip!=NULL)
		{
		  WriteZipCentralDir(TempZip,NULL);
		  Zipfclose(TempZip);
		  TempZip=NULL;	       
		  XFREE(temppath);
		};
	      Zipfclose(SourceZip);
	      SourceZip=NULL;
	    };
	     
	  
	  // Import required roms from other games
	  // Update the status of the DiffRoms in the DiffGame
	  ThisDiffGame->Roms=CLFirst(ThisDiffGame->Roms);
	  do
	    {
	      ThisDiffRom=DIFF_ROM(CLGetData(ThisDiffGame->Roms));
	      
	      // Skip DiffRom that don't require any action
	      if(ThisDiffRom->Type==SAME || ThisDiffRom->Type==CHANGEDNAME)
		{
		  if(ThisDiffGame->Roms->Next!=NULL)
		    ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
		  else
		    break;
		  continue;
		};
	      
	      switch(ThisDiffRom->Type)
		{
		case ADDEDTRANSFERT:
		  if(ThisDiffRom->Rom->Content->CRC==0 &&
		     ThisDiffRom->SourceRom==NULL)
		    {
		      if(!(Diff->OptionFlags & 
			 OPTION_IGNORE_MISSING_NO_DUMP_ROMS))
			{
			  LPRINTFW(OUTPUT,
				   "\tAutomagically synthesizing rom %s ...",
				   ThisDiffRom->Rom->Name);
			  // VERY UGLY QUICK HACK :( 
			  // will have to wait for next major version, if ever
			  
			  if(Test==FALSE)
			    // synthesized rom
			    {
			      FILE *ugly=NULL;
			      ugly=fopen("/tmp/mamory_temp_ugly","w");
			      for(i=0;i<ThisDiffRom->Rom->Content->Size;i++)
				fputc(0,ugly);
			      fclose(ugly);
			      system("zip -q -9 -m /tmp/mamory_temp_ugly.zip /tmp/mamory_temp_ugly");
			      SourceZip=Zipfopen("/tmp/mamory_temp_ugly.zip",
						 "r");			      
			      CopyZipLocalFile(DestZip,SourceZip,
					       "tmp/mamory_temp_ugly",
					       ThisDiffRom->Rom->Name);
			      Zipfclose(SourceZip);
			      SourceZip=NULL;
			      unlink("/tmp/mamory_temp_ugly.zip");
			    };
			  if(Test==TRUE)
			    LPRINTFW(OUTPUT," test");
			  LPRINTF(OUTPUT," done");
			  // Change the status of the DiffRom
			  ThisDiffRom->Type=SAME;
			  ThisDiffGame->NbrSAMERoms++;
			}
		      else
			{
			  ThisDiffRom->Type=ADDEDNEW;
			  ThisDiffGame->NbrMISSINGRoms++;
			};
		      break;
		    };

		  if(strcmp(ThisDiffRom->SourceRom->Name,ThisDiffRom->Rom->Name)!=0)
		    LPRINTFW(OUTPUT,
			    "\tImporting rom %s and changing its name to %s ...",
			    ThisDiffRom->SourceRom->Name,
			    ThisDiffRom->Rom->Name);
		  else
		    LPRINTFW(OUTPUT,
			    "\tImporting rom %s ...",
			    ThisDiffRom->Rom->Name);
		  
		  // Considering that synchronization is applied to GamesLists
		  // coming from directories  analysis, 
		  // ThisDiffRom->SourceRom->GameName will always refers to 
		  // an existing game on disk.
		  // If GamesList came from an executable or a any sources
		  // that could describe a MERGED or FULL GamesList, 
		  // ThisDiffRom->SourceRom->GameName could refer to the clone
		  // of an existing game that would not be describe in 
		  // GamesList->Games. It will then bring the need to retrieve
		  // the correct game name where the rom is stored

		  if(ThisDiffRom->SourceRom->Source==Diff->GamesList1->Sources[0] &&
		     strcmp(ThisDiffRom->SourceRom->GameName,
			    ThisDiffGame->Game->Name)==0)
		    // the source of the rom to import is in the game that is
		    // currently modified (ie, two identical roms with a
		    // different name in the same game)
		    sourcepath=XSTRDUP(gamebackuppath);
		  else
		    {
		      sourcepath=XCALLOC(unsigned char,
					 strlen(ThisDiffRom->SourceRom->Source->Source)+1+
					 strlen(ThisDiffRom->SourceRom->GameName)+
					 strlen(".zip")+1);
		      
		      strcat(sourcepath,ThisDiffRom->SourceRom->Source->Source);
		      strcat(sourcepath,"/");
		      strcat(sourcepath,ThisDiffRom->SourceRom->GameName);
		      strcat(sourcepath,".zip");
		    };

		  SourceZip=Zipfopen(sourcepath,"r");

		  XFREE(sourcepath);
		  
		  sourcepath=XCALLOC(unsigned char,
				     strlen(ThisDiffRom->SourceRom->Source->Source)+1+
				     strlen(".mamory_")+
				     strlen(ThisDiffRom->SourceRom->GameName)+
				     strlen(".zip")+1);
		  
		  strcat(sourcepath,ThisDiffRom->SourceRom->Source->Source);
		  strcat(sourcepath,"/");
		  strcat(sourcepath,".mamory_");
		  strcat(sourcepath,ThisDiffRom->SourceRom->GameName);
		  strcat(sourcepath,".zip");
		  
		  if(SourceZip==NULL)
		    {
		      // this mean that we were right in saving previous 
		      // suppressed roms in a temp file :)
		      SourceZip=Zipfopen(sourcepath,"r");
		  
		      if (SourceZip==NULL)
			{
			  // Here is a really big problem !
			  // Where is my source ????
			  if (Test==TRUE)
			    LPRINTFW(OUTPUT,"** test");
			  else
			    LPRINTFW(OUTPUT,"**");
			  LPRINTF(OUTPUT," failed **");
			  LPRINTF(ERROR,
				  "source file %s not found for rom %s",
				  ThisDiffRom->SourceRom->GameName,
				  ThisDiffRom->Rom->Name);
			  
			  break;
			};
		    };
		  
		  SourceRomName=XSTRDUP(ThisDiffRom->SourceRom->Name);
		  // Check if rom is part of the game in the sourcepath
		  if(isFileNameinZip(SourceZip,ThisDiffRom->SourceRom->Name)==FALSE)
		    {
		      // If the rom is imported and its name changed, 
		      // it is possible that the name has already been changed 
		      // in the SourceZip, therefore we are looking for its CRC
		      XFREE(SourceRomName);
		      SourceRomName=isFileCRCinZip(SourceZip,ThisDiffRom->SourceRom->Content->CRC,
						   ThisDiffRom->SourceRom->Content->Size);
		      if(!SourceRomName)
			{
			  // Otherwise, let's have a look into the previously saved
			  // rom in .mamory_file.zip
			  Zipfclose(SourceZip);
			  SourceZip=NULL;
			  SourceZip=Zipfopen(sourcepath,"r");
			  if(isFileNameinZip(SourceZip,ThisDiffRom->SourceRom->Name)==FALSE)
			    {
			      //			      Games[0]=NULL;
			      //			      XFREE(Games);
			      if (Test==TRUE)
				LPRINTFW(OUTPUT,"** test");
			      else
				LPRINTFW(OUTPUT,"**");
			      LPRINTF(OUTPUT," failed **");
			      LPRINTF(ERROR,
				      "source file %s not found for rom %s",
				      ThisDiffRom->SourceRom->GameName,
				      ThisDiffRom->SourceRom->Name);
			      break;
			    }
			  else
			    SourceRomName=XSTRDUP(ThisDiffRom->SourceRom->Name);
			};
		    };
		  
		  if(Test==FALSE)
		    // Import rom
		    CopyZipLocalFile(DestZip,SourceZip,
				     SourceRomName,
				     ThisDiffRom->Rom->Name);
		  

		  XFREE(SourceRomName);

		  if(Test==TRUE)
		    LPRINTFW(OUTPUT," test");
		  LPRINTF(OUTPUT," done");

		  // Change the status of the DiffRom
		  ThisDiffRom->Type=SAME;
		  ThisDiffGame->NbrSAMERoms++;

		  XFREE(sourcepath);
		  Zipfclose(SourceZip);
		  SourceZip=NULL;
		  break;
		case SUPPRTRANSFERT:
		case SUPPRESSED:
		  if(ThisDiffRom->Type==SUPPRTRANSFERT)
		    LPRINTFW(OUTPUT,"\tRemoving rom %s ...",
			    ThisDiffRom->Rom->Name);
		  else
		    LPRINTFW(OUTPUT,"\tDeleting %s ...",
			    ThisDiffRom->Rom->Name);

		  if(ThisDiffGame->Roms->Next!=NULL)
		    {
		      ThisDiffGame->Roms=CLRemove(ThisDiffGame->Roms,
						  ThisDiffRom);
		      ThisDiffRom->Rom=NULL;
		      XFREE(ThisDiffRom);
		      ThisDiffGame->NbrRoms--;
		      if(Test==TRUE)
			LPRINTFW(OUTPUT," test");
		      LPRINTF(OUTPUT," done");
		      continue;
		    }
		  else
		    {
		      ThisDiffGame->Roms=CLRemove(ThisDiffGame->Roms,
						  ThisDiffRom);
		      ThisDiffRom->Rom=NULL;
		      XFREE(ThisDiffRom);
		      ThisDiffGame->NbrRoms--;
		      if(Test==TRUE)
			LPRINTFW(OUTPUT," test");
		      LPRINTF(OUTPUT," done");
		      break;
		    };
		}

	      if(ThisDiffGame->Roms!=NULL && ThisDiffGame->Roms->Next!=NULL)
		ThisDiffGame->Roms=ThisDiffGame->Roms->Next;
	      else
		break;
	    }while(ThisDiffGame->Roms!=NULL);  
	  
	  // All roms have been treated...
	  if(Test==FALSE)
	    // ... so let's close the zip file
	    {
	      WriteZipCentralDir(DestZip,NULL);
	      Zipfclose(DestZip);
	      DestZip=NULL;
	    };

	  if(BackUpPath==NULL && gamebackuppath!=NULL)
	    // remove the backup game if required
	    unlink(gamebackuppath);

	  XFREE(gamebackuppath);

	  if(ThisDiffGame->NbrRoms==ThisDiffGame->NbrSAMERoms)
	    {
	      if(ThisDiffGame->NbrRoms==0)
		// delete empty zip file
		{
		  LPRINTFW(OUTPUT,"\tGame deleted ...");
		  if(Test==FALSE)
		    unlink(rompath);
		  if(Test==TRUE)
		    LPRINTFW(OUTPUT," test");
		  LPRINTF(OUTPUT," done");

		};

	      if(Diff->Games->Next==NULL)
		{
		  Diff->Games=CLRemove(Diff->Games,ThisDiffGame);
		  FreeDiffGame(ThisDiffGame);
		  Diff->NbrGames--;
		  XFREE(rompath);
		  break;
		}
	      else
		{
		  Diff->Games=CLRemove(Diff->Games,ThisDiffGame);
		  FreeDiffGame(ThisDiffGame);
		  Diff->NbrGames--;  
		  XFREE(rompath);
		  continue;
		};
	    };

	  XFREE(rompath);

	  if(Diff->Games->Next!=NULL)
	    Diff->Games=Diff->Games->Next;
	  else
	    break;
	}while(Diff->Games!=NULL); 
    };
  
  XFREE(rompath);

  // Remove all the temporary files
  dirp=opendir(path);
  
  while((DirEntry=readdir(dirp))!=NULL)
    {
      if(strncmp(DirEntry->d_name,".mamory_",8)==0)
	{
	  rompath=XSTRDUP(path);
	  rompath=XREALLOC(rompath,
			   unsigned char,
			   strlen(rompath)+
			   strlen(DirEntry->d_name)+1);

	  strcat(rompath,DirEntry->d_name);
	  unlink(rompath);
	  XFREE(rompath);
	};
    };
  closedir(dirp);

  XFREE(path);
  XFREE(gamebackuppath);
  XFREE(BackUpPath);
  return TRUE;
};

