/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  context.c

  context functions
*/

#include "common.h"
#include <stdio.h>
#include <scapi/scapi.h>

s_Context *InitContext(unsigned char *Title,
		       unsigned char *Author,
		       unsigned char *Copyright,
		       unsigned int VerMaj,
		       unsigned int VerMin,
		       unsigned int VerMisc)
{
  s_Context *Context;

  Context=XCALLOC(s_Context,1);

  Context->Title=XSTRDUP(Title);
  Context->Author=XSTRDUP(Author);
  Context->Copyright=XSTRDUP(Copyright);
  Context->VerMaj=VerMaj;
  Context->VerMin=VerMin;
  Context->VerMisc=VerMisc;
  Context->GlobalConfFile=NULL;
  Context->LocalConfFile=NULL;
  Context->NbrOptions=0;
  Context->Options=NULL;

  return Context;
};
				     
s_Context *SetGlobalConfFile(s_Context *Context,unsigned char *File)
{
  if (!Context)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return NULL;
    }
  else
    Context->GlobalConfFile=XSTRDUP(File);

  return Context;
}

s_Context *SetLocalConfFile(s_Context *Context,unsigned char *File)
{
  if (!Context)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return NULL;
    }
  else
    Context->LocalConfFile=XSTRDUP(File);

  return Context;
}
  
s_Context *AddOption(s_Options *Options, 
		     s_Context *Context, 
		     unsigned char *Name,
		     e_OptionType Type)
{
  unsigned int i,j;

  if(!Context)
    {
      LPRINTF(ERROR,"This option context has not been defined");
      return NULL;
    }      ;
  
  if(!Name)
    {
      LPRINTF(ERROR,"No name for this option");
      return Context;
    };

  for(i=0;i<Options->NbrContexts;i++)
    for(j=0;j<Options->Contexts[i]->NbrOptions;j++)
      if(strcmp(Options->Contexts[i]->Options[j]->Name,Name)==0 || 
	 (Options->Contexts[i]->Options[j]->ShortName!=NULL && 
	  strcmp(Options->Contexts[i]->Options[j]->ShortName,Name)==0))
	{
	  LPRINTF(ERROR,"This option has already been defined by context '%s'",
		  Options->Contexts[i]->Title);
	  return Context;
	};

  Context->Options=XREALLOC(Context->Options,
			    s_Option*,
			    Context->NbrOptions+1);

  Context->Options[Context->NbrOptions]=XMALLOC(s_Option,1);

  Context->Options[Context->NbrOptions]->Type=Type;
  Context->Options[Context->NbrOptions]->Name=XSTRDUP(Name);
  Context->Options[Context->NbrOptions]->ShortName=NULL;
  Context->Options[Context->NbrOptions]->ConfFileName=NULL;
  Context->Options[Context->NbrOptions]->ConfFileFunc=NULL;
  Context->Options[Context->NbrOptions]->NbrSetArgs=0;
  Context->Options[Context->NbrOptions]->SetArgs=NULL;
  Context->Options[Context->NbrOptions]->ArgDescription=NULL;
  Context->Options[Context->NbrOptions]->NbrArgs=0;
  Context->Options[Context->NbrOptions]->Args=NULL;
  Context->Options[Context->NbrOptions]->Flags=SCAPI_NO_FLAGS;
  Context->Options[Context->NbrOptions]->Separator=NULL;
  Context->Options[Context->NbrOptions]->Help=NULL;
  Context->NbrOptions++;

  return Context;
};

s_Context *SetOptionFlags(s_Context *Context,
			  unsigned char *Option,
			  e_OptionFlags Flags)
{
  unsigned int i;

  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0 )
       )
       break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return Context;
    }      

  Context->Options[i]->Flags=Flags;
  return Context;
};
  
s_Context *SetOptionSeparator(s_Context *Context,
			      unsigned char *Option,
			      unsigned char *Separator)
{
  unsigned int i;

  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0 )
       )
       break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return Context;
    }

  if(!(Context->Options[i]->Flags & SCAPI_MULTIPLE_ARGS))
    {
      LPRINTF(WARNING,"Flag SCAPI_MULTIPLE_ARGS not set for this option");
      return Context;
    }

  Context->Options[i]->Separator=XSTRDUP(Separator);
  return Context;
}
s_Context *SetOptionShortName(s_Options *Options,
			      s_Context *Context, 
			      unsigned char *Option,
			      unsigned char *ShortName)
{
  unsigned int i,j;

  for(i=0;i<Options->NbrContexts;i++)
    for(j=0;j<Options->Contexts[i]->NbrOptions;j++)
      if(strcmp(Options->Contexts[i]->Options[j]->ShortName,ShortName)==0 || 
	 (Options->Contexts[i]->Options[j]->Name!=NULL && 
	  strcmp(Options->Contexts[i]->Options[j]->Name,ShortName)==0))
	{
	  LPRINTF(ERROR,"This option has already been defined by context '%s'",
		  Options->Contexts[i]->Title);
	  return Context;
	};

  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0 )
       )
       break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(ERROR,"This option has not been defined");
      return Context;
    }

  Context->Options[i]->ShortName=XSTRDUP(ShortName);

  return Context;
};

s_Context *SetContextConfFileName(s_Context *Context, 
				  unsigned char *Option,
				  unsigned char *ConfFileName)
{
  unsigned int i;

  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0 )
       )
       break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option context has not been defined");
      return Context;
    }      

  Context->Options[i]->ConfFileName=XSTRDUP(ConfFileName);
  return Context;
};

s_Context *SetOptionDefaultValue(s_Context *Context,
				 unsigned char *Option,
				 unsigned char *Value)
{
  unsigned int i,j;
  
  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0)
       )
      break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option has not been defined");
      return Context;
    }   

  switch(Context->Options[i]->Type)
    {
    case SET_ARG:
      for(j=0;j<Context->Options[i]->NbrSetArgs;j++)
	if(strcmp(Context->Options[i]->SetArgs[j],Value)==0)
	  break;
      if(j==Context->Options[i]->NbrSetArgs)
	return NULL;
      else
	{
	  Context->Options[i]->Flags|=SCAPI_OPTION_USED;
	  Context->Options[i]->Args=XREALLOC(Context->Options[i]->Args,
					     unsigned char*,
					     Context->Options[i]->NbrArgs+1);
	  Context->Options[i]->Args[Context->Options[i]->NbrArgs]=XSTRDUP(Value);
	  Context->Options[i]->NbrArgs++;
	}
      break;

    case FREE_ARG:
      Context->Options[i]->Flags|=SCAPI_OPTION_USED;
      Context->Options[i]->Args=XREALLOC(Context->Options[i]->Args,
					     unsigned char*,
					     Context->Options[i]->NbrArgs+1);
      Context->Options[i]->Args[Context->Options[i]->NbrArgs]=XSTRDUP(Value);
      Context->Options[i]->NbrArgs++;
      break;

    case NO_ARG:
      Context->Options[i]->Flags|=SCAPI_OPTION_USED;
    };
  return Context;
}

s_Context *SetOptionHelp(s_Context *Context, 
			 unsigned char *Option,
			 unsigned char *ArgDescription,
			 unsigned char *Help)
{
  unsigned int i;
  
  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0)
       )
      break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option has not been defined");
      return Context;
    } 

  Context->Options[i]->Help=XSTRDUP(Help);

  if(Context->Options[i]->Type==FREE_ARG)
    Context->Options[i]->ArgDescription=XSTRDUP(ArgDescription);

  return Context;
};

s_Context *AddOptionArg(s_Context *Context,
			unsigned char *Option,
			unsigned char *Arg)
{
  unsigned int i;

  if(!Context)
    {
      LPRINTF(ERROR,"This option context has not been defined");
      return NULL;
    } 

  for(i=0;i<Context->NbrOptions;i++)
    if(strcmp(Context->Options[i]->Name,Option)==0 ||
       (Context->Options[i]->ShortName!=NULL &&
	strcmp(Context->Options[i]->ShortName,Option)==0)
       )      
      break;

  if(i==Context->NbrOptions)
    {
      LPRINTF(WARNING,"This option has not been defined");
      return Context;
    } 

  if(Context->Options[i]->Type!=SET_ARG)
    {
      LPRINTF(WARNING,"'%s' option has not been defined as SetArg",
	      Context->Options[i]->Name);
      return Context;
    };

  Context->Options[i]->SetArgs=XREALLOC(Context->Options[i]->SetArgs,
					unsigned char*,
					Context->Options[i]->NbrSetArgs+1);

  Context->Options[i]->SetArgs[Context->Options[i]->NbrSetArgs]=XSTRDUP(Arg);
  Context->Options[i]->NbrSetArgs++;

  return Context;
};

void FreeContext(s_Context *Context)
{
  unsigned int i,j;

  XFREE(Context->Title);
  XFREE(Context->Author);
  XFREE(Context->Copyright);
  Context->VerMaj=0;
  Context->VerMin=0;
  Context->VerMisc=0;
  for(i=0;i<Context->NbrOptions;i++)
    {
      XFREE(Context->Options[i]->Name);
      XFREE(Context->Options[i]->ShortName);
      XFREE(Context->Options[i]->ConfFileName); 
      XFREE(Context->Options[i]->ArgDescription);
      XFREE(Context->Options[i]->Help);
      XFREE(Context->Options[i]->Separator);
      for(j=0;j<Context->Options[i]->NbrArgs;j++)
	XFREE(Context->Options[i]->Args[j]);
      XFREE(Context->Options[i]->Args);
      for(j=0;j<Context->Options[i]->NbrSetArgs;j++)
	XFREE(Context->Options[i]->SetArgs[j]);
      XFREE(Context->Options[i]->SetArgs);
      XFREE(Context->Options[i]);
    };
  XFREE(Context->Options);
  XFREE(Context->GlobalConfFile);
  XFREE(Context->LocalConfFile);
  XFREE(Context);
};
