/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  sortbox.c

  Sorting functions 
*/

#include "common.h"
#include <stdio.h>
#include <sortbox/sortbox.h>

/*
 * Function: SortBox_CreateBox
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Mother -> pointer to the parent box
 *    MotherLetter -> letter of the parent box
 *    NbrLetters -> Number of letters of the words or partial words
 *                  whose last letter is in this box
 * Output:
 *    Pointer to a s_SBBox structure
 * Status: PRIVATE
 * Description:
 *    Create a new box for the new letter of a word or partial word
 *
 */
unsigned int Libsortbox_WARNING_On=TRUE;
unsigned int Libsortbox_ERROR_On=TRUE;
unsigned int Libsortbox_OUTPUT_On=TRUE;
unsigned int Libsortbox_HEADER_OUTPUT_On=TRUE;         // Print [OUTPUT] 
unsigned int Libsortbox_HEADER_WARNING_On=TRUE;        // Print [WARNING]
unsigned int Libsortbox_HEADER_ERROR_On=TRUE;          // Print [ERROR]
unsigned int Libsortbox_HEADER_DEBUG_On=TRUE; 
unsigned int Libsortbox_HEADER_OUTPUT_LIBNAME_On=TRUE; // 
unsigned int Libsortbox_HEADER_WARNING_LIBNAME_On=TRUE;// Print the lib name
unsigned int Libsortbox_HEADER_ERROR_LIBNAME_On=TRUE;  //
unsigned int Libsortbox_HEADER_DEBUG_LIBNAME_On=TRUE;

FILE *Libsortbox_err=NULL;
FILE *Libsortbox_out=NULL;

s_SBBox *SortBox_CreateBox( s_SortBox *SortBox,
			    s_SBBox *Mother,unsigned char MotherLetter,
			    unsigned short NbrLetters)
{
  SortBox->Boxes=XREALLOC(SortBox->Boxes,
			  s_SBBox*,
			  SortBox->NbrBoxes+1);

  
  SortBox->Boxes[SortBox->NbrBoxes]=XMALLOC(s_SBBox,1);

  SortBox->Boxes[SortBox->NbrBoxes]->Mother=Mother;
  SortBox->Boxes[SortBox->NbrBoxes]->MotherLetter=MotherLetter;
  SortBox->Boxes[SortBox->NbrBoxes]->NbrLetters=NbrLetters;
  SortBox->Boxes[SortBox->NbrBoxes]->NbrWords=0;
  SortBox->Boxes[SortBox->NbrBoxes]->Words=NULL;
  SortBox->NbrBoxes++;
  return(SortBox->Boxes[SortBox->NbrBoxes-1]);
};

/*
 * Function: SortBox_AddLetter
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Box -> pointer to the Box where the letter is added
 *    WordID -> pointer to the structure associated  to the Word 
 *              NULL if the Letter is not the last of the Word
 *    LetterDaughter -> pointer to the Box containing the next Letter of the 
 *                      partial word
 * Output:
 *    None
 * Status: PUBLIC
 * Description:
 *    Add a new letter into a Box
 */

void SortBox_AddLetter( s_SortBox *SortBox, s_SBBox *Box,
			void *WordID, s_SBBox *LetterDaughter,
			unsigned char Letter)
{
  unsigned char *order;

  /* increase number of words or partial words in the Box */
  Box->Words=XREALLOC(Box->Words,
		      s_SBWord*,
		      Box->NbrWords+1);


  /* update the alphabet */
  if (SortBox->Alphabet->Order!=NULL)
    /* if an alphabet exists */
    {
      if(!strchr(SortBox->Alphabet->Order,Letter))
	/* if the new letter is not in the alphabet */
	{
	  /* increase alphabet length of one and add the new letter at the end
	     of the alphabet */
	  order=XCALLOC(char,SortBox->Alphabet->NbrLetters+1+1);

	  strncpy(order,SortBox->Alphabet->Order,SortBox->Alphabet->NbrLetters);
	  XFREE(SortBox->Alphabet->Order);
	  SortBox->Alphabet->Order=order;
	  SortBox->Alphabet->Order[SortBox->Alphabet->NbrLetters]=Letter;
	  SortBox->Alphabet->NbrLetters++;
	};
    }  
  else
    /* if alphabet does not exist */
    {
      /* initialize alphabet with the new letter */
      SortBox->Alphabet->Order=XCALLOC(char,SortBox->Alphabet->NbrLetters+1+1);

      SortBox->Alphabet->Order[SortBox->Alphabet->NbrLetters]=Letter;
      SortBox->Alphabet->NbrLetters++;
    };
   
  /* initialize the new word (or partial word) in the box */
  Box->Words[Box->NbrWords]=XMALLOC(s_SBWord,1);

  Box->Words[Box->NbrWords]->NbrIDs=0;
  Box->Words[Box->NbrWords]->IDs=NULL;
  if(WordID!=NULL)
    /* if the new word is a complete word */
    {
      /* initialize the IDs associated to the new word */
      Box->Words[Box->NbrWords]->IDs=
	XREALLOC(Box->Words[Box->NbrWords]->IDs,
		 void*,
		 Box->Words[Box->NbrWords]->NbrIDs+1);

      Box->Words[Box->NbrWords]->IDs[Box->Words[Box->NbrWords]->NbrIDs]=WordID;
      Box->Words[Box->NbrWords]->NbrIDs++;
    };
  /* set the box corresponding to the next part of the partial word */
  Box->Words[Box->NbrWords]->Daughter=LetterDaughter;
  /* set the letter representing the word in the box */
  Box->Words[Box->NbrWords]->Letter=Letter;

  Box->NbrWords++;
};

/*
 * Function: SortAlphabet
 * Input:   
 *    Alphabet -> string of letters to sort
 * Output:
 *    None
 * Status: PRIVATE
 * Description:
 *    Sort the Alphabet used in the SortBox, respecting the priority
 */

inline void SortAlphabet(s_SBAlphabet *Alphabet)
{
  unsigned char ASCII[256]={0};
  unsigned char *order;
  unsigned int i,j;
  unsigned char *pointer;
  
  /* set to 1 the letter in the ASCII table that are in the alphabet */
  for(i=0;i<Alphabet->NbrLetters;i++)
    ASCII[Alphabet->Order[i]]=1;

  /* initialize the futur sorted alphabet string */
  order=XCALLOC(char,Alphabet->NbrLetters+1);

  j=0;

  if (Alphabet->Priority!=NULL)
    /* if priority string exists */
    for (i=0;i<strlen(Alphabet->Priority);i++)
      /* for each letter in priority string */
      if ((pointer=strchr(Alphabet->Order,Alphabet->Priority[i]))!=NULL)
	/* if this letter is in the alphabet */
	{
	  /* place this letter at the beginning of the alphabet in the 
	     priority order */
	  order[j]=Alphabet->Priority[i];
	  /* remove this letter from the ASCII table */
	  ASCII[Alphabet->Order[pointer-Alphabet->Order]]=0;
	  j++;
	};
	
  
  /* add the letter used in the alphabet in the order defined by the ASCII 
     table */
  for(i=0;i<256;i++)
    if(ASCII[i]==1)
      {
	order[j]=i;
	j++;
      };
  XFREE(Alphabet->Order);
  Alphabet->Order=XSTRDUP(order);
  XFREE(order);
};
    
/*
 * Function: InitSortBox
 * Input:
 *    Type -> SB_NUM or SB_ALPHANUM
 *    PriorityOrder -> string of letter giving the priority for Alphabet 
 *                     sorting
 * Output:
 *    Pointer to a s_SortBox structure      
 * Status: PUBLIC
 * Description:
 *    Initialize a SortBox
 */

s_SortBox *InitSortBox (e_SortBoxType Type,const char *PriorityOrder)
{
  s_SortBox *SortBox;
 
  Libsortbox_err=stderr;
  Libsortbox_out=stdout;

  /* reserve memory for the SortBox */
  SortBox=XMALLOC(s_SortBox,1);

  if (!SortBox)
    return(NULL);

  /* initialize SortBox */
  SortBox->Type=Type;
  SortBox->Alphabet=XMALLOC(s_SBAlphabet,1);

  if (PriorityOrder!=NULL)
    SortBox->Alphabet->Priority=XSTRDUP(PriorityOrder);
  else
    SortBox->Alphabet->Priority=NULL;
  SortBox->Alphabet->NbrLetters=0;
  SortBox->Alphabet->Order=NULL;

  SortBox->NbrBoxes=0;
  SortBox->Boxes=XMALLOC(s_SBBox*,SortBox->NbrBoxes+1);

  SortBox->Boxes[0]=XMALLOC(s_SBBox,1);

  SortBox->Boxes[0]->Mother=NULL;
  SortBox->Boxes[0]->MotherLetter=0;
  SortBox->Boxes[0]->NbrLetters=1;
  SortBox->Boxes[0]->NbrWords=0;
  SortBox->NbrBoxes++;
  SortBox->Boxes[0]->Words=NULL;

  return(SortBox);
};

/*
 * Function: SortBox_AddWord
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Name -> Word to add
 *    ID: pointer to the structure associated to the Word
 * Output:
 *    None
 * Status: PRIVATE
 * Description:
 *    Add a new (Word ,ID) couple in the SortBox
 */

void SortBox_AddWord (s_SortBox *SortBox, unsigned char *Name, void *ID)
{
  unsigned long i;
  unsigned short WordLength;
  s_SBBox  *Box, *NewBox;
  unsigned char word=0;


  Box=SortBox->Boxes[0];
  WordLength=strlen(Name);
      
  for(i=0;i<WordLength;i++)
    /* for each letter of the new word */
    {
      if (i!=0)
	/* if word letter is not the first one, set the current Box to the 
	   following */
	Box=Box->Words[word]->Daughter;

      for(word=0;word<Box->NbrWords;word++)
	if (Box->Words[word]->Letter ==Name[i])
	  break;
      
      if (word==Box->NbrWords)
	/* if the word letter does not exist in the box, add it (it makes a 
	   partial word) */
	SortBox_AddLetter(SortBox,Box,NULL,NULL,Name[i]);
      
      if(Box->NbrLetters!=WordLength)
	/* if this box contains words or partial words whose lenght is inferior
	   to the new word then we need to continue with the following box */
	{
	  if (Box->Words[word]->Daughter==NULL)
	    /* if the following box does not exist */
	    {
	      /* Create the new box */
	      NewBox=SortBox_CreateBox(SortBox,Box,Name[i],i+2); 
	               /* +2=+1+1 because for 1 letter, i=0 */
	      SortBox_AddLetter(SortBox,NewBox,NULL,NULL,Name[i+1]);
	      Box->Words[word]->Daughter=NewBox;
	    };	  
	}
    };   
      
  /* the last partial word added to the last box need to be set as a word by 
     setting its associated ID */
  Box->Words[word]->IDs=XREALLOC(Box->Words[word]->IDs,
				 void*,Box->Words[word]->NbrIDs+1);


  Box->Words[word]->IDs[Box->Words[word]->NbrIDs]=ID;
  Box->Words[word]->NbrIDs++;
};


/*
 * Function: SortBox_AddOneWord
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Name -> Word to add
 *    ID -> pointer to the structure associated to the Word
 * Output:
 *    None
 * Status: PUBLIC
 * Description:
 *    Add a new Word to the SortBox and update Alphabet
 */

void SortBox_AddOneWord(s_SortBox *SortBox,unsigned char *Name, void *ID)
{
  SortBox_AddWord(SortBox,Name,ID);
  SortAlphabet(SortBox->Alphabet);
};

void SortBox_AddOneUint(s_SortBox *SortBox,unsigned int Number, void *ID)
{
  unsigned char str[BUFSIZ]={0};
  unsigned char *string=NULL;
  unsigned int strl=0;
  unsigned int i;

  sprintf(str,"%X",Number);
  
  strl=strlen(str);
  string=XCALLOC(unsigned char,strl+1);

  for(i=0;i<strl;i++)
    string[i]=str[strl-i-1];
  
  SortBox_AddWord(SortBox,string,ID);
  SortAlphabet(SortBox->Alphabet);
  XFREE(string);
};
    
/*
 * Function: SortBox_AddWordsTable
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    NbrWords -> Number of words to add
 *    WordsTable -> pointer to an array of pointer to (Word,ID) couple
 * Output:
 *    None
 * Status: PUBLIC
 * Description:
 *    Add a NbrWords of new words to the SortBox and update Alphabet
 */

void SortBox_AddWordsTable(s_SortBox *SortBox, unsigned int NbrWords, s_NameID **WordsTable)
{
  unsigned long j;		    
  unsigned char *name=NULL;
  for(j=0;j<NbrWords;j++)
    {
      /*      if (SortBox->Type==SB_NUM)
	name=InvertString(WordsTable[j]->Name);
	else*/
      name=XSTRDUP(WordsTable[j]->Name);
      SortBox_AddWord(SortBox,name,WordsTable[j]->ID);
      XFREE(name);
    };
  SortAlphabet(SortBox->Alphabet);
};

/*
 * Function: SortBox_Free
 * Input:
 *    SortBox -> pointer to the SortBox in use
 * Output:
 *    None
 * Status: PUBLIC
 * Description:
 *    Destroy the SortBox and free the memory it used
 */

void FreeSortBox( s_SortBox *SortBox)
{
  unsigned int i,j;
  
  for(i=0; i<SortBox->NbrBoxes; i++)
    {
      for (j=0;j<SortBox->Boxes[i]->NbrWords;j++)
	{
	  XFREE(SortBox->Boxes[i]->Words[j]->IDs);
	  XFREE(SortBox->Boxes[i]->Words[j]);
	};
      XFREE(SortBox->Boxes[i]->Words);
      XFREE(SortBox->Boxes[i]);
    };
  XFREE(SortBox->Boxes);
  XFREE(SortBox->Alphabet->Order);
  XFREE(SortBox->Alphabet);
  XFREE(SortBox);
};
	  
/*
 * Function: ExploreBox
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Box -> pointer to the Box to explore
 *    FromLetter -> letter to start from in scanning the box
 *    MaxNbrData -> maximum number of data to store in TabID
 *    TabID_ptr -> pointer to the array where to store the Word IDs found
 *    nbrdata_ptr -> Maximum number of data to store in TabID
 * Output:
 *    None
 * Status: PRIVATE
 * Description:
 *    Scan recursively the Box and its children to find Word IDs
 */

void ExploreBox(  s_SortBox *SortBox,  s_SBBox *Box,unsigned char FromLetter,
		  unsigned int MaxNbrData,void ***TabID_ptr, 
		  unsigned int *nbrdata_ptr)
{

 unsigned int i,j,k;

 if (*nbrdata_ptr>=MaxNbrData) 
   /* if the number of stored ID is superior  or equal to the maximum number 
      of data wanted get out */
   return;

 for(j=FromLetter;j<SortBox->Alphabet->NbrLetters;j++) 
   /* for each letters used in the alphabet */
   for(i=0;i<Box->NbrWords;i++)
     /* for each word or partial word in the Box */
     if (Box->Words[i]->Letter==SortBox->Alphabet->Order[j])
       {
	 /* if the word last letter is the alphabet letter */
	 if (Box->Words[i]->NbrIDs!=0)
	   /* if it's not a partial word */
	   {
	     for(k=0;k<Box->Words[i]->NbrIDs;k++)
	       /* for each IDs associated to the  word */
	       {
		 /* store the ID in TabID */
		 *TabID_ptr=XREALLOC(*TabID_ptr,
				     void*,(*nbrdata_ptr)+1);

		 (*TabID_ptr)[*nbrdata_ptr]=Box->Words[i]->IDs[k];
		 (*nbrdata_ptr)++;
		 if (*nbrdata_ptr>=MaxNbrData) 
		   return;
	       };
	   };	 
	 if (Box->Words[i]->Daughter!=NULL)
	   /* if it's a partial word and we have the next Box */
	   {
	     /* scan the following box */
	     ExploreBox(SortBox, Box->Words[i]->Daughter,0,MaxNbrData,TabID_ptr,nbrdata_ptr);
	     if (*nbrdata_ptr>=MaxNbrData) 
	       return;
	   };
       };
};


/*
 * Function: SortBox_FindOrigin
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Origin -> sequence of letter to look for
 *    MaxNbrData_ptr -> pointer to maximum number of data to store in TabID
 *    OriginBox_ptr -> pointer where to store the box containing Origin word
 * Output:
 *    array containing fitting ID
 * Status: PRIVATE
 * Description:
 *    Scan recursively the Box and its children to find Word IDs
 */

void** SortBox_FindOrigin (s_SortBox *SortBox,unsigned char *Origin,unsigned int *MaxNbrData_ptr,s_SBBox **OriginBox_ptr)
{
  unsigned int i=0,k=0;
  int j=-1;
  unsigned int nbrdata=0;
  s_SBBox *Box=NULL;
  s_SBBox *Daughter=NULL;

  void **TabID=NULL;

  Box=SortBox->Boxes[0];
  if (Origin==NULL)
    /* if no pattern given */
    {
      /* set pattern to the first letter used in the alphabet */
      Origin=XCALLOC(char,2);

      Origin[0]=SortBox->Alphabet->Order[0];
    };

  /* seek for the box containing the partial word Origin */
  for(i=0;i<strlen(Origin);i++)
    /* for each letter in the pattern */
    {      
      if (i!=0)
	/* if the letter is not the first of the pattern */
	{	 
	  if (Daughter==NULL)
	    /* if the pattern is not found */
	    {
	      *MaxNbrData_ptr=nbrdata;
	      return NULL;
	    }
	  /* set the current Box to the following one */
	  Box=Daughter;
	};

      for(j=0;j<Box->NbrWords;j++)
	/* for each words or partial words in the Box */
	if (Origin[i]==Box->Words[j]->Letter)
	  break;

      if(j==Box->NbrWords)
	Daughter=NULL;
      else
	Daughter=Box->Words[j]->Daughter;
    };

 if(OriginBox_ptr==NULL && (j>=0 && j<Box->NbrWords) && Box->Words[j]->NbrIDs!=0)
   {
     for(k=0;k<Box->Words[j]->NbrIDs;k++)
       /* for each IDs associated to the  word */
       {
	 /* store the ID in TabID */
	 TabID=XREALLOC(TabID,void*,nbrdata+1);

	 TabID[nbrdata]=Box->Words[j]->IDs[k];
	 nbrdata++; 	 
       };
   };	 

 if (OriginBox_ptr!=NULL)
   *OriginBox_ptr=Box;
     
 *MaxNbrData_ptr=nbrdata;
 return TabID;
};

/*
 * Function: SortBox_FindTheWord
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Origin -> sequence of letter to look for
 *    NbrData_ptr -> pointer to maximum number of data to store in TabID
 * Output:
 *    array containing fitting ID
 * Status: PRIVATE
 * Description:
 *    Scan recursively the Box and its children to find Word IDs
 */

void** SortBox_FindTheWord (s_SortBox *SortBox,unsigned char *Origin,unsigned int *NbrData_ptr)
{
  void **TabID=NULL;

  TabID=SortBox_FindOrigin (SortBox, Origin, NbrData_ptr,NULL);
  return TabID;
};

void** SortBox_FindTheUint (s_SortBox *SortBox,unsigned int Number,unsigned int *NbrData_ptr)
{
  void **TabID=NULL;
  unsigned char str[BUFSIZ]={0};
  unsigned char *Origin=NULL;
  unsigned int strl=0;
  unsigned int i;

  sprintf(str,"%X",Number);
  
  strl=strlen(str);
  Origin=XCALLOC(unsigned char,strl+1);

  for(i=0;i<strl;i++)
    Origin[i]=str[strl-i-1];

  TabID=SortBox_FindOrigin(SortBox, Origin, NbrData_ptr,NULL);
  XFREE(Origin);
  return TabID;
};

/*
 * Function: SortBox_Find
 * Input:
 *    SortBox -> pointer to the SortBox in use
 *    Origin -> pattern of the words to look for
 *    NbrData -> Maximal number of data to store in TabID
 * Output:
 *    array containing fitting ID    
 * Status: PUBLIC
 * Description:
 *    Find every Words ID fitting the pattern and store ID in TabID 
 *    in the order given by the Alphabet
 * TO DO:
 *    - transform Origin in a real pattern using wildcards (for the moment 
 *      it is just the start of the word)
 *    - being  able to seek the Boxes upper the OriginBox
 */

void** SortBox_Find (  s_SortBox *SortBox,unsigned char *Origin,unsigned int *MaxNbrData_ptr)
{
  unsigned char i;
  unsigned int nbrdata=0;
  s_SBBox *OriginBox=NULL;
  void **TabID=NULL;

  nbrdata=*MaxNbrData_ptr;

  if(Origin==NULL)
    {
      Origin=XCALLOC(char,2);

      Origin[0]=SortBox->Alphabet->Order[0];
    };

  TabID=SortBox_FindOrigin(SortBox,Origin,&nbrdata,&OriginBox);

  if (OriginBox!=NULL)
    {      
      for (i=0; i<SortBox->Alphabet->NbrLetters;i++)
	if (SortBox->Alphabet->Order[i]==Origin[strlen(Origin)-1])
	  break;
      ExploreBox(SortBox,OriginBox,i,*MaxNbrData_ptr,&TabID,&nbrdata);
      
      for (i=0; i<SortBox->Alphabet->NbrLetters;i++)
	if (SortBox->Alphabet->Order[i]==OriginBox->MotherLetter)
	  break; 
      i++;
      if(OriginBox->Mother!=NULL)
	OriginBox=OriginBox->Mother;
      else
	goto fin;
      
      while (nbrdata<*MaxNbrData_ptr)
	/* while the number of stored ID corresponding to the pattern is less 
	   than the maximum number of data wanted  */
	{
	  /* scan the SortBox from the OriginBox */
	  ExploreBox(SortBox,OriginBox,i,*MaxNbrData_ptr,&TabID,&nbrdata);
	  for (i=0; i<SortBox->Alphabet->NbrLetters;i++)
	    if (SortBox->Alphabet->Order[i]==OriginBox->MotherLetter)
	      break; 
	  i++;
	  if(OriginBox->Mother!=NULL)
	    OriginBox=OriginBox->Mother;
	  else
	    break;
	};
    };
 fin:
  *MaxNbrData_ptr=nbrdata;

  return TabID;
};

/*
 * Function: SortBox_GetUsedMemory
 * Input:
 *    SortBox -> pointer to the SortBox in use
 * Output:
 *    Memory used by the SortBox
 * Status: PUBLIC
 * Description:
 *    Get the memory used by the SortBox
 * TO DO:
 */

unsigned int SortBox_GetUsedMemory(s_SortBox *SortBox)
{
  unsigned int Mem=0;
  unsigned int i,j,k;

  Mem+=sizeof(SortBox);
  Mem+=sizeof(s_SortBox);
  Mem+=sizeof(s_SBAlphabet);
  if (SortBox->Alphabet->Order!=NULL)
    Mem+=(strlen(SortBox->Alphabet->Order) + 1);
  if (SortBox->Alphabet->Priority!=NULL)
    Mem+=(strlen(SortBox->Alphabet->Priority) + 1);
  for(i=0;i<SortBox->NbrBoxes;i++)
    {
      Mem+=sizeof(SortBox->Boxes[i]);
      Mem+=sizeof(s_SBBox);
      for(j=0;j<SortBox->Boxes[i]->NbrWords;j++)
	{
	  Mem+=sizeof(SortBox->Boxes[i]->Words[j]);
	  Mem+=sizeof(s_SBWord);
	  for(k=0;k<SortBox->Boxes[i]->Words[j]->NbrIDs;k++)
	    Mem+=sizeof(SortBox->Boxes[i]->Words[j]->IDs[k]);
	};
    };
  return Mem;
};
