/* Copyright (C) 2005 to 2015 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/thread.h>

extern "C" {

static void cgu_thread_cleanup_handler(void* arg) {
  delete static_cast<Cgu::Callback::Callback*>(arg);
}

static void* cgu_thread_func(void* arg) {

  const Cgu::Callback::Callback* callback_p = static_cast<Cgu::Callback::Callback*>(arg);

  // we don't need to worry about the thread being cancelled between it starting
  // and pthread_cleanup_push() executing, because a new thread always starts
  // in deferred cancellation state and pthread_cleanup_push() is not a cancellation
  // point - this new thread must therefore run as a minimum until the call to
  // Callback::dispatch() is made
  pthread_cleanup_push(cgu_thread_cleanup_handler, const_cast<Cgu::Callback::Callback*>(callback_p));

  try {
    callback_p->dispatch();
  }
  // don't bother to deal with uncaught exceptions, except Thread::Exit.  If we have
  // reached here with an uncaught exception we are in trouble anyway, and catching
  // general exceptions could have unintended consequences (eg with NPTL a catch with
  // an elipsis (...) argument without rethrowing will terminate the whole application
  // on thread cancellation).  So it is best to do nothing here and rely on the user
  // to stop uncaught exceptions reaching this far.
  catch (Cgu::Thread::Exit&) {;} // just continue to thread termination

  pthread_cleanup_pop(true);

  return 0;
}

} // extern "C"

namespace Cgu {

namespace Thread {

std::unique_ptr<Thread> Thread::start(const Cgu::Callback::Callback* cb, bool joinable) {

  // take ownership of the first argument
  std::unique_ptr<Cgu::Callback::Callback> cb_u(const_cast<Cgu::Callback::Callback*>(cb));
  std::unique_ptr<Thread> return_val;

  // set joinable attribute
  int detach_state;
  if (joinable) detach_state = PTHREAD_CREATE_JOINABLE;
  else detach_state = PTHREAD_CREATE_DETACHED;
  pthread_attr_t attr;
  if (pthread_attr_init(&attr))
    return return_val;
  pthread_attr_setdetachstate(&attr, detach_state); // always returns 0 if attr and detach_state
                                                    // are valid, which we now know they are

  try {
    return_val.reset(new Thread);
  }
  catch (...) {
    pthread_attr_destroy(&attr);
    throw;
  }

  pthread_t thread;
  // start the new thread and release ownership of the callback
  // to the cgu_thread_func() function in the new thread
  if (!pthread_create(&thread, &attr, cgu_thread_func, cb_u.release())) {
    return_val->thread = thread;
  }
  else {
    return_val.reset();
    // we have released in initialising thread_args
    delete cb;
  }
  pthread_attr_destroy(&attr);
  return return_val;
}

void JoinableHandle::cancel() {
  Mutex::Lock lock{mutex};
  if (thread.get()) thread->cancel();
}

bool JoinableHandle::join() {
  mutex.lock();
  if (thread.get() && !detached) {
    detached = true;
    mutex.unlock(); // Thread::Thread::join() might block
    thread->join();
    return true;
  }
  mutex.unlock();
  return false;
}

void JoinableHandle::detach() {
  Mutex::Lock lock{mutex};
  if (thread.get() && !detached) {
    thread->detach();
    detached = true;
  }
}

bool JoinableHandle::is_caller() {
  Mutex::Lock lock{mutex};
  return thread.get() ? thread->is_caller() : false;
}

bool JoinableHandle::is_managing() {
  Mutex::Lock lock{mutex};
  return (thread.get() && !detached);
}

JoinableHandle& JoinableHandle::operator=(JoinableHandle&& h) {
  Mutex::Lock lock{mutex};
  if (thread.get() && !detached)
    thread->detach();
  action = h.action;
  detached = h.detached;
  thread = std::move(h.thread);
  return *this;
}

JoinableHandle::~JoinableHandle() {
  Mutex::Lock lock{mutex};
  if (thread.get() && !detached) {
    if (action == detach_on_exit) thread->detach(); 
    else thread->join();
  }
}

CancelBlock::CancelBlock(bool blocking) {
  // the default value of blocking is true
  if (blocking) block(starting_state);
  else unblock(starting_state);
}

} // namespace Thread

} // namespace Cgu
