#******************************************************************************
#
#       Copyright:      2006-2022 Paul Obermeier (obermeier@tcl3d.org)
#
#                       See the file "Tcl3D_License.txt" for information on
#                       usage and redistribution of this file, and for a
#                       DISCLAIMER OF ALL WARRANTIES.
#
#       Module:         Tcl3D -> tcl3dSDL
#       Filename:       tcl3dSDLUtil.tcl
#
#       Author:         Paul Obermeier
#
#       Description:    Tcl module with miscellaneous utility
#                       procedures related to the SDL module.
#
#******************************************************************************

###############################################################################
#[@e
#       Name:           tcl3dSDLGetFocusName - Convert focus state bitfield.
#
#       Synopsis:       tcl3dSDLGetFocusName { state }
#
#       Description:    state : int
#
#                       Return a SDL focus state bitfield as the corresponding
#                       string representation.
#                       See file SDL_active.h for the definition of possible
#                       bitfield values. 
#
#       See also:       tcl3dSDLGetButtonName
#                       tcl3dSDLGetHatName
#                       tcl3dSDLGetEventName
#
###############################################################################

proc tcl3dSDLGetFocusName { state } {
    set stateStr ""
    if { $state & $::SDL_APPMOUSEFOCUS } {
        append stateStr "MouseFocus "
    } 
    if { $state & $::SDL_APPINPUTFOCUS } {
        append stateStr "InputFocus "
    } 
    if { $state & $::SDL_APPACTIVE } {
        append stateStr "Active "
    } 
    return $stateStr
}

###############################################################################
#[@e
#       Name:           tcl3dSDLGetButtonName - Convert button state bitfield.
#
#       Synopsis:       tcl3dSDLGetButtonName { state }
#
#       Description:    state : int
#
#                       Return a SDL button state bitfield as the corresponding
#                       string representation.
#                       See file SDL_mouse.h for the definition of possible
#                       bitfield values.
#
#       See also:       tcl3dSDLGetFocusName
#                       tcl3dSDLGetHatName
#                       tcl3dSDLGetEventName
#
###############################################################################

proc tcl3dSDLGetButtonName { state } {
    set stateStr ""
    if { $state & [expr 1 << ($::SDL_BUTTON_LEFT-1)] } {
        append stateStr "Left "
    }
    if { $state & [expr 1 << ($::SDL_BUTTON_MIDDLE-1)] } {
        append stateStr "Middle "
    }
    if { $state & [expr 1 << ($::SDL_BUTTON_RIGHT-1)] } {
        append stateStr "Right "
    }
    if { $state & [expr 1 << ($::SDL_BUTTON_WHEELUP-1)] } {
        append stateStr "WheelUp "
    } 
    if { $state & [expr 1 << ($::SDL_BUTTON_WHEELDOWN-1)] } {
        append stateStr "WheelDown "
    }
    return $stateStr
}

array set ::__tcl3dSDLHatNames [list \
    $::SDL_HAT_CENTERED         "HAT_CENTERED" \
    $::SDL_HAT_UP               "HAT_UP" \
    $::SDL_HAT_RIGHT            "HAT_RIGHT" \
    $::SDL_HAT_DOWN             "HAT_DOWN" \
    $::SDL_HAT_LEFT             "HAT_LEFT" \
    $::SDL_HAT_RIGHTUP          "HAT_RIGHTUP" \
    $::SDL_HAT_RIGHTDOWN        "HAT_RIGHTDOWN" \
    $::SDL_HAT_LEFTUP           "HAT_LEFTUP" \
    $::SDL_HAT_LEFTDOWN         "HAT_LEFTDOWN" \
]

###############################################################################
#[@e
#       Name:           tcl3dSDLGetHatName - Convert hat state bitfield.
#
#       Synopsis:       tcl3dSDLGetHatName { state }
#
#       Description:    state : int
#
#                       Return a SDL hat state bitfield as the corresponding
#                       string representation.
#                       See file SDL_joystick.h for the definition of possible
#                       bitfield values.
#
#       See also:       tcl3dSDLGetFocusName
#                       tcl3dSDLGetButtonName
#                       tcl3dSDLGetEventName
#
###############################################################################

proc tcl3dSDLGetHatName { state } {
    if { [info exists ::__tcl3dSDLHatNames($state)] } {
        return $::__tcl3dSDLHatNames($state)
    } else {
        return ""
    }
}

array set ::__tcl3dSDLEventNames [list \
    $::SDL_FIRSTEVENT           "NOEVENT" \
    $::SDL_KEYDOWN              "KEYDOWN" \
    $::SDL_KEYUP                "KEYUP" \
    $::SDL_MOUSEMOTION          "MOUSEMOTION" \
    $::SDL_MOUSEBUTTONDOWN      "MOUSEBUTTONDOWN" \
    $::SDL_MOUSEBUTTONUP        "MOUSEBUTTONUP" \
    $::SDL_JOYAXISMOTION        "JOYAXISMOTION" \
    $::SDL_JOYBALLMOTION        "JOYBALLMOTION" \
    $::SDL_JOYHATMOTION         "JOYHATMOTION" \
    $::SDL_JOYBUTTONDOWN        "JOYBUTTONDOWN" \
    $::SDL_JOYBUTTONUP          "JOYBUTTONUP" \
    $::SDL_QUIT                 "QUIT" \
    $::SDL_SYSWMEVENT           "SYSWMEVENT" \
    $::SDL_USEREVENT            "USEREVENT" \
]

###############################################################################
#[@e
#       Name:           tcl3dSDLGetEventName - Convert event enumeration.
#
#       Synopsis:       tcl3dSDLGetEventName { state }
#
#       Description:    state : int (SDL event enumeration)
#
#                       Return SDL event related enumeration as the 
#                       corresponding string representation.
#                       See file SDL_events.h for the definition of possible
#                       enumeration values.
#
#       See also:       tcl3dSDLGetFocusName
#                       tcl3dSDLGetButtonName
#                       tcl3dSDLGetHatName
#
###############################################################################

proc tcl3dSDLGetEventName { state } {
    if { [info exists ::__tcl3dSDLEventNames($state)] } {
        return $::__tcl3dSDLEventNames($state)
    } else {
        return ""
    }
}
