/*
   Unix SMB/CIFS implementation.
   Parameter loading functions
   Copyright (C) Karl Auer 1993-1998

   Largely re-written by Andrew Tridgell, September 1994

   Copyright (C) Simo Sorce 2001
   Copyright (C) Alexander Bokovoy 2002
   Copyright (C) Stefan (metze) Metzmacher 2002
   Copyright (C) Jim McDonough <jmcd@us.ibm.com> 2003
   Copyright (C) Michael Adam 2008
   Copyright (C) Jelmer Vernooij <jelmer@samba.org> 2007
   Copyright (C) Andrew Bartlett 2011

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

static const struct enum_list enum_protocol[] = {
	{PROTOCOL_SMB2_10, "SMB2"}, /* for now keep PROTOCOL_SMB2_10 */
	{PROTOCOL_SMB3_00, "SMB3"}, /* for now keep PROTOCOL_SMB3_00 */
	{PROTOCOL_SMB3_00, "SMB3_00"},
	{PROTOCOL_SMB2_24, "SMB2_24"},
	{PROTOCOL_SMB2_22, "SMB2_22"},
	{PROTOCOL_SMB2_10, "SMB2_10"},
	{PROTOCOL_SMB2_02, "SMB2_02"},
	{PROTOCOL_NT1, "NT1"},
	{PROTOCOL_LANMAN2, "LANMAN2"},
	{PROTOCOL_LANMAN1, "LANMAN1"},
	{PROTOCOL_CORE, "CORE"},
	{PROTOCOL_COREPLUS, "COREPLUS"},
	{PROTOCOL_COREPLUS, "CORE+"},
	{-1, NULL}
};

static const struct enum_list enum_security[] = {
	{SEC_AUTO, "AUTO"},
	{SEC_USER, "USER"},
	{SEC_DOMAIN, "DOMAIN"},
	{SEC_ADS, "ADS"},
	{-1, NULL}
};

static const struct enum_list enum_bool_auto[] = {
	{false, "No"},
	{false, "False"},
	{false, "0"},
	{true, "Yes"},
	{true, "True"},
	{true, "1"},
	{Auto, "Auto"},
	{-1, NULL}
};

static const struct enum_list enum_csc_policy[] = {
	{CSC_POLICY_MANUAL, "manual"},
	{CSC_POLICY_DOCUMENTS, "documents"},
	{CSC_POLICY_PROGRAMS, "programs"},
	{CSC_POLICY_DISABLE, "disable"},
	{-1, NULL}
};

/* Server role options */
static const struct enum_list enum_server_role[] = {
	{ROLE_AUTO, "auto"},
	{ROLE_STANDALONE, "standalone server"},
	{ROLE_STANDALONE, "standalone"},
	{ROLE_DOMAIN_MEMBER, "member server"},
	{ROLE_DOMAIN_MEMBER, "member"},
	{ROLE_DOMAIN_PDC, "classic primary domain controller"},
	{ROLE_DOMAIN_BDC, "classic backup domain controller"},
	{ROLE_ACTIVE_DIRECTORY_DC, "active directory domain controller"},
	{ROLE_ACTIVE_DIRECTORY_DC, "domain controller"},
	{ROLE_ACTIVE_DIRECTORY_DC, "dc"},
	{-1, NULL}
};

/* SMB signing types. */
static const struct enum_list enum_smb_signing_vals[] = {
	{SMB_SIGNING_DEFAULT, "default"},
	{SMB_SIGNING_OFF, "No"},
	{SMB_SIGNING_OFF, "False"},
	{SMB_SIGNING_OFF, "0"},
	{SMB_SIGNING_OFF, "Off"},
	{SMB_SIGNING_OFF, "disabled"},
	{SMB_SIGNING_IF_REQUIRED, "if_required"},
	{SMB_SIGNING_IF_REQUIRED, "Yes"},
	{SMB_SIGNING_IF_REQUIRED, "True"},
	{SMB_SIGNING_IF_REQUIRED, "1"},
	{SMB_SIGNING_IF_REQUIRED, "On"},
	{SMB_SIGNING_IF_REQUIRED, "enabled"},
	{SMB_SIGNING_IF_REQUIRED, "auto"},
	{SMB_SIGNING_DESIRED, "desired"},
	{SMB_SIGNING_REQUIRED, "required"},
	{SMB_SIGNING_REQUIRED, "mandatory"},
	{SMB_SIGNING_REQUIRED, "force"},
	{SMB_SIGNING_REQUIRED, "forced"},
	{SMB_SIGNING_REQUIRED, "enforced"},
	{-1, NULL}
};

/* DNS update options. */
static const struct enum_list enum_dns_update_settings[] = {
	{DNS_UPDATE_OFF, "disabled"},
	{DNS_UPDATE_OFF, "No"},
	{DNS_UPDATE_OFF, "False"},
	{DNS_UPDATE_OFF, "0"},
	{DNS_UPDATE_OFF, "Off"},
	{DNS_UPDATE_ON, "nonsecure and secure"},
	{DNS_UPDATE_ON, "nonsecure"},
	{DNS_UPDATE_SIGNED, "secure only"},
	{DNS_UPDATE_SIGNED, "secure"},
	{DNS_UPDATE_SIGNED, "signed"},
	{-1, NULL}
};

/*
   Do you want session setups at user level security with a invalid
   password to be rejected or allowed in as guest? WinNT rejects them
   but it can be a pain as it means "net view" needs to use a password

   You have 3 choices in the setting of map_to_guest:

   "Never" means session setups with an invalid password
   are rejected. This is the default.

   "Bad User" means session setups with an invalid password
   are rejected, unless the username does not exist, in which case it
   is treated as a guest login

   "Bad Password" means session setups with an invalid password
   are treated as a guest login

   Note that map_to_guest only has an effect in user or server
   level security.
*/

static const struct enum_list enum_map_to_guest[] = {
	{NEVER_MAP_TO_GUEST, "Never"},
	{MAP_TO_GUEST_ON_BAD_USER, "Bad User"},
	{MAP_TO_GUEST_ON_BAD_PASSWORD, "Bad Password"},
        {MAP_TO_GUEST_ON_BAD_UID, "Bad Uid"},
	{-1, NULL}
};

/* Config backend options */

static const struct enum_list enum_config_backend[] = {
	{CONFIG_BACKEND_FILE, "file"},
	{CONFIG_BACKEND_REGISTRY, "registry"},
	{-1, NULL}
};


/* ADS kerberos ticket verification options */

static const struct enum_list enum_kerberos_method[] = {
	{KERBEROS_VERIFY_SECRETS, "default"},
	{KERBEROS_VERIFY_SECRETS, "secrets only"},
	{KERBEROS_VERIFY_SYSTEM_KEYTAB, "system keytab"},
	{KERBEROS_VERIFY_DEDICATED_KEYTAB, "dedicated keytab"},
	{KERBEROS_VERIFY_SECRETS_AND_KEYTAB, "secrets and keytab"},
	{-1, NULL}
};

static const struct enum_list enum_printing[] = {
	{PRINT_SYSV, "sysv"},
	{PRINT_AIX, "aix"},
	{PRINT_HPUX, "hpux"},
	{PRINT_BSD, "bsd"},
	{PRINT_QNX, "qnx"},
	{PRINT_PLP, "plp"},
	{PRINT_LPRNG, "lprng"},
#ifdef HAVE_CUPS
	{PRINT_CUPS, "cups"},
#endif
#ifdef HAVE_IPRINT
	{PRINT_IPRINT, "iprint"},
#endif
	{PRINT_LPRNT, "nt"},
	{PRINT_LPROS2, "os2"},
#if defined(DEVELOPER) || defined(ENABLE_SELFTEST)
	{PRINT_TEST, "test"},
	{PRINT_VLP, "vlp"},
#endif /* DEVELOPER */
	{-1, NULL}
};

static const struct enum_list enum_ldap_sasl_wrapping[] = {
	{0, "plain"},
	{ADS_AUTH_SASL_SIGN, "sign"},
	{ADS_AUTH_SASL_SEAL, "seal"},
	{-1, NULL}
};

static const struct enum_list enum_ldap_ssl[] = {
	{LDAP_SSL_OFF, "no"},
	{LDAP_SSL_OFF, "off"},
	{LDAP_SSL_START_TLS, "start tls"},
	{LDAP_SSL_START_TLS, "start_tls"},
	{-1, NULL}
};

/* LDAP Dereferencing Alias types */
#define SAMBA_LDAP_DEREF_NEVER		0
#define SAMBA_LDAP_DEREF_SEARCHING	1
#define SAMBA_LDAP_DEREF_FINDING	2
#define SAMBA_LDAP_DEREF_ALWAYS		3

static const struct enum_list enum_ldap_deref[] = {
	{SAMBA_LDAP_DEREF_NEVER, "never"},
	{SAMBA_LDAP_DEREF_SEARCHING, "searching"},
	{SAMBA_LDAP_DEREF_FINDING, "finding"},
	{SAMBA_LDAP_DEREF_ALWAYS, "always"},
	{-1, "auto"}
};

static const struct enum_list enum_ldap_passwd_sync[] = {
	{LDAP_PASSWD_SYNC_OFF, "no"},
	{LDAP_PASSWD_SYNC_OFF, "off"},
	{LDAP_PASSWD_SYNC_ON, "yes"},
	{LDAP_PASSWD_SYNC_ON, "on"},
	{LDAP_PASSWD_SYNC_ONLY, "only"},
	{-1, NULL}
};

static const struct enum_list enum_map_readonly[] = {
	{MAP_READONLY_NO, "no"},
	{MAP_READONLY_NO, "false"},
	{MAP_READONLY_NO, "0"},
	{MAP_READONLY_YES, "yes"},
	{MAP_READONLY_YES, "true"},
	{MAP_READONLY_YES, "1"},
	{MAP_READONLY_PERMISSIONS, "permissions"},
	{MAP_READONLY_PERMISSIONS, "perms"},
	{-1, NULL}
};

static const struct enum_list enum_case[] = {
	{CASE_LOWER, "lower"},
	{CASE_UPPER, "upper"},
	{-1, NULL}
};


/* Note: We do not initialise the defaults union - it is not allowed in ANSI C
 *
 * The FLAG_HIDE is explicit. Parameters set this way do NOT appear in any edit
 * screen in SWAT. This is used to exclude parameters as well as to squash all
 * parameters that have been duplicated by pseudonyms.
 *
 * NOTE: To display a parameter in BASIC view set FLAG_BASIC
 *       Any parameter that does NOT have FLAG_ADVANCED will not disply at all
 *	 Set FLAG_SHARE and FLAG_PRINT to specifically display parameters in
 *        respective views.
 *
 * NOTE2: Handling of duplicated (synonym) parameters:
 *	Only the first occurance of a parameter should be enabled by FLAG_BASIC
 *	and/or FLAG_ADVANCED. All duplicates following the first mention should be
 *	set to FLAG_HIDE. ie: Make you must place the parameter that has the preferred
 *	name first, and all synonyms must follow it with the FLAG_HIDE attribute.
 */

#define GLOBAL_VAR(name) offsetof(struct loadparm_global, name)
#define LOCAL_VAR(name) offsetof(struct loadparm_service, name)


static struct parm_struct parm_table[] = {
	{N_("Base Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "dos charset",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(dos_charset),
		.special	= handle_dos_charset,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED
	},
	{
		.label		= "unix charset",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(unix_charset),
		.special	= handle_charset,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED
	},
	{
		.label		= "comment",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(comment),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT
	},
	{
		.label		= "path",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPath),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "directory",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPath),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "workgroup",
		.type		= P_USTRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWorkgroup),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "realm",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRealm),
		.special	= handle_realm,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "netbios name",
		.type		= P_USTRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNetbiosName),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "netbios aliases",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNetbiosAliases),
		.special	= handle_netbios_aliases,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "netbios scope",
		.type		= P_USTRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNetbiosScope),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "server string",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szServerString),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "interfaces",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szInterfaces),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "bind interfaces only",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bBindInterfacesOnly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "config backend",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ConfigBackend),
		.special	= NULL,
		.enum_list	= enum_config_backend,
		.flags		= FLAG_HIDE|FLAG_ADVANCED|FLAG_META,
	},
	{
		.label		= "server role",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(server_role),
		.special	= NULL,
		.enum_list	= enum_server_role,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},

	{N_("Security Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "security",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(security),
		.special	= NULL,
		.enum_list	= enum_security,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "auth methods",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(AuthMethods),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "encrypt passwords",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bEncryptPasswords),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "client schannel",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(clientSchannel),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "server schannel",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(serverSchannel),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "allow trusted domains",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bAllowTrustedDomains),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "map to guest",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(map_to_guest),
		.special	= NULL,
		.enum_list	= enum_map_to_guest,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "null passwords",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNullPasswords),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "obey pam restrictions",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bObeyPamRestrictions),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "password server",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPasswordServer),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "smb passwd file",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSMBPasswdFile),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "private dir",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPrivateDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "private directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPrivateDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "passdb backend",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(passdb_backend),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "algorithmic rid base",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(AlgorithmicRidBase),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "root directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRootdir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "root dir",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRootdir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "root",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRootdir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "guest account",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szGuestaccount),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "enable privileges",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bEnablePrivileges),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},

	{
		.label		= "pam password change",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bPamPasswordChange),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "passwd program",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPasswdProgram),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "passwd chat",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPasswdChat),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "passwd chat debug",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bPasswdChatDebug),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "passwd chat timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iPasswdChatTimeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "check password script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szCheckPasswordScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "username map",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsernameMap),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "username level",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(unamelevel),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "unix password sync",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUnixPasswdSync),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "restrict anonymous",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(restrict_anonymous),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "lanman auth",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bLanmanAuth),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ntlm auth",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNTLMAuth),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client NTLMv2 auth",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bClientNTLMv2Auth),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client lanman auth",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bClientLanManAuth),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client plaintext auth",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bClientPlaintextAuth),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client use spnego principal",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(client_use_spnego_principal),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "username",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szUsername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE | FLAG_DEPRECATED,
	},
	{
		.label		= "user",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szUsername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "users",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szUsername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "invalid users",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szInvalidUsers),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "valid users",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szValidUsers),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "admin users",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szAdminUsers),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "read list",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(readlist),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "write list",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(writelist),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "force user",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(force_user),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "force group",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(force_group),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "group",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(force_group),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "read only",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bRead_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "write ok",
		.type		= P_BOOLREV,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bRead_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "writeable",
		.type		= P_BOOLREV,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bRead_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "writable",
		.type		= P_BOOLREV,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bRead_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "acl check permissions",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAclCheckPermissions),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE | FLAG_DEPRECATED,
	},
	{
		.label		= "acl group control",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAclGroupControl),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "acl map full control",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAclMapFullControl),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "acl allow execute always",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAclAllowExecuteAlways),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},

	{
		.label		= "create mask",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCreate_mask),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "create mode",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCreate_mask),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "force create mode",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCreate_force_mode),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "directory mask",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDir_mask),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "directory mode",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDir_mask),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "force directory mode",
		.type		= P_OCTAL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDir_force_mode),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "force unknown acl user",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bForceUnknownAclUser),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "inherit permissions",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bInheritPerms),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "inherit acls",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bInheritACLS),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "inherit owner",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bInheritOwner),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "guest only",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bGuest_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "only guest",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bGuest_only),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "administrative share",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAdministrative_share),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},

	{
		.label		= "guest ok",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bGuest_ok),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "public",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bGuest_ok),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "only user",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bOnlyUser),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_DEPRECATED,
	},
	{
		.label		= "hosts allow",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szHostsallow),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_GLOBAL | FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "allow hosts",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szHostsallow),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "hosts deny",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szHostsdeny),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_GLOBAL | FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "deny hosts",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szHostsdeny),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "preload modules",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPreloadModules),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "dedicated keytab file",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDedicatedKeytabFile),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "kerberos method",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iKerberosMethod),
		.special	= NULL,
		.enum_list	= enum_kerberos_method,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "map untrusted to domain",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bMapUntrustedToDomain),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},


	{N_("Logging Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "log level",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(loglevel),
		.special	= handle_debug_list,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debuglevel",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(loglevel),
		.special	= handle_debug_list,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "syslog",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(syslog),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "syslog only",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bSyslogOnly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "log file",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(logfile),
		.special	= handle_logfile,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max log size",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_log_size),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug timestamp",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bTimestampLogs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "timestamp logs",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bTimestampLogs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug prefix timestamp",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDebugPrefixTimestamp),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug hires timestamp",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDebugHiresTimestamp),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug pid",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDebugPid),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug uid",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDebugUid),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "debug class",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDebugClass),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "enable core files",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bEnableCoreFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("Protocol Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "allocation roundup size",
		.type		= P_BYTES,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iallocation_roundup_size),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "aio read size",
		.type		= P_BYTES,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iAioReadSize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "aio write size",
		.type		= P_BYTES,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iAioWriteSize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "aio write behind",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szAioWriteBehind),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "smb ports",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(smb_ports),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "large readwrite",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bLargeReadwrite),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "server max protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(srv_maxprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(srv_maxprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(srv_maxprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "server min protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(srv_minprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "min protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(srv_minprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client max protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(cli_maxprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client min protocol",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(cli_minprotocol),
		.special	= NULL,
		.enum_list	= enum_protocol,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "unicode",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUnicode),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "min receivefile size",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iminreceivefile),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "read raw",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bReadRaw),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "write raw",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWriteRaw),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "disable netbios",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDisableNetbios),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "reset on zero vc",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bResetOnZeroVC),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "log writeable files on exit",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bLogWriteableFilesOnExit),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "defer sharing violations",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDeferSharingViolations),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "ea support",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bEASupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "nt acl support",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bNTAclSupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "nt pipe support",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNTPipeSupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "nt status support",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNTStatusSupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "profile acls",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bProfileAcls),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},
	{
		.label		= "map acl inherit",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMap_acl_inherit),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "afs share",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAfs_Share),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "max mux",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_mux),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max xmit",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_xmit),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "name resolve order",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNameResolveOrder),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "max ttl",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_ttl),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max wins ttl",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_wins_ttl),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "min wins ttl",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(min_wins_ttl),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "time server",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bTimeServer),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "unix extensions",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUnixExtensions),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "use spnego",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUseSpnego),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "client signing",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(client_signing),
		.special	= NULL,
		.enum_list	= enum_smb_signing_vals,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "server signing",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(server_signing),
		.special	= NULL,
		.enum_list	= enum_smb_signing_vals,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "smb encrypt",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(ismb_encrypt),
		.special	= NULL,
		.enum_list	= enum_smb_signing_vals,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client use spnego",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bClientUseSpnego),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "client ldap sasl wrapping",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(client_ldap_sasl_wrapping),
		.special	= NULL,
		.enum_list	= enum_ldap_sasl_wrapping,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "enable asu support",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bASUSupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "svcctl list",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szServicesList),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "cldap port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(cldap_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "dgram port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(dgram_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "nbt port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(nbt_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "krb5 port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(krb5_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "kpasswd port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(kpasswd_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "web port",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(web_port),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "rpc big endian",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bRpcBigEndian),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "durable handles",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDurableHandles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("Tuning Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "block size",
		.type		= P_BYTES,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iBlock_size),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "deadtime",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(deadtime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "getwd cache",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(getwd_cache),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "keepalive",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iKeepalive),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "change notify",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bChangeNotify),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "directory name cache size",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDirectoryNameCacheSize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "kernel change notify",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bKernelChangeNotify),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "lpq cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(lpqcachetime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max smbd processes",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iMaxSmbdProcesses),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max connections",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iMaxConnections),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "max disk size",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(maxdisksize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "max open files",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(max_open_files),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "min print space",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iMinPrintSpace),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "socket options",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(socket_options),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "strict allocate",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bStrictAllocate),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "strict sync",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bStrictSync),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "sync always",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bSyncAlways),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "use mmap",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUseMmap),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "use ntdb",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUseNtdb),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "use sendfile",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bUseSendfile),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "hostname lookups",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bHostnameLookups),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "write cache size",
		.type		= P_BYTES,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iWriteCacheSize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "name cache timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(name_cache_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ctdbd socket",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ctdbdSocket),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "cluster addresses",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szClusterAddresses),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "clustering",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(clustering),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "ctdb timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ctdb_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "ctdb locktime warn threshold",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ctdb_locktime_warn_threshold),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "smb2 max read",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ismb2_max_read),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "smb2 max write",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ismb2_max_write),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "smb2 max trans",
		.type		= P_BYTES,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ismb2_max_trans),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "smb2 max credits",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ismb2_max_credits),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("Printing Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "max reported print jobs",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iMaxReportedPrintJobs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "max print jobs",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iMaxPrintJobs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "load printers",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bLoadPrinters),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "printcap cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(PrintcapCacheTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "printcap name",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPrintcapname),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "printcap",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPrintcapname),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "printable",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bPrint_ok),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "print notify backchannel",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bPrintNotifyBackchannel),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "print ok",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bPrint_ok),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "printing",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iPrinting),
		.special	= handle_printing,
		.enum_list	= enum_printing,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "cups options",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szCupsOptions),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "cups server",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szCupsServer),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label          = "cups encrypt",
		.type           = P_ENUM,
		.p_class        = P_GLOBAL,
		.offset         = GLOBAL_VAR(CupsEncrypt),
		.special        = NULL,
		.enum_list      = enum_bool_auto,
		.flags          = FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{

		.label		= "cups connection timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(cups_connection_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "iprint server",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIPrintServer),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "print command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPrintcommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "disable spoolss",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDisableSpoolss),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "enable spoolss",
		.type		= P_BOOLREV,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDisableSpoolss),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "lpq command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szLpqcommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "lprm command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szLprmcommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "lppause command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szLppausecommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "lpresume command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szLpresumecommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "queuepause command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szQueuepausecommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "queueresume command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szQueueresumecommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT | FLAG_GLOBAL,
	},
	{
		.label		= "addport command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddPortCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "enumports command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szEnumPortsCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "addprinter command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddPrinterCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "deleteprinter command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDeletePrinterCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "show add printer wizard",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bMsAddPrinterWizard),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "os2 driver map",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szOs2DriverMap),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{
		.label		= "printer name",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPrintername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "printer",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPrintername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "use client driver",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bUseClientDriver),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "default devmode",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDefaultDevmode),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "force printername",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bForcePrintername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},
	{
		.label		= "printjob username",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPrintjobUsername),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_PRINT,
	},

	{N_("Filename Handling"), P_SEP, P_SEPARATOR},

	{
		.label		= "mangling method",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szManglingMethod),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "mangle prefix",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(mangle_prefix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{
		.label		= "default case",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDefaultCase),
		.special	= NULL,
		.enum_list	= enum_case,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "case sensitive",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCaseSensitive),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "casesignames",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCaseSensitive),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL | FLAG_HIDE,
	},
	{
		.label		= "preserve case",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bCasePreserve),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "short preserve case",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bShortCasePreserve),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "mangling char",
		.type		= P_CHAR,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(magic_char),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "hide dot files",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bHideDotFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "hide special files",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bHideSpecialFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "hide unreadable",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bHideUnReadable),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "hide unwriteable files",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bHideUnWriteableFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "delete veto files",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDeleteVetoFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "veto files",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szVetoFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "hide files",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szHideFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "veto oplock files",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szVetoOplockFiles),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "map archive",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMap_archive),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "map hidden",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMap_hidden),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "map system",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMap_system),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "map readonly",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iMap_readonly),
		.special	= NULL,
		.enum_list	= enum_map_readonly,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "mangled names",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMangledNames),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "max stat cache size",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iMaxStatCacheSize),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "stat cache",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bStatCache),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "store dos attributes",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bStoreDosAttributes),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "dmapi support",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDmapiSupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},


	{N_("Domain Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "machine password timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(machine_password_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_WIZARD,
	},

	{N_("Logon Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "add user script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddUserScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "rename user script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRenameUserScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "delete user script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDelUserScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "add group script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddGroupScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "delete group script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDelGroupScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "add user to group script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddUserToGroupScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "delete user from group script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDelUserFromGroupScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "set primary group script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSetPrimaryGroupScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "add machine script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddMachineScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "shutdown script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szShutdownScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "abort shutdown script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAbortShutdownScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "username map script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsernameMapScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "username map cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iUsernameMapCacheTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "logon script",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLogonScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "logon path",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLogonPath),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "logon drive",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLogonDrive),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "logon home",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLogonHome),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "domain logons",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bDomainLogons),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{
		.label		= "init logon delayed hosts",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szInitLogonDelayedHosts),
		.special        = NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{
		.label		= "init logon delay",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(InitLogonDelay),
		.special        = NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,

	},

	{N_("Browse Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "os level",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(os_level),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "lm announce",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(lm_announce),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "lm interval",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(lm_interval),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "preferred master",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iPreferredMaster),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "prefered master",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iPreferredMaster),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "local master",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bLocalMaster),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "domain master",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(domain_master),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_BASIC | FLAG_ADVANCED,
	},
	{
		.label		= "browse list",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bBrowseList),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "browseable",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bBrowseable),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "browsable",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bBrowseable),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "access based share enum",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAccessBasedShareEnum),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE
	},
	{
		.label		= "enhanced browsing",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(enhanced_browsing),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("WINS Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "dns proxy",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWINSdnsProxy),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "wins proxy",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWINSproxy),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "wins server",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWINSservers),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "wins support",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWINSsupport),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_WIZARD,
	},
	{
		.label		= "wins hook",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWINSHook),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("Locking Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "blocking locks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bBlockingLocks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "csc policy",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iCSCPolicy),
		.special	= NULL,
		.enum_list	= enum_csc_policy,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "fake oplocks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bFakeOplocks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "kernel oplocks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bKernelOplocks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "kernel share modes",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bKernelShareModes),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "locking",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bLocking),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "lock spin time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iLockSpinTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "oplocks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bOpLocks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "level2 oplocks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bLevel2OpLocks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "oplock break wait time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(oplock_break_wait_time),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "oplock contention limit",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iOplockContentionLimit),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "posix locking",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bPosixLocking),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "strict locking",
		.type		= P_ENUM,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iStrictLocking),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},

	{N_("Ldap Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "ldap admin dn",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapAdminDn),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap delete dn",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_delete_dn),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap group suffix",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapGroupSuffix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap idmap suffix",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapIdmapSuffix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap machine suffix",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapMachineSuffix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap passwd sync",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_passwd_sync),
		.special	= NULL,
		.enum_list	= enum_ldap_passwd_sync,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap password sync",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_passwd_sync),
		.special	= NULL,
		.enum_list	= enum_ldap_passwd_sync,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "ldap replication sleep",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_replication_sleep),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap suffix",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapSuffix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap ssl",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_ssl),
		.special	= NULL,
		.enum_list	= enum_ldap_ssl,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap ssl ads",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_ssl_ads),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap deref",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_deref),
		.special	= NULL,
		.enum_list	= enum_ldap_deref,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap follow referral",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_follow_referral),
		.special	= NULL,
		.enum_list	= enum_bool_auto,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap connection timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_connection_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap page size",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_page_size),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap user suffix",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLdapUserSuffix),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap debug level",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_debug_level),
		.special	= handle_ldap_debug_level,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ldap debug threshold",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ldap_debug_threshold),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("EventLog Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "eventlog list",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szEventLogs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL | FLAG_SHARE,
	},

	{N_("Miscellaneous Options"), P_SEP, P_SEPARATOR},

	{
		.label		= "add share command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAddShareCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "change share command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szChangeShareCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "delete share command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDeleteShareCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "config file",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szConfigFile),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE|FLAG_META,
	},
	{
		.label		= "preload",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAutoServices),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "auto services",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAutoServices),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "lock directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLockDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "lock dir",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLockDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "state directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szStateDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "cache directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szCacheDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "pid directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPidDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ntp signd socket directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNTPSignDSocketDirectory),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

#ifdef WITH_UTMP
	{
		.label		= "utmp directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUtmpDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "wtmp directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWtmpDir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "utmp",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUtmp),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
#endif
	{
		.label		= "default service",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDefaultService),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "default",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDefaultService),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "message command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szMsgCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "dfree cache time",
		.type		= P_INTEGER,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(iDfreeCacheTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "dfree command",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szDfree),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "get quota command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szGetQuota),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "set quota command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSetQuota),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "remote announce",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRemoteAnnounce),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "remote browse sync",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRemoteBrowseSync),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "nbt client socket address",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(nbt_client_socket_address),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "socket address",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(nbt_client_socket_address),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "nmbd bind explicit broadcast",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNmbdBindExplicitBroadcast),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "homedir map",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNISHomeMapName),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "afs username map",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szAfsUsernameMap),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "afs token lifetime",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iAfsTokenLifetime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "log nt token command",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szLogNtTokenCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "NIS homedir",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bNISHomeMap),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "-valid",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(valid),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "copy",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szCopy),
		.special	= handle_copy,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "include",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szInclude),
		.special	= handle_include,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE|FLAG_META,
	},
	{
		.label		= "preexec",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPreExec),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "exec",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPreExec),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "preexec close",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bPreexecClose),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "postexec",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szPostExec),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "root preexec",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szRootPreExec),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "root preexec close",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bRootpreexecClose),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "root postexec",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szRootPostExec),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "available",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bAvailable),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_BASIC | FLAG_ADVANCED | FLAG_SHARE | FLAG_PRINT,
	},
	{
		.label		= "registry shares",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bRegistryShares),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare allow guests",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUsershareAllowGuests),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare max shares",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iUsershareMaxShares),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare owner only",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bUsershareOwnerOnly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare path",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsersharePath),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare prefix allow list",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsersharePrefixAllowList),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare prefix deny list",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsersharePrefixDenyList),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "usershare template share",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szUsershareTemplateShare),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "volume",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(volume),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "fstype",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(fstype),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "allow insecure wide links",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bAllowInsecureWidelinks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "wide links",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bWidelinks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "follow symlinks",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bSymlinks),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "dont descend",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szDontdescend),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "magic script",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szMagicScript),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "magic output",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szMagicOutput),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "delete readonly",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDeleteReadonly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "dos filemode",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDosFilemode),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "dos filetimes",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDosFiletimes),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "dos filetime resolution",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bDosFiletimeResolution),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE | FLAG_GLOBAL,
	},
	{
		.label		= "fake directory create times",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bFakeDirCreateTimes),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "async smb echo handler",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bAsyncSMBEchoHandler),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},
	{
		.label		= "panic action",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szPanicAction),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "perfcount module",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSMBPerfcountModule),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("VFS module options"), P_SEP, P_SEPARATOR},

	{
		.label		= "vfs objects",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szVfsObjects),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "vfs object",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szVfsObjects),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},


	{N_("MSDFS options"), P_SEP, P_SEPARATOR},

	{
		.label		= "msdfs root",
		.type		= P_BOOL,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(bMSDfsRoot),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "msdfs proxy",
		.type		= P_STRING,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(szMSDfsProxy),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_SHARE,
	},
	{
		.label		= "host msdfs",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bHostMSDfs),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("Winbind options"), P_SEP, P_SEPARATOR},

	{
		.label		= "passdb expand explicit",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bPassdbExpandExplicit),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "idmap backend",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIdmapBackend),
		.special	= handle_idmap_backend,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "idmap cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iIdmapCacheTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "idmap negative cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(iIdmapNegativeCacheTime),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "idmap uid",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIdmapUID),
		.special	= handle_idmap_uid,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "winbind uid",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIdmapUID),
		.special	= handle_idmap_uid,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "idmap gid",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIdmapGID),
		.special	= handle_idmap_gid,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_DEPRECATED,
	},
	{
		.label		= "winbind gid",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szIdmapGID),
		.special	= handle_idmap_gid,
		.enum_list	= NULL,
		.flags		= FLAG_HIDE,
	},
	{
		.label		= "template homedir",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szTemplateHomedir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "template shell",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szTemplateShell),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind separator",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWinbindSeparator),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind cache time",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbind_cache_time),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind reconnect delay",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbind_reconnect_delay),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind request timeout",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbind_request_timeout),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind max clients",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbind_max_clients),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind enum users",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindEnumUsers),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind enum groups",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindEnumGroups),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind use default domain",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindUseDefaultDomain),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind trusted domains only",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindTrustedDomainsOnly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind nested groups",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindNestedGroups),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind expand groups",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbind_expand_groups),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind nss info",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWinbindNssInfo),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind refresh tickets",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindRefreshTickets),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind offline logon",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindOfflineLogon),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind normalize names",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindNormalizeNames),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind rpc only",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindRpcOnly),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "create krb5 conf",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bCreateKrb5Conf),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "ncalrpc dir",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(ncalrpc_dir),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind max domain connections",
		.type		= P_INTEGER,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(winbindMaxDomainConnections),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbindd socket directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWinbinddSocketDirectory),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbindd privileged socket directory",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szWinbinddPrivilegedSocketDirectory),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "winbind sealed pipes",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bWinbindSealedPipes),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},

	{N_("DNS options"), P_SEP, P_SEPARATOR},
	{
		.label		= "allow dns updates",
		.type		= P_ENUM,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(allow_dns_updates),
		.special	= NULL,
		.enum_list	= enum_dns_update_settings,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "dns forwarder",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(dns_forwarder),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "dns update command",
		.type		= P_CMDLIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szDNSUpdateCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "nsupdate command",
		.type		= P_CMDLIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szNSUpdateCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "rndc command",
		.type		= P_CMDLIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szRNDCCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "multicast dns register",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(bMulticastDnsRegister),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED | FLAG_GLOBAL,
	},

	{N_("AD DC options"), P_SEP, P_SEPARATOR},

	{
		.label		= "samba kcc command",
		.type		= P_CMDLIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSambaKCCCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "server services",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(server_services),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "dcerpc endpoint servers",
		.type		= P_LIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(dcerpc_ep_servers),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "spn update command",
		.type		= P_CMDLIST,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szSPNUpdateCommand),
		.special	= NULL,
		.enum_list	= NULL,
		.flags		= FLAG_ADVANCED,
	},
	{
		.label		= "share backend",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(szShareBackend),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "ntvfs handler",
		.type		= P_LIST,
		.p_class	= P_LOCAL,
		.offset		= LOCAL_VAR(ntvfs_handler),
		.special	= NULL,
		.enum_list	= NULL
	},

	{N_("TLS options"), P_SEP, P_SEPARATOR},

	{
		.label		= "tls enabled",
		.type		= P_BOOL,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_enabled),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "tls keyfile",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_keyfile),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "tls certfile",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_certfile),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "tls cafile",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_cafile),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "tls crlfile",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_crlfile),
		.special	= NULL,
		.enum_list	= NULL
	},
	{
		.label		= "tls dh params file",
		.type		= P_STRING,
		.p_class	= P_GLOBAL,
		.offset		= GLOBAL_VAR(tls_dhpfile),
		.special	= NULL,
		.enum_list	= NULL
	},

	{NULL,  P_BOOL,  P_NONE,  0,  NULL,  NULL,  0}
};

