# -*- coding: utf-8 -*-
# $Id: vboxwrappers.py $
# pylint: disable=C0302

"""
VirtualBox Wrapper Classes
"""

__copyright__ = \
"""
Copyright (C) 2010-2015 Oracle Corporation

This file is part of VirtualBox Open Source Edition (OSE), as
available from http://www.virtualbox.org. This file is free software;
you can redistribute it and/or modify it under the terms of the GNU
General Public License (GPL) as published by the Free Software
Foundation, in version 2 as it comes in the "COPYING" file of the
VirtualBox OSE distribution. VirtualBox OSE is distributed in the
hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.

The contents of this file may alternatively be used under the terms
of the Common Development and Distribution License Version 1.0
(CDDL) only, as it comes in the "COPYING.CDDL" file of the
VirtualBox OSE distribution, in which case the provisions of the
CDDL are applicable instead of those of the GPL.

You may elect to license modified versions of this file under the
terms and conditions of either the GPL or the CDDL or both.
"""
__version__ = "$Revision: 100512 $"


# Standard Python imports.
import array
import os
import socket
import sys

# Validation Kit imports.
from common     import utils;
from testdriver import base;
from testdriver import reporter;
from testdriver import txsclient;
from testdriver import vboxcon;
from testdriver import vbox;
from testdriver.base    import TdTaskBase;


def _ControllerNameToBus(sController):
    """ Translate a controller name to a storage bus. """
    if sController == "IDE Controller":
        iType = vboxcon.StorageBus_IDE;
    elif sController == "SATA Controller":
        iType = vboxcon.StorageBus_SATA;
    elif sController == "Floppy Controller":
        iType = vboxcon.StorageBus_Floppy;
    elif sController == "SAS Controller":
        iType = vboxcon.StorageBus_SAS;
    elif sController == "SCSI Controller":
        iType = vboxcon.StorageBus_SCSI;
    else:
        iType = vboxcon.StorageBus_Null;
    return iType;

def _nameMachineState(eState):
    """ Gets the name (string) of a machine state."""
    if eState == vboxcon.MachineState_PoweredOff: return 'PoweredOff';
    if eState == vboxcon.MachineState_Saved: return 'Saved';
    if eState == vboxcon.MachineState_Teleported: return 'Teleported';
    if eState == vboxcon.MachineState_Aborted: return 'Aborted';
    if eState == vboxcon.MachineState_Running: return 'Running';
    if eState == vboxcon.MachineState_Paused: return 'Paused';
    if eState == vboxcon.MachineState_Stuck: return 'GuruMeditation';
    if eState == vboxcon.MachineState_Teleporting: return 'Teleporting';
    if eState == vboxcon.MachineState_LiveSnapshotting: return 'LiveSnapshotting';
    if eState == vboxcon.MachineState_Starting: return 'Starting';
    if eState == vboxcon.MachineState_Stopping: return 'Stopping';
    if eState == vboxcon.MachineState_Saving: return 'Saving';
    if eState == vboxcon.MachineState_Restoring: return 'Restoring';
    if eState == vboxcon.MachineState_TeleportingPausedVM: return 'TeleportingPausedVM';
    if eState == vboxcon.MachineState_TeleportingIn: return 'TeleportingIn';
    if eState == vboxcon.MachineState_FaultTolerantSyncing: return 'FaultTolerantSyncing';
    if eState == vboxcon.MachineState_DeletingSnapshotOnline: return 'DeletingSnapshotOnline';
    if eState == vboxcon.MachineState_DeletingSnapshotPaused: return 'DeletingSnapshotPaused';
    if eState == vboxcon.MachineState_RestoringSnapshot: return 'RestoringSnapshot';
    if eState == vboxcon.MachineState_DeletingSnapshot: return 'DeletingSnapshot';
    if eState == vboxcon.MachineState_SettingUp: return 'SettingUp';
    return 'Unknown-%s' % (eState,);


class VirtualBoxWrapper(object): # pylint: disable=R0903
    """
    Wrapper around the IVirtualBox object that adds some (hopefully) useful
    utility methods

    The real object can be accessed thru the o member.  That said, members can
    be accessed directly as well.
    """

    def __init__(self, oVBox, oVBoxMgr, fpApiVer, oTstDrv):
        self.o          = oVBox;
        self.oVBoxMgr   = oVBoxMgr;
        self.fpApiVer   = fpApiVer;
        self.oTstDrv    = oTstDrv;

    def __getattr__(self, sName):
        # Try ourselves first.
        try:
            oAttr = self.__dict__[sName];
        except:
            #try:
            #    oAttr = dir(self)[sName];
            #except AttributeError:
            oAttr = getattr(self.o, sName);
        return oAttr;

    #
    # Utilities.
    #

    def registerDerivedEventHandler(self, oSubClass, dArgs = None):
        """
        Create an instance of the given VirtualBoxEventHandlerBase sub-class
        and register it.

        The new instance is returned on success.  None is returned on error.
        """
        dArgsCopy = dArgs.copy() if dArgs is not None else dict();
        dArgsCopy['oVBox'] = self;
        return oSubClass.registerDerivedEventHandler(self.oVBoxMgr, self.fpApiVer, oSubClass, dArgsCopy,
                                                     self.o, 'IVirtualBox', 'IVirtualBoxCallback');

    def deleteHdByLocation(self, sHdLocation):
        """
        Deletes a disk image from the host, given it's location.
        Returns True on success and False on failure. Error information is logged.
        """
        try:
            oIMedium = self.oVBox.findHardDisk(sHdLocation);
        except:
            try:
                if self.fpApiVer >= 4.1:
                    oIMedium = self.oVBox.openMedium(sHdLocation, vboxcon.DeviceType_HardDisk,
                                                     vboxcon.AccessMode_ReadWrite, False);
                elif self.fpApiVer >= 4.0:
                    oIMedium = self.oVBox.openMedium(sHdLocation, vboxcon.DeviceType_HardDisk,
                                                     vboxcon.AccessMode_ReadWrite);
                else:
                    oIMedium = self.oVBox.openHardDisk(sHdLocation, vboxcon.AccessMode_ReadOnly, False, "", False, "");
            except:
                return reporter.errorXcpt('failed to open hd "%s"' % (sHdLocation));
        return self.deleteHdByMedium(oIMedium)

    def deleteHdByMedium(self, oIMedium):
        """
        Deletes a disk image from the host, given an IMedium reference.
        Returns True on success and False on failure. Error information is logged.
        """
        try:    oProgressCom = oIMedium.deleteStorage();
        except: return reporter.errorXcpt('deleteStorage() for disk %s failed' % (oIMedium,));
        try:    oProgress = ProgressWrapper(oProgressCom, self.oVBoxMgr, self.oTstDrv, 'delete disk %s' % (oIMedium.location));
        except: return reporter.errorXcpt();
        oProgress.wait();
        oProgress.logResult();
        return oProgress.isSuccess();



class ProgressWrapper(TdTaskBase):
    """
    Wrapper around a progress object for making it a task and providing useful
    utility methods.
    The real progress object can be accessed thru the o member.
    """

    def __init__(self, oProgress, oVBoxMgr, oTstDrv, sName):
        TdTaskBase.__init__(self, utils.getCallerName());
        self.o          = oProgress;
        self.oVBoxMgr   = oVBoxMgr;
        self.oTstDrv    = oTstDrv;
        self.sName      = sName;

    def toString(self):
        return '<%s sName=%s, oProgress=%s >' \
             % (TdTaskBase.toString(self), self.sName, self.o);

    #
    # TdTaskBase overrides.
    #

    def pollTask(self, fLocked = False):
        """
        Overrides TdTaskBase.pollTask().

        This method returns False until the progress object has completed.
        """
        self.doQuickApiTest();
        try:
            try:
                if self.o.completed:
                    return True;
            except:
                pass;
        finally:
            self.oTstDrv.processPendingEvents();
        return False;

    def waitForTask(self, cMsTimeout = 0):
        """
        Overrides TdTaskBase.waitForTask().
        Process XPCOM/COM events while waiting.
        """
        msStart = base.timestampMilli();
        fState  = self.pollTask(False);
        while not fState:
            cMsElapsed = base.timestampMilli() - msStart;
            if cMsElapsed > cMsTimeout:
                break;
            cMsToWait = cMsTimeout - cMsElapsed;
            if cMsToWait > 500:
                cMsToWait = 500;
            try:
                self.o.waitForCompletion(cMsToWait);
            except KeyboardInterrupt: raise;
            except: pass;
            fState = self.pollTask(False);
        return fState;

    #
    # Utility methods.
    #

    def isSuccess(self):
        """
        Tests if the progress object completed successfully.
        Returns True on success, False on failure or incomplete.
        """
        if not self.isCompleted():
            return False;
        return self.getResult() >= 0;

    def isCompleted(self):
        """
        Wrapper around IProgress.completed.
        """
        return self.pollTask();

    def isCancelable(self):
        """
        Wrapper around IProgress.cancelable.
        """
        try:
            fRc = self.o.cancelable;
        except:
            reporter.logXcpt();
            fRc = False;
        return fRc;

    def wasCanceled(self):
        """
        Wrapper around IProgress.canceled.
        """
        try:
            fRc = self.o.canceled;
        except:
            reporter.logXcpt(self.sName);
            fRc = False;
        return fRc;

    def cancel(self):
        """
        Wrapper around IProgress.cancel()
        Returns True on success, False on failure (logged as error).
        """
        try:
            self.o.cancel();
        except:
            reporter.errorXcpt(self.sName);
            return False;
        return True;

    def getResult(self):
        """
        Wrapper around IProgress.resultCode.
        """
        try:
            iRc = self.o.resultCode;
        except:
            reporter.logXcpt(self.sName);
            iRc = -1;
        return iRc;

    def getErrInfoResultCode(self):
        """
        Wrapper around IProgress.errorInfo.resultCode.

        Returns the string on success, -1 on bad objects (logged as error), and
        -2 on missing errorInfo object.
        """
        iRc = -1;
        try:
            oErrInfo = self.o.errorInfo;
        except:
            reporter.errorXcpt(self.sName);
        else:
            if oErrInfo is None:
                iRc = -2;
            else:
                try:
                    iRc = oErrInfo.resultCode;
                except:
                    reporter.errorXcpt();
        return iRc;

    def getErrInfoText(self):
        """
        Wrapper around IProgress.errorInfo.text.

        Returns the string on success, None on failure.  Missing errorInfo is
        not logged as an error, all other failures are.
        """
        sText = None;
        try:
            oErrInfo = self.o.errorInfo;
        except:
            reporter.log2Xcpt(self.sName);
        else:
            if oErrInfo is not None:
                try:
                    sText = oErrInfo.text;
                except:
                    reporter.errorXcpt();
        return sText;

    def stringifyErrorInfo(self):
        """
        Formats IProgress.errorInfo into a string.
        """
        try:
            oErrInfo = self.o.errorInfo;
        except:
            reporter.logXcpt(self.sName);
            sErr = 'no error info';
        else:
            sErr = vbox.stringifyErrorInfo(oErrInfo);
        return sErr;

    def stringifyResult(self):
        """
        Stringify the result.
        """
        if self.isCompleted():
            if self.wasCanceled():
                sRet = 'Progress %s: Canceled, hrc=%s' % (self.sName, vbox.ComError.toString(self.getResult()));
            elif self.getResult() == 0:
                sRet = 'Progress %s: Success' % (self.sName,);
            elif self.getResult() > 0:
                sRet = 'Progress %s: Success (hrc=%s)' % (self.sName, vbox.ComError.toString(self.getResult()));
            else:
                sRet = 'Progress %s: Failed! %s' % (self.sName, self.stringifyErrorInfo());
        else:
            sRet = 'Progress %s: Not completed yet...' % (self.sName);
        return sRet;

    def logResult(self, fIgnoreErrors = False):
        """ Logs the result. """
        sText = self.stringifyResult();
        if      self.isCompleted() and self.getResult() < 0 \
            and fIgnoreErrors is False:
            return reporter.error(sText);
        return reporter.log(sText);

    def waitOnProgress(self, cMsInterval = 1000):
        """
        See vbox.TestDriver.waitOnProgress.
        """
        self.doQuickApiTest();
        return self.oTstDrv.waitOnProgress(self.o, cMsInterval);

    def wait(self, cMsTimeout = 60000, fErrorOnTimeout = True, cMsInterval = 1000):
        """
        Wait on the progress object for a while.

        Returns the resultCode of the progress object if completed.
        Returns -1 on timeout, logged as error if fErrorOnTimeout is set.
        Returns -2 is the progress object is invalid or waitForCompletion
        fails (logged as errors).
        """
        msStart = base.timestampMilli();
        while True:
            self.oTstDrv.processPendingEvents();
            self.doQuickApiTest();
            try:
                if self.o.completed:
                    break;
            except:
                reporter.errorXcpt(self.sName);
                return -2;
            self.oTstDrv.processPendingEvents();

            cMsElapsed = base.timestampMilli() - msStart;
            if cMsElapsed > cMsTimeout:
                if fErrorOnTimeout:
                    reporter.error('Timing out after waiting for %u s on "%s"' % (cMsTimeout / 1000, self.sName))
                return -1;

            try:
                self.o.waitForCompletion(cMsInterval);
            except:
                reporter.errorXcpt(self.sName);
                return -2;

        try:
            rc = self.o.resultCode;
        except:
            rc = -2;
            reporter.errorXcpt(self.sName);
        self.oTstDrv.processPendingEvents();
        return rc;

    def waitForOperation(self, iOperation, cMsTimeout = 60000, fErrorOnTimeout = True, cMsInterval = 1000, \
                         fIgnoreErrors = False):
        """
        Wait for the completion of a operation.

        Negative iOperation values are relative to operationCount (this
        property may changed at runtime).

        Returns 0 if the operation completed normally.
        Returns -1 on timeout, logged as error if fErrorOnTimeout is set.
        Returns -2 is the progress object is invalid or waitForCompletion
        fails (logged as errors).
        Returns -3 if if the operation completed with an error, this is logged
        as an error.
        """
        msStart = base.timestampMilli();
        while True:
            self.oTstDrv.processPendingEvents();
            self.doQuickApiTest();
            try:
                iCurrentOperation = self.o.operation;
                cOperations       = self.o.operationCount;
                if iOperation >= 0:
                    iRealOperation = iOperation;
                else:
                    iRealOperation = cOperations + iOperation;

                if iCurrentOperation > iRealOperation:
                    return 0;
                if iCurrentOperation == iRealOperation \
                   and iRealOperation >= cOperations - 1 \
                   and self.o.completed:
                    if self.o.resultCode < 0:
                        self.logResult(fIgnoreErrors);
                        return -3;
                    return 0;
            except:
                if fIgnoreErrors:
                    reporter.logXcpt();
                else:
                    reporter.errorXcpt();
                return -2;
            self.oTstDrv.processPendingEvents();

            cMsElapsed = base.timestampMilli() - msStart;
            if cMsElapsed > cMsTimeout:
                if fErrorOnTimeout:
                    if fIgnoreErrors:
                        reporter.log('Timing out after waiting for %u s on "%s" operation %d' \
                                     % (cMsTimeout / 1000, self.sName, iOperation))
                    else:
                        reporter.error('Timing out after waiting for %u s on "%s" operation %d' \
                                       % (cMsTimeout / 1000, self.sName, iOperation))
                return -1;

            try:
                self.o.waitForOperationCompletion(iRealOperation, cMsInterval);
            except:
                if fIgnoreErrors:
                    reporter.logXcpt(self.sName);
                else:
                    reporter.errorXcpt(self.sName);
                return -2;
        # Not reached.

    def doQuickApiTest(self):
        """
        Queries everything that is stable and easy to get at and checks that
        they don't throw errors.
        """
        if True:
            try:
                iPct        = self.o.operationPercent;
                sDesc       = self.o.description;
                fCancelable = self.o.cancelable;
                cSecsRemain = self.o.timeRemaining;
                fCanceled   = self.o.canceled;
                fCompleted  = self.o.completed;
                iOp         = self.o.operation;
                cOps        = self.o.operationCount;
                iOpPct      = self.o.operationPercent;
                sOpDesc     = self.o.operationDescription;
            except:
                reporter.errorXcpt('%s: %s' % (self.sName, self.o,));
                return False;
            try:
                # Very noisy -- only enable for debugging purposes.
                #reporter.log2('%s: op=%u/%u/%s: %u%%; total=%u%% cancel=%s/%s compl=%s rem=%us; desc=%s' \
                #              % (self.sName, iOp, cOps, sOpDesc, iOpPct, iPct, fCanceled, fCancelable, fCompleted, \
                #                 cSecsRemain, sDesc));
                _ = iPct; _ = sDesc;  _ = fCancelable; _ = cSecsRemain; _ = fCanceled; _ = fCompleted; _ = iOp;
                _ = cOps; _ = iOpPct; _ = sOpDesc;
            except:
                reporter.errorXcpt();
                return False;

        return True;


class SessionWrapper(TdTaskBase):
    """
    Wrapper around a machine session.  The real session object can be accessed
    thru the o member (short is good, right :-).
    """

    def __init__(self, oSession, oVM, oVBox, oVBoxMgr, oTstDrv, fRemoteSession, sFallbackName = None, sLogFile = None):
        """
        Initializes the session wrapper.
        """
        TdTaskBase.__init__(self, utils.getCallerName());
        self.o                      = oSession;
        self.oVBox                  = oVBox;
        self.oVBoxMgr               = oVBoxMgr;
        self.oVM                    = oVM;  # Not the session machine. Useful backdoor...
        self.oTstDrv                = oTstDrv;
        self.fpApiVer               = oTstDrv.fpApiVer;
        self.fRemoteSession         = fRemoteSession;
        self.sLogFile               = sLogFile;
        self.oConsoleEventHandler   = None;
        self.uPid                   = None;

        try:
            self.sName              = oSession.machine.name;
        except:
            if sFallbackName is not None:
                self.sName          = sFallbackName;
            else:
                try:    self.sName  = str(oSession.machine);
                except: self.sName  = 'is-this-vm-already-off'

        try:
            self.sUuid              = oSession.machine.id;
        except:
            self.sUuid              = None;

        # Try cache the SessionPID.
        self.getPid();

    def __del__(self):
        """
        Destructor that makes sure the callbacks are deregistered and
        that the session is closed.
        """
        if self.oConsoleEventHandler is not None:
            self.oConsoleEventHandler.unregister();
            self.oConsoleEventHandler = None;

        if self.o is not None:
            try:
                self.close();
                reporter.log('close session %s' % (self.o));
            except:
                pass;
            self.o = None;

        TdTaskBase.__del__(self);

    def toString(self):
        return '<%s: sUuid=%s, sName=%s, uPid=%s, sDbgCreated=%s, fRemoteSession=%s, oSession=%s,' \
               ' oConsoleEventHandler=%s, oVM=%s >' \
             % (type(self).__name__, self.sUuid, self.sName, self.uPid, self.sDbgCreated, self.fRemoteSession,
                self.o, self.oConsoleEventHandler, self.oVM,);

    def __str__(self):
        return self.toString();

    #
    # TdTaskBase overrides.
    #

    def __pollTask(self):
        """ Internal poller """
        # Poll for events after doing the remote GetState call, otherwise we
        # might end up sleepless because XPCOM queues a cleanup event.
        try:
            try:
                eState = self.o.machine.state;
            except Exception, oXcpt:
                if vbox.ComError.notEqual(oXcpt, vbox.ComError.E_UNEXPECTED):
                    reporter.logXcpt();
                return True;
        finally:
            self.oTstDrv.processPendingEvents();

        # Switch
        if eState == vboxcon.MachineState_Running:
            return False;
        if eState == vboxcon.MachineState_Paused:
            return False;
        if eState == vboxcon.MachineState_Teleporting:
            return False;
        if eState == vboxcon.MachineState_LiveSnapshotting:
            return False;
        if eState == vboxcon.MachineState_Starting:
            return False;
        if eState == vboxcon.MachineState_Stopping:
            return False;
        if eState == vboxcon.MachineState_Saving:
            return False;
        if eState == vboxcon.MachineState_Restoring:
            return False;
        if eState == vboxcon.MachineState_TeleportingPausedVM:
            return False;
        if eState == vboxcon.MachineState_TeleportingIn:
            return False;

        # *Beeep* fudge!
        if self.fpApiVer < 3.2 \
          and eState == vboxcon.MachineState_PoweredOff \
          and self.getAgeAsMs() < 3000:
            return False;

        reporter.log('SessionWrapper::pollTask: eState=%s' % (eState));
        return True;


    def pollTask(self, fLocked = False):
        """
        Overrides TdTaskBase.pollTask().

        This method returns False while the VM is online and running normally.
        """
        fRc = self.__pollTask();

        # HACK ALERT: Lazily try registering the console event handler if
        #             we're not ready.
        if not fRc and self.oConsoleEventHandler is None:
            self.registerEventHandlerForTask();

        # HACK ALERT: Lazily try get the PID and add it to the PID file.
        if not fRc and self.uPid is None:
            self.getPid();

        return fRc;

    def waitForTask(self, cMsTimeout = 0):
        """
        Overrides TdTaskBase.waitForTask().
        Process XPCOM/COM events while waiting.
        """
        msStart = base.timestampMilli();
        fState  = self.pollTask(False);
        while not fState:
            cMsElapsed = base.timestampMilli() - msStart;
            if cMsElapsed > cMsTimeout:
                break;
            try:    self.oVBoxMgr.waitForEvents(cMsTimeout - cMsElapsed);
            except KeyboardInterrupt: raise;
            except: pass;
            fState = self.pollTask(False);
        return fState;

    def setTaskOwner(self, oOwner):
        """
        HACK ALERT!
        Overrides TdTaskBase.setTaskOwner() so we can try call
        registerEventHandlerForTask() again when when the testdriver calls
        addTask() after VM has been spawned.  Related to pollTask() above.

        The testdriver must not add the task too early for this to work!
        """
        if oOwner is not None:
            self.registerEventHandlerForTask()
        return TdTaskBase.setTaskOwner(self, oOwner);


    #
    # Task helpers.
    #

    def registerEventHandlerForTask(self):
        """
        Registers the console event handlers for working the task state.
        """
        if self.oConsoleEventHandler is not None:
            return True;
        self.oConsoleEventHandler = self.registerDerivedEventHandler(vbox.SessionConsoleEventHandler, {}, False);
        return self.oConsoleEventHandler is not None;


    def assertPoweredOff(self):
        """
        Asserts that the VM is powered off, reporting an error if not.
        Returns True if powered off, False + error msg if not.
        """
        try:
            try:
                eState = self.oVM.state;
            except Exception:
                reporter.errorXcpt();
                return True;
        finally:
            self.oTstDrv.processPendingEvents();

        if eState == vboxcon.MachineState_PoweredOff:
            return True;
        reporter.error('Expected machine state "PoweredOff", machine is in the "%s" state instead.'
                       % (_nameMachineState(eState),));
        return False;


    def getMachineStateWithName(self):
        """
        Gets the current machine state both as a constant number/whatever and
        as a human readable string.  On error, the constants will be set to
        None and the string will be the error message.
        """
        try:
            eState = self.oVM.state;
        except:
            return (None, '[error getting state: %s]' % (self.oVBoxMgr.xcptToString(),));
        finally:
            self.oTstDrv.processPendingEvents();
        return (eState, _nameMachineState(eState));

    def reportPrematureTermination(self, sPrefix = ''):
        """
        Reports a premature virtual machine termination.
        Returns False to facilitate simpler error paths.
        """

        reporter.error(sPrefix + 'The virtual machine terminated prematurely!!');
        (enmState, sStateNm) = self.getMachineStateWithName();
        reporter.error(sPrefix + 'Machine state: %s' % (sStateNm,));

        if    enmState is not None \
          and enmState == vboxcon.MachineState_Aborted \
          and self.uPid is not None:
            #
            # Look for process crash info.
            #
            def addCrashFile(sLogFile, fBinary):
                """ processCollectCrashInfo callback. """
                reporter.addLogFile(sLogFile, 'crash/dump/vm' if fBinary else 'crash/report/vm');
            utils.processCollectCrashInfo(self.uPid, reporter.log, addCrashFile);

        return False;



    #
    # ISession / IMachine / ISomethingOrAnother wrappers.
    #

    def close(self):
        """
        Closes the session if it's open and removes it from the
        vbox.TestDriver.aoRemoteSessions list.
        Returns success indicator.
        """
        fRc = True;
        if self.o is not None:
            # Get the pid in case we need to kill the process later on.
            self.getPid();

            # Try close it.
            try:
                if self.fpApiVer < 3.3:
                    self.o.close();
                else:
                    self.o.unlockMachine();
                self.o = None;
            except KeyboardInterrupt:
                raise;
            except:
                # Kludge to ignore VBoxSVC's closing of our session when the
                # direct session closes / VM process terminates.  Fun!
                try:    fIgnore = self.o.state == vboxcon.SessionState_Unlocked;
                except: fIgnore = False;
                if not fIgnore:
                    reporter.errorXcpt('ISession::unlockMachine failed on %s' % (self.o));
                    fRc = False;

            # Remove it from the remote session list if applicable (not 100% clean).
            if fRc and self.fRemoteSession:
                try:
                    if self in self.oTstDrv.aoRemoteSessions:
                        reporter.log2('SessionWrapper::close: Removing myself from oTstDrv.aoRemoteSessions');
                        self.oTstDrv.aoRemoteSessions.remove(self)
                except:
                    reporter.logXcpt();

                if self.uPid is not None:
                    self.oTstDrv.pidFileRemove(self.uPid);

        self.oTstDrv.processPendingEvents();
        return fRc;

    def saveSettings(self, fClose = False):
        """
        Saves the settings and optionally closes the session.
        Returns success indicator.
        """
        try:
            try:
                self.o.machine.saveSettings();
            except:
                reporter.errorXcpt('saveSettings failed on %s' % (self.o));
                return False;
        finally:
            self.oTstDrv.processPendingEvents();
        if fClose:
            return self.close();
        return True;

    def discardSettings(self, fClose = False):
        """
        Discards the settings and optionally closes the session.
        """
        try:
            try:
                self.o.machine.discardSettings();
            except:
                reporter.errorXcpt('discardSettings failed on %s' % (self.o));
                return False;
        finally:
            self.oTstDrv.processPendingEvents();
        if fClose:
            return self.close();
        return True;

    def enableVirtEx(self, fEnable):
        """
        Enables or disables AMD-V/VT-x.
        Returns True on success and False on failure.  Error information is logged.
        """
        # Enable/disable it.
        fRc = True;
        try:
            self.o.machine.setHWVirtExProperty(vboxcon.HWVirtExPropertyType_Enabled, fEnable);
        except:
            reporter.errorXcpt('failed to set HWVirtExPropertyType_Enabled=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set HWVirtExPropertyType_Enabled=%s for "%s"' % (fEnable, self.sName));

        # Force/unforce it.
        if fRc and hasattr(vboxcon, 'HWVirtExPropertyType_Force'):
            try:
                self.o.machine.setHWVirtExProperty(vboxcon.HWVirtExPropertyType_Force, fEnable);
            except:
                reporter.errorXcpt('failed to set HWVirtExPropertyType_Force=%s for "%s"' % (fEnable, self.sName));
                fRc = False;
            else:
                reporter.log('set HWVirtExPropertyType_Force=%s for "%s"' % (fEnable, self.sName));
        else:
            reporter.log('Warning! vboxcon has no HWVirtExPropertyType_Force attribute.');
            ## @todo Modify CFGM to do the same for old VBox versions?

        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableNestedPaging(self, fEnable):
        """
        Enables or disables nested paging..
        Returns True on success and False on failure.  Error information is logged.
        """
        ## @todo Add/remove force CFGM thing, we don't want fallback logic when testing.
        fRc = True;
        try:
            self.o.machine.setHWVirtExProperty(vboxcon.HWVirtExPropertyType_NestedPaging, fEnable);
        except:
            reporter.errorXcpt('failed to set HWVirtExPropertyType_NestedPaging=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set HWVirtExPropertyType_NestedPaging=%s for "%s"' % (fEnable, self.sName));
            self.oTstDrv.processPendingEvents();
        return fRc;

    def enableLongMode(self, fEnable):
        """
        Enables or disables LongMode.
        Returns True on success and False on failure.  Error information is logged.
        """
        # Supported.
        if self.fpApiVer < 4.2  or  not hasattr(vboxcon, 'HWVirtExPropertyType_LongMode'):
            return True;

        # Enable/disable it.
        fRc = True;
        try:
            self.o.machine.setCPUProperty(vboxcon.CPUPropertyType_LongMode, fEnable);
        except:
            reporter.errorXcpt('failed to set CPUPropertyType_LongMode=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set CPUPropertyType_LongMode=%s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enablePae(self, fEnable):
        """
        Enables or disables PAE
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            if self.fpApiVer >= 3.2:    # great, ain't it?
                self.o.machine.setCPUProperty(vboxcon.CPUPropertyType_PAE, fEnable);
            else:
                self.o.machine.setCpuProperty(vboxcon.CpuPropertyType_PAE, fEnable);
        except:
            reporter.errorXcpt('failed to set CPUPropertyType_PAE=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set CPUPropertyType_PAE=%s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableIoApic(self, fEnable):
        """
        Enables or disables the IO-APIC
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.BIOSSettings.IOAPICEnabled = fEnable;
        except:
            reporter.errorXcpt('failed to set BIOSSettings.IOAPICEnabled=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set BIOSSettings.IOAPICEnabled=%s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableHpet(self, fEnable):
        """
        Enables or disables the HPET
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            if self.fpApiVer >= 4.2:
                self.o.machine.HPETEnabled = fEnable;
            else:
                self.o.machine.hpetEnabled = fEnable;
        except:
            reporter.errorXcpt('failed to set HpetEnabled=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set HpetEnabled=%s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableUsbHid(self, fEnable):
        """
        Enables or disables the USB HID
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            if fEnable:
                if self.fpApiVer >= 4.3:
                    cOhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_OHCI);
                    if cOhciCtls == 0:
                        self.o.machine.AddUSBController('OHCI', vboxcon.USBControllerType_OHCI);
                else:
                    self.o.machine.usbController.enabled = True;

                if self.fpApiVer >= 4.2:
                    self.o.machine.pointingHIDType = vboxcon.PointingHIDType_ComboMouse;
                    self.o.machine.keyboardHIDType = vboxcon.KeyboardHIDType_ComboKeyboard;
                else:
                    self.o.machine.pointingHidType = vboxcon.PointingHidType_ComboMouse;
                    self.o.machine.keyboardHidType = vboxcon.KeyboardHidType_ComboKeyboard;
            else:
                if self.fpApiVer >= 4.2:
                    self.o.machine.pointingHIDType = vboxcon.PointingHIDType_PS2Mouse;
                    self.o.machine.keyboardHIDType = vboxcon.KeyboardHIDType_PS2Keyboard;
                else:
                    self.o.machine.pointingHidType = vboxcon.PointingHidType_PS2Mouse;
                    self.o.machine.keyboardHidType = vboxcon.KeyboardHidType_PS2Keyboard;
        except:
            reporter.errorXcpt('failed to change UsbHid to %s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('changed UsbHid to %s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableUsbOhci(self, fEnable):
        """
        Enables or disables the USB OHCI controller
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            if fEnable:
                if self.fpApiVer >= 4.3:
                    cOhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_OHCI);
                    if cOhciCtls == 0:
                        self.o.machine.AddUSBController('OHCI', vboxcon.USBControllerType_OHCI);
                else:
                    self.o.machine.usbController.enabled = True;
            else:
                if self.fpApiVer >= 4.3:
                    cOhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_OHCI);
                    if cOhciCtls == 1:
                        self.o.machine.RemoveUSBController('OHCI');
                else:
                    self.o.machine.usbController.enabled = False;
        except:
            reporter.errorXcpt('failed to change OHCI to %s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('changed OHCI to %s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableUsbEhci(self, fEnable):
        """
        Enables or disables the USB EHCI controller, enables also OHCI if it is still disabled.
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            if fEnable:
                if self.fpApiVer >= 4.3:
                    cOhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_OHCI);
                    if cOhciCtls == 0:
                        self.o.machine.addUSBController('OHCI', vboxcon.USBControllerType_OHCI);

                    cEhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_EHCI);
                    if cEhciCtls == 0:
                        self.o.machine.addUSBController('EHCI', vboxcon.USBControllerType_EHCI);
                else:
                    self.o.machine.usbController.enabled = True;
                    self.o.machine.usbController.enabledEHCI = True;
            else:
                if self.fpApiVer >= 4.3:
                    cEhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_EHCI);
                    if cEhciCtls == 1:
                        self.o.machine.RemoveUSBController('EHCI');
                else:
                    self.o.machine.usbController.enabledEHCI = False;
        except:
            reporter.errorXcpt('failed to change EHCI to %s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('changed EHCI to %s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def enableUsbXhci(self, fEnable):
        """
        Enables or disables the USB XHCI controller. Error information is logged.
        """
        fRc = True;
        try:
            if fEnable:
                cXhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_XHCI);
                if cXhciCtls == 0:
                    self.o.machine.addUSBController('XHCI', vboxcon.USBControllerType_XHCI);
            else:
                cXhciCtls = self.o.machine.getUSBControllerCountByType(vboxcon.USBControllerType_XHCI);
                if cXhciCtls == 1:
                    self.o.machine.RemoveUSBController('XHCI');
        except:
            reporter.errorXcpt('failed to change XHCI to %s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('changed XHCI to %s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setFirmwareType(self, eType):
        """
        Sets the firmware type.
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.firmwareType = eType;
        except:
            reporter.errorXcpt('failed to set firmwareType=%s for "%s"' % (eType, self.sName));
            fRc = False;
        else:
            reporter.log('set firmwareType=%s for "%s"' % (eType, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setupBootLogo(self, fEnable, cMsLogoDisplay = 0):
        """
        Sets up the boot logo.  fEnable toggles the fade and boot menu
        settings as well as the mode.
        """
        fRc = True;
        try:
            self.o.machine.BIOSSettings.logoFadeIn       = not fEnable;
            self.o.machine.BIOSSettings.logoFadeOut      = not fEnable;
            self.o.machine.BIOSSettings.logoDisplayTime  = cMsLogoDisplay;
            if fEnable:
                self.o.machine.BIOSSettings.bootMenuMode = vboxcon.BIOSBootMenuMode_Disabled;
            else:
                self.o.machine.BIOSSettings.bootMenuMode = vboxcon.BIOSBootMenuMode_MessageAndMenu;
        except:
            reporter.errorXcpt('failed to set logoFadeIn/logoFadeOut/bootMenuMode=%s for "%s"' % (fEnable, self.sName));
            fRc = False;
        else:
            reporter.log('set logoFadeIn/logoFadeOut/bootMenuMode=%s for "%s"' % (fEnable, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setupVrdp(self, fEnable, uPort = None):
        """
        Configures VRDP.
        """
        fRc = True;
        try:
            if self.fpApiVer >= 4.0:
                self.o.machine.VRDEServer.enabled = fEnable;
            else:
                self.o.machine.VRDPServer.enabled = fEnable;
        except:
            reporter.errorXcpt('failed to set VRDEServer::enabled=%s for "%s"' % (fEnable, self.sName));
            fRc = False;

        if uPort is not None and fRc:
            try:
                if self.fpApiVer >= 4.0:
                    self.o.machine.VRDEServer.setVRDEProperty("TCP/Ports", str(uPort));
                else:
                    self.o.machine.VRDPServer.ports = str(uPort);
            except:
                reporter.errorXcpt('failed to set VRDEServer::ports=%s for "%s"' % (uPort, self.sName));
                fRc = False;
        if fRc:
            reporter.log('set VRDEServer.enabled/ports=%s/%s for "%s"' % (fEnable, uPort, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def getNicDriverNameFromType(self, eNicType):
        """
        Helper that translate the adapter type into a driver name.
        """
        if    eNicType == vboxcon.NetworkAdapterType_Am79C970A \
           or eNicType == vboxcon.NetworkAdapterType_Am79C973:
            sName = 'pcnet';
        elif    eNicType == vboxcon.NetworkAdapterType_I82540EM \
             or eNicType == vboxcon.NetworkAdapterType_I82543GC \
             or eNicType == vboxcon.NetworkAdapterType_I82545EM:
            sName = 'e1000';
        elif eNicType == vboxcon.NetworkAdapterType_Virtio:
            sName = 'virtio-net';
        else:
            reporter.error('Unknown adapter type "%s" (VM: "%s")' % (eNicType, self.sName));
            sName = 'pcnet';
        return sName;

    def setupNatForwardingForTxs(self, iNic = 0, iHostPort = 5042):
        """
        Sets up NAT forwarding for port 5042 if applicable, cleans up if not.
        """
        try:
            oNic = self.o.machine.getNetworkAdapter(iNic);
        except:
            reporter.errorXcpt('getNetworkAdapter(%s) failed for "%s"' % (iNic, self.sName));
            return False;

        # Nuke the old setup for all possible adapter types (in case we're
        # called after it changed).
        for sName in ('pcnet', 'e1000', 'virtio-net'):
            for sConfig in ('VBoxInternal/Devices/%s/%u/LUN#0/AttachedDriver/Config' % (sName, iNic), \
                            'VBoxInternal/Devices/%s/%u/LUN#0/Config' % (sName, iNic)):
                try:
                    self.o.machine.setExtraData('%s/txs/Protocol'  % (sConfig), '');
                    self.o.machine.setExtraData('%s/txs/HostPort'  % (sConfig), '');
                    self.o.machine.setExtraData('%s/txs/GuestPort' % (sConfig), '');
                except:
                    reporter.errorXcpt();

        # Set up port forwarding if NAT attachment.
        try:
            eAttType = oNic.attachmentType;
        except:
            reporter.errorXcpt('attachmentType on %s failed for "%s"' % (iNic, self.sName));
            return False;
        if eAttType != vboxcon.NetworkAttachmentType_NAT:
            return True;

        try:
            eNicType = oNic.adapterType;
            fTraceEnabled = oNic.traceEnabled;
        except:
            reporter.errorXcpt('attachmentType/traceEnabled on %s failed for "%s"' % (iNic, self.sName));
            return False;

        if self.fpApiVer >= 4.1:
            try:
                if self.fpApiVer >= 4.2:
                    oNatEngine = oNic.NATEngine;
                else:
                    oNatEngine = oNic.natDriver;
            except:
                reporter.errorXcpt('Failed to get INATEngine data on "%s"' % (self.sName));
                return False;
            try:    oNatEngine.removeRedirect('txs');
            except: pass;
            try:
                oNatEngine.addRedirect('txs', vboxcon.NATProtocol_TCP, '127.0.0.1', '%s' % (iHostPort), '', '5042');
            except:
                reporter.errorXcpt('Failed to add a addRedirect redirect on "%s"' % (self.sName));
                return False;

        else:
            sName = self.getNicDriverNameFromType(eNicType);
            if fTraceEnabled:
                sConfig = 'VBoxInternal/Devices/%s/%u/LUN#0/AttachedDriver/Config' % (sName, iNic)
            else:
                sConfig = 'VBoxInternal/Devices/%s/%u/LUN#0/Config' % (sName, iNic)

            try:
                self.o.machine.setExtraData('%s/txs/Protocol'  % (sConfig), 'TCP');
                self.o.machine.setExtraData('%s/txs/HostPort'  % (sConfig), '%s' % (iHostPort));
                self.o.machine.setExtraData('%s/txs/GuestPort' % (sConfig), '5042');
            except:
                reporter.errorXcpt('Failed to set NAT extra data on "%s"' % (self.sName));
                return False;
        return True;

    def setNicType(self, eType, iNic = 0):
        """
        Sets the NIC type of the specified NIC.
        Returns True on success and False on failure.  Error information is logged.
        """
        try:
            try:
                oNic = self.o.machine.getNetworkAdapter(iNic);
            except:
                reporter.errorXcpt('getNetworkAdapter(%s) failed for "%s"' % (iNic, self.sName));
                return False;
            try:
                oNic.adapterType = eType;
            except:
                reporter.errorXcpt('failed to set NIC type on slot %s to %s for VM "%s"' % (iNic, eType, self.sName));
                return False;
        finally:
            self.oTstDrv.processPendingEvents();

        if not self.setupNatForwardingForTxs(iNic):
            return False;
        reporter.log('set NIC type on slot %s to %s for VM "%s"' % (iNic, eType, self.sName));
        return True;

    def setNicTraceEnabled(self, fTraceEnabled, sTraceFile, iNic = 0):
        """
        Sets the NIC trace enabled flag and file path.
        Returns True on success and False on failure.  Error information is logged.
        """
        try:
            try:
                oNic = self.o.machine.getNetworkAdapter(iNic);
            except:
                reporter.errorXcpt('getNetworkAdapter(%s) failed for "%s"' % (iNic, self.sName));
                return False;
            try:
                oNic.traceEnabled = fTraceEnabled;
                oNic.traceFile = sTraceFile;
            except:
                reporter.errorXcpt('failed to set NIC trace flag on slot %s to %s for VM "%s"' \
                                   % (iNic, fTraceEnabled, self.sName));
                return False;
        finally:
            self.oTstDrv.processPendingEvents();

        if not self.setupNatForwardingForTxs(iNic):
            return False;
        reporter.log('set NIC trace on slot %s to "%s" (path "%s") for VM "%s"' %
                        (iNic, fTraceEnabled, sTraceFile, self.sName));
        return True;

    def getDefaultNicName(self, eAttachmentType):
        """
        Return the default network / interface name for the NIC attachment type.
        """
        sRetName = '';
        if eAttachmentType == vboxcon.NetworkAttachmentType_Bridged:
            if self.oTstDrv.sDefBridgedNic is not None:
                sRetName = self.oTstDrv.sDefBridgedNic;
            else:
                sRetName = 'eth0';
                try:
                    aoHostNics = self.oVBoxMgr.getArray(self.oVBox.host, 'networkInterfaces');
                    for oHostNic in aoHostNics:
                        if   oHostNic.interfaceType == vboxcon.HostNetworkInterfaceType_Bridged \
                         and oHostNic.status == vboxcon.HostNetworkInterfaceStatus_Up:
                            sRetName = oHostNic.name;
                            break;
                except:
                    reporter.errorXcpt();
        elif eAttachmentType == vboxcon.NetworkAttachmentType_HostOnly:
            try:
                aoHostNics = self.oVBoxMgr.getArray(self.oVBox.host, 'networkInterfaces');
                for oHostNic in aoHostNics:
                    if oHostNic.interfaceType == vboxcon.HostNetworkInterfaceType_HostOnly:
                        if oHostNic.status == vboxcon.HostNetworkInterfaceStatus_Up:
                            sRetName = oHostNic.name;
                            break;
                        if sRetName == '':
                            sRetName = oHostNic.name;
            except:
                reporter.errorXcpt();
            if sRetName == '':
                sRetName = 'HostInterfaceNetwork-vboxnet0';
        elif eAttachmentType == vboxcon.NetworkAttachmentType_Internal:
            sRetName = 'VBoxTest';
        elif eAttachmentType == vboxcon.NetworkAttachmentType_NAT:
            sRetName = '';
        else:
            reporter.error('eAttachmentType=%s is not known' % (eAttachmentType));
        return sRetName;

    def setNicAttachment(self, eAttachmentType, sName = None, iNic = 0):
        """
        Sets the attachment type of the specified NIC.
        Returns True on success and False on failure.  Error information is logged.
        """
        try:
            oNic = self.o.machine.getNetworkAdapter(iNic);
        except:
            reporter.errorXcpt('getNetworkAdapter(%s) failed for "%s"' % (iNic, self.sName));
            return False;

        try:
            if eAttachmentType is not None:
                try:
                    if self.fpApiVer >= 4.1:
                        oNic.attachmentType = eAttachmentType;
                    else:
                        if eAttachmentType == vboxcon.NetworkAttachmentType_NAT:
                            oNic.attachToNAT();
                        elif eAttachmentType == vboxcon.NetworkAttachmentType_Bridged:
                            oNic.attachToBridgedInterface();
                        elif eAttachmentType == vboxcon.NetworkAttachmentType_Internal:
                            oNic.attachToInternalNetwork();
                        elif eAttachmentType == vboxcon.NetworkAttachmentType_HostOnly:
                            oNic.attachToHostOnlyInterface();
                        else:
                            raise base.GenError("eAttachmentType=%s is invalid" % (eAttachmentType));
                except:
                    reporter.errorXcpt('failed to set the attachment type on slot %s to %s for VM "%s"' \
                        % (iNic, eAttachmentType, self.sName));
                    return False;
            else:
                try:
                    eAttachmentType = oNic.attachmentType;
                except:
                    reporter.errorXcpt('failed to get the attachment type on slot %s for VM "%s"' % (iNic, self.sName));
                    return False;
        finally:
            self.oTstDrv.processPendingEvents();

        if sName is not None:
            # Resolve the special 'default' name.
            if sName == 'default':
                sName = self.getDefaultNicName(eAttachmentType);

            # The name translate to different attributes depending on the
            # attachment type.
            try:
                if eAttachmentType == vboxcon.NetworkAttachmentType_Bridged:
                    ## @todo check this out on windows, may have to do a
                    # translation of the name there or smth IIRC.
                    try:
                        if self.fpApiVer >= 4.1:
                            oNic.bridgedInterface = sName;
                        else:
                            oNic.hostInterface = sName;
                    except:
                        reporter.errorXcpt('failed to set the hostInterface property on slot %s to "%s" for VM "%s"' \
                            % (iNic, sName, self.sName));
                        return False;
                elif eAttachmentType == vboxcon.NetworkAttachmentType_HostOnly:
                    try:
                        if self.fpApiVer >= 4.1:
                            oNic.hostOnlyInterface = sName;
                        else:
                            oNic.hostInterface = sName;
                    except:
                        reporter.errorXcpt('failed to set the internalNetwork property on slot %s to "%s" for VM "%s"' \
                            % (iNic, sName, self.sName));
                        return False;
                elif eAttachmentType == vboxcon.NetworkAttachmentType_Internal:
                    try:
                        oNic.internalNetwork = sName;
                    except:
                        reporter.errorXcpt('failed to set the internalNetwork property on slot %s to "%s" for VM "%s"' \
                            % (iNic, sName, self.sName));
                        return False;
                elif eAttachmentType == vboxcon.NetworkAttachmentType_NAT:
                    try:
                        oNic.NATNetwork = sName;
                    except:
                        reporter.errorXcpt('failed to set the NATNetwork property on slot %s to "%s" for VM "%s"' \
                            % (iNic, sName, self.sName));
                        return False;
            finally:
                self.oTstDrv.processPendingEvents();

        if not self.setupNatForwardingForTxs(iNic):
            return False;
        reporter.log('set NIC type on slot %s to %s for VM "%s"' % (iNic, eAttachmentType, self.sName));
        return True;

    def setNicMacAddress(self, sMacAddr, iNic = 0):
        """
        Sets the MAC address of the specified NIC.
        Returns True on success and False on failure.  Error information is logged.
        """

        # Resolve missing MAC address prefix
        cchMacAddr = len(sMacAddr) > 0;
        if cchMacAddr > 0 and cchMacAddr < 12:
            sHostName = '';
            try:
                sHostName = socket.getfqdn();
                if sys.platform == 'win32' \
                 and sHostName.endswith('.sun.com') \
                 and not sHostName.endswith('.germany.sun.com'):
                    sHostName = socket.gethostname(); # klugde.
                abHostIP = socket.inet_aton(socket.gethostbyname(sHostName));
            except:
                reporter.errorXcpt('failed to determin the host IP for "%s".' % (sHostName,));
                abHostIP = array.array('B', (0x80, 0x86, 0x00, 0x00)).tostring();
            sDefaultMac = '%02X%02X%02X%02X%02X%02X' \
                % (0x02, ord(abHostIP[0]), ord(abHostIP[1]), ord(abHostIP[2]), ord(abHostIP[3]), iNic);
            sMacAddr = sDefaultMac[0:(11 - cchMacAddr)] + sMacAddr;

        # Get the NIC object and try set it address.
        try:
            oNic = self.o.machine.getNetworkAdapter(iNic);
        except:
            reporter.errorXcpt('getNetworkAdapter(%s) failed for "%s"' % (iNic, self.sName));
            return False;

        try:
            oNic.MACAddress = sMacAddr;
        except:
            reporter.errorXcpt('failed to set the MAC address on slot %s to "%s" for VM "%s"' \
                % (iNic, sMacAddr, self.sName));
            return False;

        reporter.log('set MAC address on slot %s to %s for VM "%s"' % (iNic, sMacAddr, self.sName));
        return True;

    def setRamSize(self, cMB):
        """
        Set the RAM size of the VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.memorySize = cMB;
        except:
            reporter.errorXcpt('failed to set the RAM size of "%s" to %s' % (self.sName, cMB));
            fRc = False;
        else:
            reporter.log('set the RAM size of "%s" to %s' % (self.sName, cMB));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setVRamSize(self, cMB):
        """
        Set the RAM size of the VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.VRAMSize = cMB;
        except:
            reporter.errorXcpt('failed to set the VRAM size of "%s" to %s' % (self.sName, cMB));
            fRc = False;
        else:
            reporter.log('set the VRAM size of "%s" to %s' % (self.sName, cMB));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setCpuCount(self, cCpus):
        """
        Set the number of CPUs.
        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.CPUCount = cCpus;
        except:
            reporter.errorXcpt('failed to set the CPU count of "%s" to %s' % (self.sName, cCpus));
            fRc = False;
        else:
            reporter.log('set the CPU count of "%s" to %s' % (self.sName, cCpus));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def ensureControllerAttached(self, sController):
        """
        Makes sure the specified controller is attached to the VM, attaching it
        if necessary.
        """
        try:
            try:
                self.o.machine.getStorageControllerByName(sController);
            except:
                iType = _ControllerNameToBus(sController);
                try:
                    self.o.machine.addStorageController(sController, iType);
                    reporter.log('added storage controller "%s" (type %s) to %s' % (sController, iType, self.sName));
                except:
                    reporter.errorXcpt('addStorageController("%s",%s) failed on "%s"' % (sController, iType, self.sName) );
                    return False;
        finally:
            self.oTstDrv.processPendingEvents();
        return True;

    def setStorageControllerPortCount(self, sController, iPortCount):
        """
        Set maximum ports count for storage controller
        """
        try:
            oCtl = self.o.machine.getStorageControllerByName(sController)
            oCtl.portCount = iPortCount
            self.oTstDrv.processPendingEvents()
            reporter.log('set controller "%s" port count to value %d' % (sController, iPortCount))
            return True
        except:
            reporter.log('unable to set storage controller "%s" ports count to %d' % (sController, iPortCount))

        return False

    def setBootOrder(self, iPosition, eType):
        """
        Set guest boot order type
        @param iPosition    boot order position
        @param eType        device type (vboxcon.DeviceType_HardDisk,
                            vboxcon.DeviceType_DVD, vboxcon.DeviceType_Floppy)
        """
        try:
            self.o.machine.setBootOrder(iPosition, eType)
        except:
            return reporter.errorXcpt('Unable to set boot order.')

        reporter.log('Set boot order [%d] for device %s' % (iPosition, str(eType)))
        self.oTstDrv.processPendingEvents();

        return True

    def setStorageControllerType(self, eType, sController = "IDE Controller"):
        """
        Similar to ensureControllerAttached, except it will change the type.
        """
        try:
            oCtl = self.o.machine.getStorageControllerByName(sController);
        except:
            iType = _ControllerNameToBus(sController);
            try:
                oCtl = self.o.machine.addStorageController(sController, iType);
                reporter.log('added storage controller "%s" (type %s) to %s' % (sController, iType, self.sName));
            except:
                reporter.errorXcpt('addStorageController("%s",%s) failed on "%s"' % (sController, iType, self.sName) );
                return False;
        try:
            oCtl.controllerType = eType;
        except:
            reporter.errorXcpt('failed to set controller type of "%s" on "%s" to %s' % (sController, self.sName, eType) );
            return False;
        reporter.log('set controller type of "%s" on "%s" to %s' % (sController, self.sName, eType) );
        self.oTstDrv.processPendingEvents();
        return True;

    def attachDvd(self, sImage = None, sController = "IDE Controller", iPort = 1, iDevice = 0):
        """
        Attaches a DVD drive to a VM, optionally with an ISO inserted.
        Returns True on success and False on failure.  Error information is logged.
        """
        # Input validation.
        if sImage is not None and not self.oTstDrv.isResourceFile(sImage)\
            and not os.path.isabs(sImage): ## fixme - testsuite unzip ++
            reporter.fatal('"%s" is not in the resource set' % (sImage));
            return None;

        if not self.ensureControllerAttached(sController):
            return False;

        # Find/register the image if specified.
        oImage = None;
        sImageUuid = "";
        if sImage is not None:
            sFullName = self.oTstDrv.getFullResourceName(sImage)
            try:
                oImage = self.oVBox.findDVDImage(sFullName);
            except:
                try:
                    if self.fpApiVer >= 4.1:
                        oImage = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_DVD, vboxcon.AccessMode_ReadOnly, False);
                    elif self.fpApiVer >= 4.0:
                        oImage = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_DVD, vboxcon.AccessMode_ReadOnly);
                    else:
                        oImage = self.oVBox.openDVDImage(sFullName, "");
                except vbox.ComException, oXcpt:
                    if oXcpt.errno != -1:
                        reporter.errorXcpt('failed to open DVD image "%s" xxx' % (sFullName));
                    else:
                        reporter.errorXcpt('failed to open DVD image "%s" yyy' % (sFullName));
                    return False;
                except:
                    reporter.errorXcpt('failed to open DVD image "%s"' % (sFullName));
                    return False;
            try:
                sImageUuid = oImage.id;
            except:
                reporter.errorXcpt('failed to get the UUID of "%s"' % (sFullName));
                return False;

        # Attach the DVD.
        fRc = True;
        try:
            if self.fpApiVer >= 4.0:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_DVD, oImage);
            else:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_DVD, sImageUuid);
        except:
            reporter.errorXcpt('attachDevice("%s",%s,%s,HardDisk,"%s") failed on "%s"' \
                               % (sController, iPort, iDevice, sImageUuid, self.sName) );
            fRc = False;
        else:
            reporter.log('attached DVD to %s, image="%s"' % (self.sName, sImage));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def attachHd(self, sHd, sController = "IDE Controller", iPort = 0, iDevice = 0, fImmutable = True, fForceResource = True):
        """
        Attaches a HD to a VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        # Input validation.
        if fForceResource and not self.oTstDrv.isResourceFile(sHd):
            reporter.fatal('"%s" is not in the resource set' % (sHd,));
            return None;

        if not self.ensureControllerAttached(sController):
            return False;

        # Find the HD, registering it if necessary (as immutable).
        if fForceResource:
            sFullName = self.oTstDrv.getFullResourceName(sHd);
        else:
            sFullName = sHd;
        try:
            oHd = self.oVBox.findHardDisk(sFullName);
        except:
            try:
                if self.fpApiVer >= 4.1:
                    oHd = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_HardDisk, vboxcon.AccessMode_ReadOnly, False);
                elif self.fpApiVer >= 4.0:
                    oHd = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_HardDisk, vboxcon.AccessMode_ReadOnly);
                else:
                    oHd = self.oVBox.openHardDisk(sFullName, vboxcon.AccessMode_ReadOnly, False, "", False, "");
            except:
                reporter.errorXcpt('failed to open hd "%s"' % (sFullName));
                return False;
        try:
            if fImmutable:
                oHd.type = vboxcon.MediumType_Immutable;
            else:
                oHd.type = vboxcon.MediumType_Normal;
        except:
            if fImmutable:
                reporter.errorXcpt('failed to set hd "%s" immutable' % (sHd));
            else:
                reporter.errorXcpt('failed to set hd "%s" normal' % (sHd));
            return False;

        # Attach it.
        fRc = True;
        try:
            if self.fpApiVer >= 4.0:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_HardDisk, oHd);
            else:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_HardDisk, oHd.id);
        except:
            reporter.errorXcpt('attachDevice("%s",%s,%s,HardDisk,"%s") failed on "%s"' \
                               % (sController, iPort, iDevice, oHd.id, self.sName) );
            fRc = False;
        else:
            reporter.log('attached "%s" to %s' % (sHd, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def createBaseHd(self, sHd, sFmt = "VDI", cb = 10*1024*1024*1024):
        """
        Creates a base HD.
        Returns Medium object on success and None on failure.  Error information is logged.
        """
        try:
            if self.fpApiVer >= 5.0:
                oHd = self.oVBox.createMedium(sFmt, sHd, vboxcon.AccessMode_ReadWrite, vboxcon.DeviceType_HardDisk);
            else:
                oHd = self.oVBox.createHardDisk(sFmt, sHd);
            oProgressXpcom = oHd.createBaseStorage(cb, (vboxcon.MediumVariant_Standard, ))
            oProgress = ProgressWrapper(oProgressXpcom, self.oVBoxMgr, self.oTstDrv, 'create base disk %s' % (sHd));
            oProgress.wait();
            oProgress.logResult();
        except:
            reporter.errorXcpt('failed to create base hd "%s"' % (sHd));
            oHd = None

        return oHd;

    def createDiffHd(self, oParentHd, sHd, sFmt = "VDI"):
        """
        Creates a differencing HD.
        Returns Medium object on success and None on failure.  Error information is logged.
        """
        try:
            if self.fpApiVer >= 5.0:
                oHd = self.oVBox.createMedium(sFmt, sHd, vboxcon.AccessMode_ReadWrite, vboxcon.DeviceType_HardDisk);
            else:
                oHd = self.oVBox.createHardDisk(sFmt, sHd);
            oProgressXpcom = oParentHd.createDiffStorage(oHd, (vboxcon.MediumVariant_Standard, ))
            oProgress = ProgressWrapper(oProgressXpcom, self.oVBoxMgr, self.oTstDrv, 'create diff disk %s' % (sHd));
            oProgress.wait();
            oProgress.logResult();
        except:
            reporter.errorXcpt('failed to create diff hd "%s"' % (sHd));
            oHd = None

        return oHd;

    def createAndAttachHd(self, sHd, sFmt = "VDI", sController = "IDE Controller", cb = 10*1024*1024*1024, \
                          iPort = 0, iDevice = 0, fImmutable = True):
        """
        Creates and attaches a HD to a VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        if not self.ensureControllerAttached(sController):
            return False;

        oHd = self.createBaseHd(sHd, sFmt, cb)
        if oHd is None:
            return False;

        fRc = True;
        try:
            if fImmutable:
                oHd.type = vboxcon.MediumType_Immutable;
            else:
                oHd.type = vboxcon.MediumType_Normal;
        except:
            if fImmutable:
                reporter.errorXcpt('failed to set hd "%s" immutable' % (sHd));
            else:
                reporter.errorXcpt('failed to set hd "%s" normal' % (sHd));
            fRc = False;

        # Attach it.
        if fRc is True:
            try:
                if self.fpApiVer >= 4.0:
                    self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_HardDisk, oHd);
                else:
                    self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_HardDisk, oHd.id);
            except:
                reporter.errorXcpt('attachDevice("%s",%s,%s,HardDisk,"%s") failed on "%s"' \
                                   % (sController, iPort, iDevice, oHd.id, self.sName) );
                fRc = False;
            else:
                reporter.log('attached "%s" to %s' % (sHd, self.sName));

        # Delete disk in case of an error
        if fRc is False:
            try:
                oProgressCom = oHd.deleteStorage();
            except:
                reporter.errorXcpt('deleteStorage() for disk %s failed' % (sHd,));
            else:
                oProgress = ProgressWrapper(oProgressCom, self.oVBoxMgr, self.oTstDrv, 'delete disk %s' % (sHd));
                oProgress.wait();
                oProgress.logResult();

        self.oTstDrv.processPendingEvents();
        return fRc;

    def detachHd(self, sController = "IDE Controller", iPort = 0, iDevice = 0):
        """
        Detaches a HD, if attached, and returns a reference to it (IMedium).

        In order to delete the detached medium, the caller must first save
        the changes made in this session.

        Returns (fRc, oHd), where oHd is None unless fRc is True, and fRc is
        your standard success indicator.  Error information is logged.
        """

        # What's attached?
        try:
            oHd = self.o.machine.getMedium(sController, iPort, iDevice);
        except:
            if    self.oVBoxMgr.xcptIsOurXcptKind() \
              and self.oVBoxMgr.xcptIsEqual(None, self.oVBoxMgr.constants.VBOX_E_OBJECT_NOT_FOUND):
                reporter.log('No HD attached (to %s %s:%s)' % (sController, iPort, iDevice));
                return (True, None);
            return (reporter.errorXcpt('Error getting media at port %s, device %s, on %s.'
                                       % (iPort, iDevice, sController)), None);
        # Detach it.
        try:
            self.o.machine.detachDevice(sController, iPort, iDevice);
        except:
            return (reporter.errorXcpt('detachDevice("%s",%s,%s) failed on "%s"' \
                                       % (sController, iPort, iDevice, self.sName) ), None);
        reporter.log('detached HD ("%s",%s,%s) from %s' % (sController, iPort, iDevice, self.sName));
        return (True, oHd);

    def attachFloppy(self, sFloppy, sController = "Floppy Controller", iPort = 0, iDevice = 0):
        """
        Attaches a floppy image to a VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        # Input validation.
        ## @todo Fix this wrt to bootsector-xxx.img from the validationkit.zip.
        ##if not self.oTstDrv.isResourceFile(sFloppy):
        ##    reporter.fatal('"%s" is not in the resource set' % (sFloppy));
        ##     return None;

        if not self.ensureControllerAttached(sController):
            return False;

        # Find the floppy image, registering it if necessary (as immutable).
        sFullName = self.oTstDrv.getFullResourceName(sFloppy);
        try:
            oFloppy = self.oVBox.findFloppyImage(sFullName);
        except:
            try:
                if self.fpApiVer >= 4.1:
                    oFloppy = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_Floppy, vboxcon.AccessMode_ReadOnly, False);
                elif self.fpApiVer >= 4.0:
                    oFloppy = self.oVBox.openMedium(sFullName, vboxcon.DeviceType_Floppy, vboxcon.AccessMode_ReadOnly);
                else:
                    oFloppy = self.oVBox.openFloppyImage(sFullName, "");
            except:
                reporter.errorXcpt('failed to open floppy "%s"' % (sFullName));
                return False;
        ## @todo the following works but causes trouble below (asserts in main).
        #try:
        #    oFloppy.type = vboxcon.MediumType_Immutable;
        #except:
        #    reporter.errorXcpt('failed to make floppy "%s" immutable' % (sFullName));
        #    return False;

        # Attach it.
        fRc = True;
        try:
            if self.fpApiVer >= 4.0:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_Floppy, oFloppy);
            else:
                self.o.machine.attachDevice(sController, iPort, iDevice, vboxcon.DeviceType_Floppy, oFloppy.id);
        except:
            reporter.errorXcpt('attachDevice("%s",%s,%s,Floppy,"%s") failed on "%s"' \
                               % (sController, iPort, iDevice, oFloppy.id, self.sName) );
            fRc = False;
        else:
            reporter.log('attached "%s" to %s' % (sFloppy, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    def setupNic(self, sType, sXXX):
        """
        Attaches a HD to a VM.
        Returns True on success and False on failure.  Error information is logged.
        """
        if sType == "PCNet":        enmType = vboxcon.NetworkAdapterType_Am79C973;
        elif sType == "PCNetOld":   enmType = vboxcon.NetworkAdapterType_Am79C970A;
        elif sType == "E1000":      enmType = vboxcon.NetworkAdapterType_I82545EM; # MT Server
        elif sType == "E1000Desk":  enmType = vboxcon.NetworkAdapterType_I82540EM; # MT Desktop
        elif sType == "E1000Srv2":  enmType = vboxcon.NetworkAdapterType_I82543GC; # T Server
        elif sType == "Virtio":     enmType = vboxcon.NetworkAdapterType_Virtio;
        else:
            reporter.error('Invalid NIC type: "%s" (sXXX=%s)' % (sType, sXXX));
            return False;
        ## @todo Implement me!
        if enmType is not None: pass
        return True;

    def setupPreferredConfig(self):                                             # pylint: disable=R0914
        """
        Configures the VM according to the preferences of the guest type.
        """
        try:
            sOsTypeId = self.o.machine.OSTypeId;
        except:
            reporter.errorXcpt('failed to obtain the OSTypeId for "%s"' % (self.sName));
            return False;

        try:
            oOsType = self.oVBox.getGuestOSType(sOsTypeId);
        except:
            reporter.errorXcpt('getGuestOSType("%s") failed for "%s"' % (sOsTypeId, self.sName));
            return False;

        # get the attributes.
        try:
            #sFamilyId       = oOsType.familyId;
            #f64Bit          = oOsType.is64Bit;
            fIoApic         = oOsType.recommendedIOAPIC;
            fVirtEx         = oOsType.recommendedVirtEx;
            cMBRam          = oOsType.recommendedRAM;
            cMBVRam         = oOsType.recommendedVRAM;
            #cMBHdd          = oOsType.recommendedHDD;
            eNicType        = oOsType.adapterType;
            if self.fpApiVer >= 3.2:
                if self.fpApiVer >= 4.2:
                    fPae            = oOsType.recommendedPAE;
                    fUsbHid         = oOsType.recommendedUSBHID;
                    fHpet           = oOsType.recommendedHPET;
                    eStorCtlType    = oOsType.recommendedHDStorageController;
                else:
                    fPae            = oOsType.recommendedPae;
                    fUsbHid         = oOsType.recommendedUsbHid;
                    fHpet           = oOsType.recommendedHpet;
                    eStorCtlType    = oOsType.recommendedHdStorageController;
                eFirmwareType   = oOsType.recommendedFirmware;
            else:
                fPae            = False;
                fUsbHid         = False;
                fHpet           = False;
                eFirmwareType   = -1;
                eStorCtlType    = vboxcon.StorageControllerType_PIIX4;
        except:
            reporter.errorXcpt('exception reading IGuestOSType(%s) attribute' % (sOsTypeId));
            self.oTstDrv.processPendingEvents();
            return False;
        self.oTstDrv.processPendingEvents();

        # Do the setting. Continue applying settings on error in case the
        # caller ignores the return code
        fRc = True;
        if not self.enableIoApic(fIoApic):              fRc = False;
        if not self.enableVirtEx(fVirtEx):              fRc = False;
        if not self.enablePae(fPae):                    fRc = False;
        if not self.setRamSize(cMBRam):                 fRc = False;
        if not self.setVRamSize(cMBVRam):               fRc = False;
        if not self.setNicType(eNicType, 0):            fRc = False;
        if self.fpApiVer >= 3.2:
            if not self.setFirmwareType(eFirmwareType): fRc = False;
            if not self.enableUsbHid(fUsbHid):          fRc = False;
            if not self.enableHpet(fHpet):              fRc = False;
        if  eStorCtlType == vboxcon.StorageControllerType_PIIX3 \
         or eStorCtlType == vboxcon.StorageControllerType_PIIX4 \
         or eStorCtlType == vboxcon.StorageControllerType_ICH6:
            if not self.setStorageControllerType(eStorCtlType, "IDE Controller"):
                fRc = False;

        return fRc;

    def addUsbDeviceFilter(self, sName, sVendorId, sProductId):
        """
        Creates a USB device filter and inserts it into the VM.
        Returns True on success.
        Returns False on failure (logged).
        """
        fRc = True;

        try:
            usbDevFilter = self.o.machine.USBDeviceFilters.createDeviceFilter(sName);
            usbDevFilter.active = True;
            usbDevFilter.vendorId = sVendorId;
            usbDevFilter.productId = sProductId;
            try:
                self.o.machine.USBDeviceFilters.insertDeviceFilter(0, usbDevFilter);
            except:
                reporter.errorXcpt('insertDeviceFilter(%s) failed on "%s"' \
                                   % (0, self.sName) );
                fRc = False;
            else:
                reporter.log('inserted USB device filter "%s" to %s' % (sName, self.sName));
        except:
            reporter.errorXcpt('createDeviceFilter("%s") failed on "%s"' \
                               % (sName, self.sName) );
            fRc = False;
        return fRc;

    def getGuestPropertyValue(self, sName):
        """
        Gets a guest property value.
        Returns the value on success, None on failure (logged).
        """
        try:
            sValue = self.o.machine.getGuestPropertyValue(sName);
        except:
            reporter.errorXcpt('IMachine::getGuestPropertyValue("%s") failed' % (sName));
            return None;
        return sValue;

    def setGuestPropertyValue(self, sName, sValue):
        """
        Sets a guest property value.
        Returns the True on success, False on failure (logged).
        """
        try:
            self.o.machine.setGuestPropertyValue(sName, sValue);
        except:
            reporter.errorXcpt('IMachine::setGuestPropertyValue("%s","%s") failed' % (sName, sValue));
            return False;
        return True;

    def delGuestPropertyValue(self, sName):
        """
        Deletes a guest property value.
        Returns the True on success, False on failure (logged).
        """
        try:
            oMachine = self.o.machine;
            if self.fpApiVer >= 4.2:
                oMachine.deleteGuestProperty(sName);
            else:
                oMachine.setGuestPropertyValue(sName, '');
        except:
            reporter.errorXcpt('Unable to delete guest property "%s"' % (sName,));
            return False;
        return True;

    def setExtraData(self, sKey, sValue):
        """
        Sets extra data.
        Returns the True on success, False on failure (logged).
        """
        try:
            self.o.machine.setExtraData(sKey, sValue);
        except:
            reporter.errorXcpt('IMachine::setExtraData("%s","%s") failed' % (sKey, sValue));
            return False;
        return True;

    def getExtraData(self, sKey):
        """
        Gets extra data.
        Returns value on success, None on failure.
        """
        try:
            sValue = self.o.machine.getExtraData(sKey)
        except:
            reporter.errorXcpt('IMachine::setExtraData("%s","%s") failed' % (sKey, sValue))
            return None
        return sValue

    def setupTeleporter(self, fEnabled=True, uPort = 6500, sAddress = '', sPassword = ''):
        """
        Sets up the teleporter for the VM.
        Returns True on success, False on failure (logged).
        """
        try:
            self.o.machine.teleporterAddress  = sAddress;
            self.o.machine.teleporterPort     = uPort;
            self.o.machine.teleporterPassword = sPassword;
            self.o.machine.teleporterEnabled  = fEnabled;
        except:
            reporter.errorXcpt('setupTeleporter(%s, %s, %s, %s)' % (fEnabled, sPassword, uPort, sAddress));
            return False;
        return True;

    def enableTeleporter(self, fEnable=True):
        """
        Enables or disables the teleporter of the VM.
        Returns True on success, False on failure (logged).
        """
        try:
            self.o.machine.teleporterEnabled = fEnable;
        except:
            reporter.errorXcpt('IMachine::teleporterEnabled=%s failed' % (fEnable));
            return False;
        return True;

    def teleport(self, sHostname = 'localhost', uPort = 6500, sPassword = 'password', cMsMaxDowntime = 250):
        """
        Wrapper around the IConsole::teleport() method.
        Returns a progress object on success, None on failure (logged).
        """
        reporter.log2('"%s"::teleport(%s,%s,%s,%s)...' % (self.sName, sHostname, uPort, sPassword, cMsMaxDowntime));
        try:
            oProgress = self.o.console.teleport(sHostname, uPort, sPassword, cMsMaxDowntime)
        except:
            reporter.errorXcpt('IConsole::teleport(%s,%s,%s,%s) failed' % (sHostname, uPort, sPassword, cMsMaxDowntime));
            return None;
        return ProgressWrapper(oProgress, self.oVBoxMgr, self.oTstDrv, 'teleport %s' % (self.sName,));

    def getOsType(self):
        """
        Gets the IGuestOSType interface for the machine.

        return IGuestOSType interface on success, None + errorXcpt on failure.
        No exceptions raised.
        """
        try:
            sOsTypeId = self.o.machine.OSTypeId;
        except:
            reporter.errorXcpt('failed to obtain the OSTypeId for "%s"' % (self.sName));
            return None;

        try:
            oOsType = self.oVBox.getGuestOSType(sOsTypeId);
        except:
            reporter.errorXcpt('getGuestOSType("%s") failed for "%s"' % (sOsTypeId, self.sName));
            return None;

        return oOsType;

    def setOsType(self, sNewTypeId):
        """
        Changes the OS type.

        returns True on success, False + errorXcpt on failure.
        No exceptions raised.
        """
        try:
            self.o.machine.OSTypeId = sNewTypeId;
        except:
            reporter.errorXcpt('failed to set the OSTypeId for "%s" to "%s"' % (self.sName, sNewTypeId));
            return False;
        return True;


    def setParavirtProvider(self, iProvider):
        """
        Sets a paravirtualisation provider.
        Returns the True on success, False on failure (logged).
        """
        try:
            self.o.machine.paravirtProvider = iProvider
        except:
            reporter.errorXcpt('Unable to set paravirtualisation provider "%s"' % (iProvider,))
            return False;
        return True;



    #
    # IConsole wrappers.
    #

    def powerOff(self, fFudgeOnFailure = True):
        """
        Powers off the VM.

        Returns True on success.
        Returns False on IConsole::powerDown() failure.
        Returns None if the progress object returns failure.
        """
        try:
            oProgress = self.o.console.powerDown();
        except:
            reporter.logXcpt('IConsole::powerDown failed on %s' % (self.sName));
            if fFudgeOnFailure:
                self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000); # fudge
                self.waitForTask(1000);                                # fudge
            return False;

        rc = self.oTstDrv.waitOnProgress(oProgress);
        if rc < 0:
            self.close();
            if fFudgeOnFailure:
                vbox.reportError(oProgress, 'powerDown for "%s" failed' % (self.sName));
                self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000); # fudge
            return None;

        # Wait for the VM to really power off or we'll fail to open a new session to it.
        self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000);         # fudge
        return self.waitForTask(30 * 1000);                            # fudge

    def restoreSnapshot(self, oSnapshot, fFudgeOnFailure = True):
        """
        Restores the given snapshot.

        Returns True on success.
        Returns False on IMachine::restoreSnapshot() failure.
        Returns None if the progress object returns failure.
        """
        try:
            if self.fpApiVer >= 5.0:
                oProgress = self.o.machine.restoreSnapshot(oSnapshot);
            else:
                oProgress = self.o.console.restoreSnapshot(oSnapshot);
        except:
            reporter.logXcpt('IMachine::restoreSnapshot failed on %s' % (self.sName));
            if fFudgeOnFailure:
                self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000); # fudge
                self.waitForTask(1000);                                # fudge
            return False;

        rc = self.oTstDrv.waitOnProgress(oProgress);
        if rc < 0:
            self.close();
            if fFudgeOnFailure:
                vbox.reportError(oProgress, 'restoreSnapshot for "%s" failed' % (self.sName));
            return None;

        return self.waitForTask(30 * 1000);

    def deleteSnapshot(self, oSnapshot, fFudgeOnFailure = True, cMsTimeout = 30 * 1000):
        """
        Deletes the given snapshot merging the diff image into the base.

        Returns True on success.
        Returns False on IMachine::deleteSnapshot() failure.
        """
        try:
            if self.fpApiVer >= 5.0:
                oProgressCom = self.o.machine.deleteSnapshot(oSnapshot);
            else:
                oProgressCom = self.o.console.deleteSnapshot(oSnapshot);
            oProgress = ProgressWrapper(oProgressCom, self.oVBoxMgr, self.oTstDrv, 'Delete Snapshot %s' % (oSnapshot));
            oProgress.wait(cMsTimeout);
            oProgress.logResult();
        except:
            reporter.logXcpt('IMachine::deleteSnapshot failed on %s' % (self.sName));
            if fFudgeOnFailure:
                self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000); # fudge
                self.waitForTask(1000);                                # fudge
            return False;

        return True;

    def takeSnapshot(self, sName, sDescription = '', fPause = True, fFudgeOnFailure = True, cMsTimeout = 30 * 1000):
        """
        Takes a snapshot with the given name

        Returns True on success.
        Returns False on IMachine::takeSnapshot() or VM state change failure.
        """
        try:
            if     fPause is True \
               and self.oVM.state is vboxcon.MachineState_Running:
                self.o.console.pause();
            if self.fpApiVer >= 5.0:
                (oProgressCom, _) = self.o.machine.takeSnapshot(sName, sDescription, True);
            else:
                oProgressCom = self.o.console.takeSnapshot(sName, sDescription);
            oProgress = ProgressWrapper(oProgressCom, self.oVBoxMgr, self.oTstDrv, 'Take Snapshot %s' % (sName));
            oProgress.wait(cMsTimeout);
            oProgress.logResult();
        except:
            reporter.logXcpt('IMachine::takeSnapshot failed on %s' % (self.sName));
            if fFudgeOnFailure:
                self.oTstDrv.waitOnDirectSessionClose(self.oVM, 5000); # fudge
                self.waitForTask(1000);                                # fudge
            return False;

        if     fPause is True \
           and self.oVM.state is vboxcon.MachineState_Paused:
            self.o.console.resume();

        return True;

    def findSnapshot(self, sName):
        """
        Returns the snapshot object with the given name

        Returns snapshot object on success.
        Returns None if there is no snapshot with the given name.
        """
        return self.oVM.findSnapshot(sName);

    def takeScreenshot(self, sFilename, iScreenId=0):
        """
        Take screenshot from the given display and save it to specified file.

        Returns True on success
        Returns False on failure.
        """
        try:
            if self.fpApiVer >= 5.0:
                iWidth, iHeight, _, _, _, _ = self.o.console.display.getScreenResolution(iScreenId)
                aPngData = self.o.console.display.takeScreenShotToArray(iScreenId, iWidth, iHeight,
                                                                        vboxcon.BitmapFormat_PNG)
            else:
                iWidth, iHeight, _, _, _ = self.o.console.display.getScreenResolution(iScreenId)
                aPngData = self.o.console.display.takeScreenShotPNGToArray(iScreenId, iWidth, iHeight)
        except:
            reporter.logXcpt("Unable to take screenshot")
            return False

        oFile = open(sFilename, 'wb')
        oFile.write(aPngData)
        oFile.close()

        return True

    #
    # Other methods.
    #

    def getPrimaryIp(self):
        """
        Tries to obtain the primary IP address of the guest via the guest
        properties.

        Returns IP address on success.
        Returns empty string on failure.
        """
        sIpAddr = self.getGuestPropertyValue('/VirtualBox/GuestInfo/Net/0/V4/IP');
        if vbox.isIpAddrValid(sIpAddr):
            return sIpAddr;
        return '';

    def getPid(self):
        """
        Gets the process ID for the direct session unless it's ourselves.
        """
        if self.uPid is None and self.o is not None and self.fRemoteSession:
            try:
                if self.fpApiVer >= 4.2:
                    uPid = self.o.machine.sessionPID;
                else:
                    uPid = self.o.machine.sessionPid;
                if uPid != os.getpid() and uPid != 0xffffffff:
                    self.uPid = uPid;
            except Exception, oXcpt:
                if vbox.ComError.equal(oXcpt, vbox.ComError.E_UNEXPECTED):
                    try:
                        if self.fpApiVer >= 4.2:
                            uPid = self.oVM.sessionPID;
                        else:
                            uPid = self.oVM.sessionPid;
                        if uPid != os.getpid() and uPid != 0xffffffff:
                            self.uPid = uPid;
                    except:
                        reporter.log2Xcpt();
                else:
                    reporter.log2Xcpt();
            if self.uPid is not None:
                reporter.log2('getPid: %u' % (self.uPid,));
                self.oTstDrv.pidFileAdd(self.uPid);
        return self.uPid;

    def addLogsToReport(self, cReleaseLogs = 1):
        """
        Retrieves and adds the release and debug logs to the test report.
        """
        fRc = True;

        # Add each of the requested release logs to the report.
        for iLog in range(0, cReleaseLogs):
            try:
                if self.fpApiVer >= 3.2:
                    sLogFile = self.oVM.queryLogFilename(iLog);
                elif iLog > 0:
                    sLogFile = '%s/VBox.log' % (self.oVM.logFolder,);
                else:
                    sLogFile = '%s/VBox.log.%u' % (self.oVM.logFolder, iLog);
            except:
                reporter.logXcpt('iLog=%s' % (iLog,));
                fRc = False;
            else:
                if sLogFile is not None and sLogFile != '': # the None bit is for a 3.2.0 bug.
                    reporter.addLogFile(sLogFile, 'log/release/vm', '%s #%u' % (self.sName, iLog),
                                        sAltName = '%s-%s' % (self.sName, os.path.basename(sLogFile),));

        # Now for the hardened windows startup log.
        try:
            sLogFile = os.path.join(self.oVM.logFolder, 'VBoxStartup.log');
        except:
            reporter.logXcpt();
            fRc = False;
        else:
            if os.path.isfile(sLogFile):
                reporter.addLogFile(sLogFile, 'log/release/vm', '%s startup log' % (self.sName, ),
                                    sAltName = '%s-%s' % (self.sName, os.path.basename(sLogFile),));

        # Now for the debug log.
        if self.sLogFile is not None and os.path.isfile(self.sLogFile):
            reporter.addLogFile(self.sLogFile, 'log/debug/vm', '%s debug' % (self.sName, ),
                                sAltName = '%s-%s' % (self.sName, os.path.basename(self.sLogFile),));

        return fRc;

    def registerDerivedEventHandler(self, oSubClass, dArgs = None, fMustSucceed = True):
        """
        Create an instance of the given ConsoleEventHandlerBase sub-class and
        register it.

        The new instance is returned on success.  None is returned on error.
        """

        # We need a console object.
        try:
            oConsole = self.o.console;
        except Exception, oXcpt:
            if fMustSucceed or vbox.ComError.notEqual(oXcpt, vbox.ComError.E_UNEXPECTED):
                reporter.errorXcpt('Failed to get ISession::console for "%s"' % (self.sName, ));
            return None;

        # Add the base class arguments.
        dArgsCopy = dArgs.copy() if dArgs is not None else dict();
        dArgsCopy['oSession'] = self;
        dArgsCopy['oConsole'] = oConsole;
        sLogSuffix = 'on %s' % (self.sName,)
        return oSubClass.registerDerivedEventHandler(self.oVBoxMgr, self.fpApiVer, oSubClass, dArgsCopy,
                                                     oConsole, 'IConsole', 'IConsoleCallback',
                                                     fMustSucceed = fMustSucceed, sLogSuffix = sLogSuffix);

    def enableVmmDevTestingPart(self, fEnabled, fEnableMMIO = False):
        """
        Enables the testing part of the VMMDev.

        Returns True on success and False on failure.  Error information is logged.
        """
        fRc = True;
        try:
            self.o.machine.setExtraData('VBoxInternal/Devices/VMMDev/0/Config/TestingEnabled',
                                        '1' if fEnabled else '');
            self.o.machine.setExtraData('VBoxInternal/Devices/VMMDev/0/Config/TestingMMIO',
                                        '1' if fEnableMMIO and fEnabled else '');
        except:
            reporter.errorXcpt('VM name "%s", fEnabled=%s' % (self.sName, fEnabled));
            fRc = False;
        else:
            reporter.log('set VMMDevTesting=%s for "%s"' % (fEnabled, self.sName));
        self.oTstDrv.processPendingEvents();
        return fRc;

    #
    # Test eXecution Service methods.
    #

    def txsConnectViaTcp(self, cMsTimeout = 10*60000, sIpAddr = None, sMacAddr = None, fNatForwardingForTxs = False):
        """
        Connects to the TXS using TCP/IP as transport.  If no IP or MAC is
        addresses are specified, we'll get the IP from the guest additions.

        Returns a TxsConnectTask object on success, None + log on failure.
        """
        # If the VM is configured with a NAT interface, connect to local host.
        fReversedSetup = False;
        fUseNatForTxs  = False;
        if sIpAddr == None:
            try:
                oNic = self.oVM.getNetworkAdapter(0);
                if oNic.attachmentType == vboxcon.NetworkAttachmentType_NAT:
                    fUseNatForTxs = True;
            except:
                reporter.errorXcpt();
                return None;
        if fUseNatForTxs:
            fReversedSetup = not fNatForwardingForTxs;
            sIpAddr = '127.0.0.1';

        # Kick off the task.
        try:
            oTask = TxsConnectTask(self, cMsTimeout, sIpAddr, sMacAddr, fReversedSetup);
        except:
            reporter.errorXcpt();
            oTask = None;
        return oTask;

    def txsTryConnectViaTcp(self, cMsTimeout, sHostname, fReversed = False):
        """
        Attempts to connect to a TXS instance.

        Returns True if a connection was established, False if not (only grave
        failures are logged as errors).

        Note!   The timeout is more of a guideline...
        """

        if sHostname is None  or  sHostname.strip() == '':
            raise base.GenError('Empty sHostname is not implemented yet');

        oTxsSession = txsclient.tryOpenTcpSession(cMsTimeout, sHostname, fReversedSetup = fReversed,
                                                  cMsIdleFudge = cMsTimeout / 2);
        if oTxsSession is None:
            return False;

        # Wait for the connect task to time out.
        self.oTstDrv.addTask(oTxsSession);
        self.oTstDrv.processPendingEvents();
        oRc = self.oTstDrv.waitForTasks(cMsTimeout);
        self.oTstDrv.removeTask(oTxsSession);
        if oRc != oTxsSession:
            if oRc is not None:
                reporter.log('oRc=%s, expected %s' % (oRc, oTxsSession));
            self.oTstDrv.processPendingEvents();
            oTxsSession.cancelTask(); # this is synchronous
            return False;

        # Check the status.
        reporter.log2('TxsSession is ready, isSuccess() -> %s.' % (oTxsSession.isSuccess(),));
        if not oTxsSession.isSuccess():
            return False;

        reporter.log2('Disconnecting from TXS...');
        return oTxsSession.syncDisconnect();



class TxsConnectTask(TdTaskBase):
    """
    Class that takes care of connecting to a VM.
    """

    class TxsConnectTaskVBoxCallback(vbox.VirtualBoxEventHandlerBase):
        """ Class for looking for IPv4 address changes on interface 0."""
        def __init__(self, dArgs):
            vbox.VirtualBoxEventHandlerBase.__init__(self, dArgs);              # pylint: disable=W0233
            self.oParentTask = dArgs['oParentTask'];
            self.sMachineId  = dArgs['sMachineId'];

        def onGuestPropertyChange(self, sMachineId, sName, sValue, sFlags):
            """Look for IP address."""
            reporter.log2('onGuestPropertyChange(,%s,%s,%s,%s)' % (sMachineId, sName, sValue, sFlags));
            if    sMachineId == self.sMachineId \
              and sName  == '/VirtualBox/GuestInfo/Net/0/V4/IP':
                self.oParentTask._setIp(sValue);                                # pylint: disable=W0212


    def __init__(self, oSession, cMsTimeout, sIpAddr, sMacAddr, fReversedSetup):
        TdTaskBase.__init__(self, utils.getCallerName());
        self.cMsTimeout         = cMsTimeout;
        self.sIpAddr            = None;
        self.sNextIpAddr        = None;
        self.sMacAddr           = sMacAddr;
        self.fReversedSetup     = fReversedSetup;
        self.oVBox              = oSession.oVBox;
        self.oVBoxEventHandler  = None;
        self.oTxsSession        = None;
        self.fpApiVer           = oSession.fpApiVer;

        # Skip things we don't implement.
        if sMacAddr is not None:
            reporter.error('TxsConnectTask does not implement sMacAddr yet');
            raise base.GenError();

        reporter.log2('TxsConnectTask: sIpAddr=%s fReversedSetup=%s' % (sIpAddr, fReversedSetup))
        if fReversedSetup is True:
            self._openTcpSession(sIpAddr, fReversedSetup = True);
        elif sIpAddr is not None and sIpAddr.strip() != '':
            self._openTcpSession(sIpAddr, cMsIdleFudge = 5000);
        else:
            #
            # If we've got no IP address, register callbacks that listens for
            # the primary network adaptor of the VM to set a IPv4 guest prop.
            # Note! The order in which things are done here is kind of important.
            #

            # 0. The caller zaps the property before starting the VM.
            #try:
            #    oSession.delGuestPropertyValue('/VirtualBox/GuestInfo/Net/0/V4/IP');
            #except:
            #    reporter.logXcpt();

            # 1. Register the callback / event listener object.
            dArgs = {'oParentTask':self, 'sMachineId':oSession.o.machine.id};
            self.oVBoxEventHandler = self.oVBox.registerDerivedEventHandler(self.TxsConnectTaskVBoxCallback, dArgs);

            # 2. Query the guest properties.
            try:
                sIpAddr = oSession.getGuestPropertyValue('/VirtualBox/GuestInfo/Net/0/V4/IP');
            except:
                reporter.errorXcpt('IMachine::getGuestPropertyValue("/VirtualBox/GuestInfo/Net/0/V4/IP") failed');
                self._deregisterEventHandler();
                raise;
            else:
                if sIpAddr is not None:
                    self._setIp(sIpAddr);
        # end __init__

    def __del__(self):
        """ Make sure we deregister the callback. """
        self._deregisterEventHandler();
        return TdTaskBase.__del__(self);

    def toString(self):
        return '<%s cMsTimeout=%s, sIpAddr=%s, sNextIpAddr=%s, sMacAddr=%s, fReversedSetup=%s,' \
               ' oTxsSession=%s oVBoxEventHandler=%s, oVBox=%s>' \
             % (TdTaskBase.toString(self), self.cMsTimeout, self.sIpAddr, self.sNextIpAddr, self.sMacAddr, self.fReversedSetup,
                self.oTxsSession, self.oVBoxEventHandler, self.oVBox);

    def _deregisterEventHandler(self):
        """Deregisters the event handler."""
        fRc = True;
        if self.oVBoxEventHandler is not None:
            fRc = self.oVBoxEventHandler.unregister();
            self.oVBoxEventHandler = None;
        return fRc;

    def _setIp(self, sIpAddr, fInitCall = False):
        """Called when we get an IP. Will create a TXS session and signal the task."""
        sIpAddr = sIpAddr.strip();

        if   sIpAddr is not None \
         and sIpAddr != '':
            if vbox.isIpAddrValid(sIpAddr) or fInitCall:
                try:
                    for s in sIpAddr.split('.'):
                        i = int(s);
                        if str(i) != s:
                            raise Exception();
                except:
                    reporter.fatalXcpt();
                else:
                    self._openTcpSession(sIpAddr, cMsIdleFudge = 5000);
                    return None;

            reporter.log('TxsConnectTask: Ignoring Bad ip "%s"' % (sIpAddr));
        else:
            reporter.log2('TxsConnectTask: Ignoring empty ip "%s"' % (sIpAddr));
        return None;

    def _openTcpSession(self, sIpAddr, uPort = None, fReversedSetup = False, cMsIdleFudge = 0):
        """
        Calls txsclient.openTcpSession and switches our task to reflect the
        state of the subtask.
        """
        self.oCv.acquire();
        if self.oTxsSession is None:
            reporter.log2('_openTcpSession: sIpAddr=%s, uPort=%d, fReversedSetup=%s' % \
                          (sIpAddr, uPort if uPort is not None else 0, fReversedSetup));
            self.sIpAddr     = sIpAddr;
            self.oTxsSession = txsclient.openTcpSession(self.cMsTimeout, sIpAddr, uPort, \
                                                        fReversedSetup, cMsIdleFudge);
            self.oTxsSession.setTaskOwner(self);
        else:
            self.sNextIpAddr = sIpAddr;
            reporter.log2('_openTcpSession: sNextIpAddr=%s' % (sIpAddr,));
        self.oCv.release();
        return None;

    def notifyAboutReadyTask(self, oTxsSession):
        """
        Called by the TXS session task when it's done.

        We'll signal the task completed or retry depending on the result.
        """

        self.oCv.acquire();

        # Disassociate ourselves with the session (avoid cyclic ref)
        oTxsSession.setTaskOwner(None);
        fSuccess = oTxsSession.isSuccess();
        if self.oTxsSession is not None:
            if not fSuccess:
                self.oTxsSession = None;
            if fSuccess and self.fReversedSetup:
                self.sIpAddr = oTxsSession.oTransport.sHostname;
        else:
            fSuccess = False;

        # Signal done, or retry?
        if   fSuccess \
          or self.fReversedSetup \
          or self.getAgeAsMs() >= self.cMsTimeout:
            self.signalTaskLocked();
        else:
            sIpAddr = self.sNextIpAddr if self.sNextIpAddr is not None else self.sIpAddr;
            self._openTcpSession(sIpAddr, cMsIdleFudge = 5000);

        self.oCv.release();
        return True;

    #
    # Public methods
    #

    def getResult(self):
        """
        Returns the connected TXS session object on success.
        Returns None on failure or if the task has not yet completed.
        """
        self.oCv.acquire();
        oTxsSession = self.oTxsSession;
        self.oCv.release();

        if oTxsSession is not None  and  not oTxsSession.isSuccess():
            oTxsSession = None;
        return oTxsSession;

    def cancelTask(self):
        """ Cancels the task. """
        self.oCv.acquire();
        if not self.fSignalled:
            oTxsSession = self.oTxsSession;
            if oTxsSession is not None:
                self.oCv.release();
                oTxsSession.setTaskOwner(None);
                oTxsSession.cancelTask();
                oTxsSession.waitForTask(1000);
                self.oCv.acquire();
            self.signalTaskLocked();
        self.oCv.release();
        return True;

