#ifndef STFT_GOOD_DATA_H
#define STFT_GOOD_DATA_H

#include <complex>

constexpr std::complex<double> operator "" _i (long double imag)
{
    return std::complex<double>{0.0, static_cast<double>(imag)};
}

/*!
    Data generated using:

        import scipy.signal
        import numpy as np
        np.set_printoptions(threshold=10_000, precision=12, linewidth=200)

        npoints     = 512
        fftsize     = 128
        wlen        = 64
        onesided    = False

        tone    = np.sin(2 * np.pi * np.arange(npoints) / npoints)
        win     = scipy.signal.get_window('hann', wlen)
        scale   = win.sum()

        f, t, Z = scipy.signal.stft(tone,
                        nfft=fftsize, nperseg=wlen, noverlap=wlen//2,
                        window=win, return_onesided=onesided)

        print(repr(Z.T * scale)
!*/
static const std::complex<double> STFT_FFT_128_WLEN_64_TONE_512[17][128] =
{{1.8561830307012523e+00+0.0000000000000000e+00_i, -1.1231248390613415e+00-1.3647140589464484e+00_i, -2.9938833406609222e-01+1.4907693782657505e+00_i,
1.0386798021125028e+00-5.3384012727501551e-01_i, -7.0345957258712155e-01-3.3704182679810218e-01_i,  6.1312189143079170e-02+4.2305593058264429e-01_i,
1.4494751578417894e-01-8.2224738731068669e-02_i,  1.3630263330467025e-02-6.9924186916001690e-02_i, -9.6735350049332869e-02-3.2690642511285034e-02_i,
4.6967388427158569e-03+9.1649439085613030e-02_i,  5.2246138542576773e-02-1.6302636524914357e-02_i,  2.0436217564538672e-03-3.3211809089683388e-02_i,
-3.9400503850194127e-02-9.3014999699242464e-03_i,  1.0303722214941771e-03+3.9586717107854399e-02_i,  2.7102794310344465e-02-5.8065391083880142e-03_i,
5.7541583529767534e-04-1.9609876109918833e-02_i, -2.1876470763211614e-02-3.8665059247377283e-03_i,  3.4649412236941075e-04+2.2276450828333058e-02_i,
1.6808546635877594e-02-2.7028938973390482e-03_i,  2.2110416902670427e-04-1.3124982483492751e-02_i, -1.4182711865637229e-02-1.9623967533114206e-03_i,
1.4774881545893785e-04+1.4481443405078877e-02_i,  1.1619263424342802e-02-1.4685466549569062e-03_i,  1.0251576011318481e-04-9.5483414452172226e-03_i,
-1.0120618008741774e-02-1.1263908912705085e-03_i,  7.3396906273871907e-05+1.0327773571517712e-02_i,  8.6514816678518840e-03-8.8169762528260334e-04_i,
5.3967339990779786e-05-7.3791991748171203e-03_i, -7.7208605005761911e-03-7.0193814140195659e-04_i,  4.0604235057117887e-05+7.8652123045483031e-03_i,
6.8060263495653877e-03-5.6678520846009618e-04_i,  3.1172526894210160e-05-5.9749922520765464e-03_i, -6.1934992250489707e-03-4.6308569550751466e-04_i,
2.4365585495242561e-05+6.2957442864748281e-03_i,  5.5902140742917867e-03-3.8206879657010046e-04_i,  1.9357299940758711e-05-5.0238558552534278e-03_i,
-5.1706035925751480e-03-3.1773025343883937e-04_i,  1.5610327484893392e-05+5.2439812572931000e-03_i,  4.7567994440496824e-03-2.6586308547893369e-04_i,
1.2766338805083699e-05-4.3600308778223244e-03_i, -4.4619850359668024e-03-2.2345760457617887e-04_i,  1.0581077101041981e-05+4.5148687178461185e-03_i,
4.1710414019030606e-03-1.8831955894074064e-04_i,  8.8847894962346824e-06-3.8892811080799769e-03_i, -3.9616128878303828e-03-1.5882099254863275e-04_i,
7.5574810990733897e-06+3.9993434864334949e-03_i,  3.7549071583387111e-03-1.3373400329445839e-04_i,  6.5130753389552182e-06-3.5553598263076275e-03_i,
-3.6070629908326619e-03-1.1211750891929571e-04_i,  5.6890634545074016e-06+3.6330630124695625e-03_i,  3.4612376444633653e-03-9.3238620330401457e-05_i,
5.0396167193909186e-06-3.3236544465719969e-03_i, -3.3602584269725594e-03-7.6517026828651184e-05_i,  4.5309309438662515e-06+3.3766875759967622e-03_i,
3.2608615513319179e-03-6.1484924932577248e-05_i,  4.1380402583208672e-06-3.1727127796792737e-03_i, -3.1976797184820882e-03-4.7757579136529366e-05_i,
3.8426183936986560e-06+3.2058142331710879e-03_i,  3.1358167052790431e-03-3.5011215880668967e-05_i,  3.6314588135258341e-06-3.0896868030566016e-03_i,
-3.1051643536058049e-03-2.2965984515677107e-05_i,  3.4954342108539649e-06+3.1056051759518755e-03_i,  3.0756893716968603e-03-1.1372385953722919e-05_i,
3.4288065915433208e-06-3.0678878722273373e-03_i, -3.0751229889927068e-03-0.0000000000000000e+00_i,  3.4288065915433208e-06+3.0678878722273373e-03_i,
3.0756893716968603e-03+1.1372385953722919e-05_i,  3.4954342108539649e-06-3.1056051759518755e-03_i, -3.1051643536058049e-03+2.2965984515677107e-05_i,
3.6314588135258341e-06+3.0896868030566016e-03_i,  3.1358167052790431e-03+3.5011215880668967e-05_i,  3.8426183936986560e-06-3.2058142331710879e-03_i,
-3.1976797184820882e-03+4.7757579136529366e-05_i,  4.1380402583208672e-06+3.1727127796792737e-03_i,  3.2608615513319179e-03+6.1484924932577248e-05_i,
4.5309309438662515e-06-3.3766875759967622e-03_i, -3.3602584269725594e-03+7.6517026828651184e-05_i,  5.0396167193909186e-06+3.3236544465719969e-03_i,
3.4612376444633653e-03+9.3238620330401457e-05_i,  5.6890634545074016e-06-3.6330630124695625e-03_i, -3.6070629908326619e-03+1.1211750891929571e-04_i,
6.5130753389552182e-06+3.5553598263076275e-03_i,  3.7549071583387111e-03+1.3373400329445839e-04_i,  7.5574810990733897e-06-3.9993434864334949e-03_i,
-3.9616128878303828e-03+1.5882099254863275e-04_i,  8.8847894962346824e-06+3.8892811080799769e-03_i,  4.1710414019030606e-03+1.8831955894074064e-04_i,
1.0581077101041981e-05-4.5148687178461185e-03_i, -4.4619850359668024e-03+2.2345760457617887e-04_i,  1.2766338805083699e-05+4.3600308778223244e-03_i,
4.7567994440496824e-03+2.6586308547893369e-04_i,  1.5610327484893392e-05-5.2439812572931000e-03_i, -5.1706035925751480e-03+3.1773025343883937e-04_i,
1.9357299940758711e-05+5.0238558552534278e-03_i,  5.5902140742917867e-03+3.8206879657010046e-04_i,  2.4365585495242561e-05-6.2957442864748281e-03_i,
-6.1934992250489707e-03+4.6308569550751466e-04_i,  3.1172526894210160e-05+5.9749922520765464e-03_i,  6.8060263495653877e-03+5.6678520846009618e-04_i,
4.0604235057117887e-05-7.8652123045483031e-03_i, -7.7208605005761911e-03+7.0193814140195659e-04_i,  5.3967339990779786e-05+7.3791991748171203e-03_i,
8.6514816678518840e-03+8.8169762528260334e-04_i,  7.3396906273871907e-05-1.0327773571517712e-02_i, -1.0120618008741774e-02+1.1263908912705085e-03_i,
1.0251576011318481e-04+9.5483414452172226e-03_i,  1.1619263424342802e-02+1.4685466549569062e-03_i,  1.4774881545893785e-04-1.4481443405078877e-02_i,
-1.4182711865637229e-02+1.9623967533114206e-03_i,  2.2110416902670427e-04+1.3124982483492751e-02_i,  1.6808546635877594e-02+2.7028938973390482e-03_i,
3.4649412236941075e-04-2.2276450828333058e-02_i, -2.1876470763211614e-02+3.8665059247377283e-03_i,  5.7541583529767534e-04+1.9609876109918833e-02_i,
2.7102794310344465e-02+5.8065391083880142e-03_i,  1.0303722214941771e-03-3.9586717107854399e-02_i, -3.9400503850194127e-02+9.3014999699242464e-03_i,
2.0436217564538672e-03+3.3211809089683388e-02_i,  5.2246138542576773e-02+1.6302636524914357e-02_i,  4.6967388427158569e-03-9.1649439085613030e-02_i,
-9.6735350049332869e-02+3.2690642511285034e-02_i,  1.3630263330467025e-02+6.9924186916001690e-02_i,  1.4494751578417894e-01+8.2224738731068669e-02_i,
6.1312189143079170e-02-4.2305593058264429e-01_i, -7.0345957258712155e-01+3.3704182679810218e-01_i,  1.0386798021125028e+00+5.3384012727501551e-01_i,
-2.9938833406609222e-01-1.4907693782657505e+00_i, -1.1231248390613415e+00+1.3647140589464484e+00_i},
{ 1.2122964709152136e+01+0.0000000000000000e+00_i, -2.0752641025276124e+00-1.0315233231602493e+01_i, -6.1327938027951330e+00+2.7545826325486065e+00_i,
1.9192300200092292e+00+2.1457730934321857e+00_i,  5.8289519203533022e-02-6.2277209075796147e-01_i,  1.1329015328550315e-01+2.7981992439779746e-01_i,
8.4778112363261098e-03-1.5193150635944502e-01_i,  2.5185442627515299e-02+9.2255822038169566e-02_i,  2.4433908589009804e-03-6.0404431041638958e-02_i,
8.6784417726719926e-03+4.1774934375973342e-02_i,  9.6043792566970557e-04-3.0123344422678975e-02_i,  3.7761206259648234e-03+2.2450726954303814e-02_i,
4.5318101000886221e-04-1.7186930887734632e-02_i,  1.9038796126131506e-03+1.3452857203858885e-02_i,  2.4148402692576775e-04-1.0729085273931630e-02_i,
1.0632298258292229e-03+8.6948058311676242e-03_i,  1.4040535125631865e-04-7.1443713723977238e-03_i,  6.4023765558462748e-04+5.9417117578475787e-03_i,
8.7188073986591138e-05-4.9942967006023058e-03_i,  4.0854723263349384e-04+4.2376833522191400e-03_i,  5.7004467792667380e-05-3.6260363901020336e-03_i,
2.7300421311067980e-04+3.1261405426053833e-03_i,  3.8850553163751454e-05-2.7135203941051770e-03_i,  1.8942442505678523e-04+2.3698675119102888e-03_i,
2.7401034077476549e-05-2.0812989801038007e-03_i,  1.3561979891217596e-04+1.8371565916130538e-03_i,  1.9891188401001902e-05-1.6291647797249573e-03_i,
9.9718641682921909e-05+1.4508309775053893e-03_i,  1.4800642651141782e-05-1.2970125638602807e-03_i,  7.5026843405139547e-05+1.1636000473283718e-03_i,
1.1252161239359978e-05-1.0472825068528280e-03_i,  5.7599319148801315e-05+9.4537200485835437e-04_i,  8.7185392332322920e-06-8.5567079175641325e-04_i,
4.5021731472841636e-05+7.7636991029951996e-04_i,  6.8716486042021217e-06-7.0597108232429377e-04_i,  3.5767626439826334e-05+6.4322602786334097e-04_i,
5.5009669056733099e-06-5.8708895602360949e-04_i,  2.8844124118166032e-05+5.3667909379744722e-04_i,  4.4677887551096982e-06-4.9125092624851077e-04_i,
2.3589118254275425e-05+4.5017040573157720e-04_i,  3.6785649226598813e-06-4.1289581450384082e-04_i,  1.9551281131243098e-05+3.7896227955623520e-04_i,
3.0688872200019526e-06-3.4796917215373317e-04_i,  1.6416950332330841e-05+3.1956979958542740e-04_i,  2.5935987190811019e-06-2.9346292869760664e-04_i,
1.3964404209987303e-05+2.6938578112007403e-04_i,  2.2205343968766198e-06-2.4710821688961153e-04_i,  1.2034593998569258e-05+2.2642788228049993e-04_i,
1.9264717175326244e-06-2.0716614345346954e-04_i,  1.0512018569678195e-05+1.8916466318881844e-04_i,  1.6944594984215182e-06-1.7228250592599571e-04_i,
9.3119974773638557e-06+1.5639367832229016e-04_i,  1.5120258777291673e-06-1.4138502995106501e-04_i,  8.3720687245282185e-06+1.2715445256515114e-04_i,
1.3699589433893017e-06-1.1360932740646021e-04_i,  7.6461013989188170e-06+1.0066517885123222e-04_i,  1.2614682101627119e-06-8.8244499773244833e-05_i,
7.1002329702744837e-06+7.6275719710638956e-05_i,  1.1816048867234474e-06-6.4692291520909717e-05_i,  6.7100609419321056e-06+5.3431875892878677e-05_i,
1.1268623086001239e-06-4.2435606076196564e-05_i,  6.4587202496246832e-06+3.1647417549462986e-05_i,  1.0949056790110490e-06-2.1013429236882430e-05_i,
6.3356084614163422e-06+1.0481364345338307e-05_i,  1.0843985052133576e-06+0.0000000000000000e+00_i,  6.3356084614163422e-06-1.0481364345338307e-05_i,
1.0949056790110490e-06+2.1013429236882430e-05_i,  6.4587202496246832e-06-3.1647417549462986e-05_i,  1.1268623086001239e-06+4.2435606076196564e-05_i,
6.7100609419321056e-06-5.3431875892878677e-05_i,  1.1816048867234474e-06+6.4692291520909717e-05_i,  7.1002329702744837e-06-7.6275719710638956e-05_i,
1.2614682101627119e-06+8.8244499773244833e-05_i,  7.6461013989188170e-06-1.0066517885123222e-04_i,  1.3699589433893017e-06+1.1360932740646021e-04_i,
8.3720687245282185e-06-1.2715445256515114e-04_i,  1.5120258777291673e-06+1.4138502995106501e-04_i,  9.3119974773638557e-06-1.5639367832229016e-04_i,
1.6944594984215182e-06+1.7228250592599571e-04_i,  1.0512018569678195e-05-1.8916466318881844e-04_i,  1.9264717175326244e-06+2.0716614345346954e-04_i,
1.2034593998569258e-05-2.2642788228049993e-04_i,  2.2205343968766198e-06+2.4710821688961153e-04_i,  1.3964404209987303e-05-2.6938578112007403e-04_i,
2.5935987190811019e-06+2.9346292869760664e-04_i,  1.6416950332330841e-05-3.1956979958542740e-04_i,  3.0688872200019526e-06+3.4796917215373317e-04_i,
1.9551281131243098e-05-3.7896227955623520e-04_i,  3.6785649226598813e-06+4.1289581450384082e-04_i,  2.3589118254275425e-05-4.5017040573157720e-04_i,
4.4677887551096982e-06+4.9125092624851077e-04_i,  2.8844124118166032e-05-5.3667909379744722e-04_i,  5.5009669056733099e-06+5.8708895602360949e-04_i,
3.5767626439826334e-05-6.4322602786334097e-04_i,  6.8716486042021217e-06+7.0597108232429377e-04_i,  4.5021731472841636e-05-7.7636991029951996e-04_i,
8.7185392332322920e-06+8.5567079175641325e-04_i,  5.7599319148801315e-05-9.4537200485835437e-04_i,  1.1252161239359978e-05+1.0472825068528280e-03_i,
7.5026843405139547e-05-1.1636000473283718e-03_i,  1.4800642651141782e-05+1.2970125638602807e-03_i,  9.9718641682921909e-05-1.4508309775053893e-03_i,
1.9891188401001902e-05+1.6291647797249573e-03_i,  1.3561979891217596e-04-1.8371565916130538e-03_i,  2.7401034077476549e-05+2.0812989801038007e-03_i,
1.8942442505678523e-04-2.3698675119102888e-03_i,  3.8850553163751454e-05+2.7135203941051770e-03_i,  2.7300421311067980e-04-3.1261405426053833e-03_i,
5.7004467792667380e-05+3.6260363901020336e-03_i,  4.0854723263349384e-04-4.2376833522191400e-03_i,  8.7188073986591138e-05+4.9942967006023058e-03_i,
6.4023765558462748e-04-5.9417117578475787e-03_i,  1.4040535125631865e-04+7.1443713723977238e-03_i,  1.0632298258292229e-03-8.6948058311676242e-03_i,
2.4148402692576775e-04+1.0729085273931630e-02_i,  1.9038796126131506e-03-1.3452857203858885e-02_i,  4.5318101000886221e-04+1.7186930887734632e-02_i,
3.7761206259648234e-03-2.2450726954303814e-02_i,  9.6043792566970557e-04+3.0123344422678975e-02_i,  8.6784417726719926e-03-4.1774934375973342e-02_i,
2.4433908589009804e-03+6.0404431041638958e-02_i,  2.5185442627515299e-02-9.2255822038169566e-02_i,  8.4778112363261098e-03+1.5193150635944502e-01_i,
1.1329015328550315e-01-2.7981992439779746e-01_i,  5.8289519203533022e-02+6.2277209075796147e-01_i,  1.9192300200092292e+00-2.1457730934321857e+00_i,
-6.1327938027951330e+00-2.7545826325486065e+00_i, -2.0752641025276124e+00+1.0315233231602493e+01_i},
{ 2.2400317936284608e+01+0.0000000000000000e+00_i, -1.5883383796386488e+00-1.9060065711515598e+01_i, -1.1331925343028967e+01+2.1082662731139310e+00_i,
1.4689150631105035e+00+3.9648716848708503e+00_i,  1.0770498750413532e-01-4.7664912254487979e-01_i,  8.6708529424927266e-02+5.1703980187996201e-01_i,
1.5664952563471779e-02-1.1628334067606193e-01_i,  1.9276103260662734e-02+1.7046653147213683e-01_i,  4.5147976089274632e-03-4.6231550002149430e-02_i,
6.6421917702935099e-03+7.7190013683927694e-02_i,  1.7746578835482222e-03-2.3055409675972999e-02_i,  2.8901176043377649e-03+4.1483534246161538e-02_i,
8.3736931933973751e-04-1.3154307407879838e-02_i,  1.4571663699294961e-03+2.4857638848884488e-02_i,  4.4620429981054571e-04-8.2116863575321421e-03_i,
8.1376087828166285e-04+1.6065906293151321e-02_i,  2.5943526056193306e-04-5.4680651177607849e-03_i,  4.9001668713756540e-04+1.0978851762312999e-02_i,
1.6110255407056286e-04-3.8224692072729330e-03_i,  3.1268851453449334e-04+7.8302178287577365e-03_i,  1.0533052211045778e-04-2.7752481032899037e-03_i,
2.0894837864701937e-04+5.7763545261343106e-03_i,  7.1786461789760680e-05-2.0768385964179035e-03_i,  1.4497917830907979e-04+4.3789441780335680e-03_i,
5.0630509107735158e-05-1.5929572749680137e-03_i,  1.0379890028870242e-04+3.3946227460197216e-03_i,  3.6754123681898986e-05-1.2469087395869940e-03_i,
7.6321344140117082e-05+2.6807860905016146e-03_i,  2.7348021626927255e-05-9.9269043951714844e-04_i,  5.7423059907825414e-05+2.1500525355117629e-03_i,
2.0791282931468576e-05-8.0155532875712083e-04_i,  4.4084610306648989e-05+1.7468196917953976e-03_i,  1.6109759902072085e-05-6.5490207112794963e-04_i,
3.4458141462790268e-05+1.4345445395663064e-03_i,  1.2697150999337563e-05-5.4032687386893592e-04_i,  2.7375356106783277e-05+1.1885267238431231e-03_i,
1.0164461466058460e-05-4.4933843358933769e-04_i,  2.2076336842472748e-05+9.9165366057242911e-04_i,  8.2553971730265447e-06-3.7598718121872204e-04_i,
1.8054329479789901e-05+8.3180644799513495e-04_i,  6.7971016823700531e-06-3.1601677500734987e-04_i,  1.4963902740676590e-05+7.0023098735237438e-04_i,
5.6705641805394088e-06-2.6632407431380822e-04_i,  1.2564989804652151e-05+5.9048799409136890e-04_i,  4.7923455438350724e-06-2.2460680165159257e-04_i,
1.0687892267696036e-05+4.9776001905264041e-04_i,  4.1030125604586253e-06-1.8912844120945764e-04_i,  9.2108794774582552e-06+4.1838417205724860e-04_i,
3.5596555788576772e-06-1.5855810169386822e-04_i,  8.0455506946720823e-06+3.4953072118950462e-04_i,  3.1309528985253368e-06-1.3185932140805361e-04_i,
7.1270943134867093e-06+2.8897783683197106e-04_i,  2.7938595217693631e-06-1.0821141709321802e-04_i,  6.4077039911430989e-06+2.3495079238519331e-04_i,
2.5313540560961342e-06-8.6952814720998514e-05_i,  5.8520726551204022e-06+1.8600499675461984e-04_i,  2.3308893211063803e-06-6.7539416121035339e-05_i,
5.4342830471086623e-06+1.4093915253651779e-04_i,  2.1833211414148956e-06-4.9513336333559255e-05_i,  5.1356583052331040e-06+9.8729233042216258e-05_i,
2.0821700454851100e-06-3.2478806775287916e-05_i,  4.9432904677404466e-06+5.8476802661244776e-05_i,  2.0231218949007257e-06-1.6082982452481076e-05_i,
4.8490647842625023e-06+1.9367035983819392e-05_i,  2.0037071681855423e-06+0.0000000000000000e+00_i,  4.8490647842625023e-06-1.9367035983819392e-05_i,
2.0231218949007257e-06+1.6082982452481076e-05_i,  4.9432904677404466e-06-5.8476802661244776e-05_i,  2.0821700454851100e-06+3.2478806775287916e-05_i,
5.1356583052331040e-06-9.8729233042216258e-05_i,  2.1833211414148956e-06+4.9513336333559255e-05_i,  5.4342830471086623e-06-1.4093915253651779e-04_i,
2.3308893211063803e-06+6.7539416121035339e-05_i,  5.8520726551204022e-06-1.8600499675461984e-04_i,  2.5313540560961342e-06+8.6952814720998514e-05_i,
6.4077039911430989e-06-2.3495079238519331e-04_i,  2.7938595217693631e-06+1.0821141709321802e-04_i,  7.1270943134867093e-06-2.8897783683197106e-04_i,
3.1309528985253368e-06+1.3185932140805361e-04_i,  8.0455506946720823e-06-3.4953072118950462e-04_i,  3.5596555788576772e-06+1.5855810169386822e-04_i,
9.2108794774582552e-06-4.1838417205724860e-04_i,  4.1030125604586253e-06+1.8912844120945764e-04_i,  1.0687892267696036e-05-4.9776001905264041e-04_i,
4.7923455438350724e-06+2.2460680165159257e-04_i,  1.2564989804652151e-05-5.9048799409136890e-04_i,  5.6705641805394088e-06+2.6632407431380822e-04_i,
1.4963902740676590e-05-7.0023098735237438e-04_i,  6.7971016823700531e-06+3.1601677500734987e-04_i,  1.8054329479789901e-05-8.3180644799513495e-04_i,
8.2553971730265447e-06+3.7598718121872204e-04_i,  2.2076336842472748e-05-9.9165366057242911e-04_i,  1.0164461466058460e-05+4.4933843358933769e-04_i,
2.7375356106783277e-05-1.1885267238431231e-03_i,  1.2697150999337563e-05+5.4032687386893592e-04_i,  3.4458141462790268e-05-1.4345445395663064e-03_i,
1.6109759902072085e-05+6.5490207112794963e-04_i,  4.4084610306648989e-05-1.7468196917953976e-03_i,  2.0791282931468576e-05+8.0155532875712083e-04_i,
5.7423059907825414e-05-2.1500525355117629e-03_i,  2.7348021626927255e-05+9.9269043951714844e-04_i,  7.6321344140117082e-05-2.6807860905016146e-03_i,
3.6754123681898986e-05+1.2469087395869940e-03_i,  1.0379890028870242e-04-3.3946227460197216e-03_i,  5.0630509107735158e-05+1.5929572749680137e-03_i,
1.4497917830907979e-04-4.3789441780335680e-03_i,  7.1786461789760680e-05+2.0768385964179035e-03_i,  2.0894837864701937e-04-5.7763545261343106e-03_i,
1.0533052211045778e-04+2.7752481032899037e-03_i,  3.1268851453449334e-04-7.8302178287577365e-03_i,  1.6110255407056286e-04+3.8224692072729330e-03_i,
4.9001668713756540e-04-1.0978851762312999e-02_i,  2.5943526056193306e-04+5.4680651177607849e-03_i,  8.1376087828166285e-04-1.6065906293151321e-02_i,
4.4620429981054571e-04+8.2116863575321421e-03_i,  1.4571663699294961e-03-2.4857638848884488e-02_i,  8.3736931933973751e-04+1.3154307407879838e-02_i,
2.8901176043377649e-03-4.1483534246161538e-02_i,  1.7746578835482222e-03+2.3055409675972999e-02_i,  6.6421917702935099e-03-7.7190013683927694e-02_i,
4.5147976089274632e-03+4.6231550002149430e-02_i,  1.9276103260662734e-02-1.7046653147213683e-01_i,  1.5664952563471779e-02+1.1628334067606193e-01_i,
8.6708529424927266e-02-5.1703980187996201e-01_i,  1.0770498750413532e-01+4.7664912254487979e-01_i,  1.4689150631105035e+00-3.9648716848708503e+00_i,
-1.1331925343028967e+01-2.1082662731139310e+00_i, -1.5883383796386488e+00+1.9060065711515598e+01_i},
{ 2.9267425817005492e+01+0.0000000000000000e+00_i, -8.5960253677296627e-01-2.4903175966776388e+01_i, -1.4805873973945676e+01+1.1409854850790166e+00_i,
7.9497110360140955e-01+5.1803545039392525e+00_i,  1.4072334780537615e-01-2.5796064625939596e-01_i,  4.6926317974181883e-02+6.7554505650337715e-01_i,
2.0467246865977323e-02-6.2932090485856074e-02_i,  1.0432151910686992e-02+2.2272525677242766e-01_i,  5.8988673497371450e-03-2.5020334564877355e-02_i,
3.5947282825056432e-03+1.0085361313772051e-01_i,  2.3187022659699233e-03-1.2477497803910057e-02_i,  1.5641203764309927e-03+5.4200849498215367e-02_i,
1.0940757405736407e-03-7.1190598692685934e-03_i,  7.8861275667058747e-04+3.2478070314224651e-02_i,  5.8299401290107591e-04-4.4441326323197018e-03_i,
4.4040421377890271e-04+2.0991118159805942e-02_i,  3.3896850323407912e-04-2.9592955170785369e-03_i,  2.6519512008401757e-04+1.4344561109506732e-02_i,
2.1049063069539990e-04-2.0687054279054304e-03_i,  1.6922580462655533e-04+1.0230672621969738e-02_i,  1.3762095926060766e-04-1.5019534504383332e-03_i,
1.1308204765587992e-04+7.5471708958427409e-03_i,  9.3793532353199176e-05-1.1239769490147922e-03_i,  7.8462165903358605e-05+5.7213662882799159e-03_i,
6.6151948092899271e-05-8.6210226491284687e-04_i,  5.6175560035641014e-05+4.4352883596766200e-03_i,  4.8021576809731387e-05-6.7482214710290630e-04_i,
4.1304813806402620e-05+3.5026158226045779e-03_i,  3.5731912220005087e-05-5.3724019451874597e-04_i,  3.1077136080702873e-05+2.8091790154378786e-03_i,
2.7165120270024230e-05-4.3379861797514935e-04_i,  2.3858419175123039e-05+2.2823299156167127e-03_i,  2.1048415661084618e-05-3.5443044687255565e-04_i,
1.8648611777198184e-05+1.8743227668638258e-03_i,  1.6589627255579702e-05-2.9242279694179779e-04_i,  1.4815435964699996e-05+1.5528850001389587e-03_i,
1.3280508908766712e-05-2.4318020790416498e-04_i,  1.1947627404571546e-05+1.2956579468881802e-03_i,  1.0786196206405088e-05-2.0348279618061048e-04_i,
9.7709327053206209e-06+1.0868074988958669e-03_i,  8.8808413266296571e-06-1.7102704621416685e-04_i,  8.0984058064192867e-06+9.1489587493314914e-04_i,
7.4089491477769562e-06-1.4413355039361855e-04_i,  6.8001234806017771e-06+7.7150974428441717e-04_i,  6.2615012029187131e-06-1.2155632512005825e-04_i,
5.7842456143981751e-06+6.5035480629023652e-04_i,  5.3608442570554768e-06-1.0235557480974705e-04_i,  4.9848920523941700e-06+5.4664526430099961e-04_i,
4.6509141472897983e-06-8.5811026284261099e-05_i,  4.3542206590169164e-06+4.5668389539212789e-04_i,  4.0907871028915410e-06-7.1361750514547032e-05_i,
3.8571556477623795e-06+3.7756773927450499e-04_i,  3.6503533803499463e-06-5.8563596922156860e-05_i,  3.4678244108313216e-06+3.0697800389906452e-04_i,
3.3073734610118179e-06-4.7058524223745138e-05_i,  3.1671188992299718e-06+2.4302724004102855e-04_i,  3.0454536620204073e-06-3.6552068611514499e-05_i,
2.9410127917314441e-06+1.8414587700578411e-04_i,  2.8526465434240700e-06-2.6796424528831297e-05_i,  2.7793982463331091e-06+1.2899595944348974e-04_i,
2.7204862683566366e-06-1.7577403563423522e-05_i,  2.6752895230242046e-06+7.6403624661214309e-05_i,  2.6433361393074506e-06-8.7040473820687581e-06_i,
2.6242949506261048e-06+2.5304251952462664e-05_i,  2.6179695815642390e-06+0.0000000000000000e+00_i,  2.6242949506261048e-06-2.5304251952462664e-05_i,
2.6433361393074506e-06+8.7040473820687581e-06_i,  2.6752895230242046e-06-7.6403624661214309e-05_i,  2.7204862683566366e-06+1.7577403563423522e-05_i,
2.7793982463331091e-06-1.2899595944348974e-04_i,  2.8526465434240700e-06+2.6796424528831297e-05_i,  2.9410127917314441e-06-1.8414587700578411e-04_i,
3.0454536620204073e-06+3.6552068611514499e-05_i,  3.1671188992299718e-06-2.4302724004102855e-04_i,  3.3073734610118179e-06+4.7058524223745138e-05_i,
3.4678244108313216e-06-3.0697800389906452e-04_i,  3.6503533803499463e-06+5.8563596922156860e-05_i,  3.8571556477623795e-06-3.7756773927450499e-04_i,
4.0907871028915410e-06+7.1361750514547032e-05_i,  4.3542206590169164e-06-4.5668389539212789e-04_i,  4.6509141472897983e-06+8.5811026284261099e-05_i,
4.9848920523941700e-06-5.4664526430099961e-04_i,  5.3608442570554768e-06+1.0235557480974705e-04_i,  5.7842456143981751e-06-6.5035480629023652e-04_i,
6.2615012029187131e-06+1.2155632512005825e-04_i,  6.8001234806017771e-06-7.7150974428441717e-04_i,  7.4089491477769562e-06+1.4413355039361855e-04_i,
8.0984058064192867e-06-9.1489587493314914e-04_i,  8.8808413266296571e-06+1.7102704621416685e-04_i,  9.7709327053206209e-06-1.0868074988958669e-03_i,
1.0786196206405088e-05+2.0348279618061048e-04_i,  1.1947627404571546e-05-1.2956579468881802e-03_i,  1.3280508908766712e-05+2.4318020790416498e-04_i,
1.4815435964699996e-05-1.5528850001389587e-03_i,  1.6589627255579702e-05+2.9242279694179779e-04_i,  1.8648611777198184e-05-1.8743227668638258e-03_i,
2.1048415661084618e-05+3.5443044687255565e-04_i,  2.3858419175123039e-05-2.2823299156167127e-03_i,  2.7165120270024230e-05+4.3379861797514935e-04_i,
3.1077136080702873e-05-2.8091790154378786e-03_i,  3.5731912220005087e-05+5.3724019451874597e-04_i,  4.1304813806402620e-05-3.5026158226045779e-03_i,
4.8021576809731387e-05+6.7482214710290630e-04_i,  5.6175560035641014e-05-4.4352883596766200e-03_i,  6.6151948092899271e-05+8.6210226491284687e-04_i,
7.8462165903358605e-05-5.7213662882799159e-03_i,  9.3793532353199176e-05+1.1239769490147922e-03_i,  1.1308204765587992e-04-7.5471708958427409e-03_i,
1.3762095926060766e-04+1.5019534504383332e-03_i,  1.6922580462655533e-04-1.0230672621969738e-02_i,  2.1049063069539990e-04+2.0687054279054304e-03_i,
2.6519512008401757e-04-1.4344561109506732e-02_i,  3.3896850323407912e-04+2.9592955170785369e-03_i,  4.4040421377890271e-04-2.0991118159805942e-02_i,
5.8299401290107591e-04+4.4441326323197018e-03_i,  7.8861275667058747e-04-3.2478070314224651e-02_i,  1.0940757405736407e-03+7.1190598692685934e-03_i,
1.5641203764309927e-03-5.4200849498215367e-02_i,  2.3187022659699233e-03+1.2477497803910057e-02_i,  3.5947282825056432e-03-1.0085361313772051e-01_i,
5.8988673497371450e-03+2.5020334564877355e-02_i,  1.0432151910686992e-02-2.2272525677242766e-01_i,  2.0467246865977323e-02+6.2932090485856074e-02_i,
4.6926317974181883e-02-6.7554505650337715e-01_i,  1.4072334780537615e-01+2.5796064625939596e-01_i,  7.9497110360140955e-01-5.1803545039392525e+00_i,
-1.4805873973945676e+01-1.1409854850790166e+00_i, -8.5960253677296627e-01+2.4903175966776388e+01_i},
{ 3.1678833426962989e+01+0.0000000000000000e+00_i, -2.2204460492503131e-16-2.6955003428947762e+01_i, -1.6025762507910954e+01-2.5456228445335822e-16_i,
-3.6082248300317588e-16+5.6071753098134209e+00_i,  1.5231785406357318e-01-2.8102520310824275e-16_i, -6.9388939039072284e-18+7.3120470010534011e-01_i,
2.2153588369192696e-02+1.0246549094611839e-16_i,  5.6725457664441592e-16+2.4107608073859582e-01_i,  6.3848880099149664e-03+6.6613381477509392e-16_i,
-2.5326962749261384e-16+1.0916316423157538e-01_i,  2.5097452474862631e-03-6.3003826807756490e-16_i,  2.7408630920433552e-16+5.8666576746090493e-02_i,
1.1842190481263278e-03+6.0444271116066872e-17_i,  4.1980308118638732e-16+3.5154009988664775e-02_i,  6.3102817238042239e-04-4.1777274341242296e-17_i,
1.8318679906315083e-15+2.2720622571589875e-02_i,  3.6689686404442901e-04-6.2803766093938451e-16_i, -2.1189647259056699e-15+1.5526441061547235e-02_i,
2.2783341689953970e-04+2.2668117859506703e-16_i, -8.2399365108898337e-18+1.1073600249765413e-02_i,  1.4895985290057026e-04+4.0308603893937645e-16_i,
-3.4694469519536142e-17+8.1689989119341552e-03_i,  1.0152138785740606e-04-2.1659826775742047e-16_i,  3.1788807697274990e-16+6.1927622454496737e-03_i,
7.1602352649819648e-05-6.6605758180984104e-16_i, -3.1474389067254194e-16+4.8007215265616675e-03_i,  5.1978180184442680e-05-8.2223944605582391e-17_i,
-3.0975656067910862e-16+3.7912040470077352e-03_i,  3.8675943088429905e-05+6.7298885241722298e-16_i,  1.0811664064025450e-15+3.0406334555357468e-03_i,
2.9403314301823408e-05-6.9057243056515914e-16_i,  1.6596153704778738e-15+2.4703760991573731e-03_i,  2.2782640940022247e-05-8.8817841970012523e-16_i,
-1.8816599754029051e-15+2.0287523436830768e-03_i,  1.7956483147252406e-05-8.9473974846718978e-17_i, -1.3747683547116196e-16+1.6808306121019029e-03_i,
1.4374719259204101e-05-2.0900537973499861e-17_i, -3.9801061751942868e-16+1.4024100559583258e-03_i,  1.1674894644832358e-05+2.0358352204022803e-17_i,
-1.9526481126463935e-16+1.1763519600243482e-03_i,  9.6125533845412028e-06+6.6621004774034681e-16_i,  8.2399365108898337e-18+9.9027615910753965e-04_i,
8.0193887697111452e-06+1.0148987835735008e-16_i,  6.9388939039072284e-18+8.3507612966370569e-04_i,  6.7774000643317937e-06+1.5328585839871622e-16_i,
5.6335144882346810e-16+7.0393896975064901e-04_i,  5.8025360104987727e-06+6.9671932726689526e-17_i,  5.6465249143045071e-16+5.9168457040528061e-04_i,
5.0341131978783775e-06-6.2803766093938451e-16_i,  1.6653345369377348e-16+4.9431108637195374e-04_i,  4.4278360527549030e-06+1.1523197152713521e-16_i,
-7.5286998857393428e-16+4.0867637607325605e-04_i,  3.9511140268820967e-06+3.1024445165672709e-16_i,  1.9775847626135601e-16+3.3227059708175466e-04_i,
3.5798752379080558e-06-1.2239803962737523e-16_i,  1.6306400674181987e-16+2.6305078907895574e-04_i,  3.2963752899148845e-06-6.6613381477509392e-16_i,
-2.9316826744008040e-16+1.9931806098676119e-04_i,  3.0876823686881716e-06-2.3343711338236733e-16_i, -6.1756155744774333e-16+1.3962422037055999e-04_i,
2.9446331176047336e-06+4.1286418728248009e-16_i,  8.3266726846886741e-17+8.2698687409177296e-05_i,  2.8611264184519314e-06+3.2517679528326908e-17_i,
0.0000000000000000e+00+2.7389124948484778e-05_i,  2.8336698534303650e-06+0.0000000000000000e+00_i,  0.0000000000000000e+00-2.7389124948484778e-05_i,
2.8611264184519314e-06-3.2517679528326908e-17_i,  8.3266726846886741e-17-8.2698687409177296e-05_i,  2.9446331176047336e-06-4.1286418728248009e-16_i,
-6.1756155744774333e-16-1.3962422037055999e-04_i,  3.0876823686881716e-06+2.3343711338236733e-16_i, -2.9316826744008040e-16-1.9931806098676119e-04_i,
3.2963752899148845e-06+6.6613381477509392e-16_i,  1.6306400674181987e-16-2.6305078907895574e-04_i,  3.5798752379080558e-06+1.2239803962737523e-16_i,
1.9775847626135601e-16-3.3227059708175466e-04_i,  3.9511140268820967e-06-3.1024445165672709e-16_i, -7.5286998857393428e-16-4.0867637607325605e-04_i,
4.4278360527549030e-06-1.1523197152713521e-16_i,  1.6653345369377348e-16-4.9431108637195374e-04_i,  5.0341131978783775e-06+6.2803766093938451e-16_i,
5.6465249143045071e-16-5.9168457040528061e-04_i,  5.8025360104987727e-06-6.9671932726689526e-17_i,  5.6335144882346810e-16-7.0393896975064901e-04_i,
6.7774000643317937e-06-1.5328585839871622e-16_i,  6.9388939039072284e-18-8.3507612966370569e-04_i,  8.0193887697111452e-06-1.0148987835735008e-16_i,
8.2399365108898337e-18-9.9027615910753965e-04_i,  9.6125533845412028e-06-6.6621004774034681e-16_i, -1.9526481126463935e-16-1.1763519600243482e-03_i,
1.1674894644832358e-05-2.0358352204022803e-17_i, -3.9801061751942868e-16-1.4024100559583258e-03_i,  1.4374719259204101e-05+2.0900537973499861e-17_i,
-1.3747683547116196e-16-1.6808306121019029e-03_i,  1.7956483147252406e-05+8.9473974846718978e-17_i, -1.8816599754029051e-15-2.0287523436830768e-03_i,
2.2782640940022247e-05+8.8817841970012523e-16_i,  1.6596153704778738e-15-2.4703760991573731e-03_i,  2.9403314301823408e-05+6.9057243056515914e-16_i,
1.0811664064025450e-15-3.0406334555357468e-03_i,  3.8675943088429905e-05-6.7298885241722298e-16_i, -3.0975656067910862e-16-3.7912040470077352e-03_i,
5.1978180184442680e-05+8.2223944605582391e-17_i, -3.1474389067254194e-16-4.8007215265616675e-03_i,  7.1602352649819648e-05+6.6605758180984104e-16_i,
3.1788807697274990e-16-6.1927622454496737e-03_i,  1.0152138785740606e-04+2.1659826775742047e-16_i, -3.4694469519536142e-17-8.1689989119341552e-03_i,
1.4895985290057026e-04-4.0308603893937645e-16_i, -8.2399365108898337e-18-1.1073600249765413e-02_i,  2.2783341689953970e-04-2.2668117859506703e-16_i,
-2.1189647259056699e-15-1.5526441061547235e-02_i,  3.6689686404442901e-04+6.2803766093938451e-16_i,  1.8318679906315083e-15-2.2720622571589875e-02_i,
6.3102817238042239e-04+4.1777274341242296e-17_i,  4.1980308118638732e-16-3.5154009988664775e-02_i,  1.1842190481263278e-03-6.0444271116066872e-17_i,
2.7408630920433552e-16-5.8666576746090493e-02_i,  2.5097452474862631e-03+6.3003826807756490e-16_i, -2.5326962749261384e-16-1.0916316423157538e-01_i,
6.3848880099149664e-03-6.6613381477509392e-16_i,  5.6725457664441592e-16-2.4107608073859582e-01_i,  2.2153588369192696e-02-1.0246549094611839e-16_i,
-6.9388939039072284e-18-7.3120470010534011e-01_i,  1.5231785406357318e-01+2.8102520310824275e-16_i, -3.6082248300317588e-16-5.6071753098134209e+00_i,
-1.6025762507910954e+01+2.5456228445335822e-16_i, -2.2204460492503131e-16+2.6955003428947762e+01_i},
{ 2.9267425817005496e+01+0.0000000000000000e+00_i,  8.5960253677296583e-01-2.4903175966776388e+01_i, -1.4805873973945680e+01-1.1409854850790164e+00_i,
-7.9497110360140966e-01+5.1803545039392533e+00_i,  1.4072334780537615e-01+2.5796064625939552e-01_i, -4.6926317974182161e-02+6.7554505650337704e-01_i,
2.0467246865977388e-02+6.2932090485855879e-02_i, -1.0432151910686097e-02+2.2272525677242722e-01_i,  5.8988673497371085e-03+2.5020334564877889e-02_i,
-3.5947282825060959e-03+1.0085361313772090e-01_i,  2.3187022659700443e-03+1.2477497803909249e-02_i, -1.5641203764305903e-03+5.4200849498215506e-02_i,
1.0940757405738207e-03+7.1190598692683583e-03_i, -7.8861275666982072e-04+3.2478070314225317e-02_i,  5.8299401290089442e-04+4.4441326323207929e-03_i,
-4.4040421377730590e-04+2.0991118159804735e-02_i,  3.3896850323376514e-04+2.9592955170758945e-03_i, -2.6519512008655417e-04+1.4344561109506927e-02_i,
2.1049063069568527e-04+2.0687054279045890e-03_i, -1.6922580462702111e-04+1.0230672621969536e-02_i,  1.3762095926041858e-04+1.5019534504389722e-03_i,
-1.1308204765493276e-04+7.5471708958429811e-03_i,  9.3793532354038945e-05+1.1239769490139996e-03_i, -7.8462165903224164e-05+5.7213662882792480e-03_i,
6.6151948093347290e-05+8.6210226491221793e-04_i, -5.6175560035963889e-05+4.4352883596767943e-03_i,  4.8021576809218858e-05+6.7482214710284840e-04_i,
-4.1304813806806919e-05+3.5026158226048533e-03_i,  3.5731912220271449e-05+5.3724019451993241e-04_i, -3.1077136079605118e-05+2.8091790154380247e-03_i,
2.7165120270758252e-05+4.3379861797352456e-04_i, -2.3858419175761634e-05+2.2823299156173029e-03_i,  2.1048415661084618e-05+3.5443044687077929e-04_i,
-1.8648611776726122e-05+1.8743227668626397e-03_i,  1.6589627255733858e-05+2.9242279694217141e-04_i, -1.4815435965353661e-05+1.5528850001395489e-03_i,
1.3280508909027707e-05+2.4318020790440773e-04_i, -1.1947627405173387e-05+1.2956579468878450e-03_i,  1.0786196206414547e-05+2.0348279618068779e-04_i,
-9.7709327057193904e-06+1.0868074988963469e-03_i,  8.8808413264956226e-06+1.7102704621455619e-04_i, -8.0984058059396357e-06+9.1489587493306414e-04_i,
7.4089491475668920e-06+1.4413355039412878e-04_i, -6.8001234811742358e-06+7.7150974428435905e-04_i,  6.2615012026220754e-06+1.2155632512098166e-04_i,
-5.7842456130997345e-06+6.5035480629034061e-04_i,  5.3608442569366482e-06+1.0235557480890549e-04_i, -4.9848920508012602e-06+5.4664526429930565e-04_i,
4.6509141476037833e-06+8.5811026282506860e-05_i, -4.3542206597810620e-06+4.5668389539258586e-04_i,  4.0907871020183246e-06+7.1361750513861816e-05_i,
-3.8571556485846384e-06+3.7756773927503234e-04_i,  3.6503533803503799e-06+5.8563596922872434e-05_i, -3.4678244104149880e-06+3.0697800389912350e-04_i,
3.3073734609029640e-06+4.7058524223441561e-05_i, -3.1671188988431284e-06+2.4302724004143100e-04_i,  3.0454536617428515e-06+3.6552068610400806e-05_i,
-2.9410127926751337e-06+1.8414587700506246e-04_i,  2.8526465432176379e-06+2.6796424528872931e-05_i, -2.7793982466835232e-06+1.2899595944293463e-04_i,
2.7204862681484698e-06+1.7577403564533745e-05_i, -2.6752895226911377e-06+7.6403624661658398e-05_i,  2.6433361410838074e-06+8.7040473817356911e-06_i,
-2.6242949504595714e-06+2.5304251956015378e-05_i,  2.6179695797878821e-06+0.0000000000000000e+00_i, -2.6242949504595714e-06-2.5304251956015378e-05_i,
2.6433361410838074e-06-8.7040473817356911e-06_i, -2.6752895226911377e-06-7.6403624661658398e-05_i,  2.7204862681484698e-06-1.7577403564533745e-05_i,
-2.7793982466835232e-06-1.2899595944293463e-04_i,  2.8526465432176379e-06-2.6796424528872931e-05_i, -2.9410127926751337e-06-1.8414587700506246e-04_i,
3.0454536617428515e-06-3.6552068610400806e-05_i, -3.1671188988431284e-06-2.4302724004143100e-04_i,  3.3073734609029640e-06-4.7058524223441561e-05_i,
-3.4678244104149880e-06-3.0697800389912350e-04_i,  3.6503533803503799e-06-5.8563596922872434e-05_i, -3.8571556485846384e-06-3.7756773927503234e-04_i,
4.0907871020183246e-06-7.1361750513861816e-05_i, -4.3542206597810620e-06-4.5668389539258586e-04_i,  4.6509141476037833e-06-8.5811026282506860e-05_i,
-4.9848920508012602e-06-5.4664526429930565e-04_i,  5.3608442569366482e-06-1.0235557480890549e-04_i, -5.7842456130997345e-06-6.5035480629034061e-04_i,
6.2615012026220754e-06-1.2155632512098166e-04_i, -6.8001234811742358e-06-7.7150974428435905e-04_i,  7.4089491475668920e-06-1.4413355039412878e-04_i,
-8.0984058059396357e-06-9.1489587493306414e-04_i,  8.8808413264956226e-06-1.7102704621455619e-04_i, -9.7709327057193904e-06-1.0868074988963469e-03_i,
1.0786196206414547e-05-2.0348279618068779e-04_i, -1.1947627405173387e-05-1.2956579468878450e-03_i,  1.3280508909027707e-05-2.4318020790440773e-04_i,
-1.4815435965353661e-05-1.5528850001395489e-03_i,  1.6589627255733858e-05-2.9242279694217141e-04_i, -1.8648611776726122e-05-1.8743227668626397e-03_i,
2.1048415661084618e-05-3.5443044687077929e-04_i, -2.3858419175761634e-05-2.2823299156173029e-03_i,  2.7165120270758252e-05-4.3379861797352456e-04_i,
-3.1077136079605118e-05-2.8091790154380247e-03_i,  3.5731912220271449e-05-5.3724019451993241e-04_i, -4.1304813806806919e-05-3.5026158226048533e-03_i,
4.8021576809218858e-05-6.7482214710284840e-04_i, -5.6175560035963889e-05-4.4352883596767943e-03_i,  6.6151948093347290e-05-8.6210226491221793e-04_i,
-7.8462165903224164e-05-5.7213662882792480e-03_i,  9.3793532354038945e-05-1.1239769490139996e-03_i, -1.1308204765493276e-04-7.5471708958429811e-03_i,
1.3762095926041858e-04-1.5019534504389722e-03_i, -1.6922580462702111e-04-1.0230672621969536e-02_i,  2.1049063069568527e-04-2.0687054279045890e-03_i,
-2.6519512008655417e-04-1.4344561109506927e-02_i,  3.3896850323376514e-04-2.9592955170758945e-03_i, -4.4040421377730590e-04-2.0991118159804735e-02_i,
5.8299401290089442e-04-4.4441326323207929e-03_i, -7.8861275666982072e-04-3.2478070314225317e-02_i,  1.0940757405738207e-03-7.1190598692683583e-03_i,
-1.5641203764305903e-03-5.4200849498215506e-02_i,  2.3187022659700443e-03-1.2477497803909249e-02_i, -3.5947282825060959e-03-1.0085361313772090e-01_i,
5.8988673497371085e-03-2.5020334564877889e-02_i, -1.0432151910686097e-02-2.2272525677242722e-01_i,  2.0467246865977388e-02-6.2932090485855879e-02_i,
-4.6926317974182161e-02-6.7554505650337704e-01_i,  1.4072334780537615e-01-2.5796064625939552e-01_i, -7.9497110360140966e-01-5.1803545039392533e+00_i,
-1.4805873973945680e+01+1.1409854850790164e+00_i,  8.5960253677296583e-01+2.4903175966776388e+01_i},
{ 2.2400317936284612e+01+0.0000000000000000e+00_i,  1.5883383796386488e+00-1.9060065711515605e+01_i, -1.1331925343028971e+01-2.1082662731139310e+00_i,
-1.4689150631105037e+00+3.9648716848708507e+00_i,  1.0770498750413518e-01+4.7664912254487979e-01_i, -8.6708529424927322e-02+5.1703980187996201e-01_i,
1.5664952563471529e-02+1.1628334067606191e-01_i, -1.9276103260663004e-02+1.7046653147213703e-01_i,  4.5147976089277338e-03+4.6231550002150380e-02_i,
-6.6421917702936470e-03+7.7190013683927278e-02_i,  1.7746578835480314e-03+2.3055409675973252e-02_i, -2.8901176043366287e-03+4.1483534246161788e-02_i,
8.3736931934041752e-04+1.3154307407878088e-02_i, -1.4571663699304268e-03+2.4857638848884617e-02_i,  4.4620429981081784e-04+8.2116863575335767e-03_i,
-8.1376087828047977e-04+1.6065906293150488e-02_i,  2.5943526056161907e-04+5.4680651177598429e-03_i, -4.9001668713826059e-04+1.0978851762314024e-02_i,
1.6110255407124666e-04+3.8224692072724160e-03_i, -3.1268851453496085e-04+7.8302178287577729e-03_i,  1.0533052211060708e-04+2.7752481032902038e-03_i,
-2.0894837864647055e-04+5.7763545261342751e-03_i,  7.1786461789817059e-05+2.0768385964173700e-03_i, -1.4497917830935095e-04+4.3789441780330701e-03_i,
5.0630509108200064e-05+1.5929572749688796e-03_i, -1.0379890028842790e-04+3.3946227460202819e-03_i,  3.6754123681896275e-05+1.2469087395867717e-03_i,
-7.6321344140127436e-05+2.6807860905012117e-03_i,  2.7348021626638126e-05+9.9269043951803163e-04_i, -5.7423059906954474e-05+2.1500525355117213e-03_i,
2.0791282931213422e-05+8.0155532875636785e-04_i, -4.4084610306325300e-05+1.7468196917953757e-03_i,  1.6109759902072085e-05+6.5490207112794963e-04_i,
-3.4458141463224979e-05+1.4345445395662845e-03_i,  1.2697150998704538e-05+5.4032687386884907e-04_i, -2.7375356106877061e-05+1.1885267238433035e-03_i,
1.0164461465688395e-05+4.4933843358905515e-04_i, -2.2076336842975872e-05+9.9165366057269236e-04_i,  8.2553971730292552e-06+3.7598718121893682e-04_i,
-1.8054329479932582e-05+8.3180644799554261e-04_i,  6.7971016826632214e-06+3.1601677500771622e-04_i, -1.4963902740624006e-05+7.0023098735225815e-04_i,
5.6705641805055817e-06+2.6632407431410572e-04_i, -1.2564989804625046e-05+5.9048799409183988e-04_i,  4.7923455444074227e-06+2.2460680165234739e-04_i,
-1.0687892267061995e-05+4.9776001905278873e-04_i,  4.1030125607740197e-06+1.8912844120894069e-04_i, -9.2108794762079532e-06+4.1838417205716273e-04_i,
3.5596555791716621e-06+1.5855810169292626e-04_i, -8.0455506953000522e-06+3.4953072118933982e-04_i,  3.1309528981421798e-06+1.3185932140771274e-04_i,
-7.1270943143878981e-06+2.8897783683243250e-04_i,  2.7938595223661079e-06+1.0821141709392058e-04_i, -6.4077039912593253e-06+2.3495079238483943e-04_i,
2.5313540560180717e-06+8.6952814721406174e-05_i, -5.8520726544941670e-06+1.8600499675450882e-04_i,  2.3308893209658677e-06+6.7539416120594720e-05_i,
-5.4342830472405013e-06+1.4093915253647615e-04_i,  2.1833211410211134e-06+4.9513336333302516e-05_i, -5.1356583058090322e-06+9.8729233041716657e-05_i,
2.0821700451867375e-06+3.2478806775232405e-05_i, -4.9432904682955581e-06+5.8476802661910909e-05_i,  2.0231218949007257e-06+1.6082982453147210e-05_i,
-4.8490647843735246e-06+1.9367035985595749e-05_i,  2.0037071699618991e-06+0.0000000000000000e+00_i, -4.8490647843735246e-06-1.9367035985595749e-05_i,
2.0231218949007257e-06-1.6082982453147210e-05_i, -4.9432904682955581e-06-5.8476802661910909e-05_i,  2.0821700451867375e-06-3.2478806775232405e-05_i,
-5.1356583058090322e-06-9.8729233041716657e-05_i,  2.1833211410211134e-06-4.9513336333302516e-05_i, -5.4342830472405013e-06-1.4093915253647615e-04_i,
2.3308893209658677e-06-6.7539416120594720e-05_i, -5.8520726544941670e-06-1.8600499675450882e-04_i,  2.5313540560180717e-06-8.6952814721406174e-05_i,
-6.4077039912593253e-06-2.3495079238483943e-04_i,  2.7938595223661079e-06-1.0821141709392058e-04_i, -7.1270943143878981e-06-2.8897783683243250e-04_i,
3.1309528981421798e-06-1.3185932140771274e-04_i, -8.0455506953000522e-06-3.4953072118933982e-04_i,  3.5596555791716621e-06-1.5855810169292626e-04_i,
-9.2108794762079532e-06-4.1838417205716273e-04_i,  4.1030125607740197e-06-1.8912844120894069e-04_i, -1.0687892267061995e-05-4.9776001905278873e-04_i,
4.7923455444074227e-06-2.2460680165234739e-04_i, -1.2564989804625046e-05-5.9048799409183988e-04_i,  5.6705641805055817e-06-2.6632407431410572e-04_i,
-1.4963902740624006e-05-7.0023098735225815e-04_i,  6.7971016826632214e-06-3.1601677500771622e-04_i, -1.8054329479932582e-05-8.3180644799554261e-04_i,
8.2553971730292552e-06-3.7598718121893682e-04_i, -2.2076336842975872e-05-9.9165366057269236e-04_i,  1.0164461465688395e-05-4.4933843358905515e-04_i,
-2.7375356106877061e-05-1.1885267238433035e-03_i,  1.2697150998704538e-05-5.4032687386884907e-04_i, -3.4458141463224979e-05-1.4345445395662845e-03_i,
1.6109759902072085e-05-6.5490207112794963e-04_i, -4.4084610306325300e-05-1.7468196917953757e-03_i,  2.0791282931213422e-05-8.0155532875636785e-04_i,
-5.7423059906954474e-05-2.1500525355117213e-03_i,  2.7348021626638126e-05-9.9269043951803163e-04_i, -7.6321344140127436e-05-2.6807860905012117e-03_i,
3.6754123681896275e-05-1.2469087395867717e-03_i, -1.0379890028842790e-04-3.3946227460202819e-03_i,  5.0630509108200064e-05-1.5929572749688796e-03_i,
-1.4497917830935095e-04-4.3789441780330701e-03_i,  7.1786461789817059e-05-2.0768385964173700e-03_i, -2.0894837864647055e-04-5.7763545261342751e-03_i,
1.0533052211060708e-04-2.7752481032902038e-03_i, -3.1268851453496085e-04-7.8302178287577729e-03_i,  1.6110255407124666e-04-3.8224692072724160e-03_i,
-4.9001668713826059e-04-1.0978851762314024e-02_i,  2.5943526056161907e-04-5.4680651177598429e-03_i, -8.1376087828047977e-04-1.6065906293150488e-02_i,
4.4620429981081784e-04-8.2116863575335767e-03_i, -1.4571663699304268e-03-2.4857638848884617e-02_i,  8.3736931934041752e-04-1.3154307407878088e-02_i,
-2.8901176043366287e-03-4.1483534246161788e-02_i,  1.7746578835480314e-03-2.3055409675973252e-02_i, -6.6421917702936470e-03-7.7190013683927278e-02_i,
4.5147976089277338e-03-4.6231550002150380e-02_i, -1.9276103260663004e-02-1.7046653147213703e-01_i,  1.5664952563471529e-02-1.1628334067606191e-01_i,
-8.6708529424927322e-02-5.1703980187996201e-01_i,  1.0770498750413518e-01-4.7664912254487979e-01_i, -1.4689150631105037e+00-3.9648716848708507e+00_i,
-1.1331925343028971e+01+2.1082662731139310e+00_i,  1.5883383796386488e+00+1.9060065711515605e+01_i},
{ 1.2122964709152141e+01+0.0000000000000000e+00_i,  2.0752641025276120e+00-1.0315233231602498e+01_i, -6.1327938027951348e+00-2.7545826325486060e+00_i,
-1.9192300200092289e+00+2.1457730934321861e+00_i,  5.8289519203532855e-02+6.2277209075796103e-01_i, -1.1329015328550321e-01+2.7981992439779774e-01_i,
8.4778112363258738e-03+1.5193150635944475e-01_i, -2.5185442627515112e-02+9.2255822038169330e-02_i,  2.4433908589010602e-03+6.0404431041639353e-02_i,
-8.6784417726725009e-03+4.1774934375973558e-02_i,  9.6043792567003603e-04+3.0123344422679818e-02_i, -3.7761206259627764e-03+2.2450726954303959e-02_i,
4.5318101000912502e-04+1.7186930887732828e-02_i, -1.9038796126145639e-03+1.3452857203859029e-02_i,  2.4148402692606092e-04+1.0729085273932706e-02_i,
-1.0632298258284939e-03+8.6948058311677821e-03_i,  1.4040535125669769e-04+7.1443713723973448e-03_i, -6.4023765558561996e-04+5.9417117578469229e-03_i,
8.7188073986696956e-05+4.9942967006022356e-03_i, -4.0854723263385661e-04+4.2376833522191643e-03_i,  5.7004467792453250e-05+3.6260363901022652e-03_i,
-2.7300421311004543e-04+3.1261405426055945e-03_i,  3.8850553163801761e-05+2.7135203941043040e-03_i, -1.8942442505764397e-04+2.3698675119095732e-03_i,
2.7401034077760827e-05+2.0812989801047748e-03_i, -1.3561979891155227e-04+1.8371565916132614e-03_i,  1.9891188400933001e-05+1.6291647797244361e-03_i,
-9.9718641683413487e-05+1.4508309775054871e-03_i,  1.4800642651010702e-05+1.2970125638606792e-03_i, -7.5026843404681147e-05+1.1636000473285867e-03_i,
1.1252161240055711e-05+1.0472825068524776e-03_i, -5.7599319147878395e-05+9.4537200485744809e-04_i,  8.7185392336763812e-06+8.5567079175508098e-04_i,
-4.5021731474208645e-05+7.7636991029950185e-04_i,  6.8716486039504783e-06+7.0597108232549777e-04_i, -3.5767626439174512e-05+6.4322602786333392e-04_i,
5.5009669052215229e-06+5.8708895602306457e-04_i, -2.8844124118819372e-05+5.3667909379760052e-04_i,  4.4677887552011506e-06+4.9125092624873888e-04_i,
-2.3589118254429002e-05+4.5017040573191655e-04_i,  3.6785649228196927e-06+4.1289581450431549e-04_i, -1.9551281130923855e-05+3.7896227955640065e-04_i,
3.0688872201494041e-06+3.4796917215403631e-04_i, -1.6416950332371933e-05+3.1956979958558287e-04_i,  2.5935987191425761e-06+2.9346292869786598e-04_i,
-1.3964404209180440e-05+2.6938578111982033e-04_i,  2.2205343968818240e-06+2.4710821688854207e-04_i, -1.2034593999020069e-05+2.2642788228073238e-04_i,
1.9264717175976765e-06+2.0716614345353459e-04_i, -1.0512018569185967e-05+1.8916466318808813e-04_i,  1.6944594980173276e-06+1.7228250592540503e-04_i,
-9.3119974779488912e-06+1.5639367832237776e-04_i,  1.5120258774802345e-06+1.4138502995156288e-04_i, -8.3720687241222932e-06+1.2715445256526910e-04_i,
1.3699589432427176e-06+1.1360932740656256e-04_i, -7.6461013983125312e-06+1.0066517885124956e-04_i,  1.2614682105270038e-06+8.8244499772249102e-05_i,
-7.1002329713795026e-06+7.6275719710333645e-05_i,  1.1816048865560466e-06+6.4692291521464829e-05_i, -6.7100609415435275e-06+5.3431875892850922e-05_i,
1.1268623086001239e-06+4.2435606075696963e-05_i, -6.4587202499577501e-06+3.1647417549240942e-05_i,  1.0949056790110490e-06+2.1013429237104475e-05_i,
-6.3356084614163422e-06+1.0481364345338307e-05_i,  1.0843985052133576e-06+0.0000000000000000e+00_i, -6.3356084614163422e-06-1.0481364345338307e-05_i,
1.0949056790110490e-06-2.1013429237104475e-05_i, -6.4587202499577501e-06-3.1647417549240942e-05_i,  1.1268623086001239e-06-4.2435606075696963e-05_i,
-6.7100609415435275e-06-5.3431875892850922e-05_i,  1.1816048865560466e-06-6.4692291521464829e-05_i, -7.1002329713795026e-06-7.6275719710333645e-05_i,
1.2614682105270038e-06-8.8244499772249102e-05_i, -7.6461013983125312e-06-1.0066517885124956e-04_i,  1.3699589432427176e-06-1.1360932740656256e-04_i,
-8.3720687241222932e-06-1.2715445256526910e-04_i,  1.5120258774802345e-06-1.4138502995156288e-04_i, -9.3119974779488912e-06-1.5639367832237776e-04_i,
1.6944594980173276e-06-1.7228250592540503e-04_i, -1.0512018569185967e-05-1.8916466318808813e-04_i,  1.9264717175976765e-06-2.0716614345353459e-04_i,
-1.2034593999020069e-05-2.2642788228073238e-04_i,  2.2205343968818240e-06-2.4710821688854207e-04_i, -1.3964404209180440e-05-2.6938578111982033e-04_i,
2.5935987191425761e-06-2.9346292869786598e-04_i, -1.6416950332371933e-05-3.1956979958558287e-04_i,  3.0688872201494041e-06-3.4796917215403631e-04_i,
-1.9551281130923855e-05-3.7896227955640065e-04_i,  3.6785649228196927e-06-4.1289581450431549e-04_i, -2.3589118254429002e-05-4.5017040573191655e-04_i,
4.4677887552011506e-06-4.9125092624873888e-04_i, -2.8844124118819372e-05-5.3667909379760052e-04_i,  5.5009669052215229e-06-5.8708895602306457e-04_i,
-3.5767626439174512e-05-6.4322602786333392e-04_i,  6.8716486039504783e-06-7.0597108232549777e-04_i, -4.5021731474208645e-05-7.7636991029950185e-04_i,
8.7185392336763812e-06-8.5567079175508098e-04_i, -5.7599319147878395e-05-9.4537200485744809e-04_i,  1.1252161240055711e-05-1.0472825068524776e-03_i,
-7.5026843404681147e-05-1.1636000473285867e-03_i,  1.4800642651010702e-05-1.2970125638606792e-03_i, -9.9718641683413487e-05-1.4508309775054871e-03_i,
1.9891188400933001e-05-1.6291647797244361e-03_i, -1.3561979891155227e-04-1.8371565916132614e-03_i,  2.7401034077760827e-05-2.0812989801047748e-03_i,
-1.8942442505764397e-04-2.3698675119095732e-03_i,  3.8850553163801761e-05-2.7135203941043040e-03_i, -2.7300421311004543e-04-3.1261405426055945e-03_i,
5.7004467792453250e-05-3.6260363901022652e-03_i, -4.0854723263385661e-04-4.2376833522191643e-03_i,  8.7188073986696956e-05-4.9942967006022356e-03_i,
-6.4023765558561996e-04-5.9417117578469229e-03_i,  1.4040535125669769e-04-7.1443713723973448e-03_i, -1.0632298258284939e-03-8.6948058311677821e-03_i,
2.4148402692606092e-04-1.0729085273932706e-02_i, -1.9038796126145639e-03-1.3452857203859029e-02_i,  4.5318101000912502e-04-1.7186930887732828e-02_i,
-3.7761206259627764e-03-2.2450726954303959e-02_i,  9.6043792567003603e-04-3.0123344422679818e-02_i, -8.6784417726725009e-03-4.1774934375973558e-02_i,
2.4433908589010602e-03-6.0404431041639353e-02_i, -2.5185442627515112e-02-9.2255822038169330e-02_i,  8.4778112363258738e-03-1.5193150635944475e-01_i,
-1.1329015328550321e-01-2.7981992439779774e-01_i,  5.8289519203532855e-02-6.2277209075796103e-01_i, -1.9192300200092289e+00-2.1457730934321861e+00_i,
-6.1327938027951348e+00+2.7545826325486060e+00_i,  2.0752641025276120e+00+1.0315233231602498e+01_i},
{ 3.8694673880246385e-15+0.0000000000000000e+00_i,  2.2462496781226831e+00-3.3074169818081530e-15_i, -1.8778005756381243e-15-2.9815387565315015e+00_i,
-2.0773596042250055e+00+6.4982061661168470e-16_i, -5.7946561586852617e-17+6.7408365359620426e-01_i, -1.2262437828615859e-01+1.4056276713024805e-16_i,
-1.9032315744297054e-16+1.6444947746213784e-01_i, -2.7260526660933669e-02+1.3699298902604108e-16_i,  5.9245023297568756e-17+6.5381285022569985e-02_i,
-9.3934776854322152e-03+1.7831081204888953e-17_i,  1.7223404903618215e-16+3.2605273049829769e-02_i, -4.0872435129058410e-03+6.5869326510991989e-17_i,
8.8637572133187582e-17+1.8602999939846269e-02_i, -2.0607444429902176e-03+1.0933774456266553e-16_i,  6.3788674848917004e-17+1.1613078216777130e-02_i,
-1.1508316705947930e-03-2.7782276216691289e-16_i,  4.4835804364773228e-17+7.7330118494754723e-03_i, -6.9298824473845852e-04-2.5381577311135421e-16_i,
-5.6557766296973989e-17+5.4057877946776722e-03_i, -4.4220833805391302e-04+7.7795550408332538e-17_i, -2.2384730329328072e-17+3.9247935066230415e-03_i,
-2.9549763091770559e-04+1.5138385089821113e-18_i,  1.0067670565213885e-16+2.9370933099133515e-03_i, -2.0503152022711122e-04-1.2112844318844539e-16_i,
-5.4683285108134405e-17+2.2527817825419320e-03_i, -1.4679381254672651e-04-2.7153144557892429e-17_i, -1.0694801037883525e-16+1.7633952505643992e-03_i,
-1.0793467998189838e-04+1.6071499374684502e-16_i,  1.2325072648978731e-16+1.4038762828040561e-03_i, -8.1208470114284089e-05-9.5174950795842692e-18_i,
5.4600061437102353e-17+1.1335704169204352e-03_i, -6.2345053787780667e-05-1.9570228265589803e-16_i, -8.2232690250527892e-17+9.2617139101460605e-04_i,
-4.8731170990680685e-05-5.2053583866563935e-17_i, -9.0954080219076617e-17+7.6413759314044379e-04_i, -3.8714599881025017e-05+6.8878411056112891e-17_i,
-7.2610395969719065e-17+6.3546050687704448e-04_i, -3.1220654970558193e-05+3.1578425938826248e-17_i, -9.3287063005517889e-17+5.3172617095840601e-04_i,
-2.5532677610022426e-05+9.1575082394265496e-17_i, -4.1594638703098982e-18+4.4691520915252319e-04_i, -2.1162154201750027e-05+1.2573630242150878e-16_i,
7.7257938726451958e-17+3.7663911788122497e-04_i, -1.7769578992790831e-05+6.2562509884911201e-17_i,  5.9147273041581862e-17+3.1764198509757688e-04_i,
-1.5114962197975367e-05-6.0035572736462233e-19_i,  1.8673720120877321e-16+2.6746800658874244e-04_i, -1.3026150678148527e-05-7.8275352320955801e-17_i,
8.9071253002181809e-17+2.2423501783898868e-04_i, -1.1378126908587194e-05-2.3131857803227999e-16_i, -1.5565384486214909e-16+1.8647724066015586e-04_i,
-1.0079233439249779e-05-3.4310954226668620e-17_i, -1.0738618065219162e-16+1.5303405365735614e-04_i, -9.0618618879692928e-06+1.2972074540539301e-16_i,
3.3456170958037579e-17+1.2296984986555001e-04_i, -8.2760805160822862e-06-1.1454191499059595e-16_i, -1.0143915741503521e-17+9.5515158272600764e-05_i,
-7.6852367877026234e-06-1.5954439421373957e-16_i, -3.0728597653104283e-17+7.0022431761407322e-05_i, -7.2629176269684015e-06-5.2543360040571016e-18_i,
-3.0190985971223704e-17+4.5931969030965636e-05_i, -6.9908684221520190e-06+1.6313198163409222e-17_i, -4.5932585006616045e-17+2.2744771907667882e-05_i,
-6.8576131826425524e-06-3.7129879875130907e-17_i,  4.5034835483122657e-18+0.0000000000000000e+00_i, -6.8576131826425524e-06+3.7129879875130907e-17_i,
-4.5932585006616045e-17-2.2744771907667882e-05_i, -6.9908684221520190e-06-1.6313198163409222e-17_i, -3.0190985971223704e-17-4.5931969030965636e-05_i,
-7.2629176269684015e-06+5.2543360040571016e-18_i, -3.0728597653104283e-17-7.0022431761407322e-05_i, -7.6852367877026234e-06+1.5954439421373957e-16_i,
-1.0143915741503521e-17-9.5515158272600764e-05_i, -8.2760805160822862e-06+1.1454191499059595e-16_i,  3.3456170958037579e-17-1.2296984986555001e-04_i,
-9.0618618879692928e-06-1.2972074540539301e-16_i, -1.0738618065219162e-16-1.5303405365735614e-04_i, -1.0079233439249779e-05+3.4310954226668620e-17_i,
-1.5565384486214909e-16-1.8647724066015586e-04_i, -1.1378126908587194e-05+2.3131857803227999e-16_i,  8.9071253002181809e-17-2.2423501783898868e-04_i,
-1.3026150678148527e-05+7.8275352320955801e-17_i,  1.8673720120877321e-16-2.6746800658874244e-04_i, -1.5114962197975367e-05+6.0035572736462233e-19_i,
5.9147273041581862e-17-3.1764198509757688e-04_i, -1.7769578992790831e-05-6.2562509884911201e-17_i,  7.7257938726451958e-17-3.7663911788122497e-04_i,
-2.1162154201750027e-05-1.2573630242150878e-16_i, -4.1594638703098982e-18-4.4691520915252319e-04_i, -2.5532677610022426e-05-9.1575082394265496e-17_i,
-9.3287063005517889e-17-5.3172617095840601e-04_i, -3.1220654970558193e-05-3.1578425938826248e-17_i, -7.2610395969719065e-17-6.3546050687704448e-04_i,
-3.8714599881025017e-05-6.8878411056112891e-17_i, -9.0954080219076617e-17-7.6413759314044379e-04_i, -4.8731170990680685e-05+5.2053583866563935e-17_i,
-8.2232690250527892e-17-9.2617139101460605e-04_i, -6.2345053787780667e-05+1.9570228265589803e-16_i,  5.4600061437102353e-17-1.1335704169204352e-03_i,
-8.1208470114284089e-05+9.5174950795842692e-18_i,  1.2325072648978731e-16-1.4038762828040561e-03_i, -1.0793467998189838e-04-1.6071499374684502e-16_i,
-1.0694801037883525e-16-1.7633952505643992e-03_i, -1.4679381254672651e-04+2.7153144557892429e-17_i, -5.4683285108134405e-17-2.2527817825419320e-03_i,
-2.0503152022711122e-04+1.2112844318844539e-16_i,  1.0067670565213885e-16-2.9370933099133515e-03_i, -2.9549763091770559e-04-1.5138385089821113e-18_i,
-2.2384730329328072e-17-3.9247935066230415e-03_i, -4.4220833805391302e-04-7.7795550408332538e-17_i, -5.6557766296973989e-17-5.4057877946776722e-03_i,
-6.9298824473845852e-04+2.5381577311135421e-16_i,  4.4835804364773228e-17-7.7330118494754723e-03_i, -1.1508316705947930e-03+2.7782276216691289e-16_i,
6.3788674848917004e-17-1.1613078216777130e-02_i, -2.0607444429902176e-03-1.0933774456266553e-16_i,  8.8637572133187582e-17-1.8602999939846269e-02_i,
-4.0872435129058410e-03-6.5869326510991989e-17_i,  1.7223404903618215e-16-3.2605273049829769e-02_i, -9.3934776854322152e-03-1.7831081204888953e-17_i,
5.9245023297568756e-17-6.5381285022569985e-02_i, -2.7260526660933669e-02-1.3699298902604108e-16_i, -1.9032315744297054e-16-1.6444947746213784e-01_i,
-1.2262437828615859e-01-1.4056276713024805e-16_i, -5.7946561586852617e-17-6.7408365359620426e-01_i, -2.0773596042250055e+00-6.4982061661168470e-16_i,
-1.8778005756381243e-15+2.9815387565315015e+00_i,  2.2462496781226831e+00+3.3074169818081530e-15_i},
{-1.2122964709152132e+01-0.0000000000000000e+00_i,  2.0752641025276124e+00+1.0315233231602491e+01_i,  6.1327938027951312e+00-2.7545826325486060e+00_i,
-1.9192300200092292e+00-2.1457730934321848e+00_i, -5.8289519203532911e-02+6.2277209075796125e-01_i, -1.1329015328550321e-01-2.7981992439779751e-01_i,
-8.4778112363262451e-03+1.5193150635944508e-01_i, -2.5185442627515112e-02-9.2255822038169191e-02_i, -2.4433908589009978e-03+6.0404431041638861e-02_i,
-8.6784417726725287e-03-4.1774934375973474e-02_i, -9.6043792566988424e-04+3.0123344422680116e-02_i, -3.7761206259630297e-03-2.2450726954303779e-02_i,
-4.5318101000895589e-04+1.7186930887732526e-02_i, -1.9038796126148606e-03-1.3452857203859105e-02_i, -2.4148402692563417e-04+1.0729085273932602e-02_i,
-1.0632298258289915e-03-8.6948058311677491e-03_i, -1.4040535125685468e-04+7.1443713723978158e-03_i, -6.4023765558466304e-04-5.9417117578477131e-03_i,
-8.7188073986629736e-05+4.9942967006023032e-03_i, -4.0854723263339843e-04-4.2376833522187836e-03_i, -5.7004467792530445e-05+3.6260363901020054e-03_i,
-2.7300421311017217e-04-3.1261405426055346e-03_i, -3.8850553163757092e-05+2.7135203941045273e-03_i, -1.8942442505792440e-04-2.3698675119096894e-03_i,
-2.7401034077613917e-05+2.0812989801051850e-03_i, -1.3561979891136156e-04-1.8371565916138137e-03_i, -1.9891188401203780e-05+1.6291647797245039e-03_i,
-9.9718641683167888e-05-1.4508309775050981e-03_i, -1.4800642650794214e-05+1.2970125638602716e-03_i, -7.5026843405079239e-05-1.1636000473285481e-03_i,
-1.1252161239624483e-05+1.0472825068528042e-03_i, -5.7599319148130804e-05-9.4537200485842160e-04_i, -8.7185392336763812e-06+8.5567079175641325e-04_i,
-4.5021731473956236e-05-7.7636991029958718e-04_i, -6.8716486039376170e-06+7.0597108232515809e-04_i, -3.5767626438887442e-05-6.4322602786329511e-04_i,
-5.5009669055073999e-06+5.8708895602298976e-04_i, -2.8844124118808232e-05-5.3667909379737804e-04_i, -4.4677887552460908e-06+4.9125092624865410e-04_i,
-2.3589118254310933e-05-4.5017040573144189e-04_i, -3.6785649229666021e-06+4.1289581450361552e-04_i, -1.9551281131168180e-05-3.7896227955657238e-04_i,
-3.0688872200084578e-06+3.4796917215391640e-04_i, -1.6416950332453356e-05-3.1956979958541569e-04_i, -2.5935987189959920e-06+2.9346292869774498e-04_i,
-1.3964404209361068e-05-2.6938578111999466e-04_i, -2.2205343968380222e-06+2.4710821688883177e-04_i, -1.2034593999421875e-05-2.2642788228052334e-04_i,
-1.9264717174406840e-06+2.0716614345400557e-04_i, -1.0512018569021385e-05-1.8916466318883232e-04_i, -1.6944594985550919e-06+1.7228250592552299e-04_i,
-9.3119974773746977e-06-1.5639367832212276e-04_i, -1.5120258775799811e-06+1.4138502995112226e-04_i, -8.3720687246600575e-06-1.2715445256520665e-04_i,
-1.3699589431395015e-06+1.1360932740635266e-04_i, -7.6461013986230467e-06-1.0066517885164508e-04_i, -1.2614682105894538e-06+8.8244499772870133e-05_i,
-7.1002329708243911e-06-7.6275719710486301e-05_i, -1.1816048867538051e-06+6.4692291521450951e-05_i, -6.7100609414741386e-06-5.3431875892739900e-05_i,
-1.1268623086070628e-06+4.2435606075696963e-05_i, -6.4587202496246832e-06-3.1647417549018897e-05_i, -1.0949056790110490e-06+2.1013429237104475e-05_i,
-6.3356084614163422e-06-1.0481364346226485e-05_i, -1.0843985061015360e-06-0.0000000000000000e+00_i, -6.3356084614163422e-06+1.0481364346226485e-05_i,
-1.0949056790110490e-06-2.1013429237104475e-05_i, -6.4587202496246832e-06+3.1647417549018897e-05_i, -1.1268623086070628e-06-4.2435606075696963e-05_i,
-6.7100609414741386e-06+5.3431875892739900e-05_i, -1.1816048867538051e-06-6.4692291521450951e-05_i, -7.1002329708243911e-06+7.6275719710486301e-05_i,
-1.2614682105894538e-06-8.8244499772870133e-05_i, -7.6461013986230467e-06+1.0066517885164508e-04_i, -1.3699589431395015e-06-1.1360932740635266e-04_i,
-8.3720687246600575e-06+1.2715445256520665e-04_i, -1.5120258775799811e-06-1.4138502995112226e-04_i, -9.3119974773746977e-06+1.5639367832212276e-04_i,
-1.6944594985550919e-06-1.7228250592552299e-04_i, -1.0512018569021385e-05+1.8916466318883232e-04_i, -1.9264717174406840e-06-2.0716614345400557e-04_i,
-1.2034593999421875e-05+2.2642788228052334e-04_i, -2.2205343968380222e-06-2.4710821688883177e-04_i, -1.3964404209361068e-05+2.6938578111999466e-04_i,
-2.5935987189959920e-06-2.9346292869774498e-04_i, -1.6416950332453356e-05+3.1956979958541569e-04_i, -3.0688872200084578e-06-3.4796917215391640e-04_i,
-1.9551281131168180e-05+3.7896227955657238e-04_i, -3.6785649229666021e-06-4.1289581450361552e-04_i, -2.3589118254310933e-05+4.5017040573144189e-04_i,
-4.4677887552460908e-06-4.9125092624865410e-04_i, -2.8844124118808232e-05+5.3667909379737804e-04_i, -5.5009669055073999e-06-5.8708895602298976e-04_i,
-3.5767626438887442e-05+6.4322602786329511e-04_i, -6.8716486039376170e-06-7.0597108232515809e-04_i, -4.5021731473956236e-05+7.7636991029958718e-04_i,
-8.7185392336763812e-06-8.5567079175641325e-04_i, -5.7599319148130804e-05+9.4537200485842160e-04_i, -1.1252161239624483e-05-1.0472825068528042e-03_i,
-7.5026843405079239e-05+1.1636000473285481e-03_i, -1.4800642650794214e-05-1.2970125638602716e-03_i, -9.9718641683167888e-05+1.4508309775050981e-03_i,
-1.9891188401203780e-05-1.6291647797245039e-03_i, -1.3561979891136156e-04+1.8371565916138137e-03_i, -2.7401034077613917e-05-2.0812989801051850e-03_i,
-1.8942442505792440e-04+2.3698675119096894e-03_i, -3.8850553163757092e-05-2.7135203941045273e-03_i, -2.7300421311017217e-04+3.1261405426055346e-03_i,
-5.7004467792530445e-05-3.6260363901020054e-03_i, -4.0854723263339843e-04+4.2376833522187836e-03_i, -8.7188073986629736e-05-4.9942967006023032e-03_i,
-6.4023765558466304e-04+5.9417117578477131e-03_i, -1.4040535125685468e-04-7.1443713723978158e-03_i, -1.0632298258289915e-03+8.6948058311677491e-03_i,
-2.4148402692563417e-04-1.0729085273932602e-02_i, -1.9038796126148606e-03+1.3452857203859105e-02_i, -4.5318101000895589e-04-1.7186930887732526e-02_i,
-3.7761206259630297e-03+2.2450726954303779e-02_i, -9.6043792566988424e-04-3.0123344422680116e-02_i, -8.6784417726725287e-03+4.1774934375973474e-02_i,
-2.4433908589009978e-03-6.0404431041638861e-02_i, -2.5185442627515112e-02+9.2255822038169191e-02_i, -8.4778112363262451e-03-1.5193150635944508e-01_i,
-1.1329015328550321e-01+2.7981992439779751e-01_i, -5.8289519203532911e-02-6.2277209075796125e-01_i, -1.9192300200092292e+00+2.1457730934321848e+00_i,
6.1327938027951312e+00+2.7545826325486060e+00_i,  2.0752641025276124e+00-1.0315233231602491e+01_i},
{-2.2400317936284605e+01-0.0000000000000000e+00_i,  1.5883383796386488e+00+1.9060065711515598e+01_i,  1.1331925343028965e+01-2.1082662731139306e+00_i,
-1.4689150631105032e+00-3.9648716848708503e+00_i, -1.0770498750413524e-01+4.7664912254487929e-01_i, -8.6708529424927655e-02-5.1703980187996168e-01_i,
-1.5664952563472070e-02+1.1628334067606255e-01_i, -1.9276103260662251e-02-1.7046653147213620e-01_i, -4.5147976089274554e-03+4.6231550002148805e-02_i,
-6.6421917702943079e-03-7.7190013683927500e-02_i, -1.7746578835469333e-03+2.3055409675973797e-02_i, -2.8901176043368993e-03-4.1483534246162572e-02_i,
-8.3736931934051771e-04+1.3154307407879078e-02_i, -1.4571663699304155e-03-2.4857638848884922e-02_i, -4.4620429981100541e-04+8.2116863575324162e-03_i,
-8.1376087828225006e-04-1.6065906293150127e-02_i, -2.5943526056193306e-04+5.4680651177601569e-03_i, -4.9001668713688886e-04-1.0978851762313949e-02_i,
-1.6110255407030981e-04+3.8224692072723046e-03_i, -3.1268851453478651e-04-7.8302178287577399e-03_i, -1.0533052211077101e-04+2.7752481032898911e-03_i,
-2.0894837864672794e-04-5.7763545261341580e-03_i, -7.1786461789632744e-05+2.0768385964175283e-03_i, -1.4497917830958329e-04-4.3789441780335567e-03_i,
-5.0630509107435593e-05+1.5929572749691263e-03_i, -1.0379890028839689e-04-3.3946227460200391e-03_i, -3.6754123682081538e-05+1.2469087395865978e-03_i,
-7.6321344139916288e-05-2.6807860905009723e-03_i, -2.7348021626989353e-05+9.9269043951668787e-04_i, -5.7423059908421454e-05-2.1500525355117668e-03_i,
-2.0791282931119056e-05+8.0155532875699842e-04_i, -4.4084610307412823e-05-1.7468196917956908e-03_i, -1.6109759902072085e-05+6.5490207112883780e-04_i,
-3.4458141461804389e-05-1.4345445395665996e-03_i, -1.2697150998798904e-05+5.4032687386814738e-04_i, -2.7375356106964393e-05-1.1885267238433491e-03_i,
-1.0164461466204529e-05+4.4933843358937661e-04_i, -2.2076336842173942e-05-9.9165366057245275e-04_i, -8.2553971731909369e-06+3.7598718121865200e-04_i,
-1.8054329480119715e-05-8.3180644799552179e-04_i, -6.7971016821717525e-06+3.1601677500681807e-04_i, -1.4963902740440234e-05-7.0023098735208598e-04_i,
-5.6705641803880542e-06+2.6632407431395871e-04_i, -1.2564989804367656e-05-5.9048799409165340e-04_i, -4.7923455442504303e-06+2.2460680165161491e-04_i,
-1.0687892267846957e-05-4.9776001905225617e-04_i, -4.1030125608227004e-06+1.8912844120894026e-04_i, -9.2108794780237750e-06-4.1838417205742121e-04_i,
-3.5596555788576772e-06+1.5855810169324025e-04_i, -8.0455506939738561e-06-3.4953072118930860e-04_i, -3.1309528988427912e-06+1.3185932140821754e-04_i,
-7.1270943130113951e-06-2.8897783683201617e-04_i, -2.7938595223838888e-06+1.0821141709249377e-04_i, -6.4077039918213757e-06-2.3495079238482902e-04_i,
-2.5313540553085698e-06+8.6952814721756588e-05_i, -5.8520726547283547e-06-1.8600499675563292e-04_i, -2.3308893225002306e-06+6.7539416121420448e-05_i,
-5.4342830466749814e-06-1.4093915253567124e-04_i, -2.1833211404660019e-06+4.9513336333059654e-05_i, -5.1356583056702543e-06-9.8729233041994213e-05_i,
-2.0821700450132652e-06+3.2478806775315672e-05_i, -4.9432904678514689e-06-5.8476802662354999e-05_i, -2.0231218940125473e-06+1.6082982452925165e-05_i,
-4.8490647843735246e-06-1.9367035983819392e-05_i, -2.0037071699618991e-06-0.0000000000000000e+00_i, -4.8490647843735246e-06+1.9367035983819392e-05_i,
-2.0231218940125473e-06-1.6082982452925165e-05_i, -4.9432904678514689e-06+5.8476802662354999e-05_i, -2.0821700450132652e-06-3.2478806775315672e-05_i,
-5.1356583056702543e-06+9.8729233041994213e-05_i, -2.1833211404660019e-06-4.9513336333059654e-05_i, -5.4342830466749814e-06+1.4093915253567124e-04_i,
-2.3308893225002306e-06-6.7539416121420448e-05_i, -5.8520726547283547e-06+1.8600499675563292e-04_i, -2.5313540553085698e-06-8.6952814721756588e-05_i,
-6.4077039918213757e-06+2.3495079238482902e-04_i, -2.7938595223838888e-06-1.0821141709249377e-04_i, -7.1270943130113951e-06+2.8897783683201617e-04_i,
-3.1309528988427912e-06-1.3185932140821754e-04_i, -8.0455506939738561e-06+3.4953072118930860e-04_i, -3.5596555788576772e-06-1.5855810169324025e-04_i,
-9.2108794780237750e-06+4.1838417205742121e-04_i, -4.1030125608227004e-06-1.8912844120894026e-04_i, -1.0687892267846957e-05+4.9776001905225617e-04_i,
-4.7923455442504303e-06-2.2460680165161491e-04_i, -1.2564989804367656e-05+5.9048799409165340e-04_i, -5.6705641803880542e-06-2.6632407431395871e-04_i,
-1.4963902740440234e-05+7.0023098735208598e-04_i, -6.7971016821717525e-06-3.1601677500681807e-04_i, -1.8054329480119715e-05+8.3180644799552179e-04_i,
-8.2553971731909369e-06-3.7598718121865200e-04_i, -2.2076336842173942e-05+9.9165366057245275e-04_i, -1.0164461466204529e-05-4.4933843358937661e-04_i,
-2.7375356106964393e-05+1.1885267238433491e-03_i, -1.2697150998798904e-05-5.4032687386814738e-04_i, -3.4458141461804389e-05+1.4345445395665996e-03_i,
-1.6109759902072085e-05-6.5490207112883780e-04_i, -4.4084610307412823e-05+1.7468196917956908e-03_i, -2.0791282931119056e-05-8.0155532875699842e-04_i,
-5.7423059908421454e-05+2.1500525355117668e-03_i, -2.7348021626989353e-05-9.9269043951668787e-04_i, -7.6321344139916288e-05+2.6807860905009723e-03_i,
-3.6754123682081538e-05-1.2469087395865978e-03_i, -1.0379890028839689e-04+3.3946227460200391e-03_i, -5.0630509107435593e-05-1.5929572749691263e-03_i,
-1.4497917830958329e-04+4.3789441780335567e-03_i, -7.1786461789632744e-05-2.0768385964175283e-03_i, -2.0894837864672794e-04+5.7763545261341580e-03_i,
-1.0533052211077101e-04-2.7752481032898911e-03_i, -3.1268851453478651e-04+7.8302178287577399e-03_i, -1.6110255407030981e-04-3.8224692072723046e-03_i,
-4.9001668713688886e-04+1.0978851762313949e-02_i, -2.5943526056193306e-04-5.4680651177601569e-03_i, -8.1376087828225006e-04+1.6065906293150127e-02_i,
-4.4620429981100541e-04-8.2116863575324162e-03_i, -1.4571663699304155e-03+2.4857638848884922e-02_i, -8.3736931934051771e-04-1.3154307407879078e-02_i,
-2.8901176043368993e-03+4.1483534246162572e-02_i, -1.7746578835469333e-03-2.3055409675973797e-02_i, -6.6421917702943079e-03+7.7190013683927500e-02_i,
-4.5147976089274554e-03-4.6231550002148805e-02_i, -1.9276103260662251e-02+1.7046653147213620e-01_i, -1.5664952563472070e-02-1.1628334067606255e-01_i,
-8.6708529424927655e-02+5.1703980187996168e-01_i, -1.0770498750413524e-01-4.7664912254487929e-01_i, -1.4689150631105032e+00+3.9648716848708503e+00_i,
1.1331925343028965e+01+2.1082662731139306e+00_i,  1.5883383796386488e+00-1.9060065711515598e+01_i},
{-2.9267425817005488e+01-0.0000000000000000e+00_i,  8.5960253677296672e-01+2.4903175966776388e+01_i,  1.4805873973945676e+01-1.1409854850790171e+00_i,
-7.9497110360140977e-01-5.1803545039392533e+00_i, -1.4072334780537604e-01+2.5796064625939608e-01_i, -4.6926317974182050e-02-6.7554505650337759e-01_i,
-2.0467246865977801e-02+6.2932090485856185e-02_i, -1.0432151910686409e-02-2.2272525677242655e-01_i, -5.8988673497367833e-03+2.5020334564876606e-02_i,
-3.5947282825062659e-03-1.0085361313772165e-01_i, -2.3187022659703132e-03+1.2477497803909637e-02_i, -1.5641203764323111e-03-5.4200849498215256e-02_i,
-1.0940757405736394e-03+7.1190598692702067e-03_i, -7.8861275666902275e-04-3.2478070314224505e-02_i, -5.8299401290020573e-04+4.4441326323192785e-03_i,
-4.4040421377890184e-04-2.0991118159806359e-02_i, -3.3896850323287707e-04+2.9592955170775950e-03_i, -2.6519512008464424e-04-1.4344561109506510e-02_i,
-2.1049063069532900e-04+2.0687054279054339e-03_i, -1.6922580462617196e-04-1.0230672621969722e-02_i, -1.3762095926052803e-04+1.5019534504382415e-03_i,
-1.1308204765594172e-04-7.5471708958428450e-03_i, -9.3793532353853086e-05+1.1239769490147427e-03_i, -7.8462165903987876e-05-5.7213662882796218e-03_i,
-6.6151948092971262e-05+8.6210226491313863e-04_i, -5.6175560034424105e-05-4.4352883596767137e-03_i, -4.8021576809718851e-05+6.7482214710241721e-04_i,
-4.1304813806337785e-05-3.5026158226046052e-03_i, -3.5731912219548096e-05+5.3724019451896498e-04_i, -3.1077136080884965e-05-2.8091790154384051e-03_i,
-2.7165120271743206e-05+4.3379861797488925e-04_i, -2.3858419175056117e-05-2.2823299156163801e-03_i, -2.1048415659308262e-05+3.5443044687255565e-04_i,
-1.8648611777098573e-05-1.8743227668634932e-03_i, -1.6589627256525261e-05+2.9242279694198178e-04_i, -1.4815435964906482e-05-1.5528850001385970e-03_i,
-1.3280508908835125e-05+2.4318020790407868e-04_i, -1.1947627404462909e-05-1.2956579468882076e-03_i, -1.0786196206284050e-05+2.0348279618005889e-04_i,
-9.7709327055071037e-06-1.0868074988959050e-03_i, -8.8808413268716511e-06+1.7102704621414463e-04_i, -8.0984058062722689e-06-9.1489587493318123e-04_i,
-7.4089491481395947e-06+1.4413355039326369e-04_i, -6.8001234808036555e-06-7.7150974428427839e-04_i, -6.2615012029220199e-06+1.2155632512006356e-04_i,
-5.7842456146150155e-06-6.5035480629022091e-04_i, -5.3608442567933167e-06+1.0235557481030519e-04_i, -4.9848920522671016e-06-5.4664526430055552e-04_i,
-4.6509141467154964e-06+8.5811026283318928e-05_i, -4.3542206595173841e-06-4.5668389539232045e-04_i, -4.0907871023184317e-06+7.1361750514457260e-05_i,
-3.8571556473842097e-06-3.7756773927435927e-04_i, -3.6503533803026750e-06+5.8563596922035430e-05_i, -3.4678244108868328e-06-3.0697800389942534e-04_i,
-3.3073734610035779e-06+4.7058524224033969e-05_i, -3.1671188993670149e-06-2.4302724004154896e-04_i, -3.0454536620681122e-06+3.6552068610449379e-05_i,
-2.9410127927670740e-06-1.8414587700545104e-04_i, -2.8526465434587644e-06+2.6796424530343976e-05_i, -2.7793982451430888e-06-1.2899595944287912e-04_i,
-2.7204862681068365e-06+1.7577403563256988e-05_i, -2.6752895231352269e-06-7.6403624662546576e-05_i, -2.6433361393074506e-06+8.7040473818467134e-06_i,
-2.6242949504595714e-06-2.5304251954239021e-05_i, -2.6179695797878821e-06-0.0000000000000000e+00_i, -2.6242949504595714e-06+2.5304251954239021e-05_i,
-2.6433361393074506e-06-8.7040473818467134e-06_i, -2.6752895231352269e-06+7.6403624662546576e-05_i, -2.7204862681068365e-06-1.7577403563256988e-05_i,
-2.7793982451430888e-06+1.2899595944287912e-04_i, -2.8526465434587644e-06-2.6796424530343976e-05_i, -2.9410127927670740e-06+1.8414587700545104e-04_i,
-3.0454536620681122e-06-3.6552068610449379e-05_i, -3.1671188993670149e-06+2.4302724004154896e-04_i, -3.3073734610035779e-06-4.7058524224033969e-05_i,
-3.4678244108868328e-06+3.0697800389942534e-04_i, -3.6503533803026750e-06-5.8563596922035430e-05_i, -3.8571556473842097e-06+3.7756773927435927e-04_i,
-4.0907871023184317e-06-7.1361750514457260e-05_i, -4.3542206595173841e-06+4.5668389539232045e-04_i, -4.6509141467154964e-06-8.5811026283318928e-05_i,
-4.9848920522671016e-06+5.4664526430055552e-04_i, -5.3608442567933167e-06-1.0235557481030519e-04_i, -5.7842456146150155e-06+6.5035480629022091e-04_i,
-6.2615012029220199e-06-1.2155632512006356e-04_i, -6.8001234808036555e-06+7.7150974428427839e-04_i, -7.4089491481395947e-06-1.4413355039326369e-04_i,
-8.0984058062722689e-06+9.1489587493318123e-04_i, -8.8808413268716511e-06-1.7102704621414463e-04_i, -9.7709327055071037e-06+1.0868074988959050e-03_i,
-1.0786196206284050e-05-2.0348279618005889e-04_i, -1.1947627404462909e-05+1.2956579468882076e-03_i, -1.3280508908835125e-05-2.4318020790407868e-04_i,
-1.4815435964906482e-05+1.5528850001385970e-03_i, -1.6589627256525261e-05-2.9242279694198178e-04_i, -1.8648611777098573e-05+1.8743227668634932e-03_i,
-2.1048415659308262e-05-3.5443044687255565e-04_i, -2.3858419175056117e-05+2.2823299156163801e-03_i, -2.7165120271743206e-05-4.3379861797488925e-04_i,
-3.1077136080884965e-05+2.8091790154384051e-03_i, -3.5731912219548096e-05-5.3724019451896498e-04_i, -4.1304813806337785e-05+3.5026158226046052e-03_i,
-4.8021576809718851e-05-6.7482214710241721e-04_i, -5.6175560034424105e-05+4.4352883596767137e-03_i, -6.6151948092971262e-05-8.6210226491313863e-04_i,
-7.8462165903987876e-05+5.7213662882796218e-03_i, -9.3793532353853086e-05-1.1239769490147427e-03_i, -1.1308204765594172e-04+7.5471708958428450e-03_i,
-1.3762095926052803e-04-1.5019534504382415e-03_i, -1.6922580462617196e-04+1.0230672621969722e-02_i, -2.1049063069532900e-04-2.0687054279054339e-03_i,
-2.6519512008464424e-04+1.4344561109506510e-02_i, -3.3896850323287707e-04-2.9592955170775950e-03_i, -4.4040421377890184e-04+2.0991118159806359e-02_i,
-5.8299401290020573e-04-4.4441326323192785e-03_i, -7.8861275666902275e-04+3.2478070314224505e-02_i, -1.0940757405736394e-03-7.1190598692702067e-03_i,
-1.5641203764323111e-03+5.4200849498215256e-02_i, -2.3187022659703132e-03-1.2477497803909637e-02_i, -3.5947282825062659e-03+1.0085361313772165e-01_i,
-5.8988673497367833e-03-2.5020334564876606e-02_i, -1.0432151910686409e-02+2.2272525677242655e-01_i, -2.0467246865977801e-02-6.2932090485856185e-02_i,
-4.6926317974182050e-02+6.7554505650337759e-01_i, -1.4072334780537604e-01-2.5796064625939608e-01_i, -7.9497110360140977e-01+5.1803545039392533e+00_i,
1.4805873973945676e+01+1.1409854850790171e+00_i,  8.5960253677296672e-01-2.4903175966776388e+01_i},
{-3.1678833426962992e+01-0.0000000000000000e+00_i,  6.9388939039072284e-16+2.6955003428947762e+01_i,  1.6025762507910954e+01-3.3306690738754696e-16_i,
-8.3266726846886741e-17-5.6071753098134209e+00_i, -1.5231785406357318e-01+2.1510571102112408e-16_i, -2.8275992658421956e-16-7.3120470010534011e-01_i,
-2.2153588369192886e-02+3.0097452308197603e-16_i, -8.6736173798840355e-17-2.4107608073859582e-01_i, -6.3848880099146351e-03-3.5908775952719907e-16_i,
8.1532003370909933e-17-1.0916316423157577e-01_i, -2.5097452474864695e-03+2.7863995832877464e-16_i, -6.0715321659188248e-16-5.8666576746090243e-02_i,
-1.1842190481264688e-03+2.2280354644577116e-16_i, -2.3765711620882257e-16-3.5154009988664699e-02_i, -6.3102817238027569e-04-1.8038413644727580e-16_i,
-1.4771170397942512e-15-2.2720622571590524e-02_i, -3.6689686404518709e-04+9.4206326767265480e-16_i,  2.2004967292765798e-15-1.5526441061546740e-02_i,
-2.2783341689964793e-04-1.2217603231196028e-17_i,  3.4087316302944259e-16-1.1073600249765109e-02_i, -1.4895985290024985e-04-3.0356305576878517e-16_i,
-4.0982842119952068e-16-8.1689989119343026e-03_i, -1.0152138785738967e-04+4.7027269231558755e-17_i, -7.2034392339936915e-16-6.1927622454495835e-03_i,
-7.1602352649996549e-05+6.4915419231443422e-16_i,  5.3104222408340007e-16-4.8007215265614880e-03_i, -5.1978180184001491e-05+1.4989095034612099e-16_i,
2.6508743117270583e-16-3.7912040470076090e-03_i, -3.8675943088435218e-05-5.9788201746771054e-16_i, -6.8039107334327831e-16-3.0406334555358292e-03_i,
-2.9403314301657677e-05+5.1736867709499231e-16_i, -1.1861171766991419e-15-2.4703760991573848e-03_i, -2.2782640940022247e-05+1.7763568394002505e-15_i,
5.7549451315530575e-16-2.0287523436830885e-03_i, -1.7956483147418136e-05+2.9532407216996100e-16_i,  5.1385761964950483e-16-1.6808306121019853e-03_i,
-1.4374719259129395e-05-4.5216524548565874e-16_i,  2.7267684638010437e-16-1.4024100559580886e-03_i, -1.1674894644376699e-05+1.8675382421062814e-16_i,
1.2121380288387940e-16-1.1763519600243495e-03_i, -9.6125533845481756e-06-2.3902422738569101e-16_i,  1.2706849461530112e-16-9.9027615910782371e-04_i,
-8.0193887699523937e-06-1.3056504662156687e-16_i,  7.3725747729014302e-18-8.3507612966308986e-04_i, -6.7774000639201357e-06-3.9723812347153276e-16_i,
-7.8496237287950521e-16-7.0393896975123361e-04_i, -5.8025360107236091e-06+9.8804699231319626e-17_i, -1.1457848558826811e-15-5.9168457040667359e-04_i,
-5.0341131980084547e-06+9.4206326767265480e-16_i,  7.5547207378789949e-16-4.9431108637093893e-04_i, -4.4278360525685286e-06-2.9140643890979145e-16_i,
6.8174632605888519e-16-4.0867637607317625e-04_i, -3.9511140271677840e-06-5.0231086651253420e-16_i, -2.9490299091605721e-16-3.3227059708190732e-04_i,
-3.5798752380921534e-06+1.9255430583342559e-16_i, -6.1929628092372013e-16-2.6305078907833818e-04_i, -3.2963752891737239e-06+5.2909066017292616e-16_i,
7.8756445809347042e-16-1.9931806098734406e-04_i, -3.0876823687766425e-06-1.4745149545802860e-17_i,  6.0541849311590568e-16-1.3962422037111510e-04_i,
-2.9446331180071894e-06-2.7061686225238191e-16_i, -1.9428902930940239e-16-8.2698687407845028e-05_i, -2.8611264184519314e-06-3.3306690738754696e-16_i,
2.4980018054066022e-16-2.7389124948484778e-05_i, -2.8336698534303650e-06-0.0000000000000000e+00_i,  2.4980018054066022e-16+2.7389124948484778e-05_i,
-2.8611264184519314e-06+3.3306690738754696e-16_i, -1.9428902930940239e-16+8.2698687407845028e-05_i, -2.9446331180071894e-06+2.7061686225238191e-16_i,
6.0541849311590568e-16+1.3962422037111510e-04_i, -3.0876823687766425e-06+1.4745149545802860e-17_i,  7.8756445809347042e-16+1.9931806098734406e-04_i,
-3.2963752891737239e-06-5.2909066017292616e-16_i, -6.1929628092372013e-16+2.6305078907833818e-04_i, -3.5798752380921534e-06-1.9255430583342559e-16_i,
-2.9490299091605721e-16+3.3227059708190732e-04_i, -3.9511140271677840e-06+5.0231086651253420e-16_i,  6.8174632605888519e-16+4.0867637607317625e-04_i,
-4.4278360525685286e-06+2.9140643890979145e-16_i,  7.5547207378789949e-16+4.9431108637093893e-04_i, -5.0341131980084547e-06-9.4206326767265480e-16_i,
-1.1457848558826811e-15+5.9168457040667359e-04_i, -5.8025360107236091e-06-9.8804699231319626e-17_i, -7.8496237287950521e-16+7.0393896975123361e-04_i,
-6.7774000639201357e-06+3.9723812347153276e-16_i,  7.3725747729014302e-18+8.3507612966308986e-04_i, -8.0193887699523937e-06+1.3056504662156687e-16_i,
1.2706849461530112e-16+9.9027615910782371e-04_i, -9.6125533845481756e-06+2.3902422738569101e-16_i,  1.2121380288387940e-16+1.1763519600243495e-03_i,
-1.1674894644376699e-05-1.8675382421062814e-16_i,  2.7267684638010437e-16+1.4024100559580886e-03_i, -1.4374719259129395e-05+4.5216524548565874e-16_i,
5.1385761964950483e-16+1.6808306121019853e-03_i, -1.7956483147418136e-05-2.9532407216996100e-16_i,  5.7549451315530575e-16+2.0287523436830885e-03_i,
-2.2782640940022247e-05-1.7763568394002505e-15_i, -1.1861171766991419e-15+2.4703760991573848e-03_i, -2.9403314301657677e-05-5.1736867709499231e-16_i,
-6.8039107334327831e-16+3.0406334555358292e-03_i, -3.8675943088435218e-05+5.9788201746771054e-16_i,  2.6508743117270583e-16+3.7912040470076090e-03_i,
-5.1978180184001491e-05-1.4989095034612099e-16_i,  5.3104222408340007e-16+4.8007215265614880e-03_i, -7.1602352649996549e-05-6.4915419231443422e-16_i,
-7.2034392339936915e-16+6.1927622454495835e-03_i, -1.0152138785738967e-04-4.7027269231558755e-17_i, -4.0982842119952068e-16+8.1689989119343026e-03_i,
-1.4895985290024985e-04+3.0356305576878517e-16_i,  3.4087316302944259e-16+1.1073600249765109e-02_i, -2.2783341689964793e-04+1.2217603231196028e-17_i,
2.2004967292765798e-15+1.5526441061546740e-02_i, -3.6689686404518709e-04-9.4206326767265480e-16_i, -1.4771170397942512e-15+2.2720622571590524e-02_i,
-6.3102817238027569e-04+1.8038413644727580e-16_i, -2.3765711620882257e-16+3.5154009988664699e-02_i, -1.1842190481264688e-03-2.2280354644577116e-16_i,
-6.0715321659188248e-16+5.8666576746090243e-02_i, -2.5097452474864695e-03-2.7863995832877464e-16_i,  8.1532003370909933e-17+1.0916316423157577e-01_i,
-6.3848880099146351e-03+3.5908775952719907e-16_i, -8.6736173798840355e-17+2.4107608073859582e-01_i, -2.2153588369192886e-02-3.0097452308197603e-16_i,
-2.8275992658421956e-16+7.3120470010534011e-01_i, -1.5231785406357318e-01-2.1510571102112408e-16_i, -8.3266726846886741e-17+5.6071753098134209e+00_i,
1.6025762507910954e+01+3.3306690738754696e-16_i,  6.9388939039072284e-16-2.6955003428947762e+01_i},
{-2.9267425817005496e+01-0.0000000000000000e+00_i, -8.5960253677296583e-01+2.4903175966776391e+01_i,  1.4805873973945680e+01+1.1409854850790162e+00_i,
7.9497110360140932e-01-5.1803545039392525e+00_i, -1.4072334780537626e-01-2.5796064625939580e-01_i,  4.6926317974182244e-02-6.7554505650337715e-01_i,
-2.0467246865977388e-02-6.2932090485856323e-02_i,  1.0432151910685729e-02-2.2272525677242719e-01_i, -5.8988673497370608e-03-2.5020334564877532e-02_i,
3.5947282825059572e-03-1.0085361313772158e-01_i, -2.3187022659704533e-03-1.2477497803908919e-02_i,  1.5641203764317456e-03-5.4200849498214902e-02_i,
-1.0940757405736394e-03-7.1190598692697634e-03_i,  7.8861275666891432e-04-3.2478070314224949e-02_i, -5.8299401290046941e-04-4.4441326323197833e-03_i,
4.4040421377794254e-04-2.0991118159805599e-02_i, -3.3896850323394902e-04-2.9592955170763386e-03_i,  2.6519512008629396e-04-1.4344561109506552e-02_i,
-2.1049063069586676e-04-2.0687054279043132e-03_i,  1.6922580462649332e-04-1.0230672621969866e-02_i, -1.3762095926046699e-04-1.5019534504385175e-03_i,
1.1308204765568606e-04-7.5471708958427270e-03_i, -9.3793532354012978e-05-1.1239769490144430e-03_i,  7.8462165903153691e-05-5.7213662882791838e-03_i,
-6.6151948093105296e-05-8.6210226491269801e-04_i,  5.6175560035149437e-05-4.4352883596771950e-03_i, -4.8021576809661253e-05-6.7482214710186720e-04_i,
4.1304813807290040e-05-3.5026158226045944e-03_i, -3.5731912220102692e-05-5.3724019452031145e-04_i,  3.1077136080027198e-05-2.8091790154387134e-03_i,
-2.7165120271756918e-05-4.3379861797386853e-04_i,  2.3858419174042930e-05-2.2823299156165019e-03_i, -2.1048415661084618e-05-3.5443044687077929e-04_i,
1.8648611778555849e-05-1.8743227668636150e-03_i, -1.6589627256511550e-05-2.9242279694184924e-04_i,  1.4815435965320159e-05-1.5528850001389053e-03_i,
-1.3280508908738496e-05-2.4318020790470350e-04_i,  1.1947627404704144e-05-1.2956579468882522e-03_i, -1.0786196206442262e-05-2.0348279618023047e-04_i,
9.7709327056872981e-06-1.0868074988961084e-03_i, -8.8808413264236316e-06-1.7102704621516638e-04_i,  8.0984058064958314e-06-9.1489587493326363e-04_i,
-7.4089491476449004e-06-1.4413355039411165e-04_i,  6.8001234805458322e-06-7.7150974428432696e-04_i, -6.2615012027124437e-06-1.2155632512025633e-04_i,
5.7842456139605911e-06-6.5035480629028163e-04_i, -5.3608442565331082e-06-1.0235557480951785e-04_i,  4.9848920501732903e-06-5.4664526429859788e-04_i,
-4.6509141465317242e-06-8.5811026282062771e-05_i,  4.3542206600325969e-06-4.5668389539311668e-04_i, -4.0907871026653764e-06-7.1361750513740385e-05_i,
3.8571556484918307e-06-3.7756773927472009e-04_i, -3.6503533802541028e-06-5.8563596923063253e-05_i,  3.4678244102172295e-06-3.0697800389924146e-04_i,
-3.3073734607507420e-06-4.7058524223558655e-05_i,  3.1671188984545504e-06-2.4302724004125753e-04_i, -3.0454536621049751e-06-3.6552068609285379e-05_i,
2.9410127939310735e-06-1.8414587700507634e-04_i, -2.8526465433321296e-06-2.6796424530746432e-05_i,  2.7793982455108501e-06-1.2899595944337872e-04_i,
-2.7204862677182584e-06-1.7577403563728833e-05_i,  2.6752895233017604e-06-7.6403624662102487e-05_i, -2.6433361401956290e-06-8.7040473814026242e-06_i,
2.6242949504595714e-06-2.5304251956015378e-05_i, -2.6179695797878821e-06-0.0000000000000000e+00_i,  2.6242949504595714e-06+2.5304251956015378e-05_i,
-2.6433361401956290e-06+8.7040473814026242e-06_i,  2.6752895233017604e-06+7.6403624662102487e-05_i, -2.7204862677182584e-06+1.7577403563728833e-05_i,
2.7793982455108501e-06+1.2899595944337872e-04_i, -2.8526465433321296e-06+2.6796424530746432e-05_i,  2.9410127939310735e-06+1.8414587700507634e-04_i,
-3.0454536621049751e-06+3.6552068609285379e-05_i,  3.1671188984545504e-06+2.4302724004125753e-04_i, -3.3073734607507420e-06+4.7058524223558655e-05_i,
3.4678244102172295e-06+3.0697800389924146e-04_i, -3.6503533802541028e-06+5.8563596923063253e-05_i,  3.8571556484918307e-06+3.7756773927472009e-04_i,
-4.0907871026653764e-06+7.1361750513740385e-05_i,  4.3542206600325969e-06+4.5668389539311668e-04_i, -4.6509141465317242e-06+8.5811026282062771e-05_i,
4.9848920501732903e-06+5.4664526429859788e-04_i, -5.3608442565331082e-06+1.0235557480951785e-04_i,  5.7842456139605911e-06+6.5035480629028163e-04_i,
-6.2615012027124437e-06+1.2155632512025633e-04_i,  6.8001234805458322e-06+7.7150974428432696e-04_i, -7.4089491476449004e-06+1.4413355039411165e-04_i,
8.0984058064958314e-06+9.1489587493326363e-04_i, -8.8808413264236316e-06+1.7102704621516638e-04_i,  9.7709327056872981e-06+1.0868074988961084e-03_i,
-1.0786196206442262e-05+2.0348279618023047e-04_i,  1.1947627404704144e-05+1.2956579468882522e-03_i, -1.3280508908738496e-05+2.4318020790470350e-04_i,
1.4815435965320159e-05+1.5528850001389053e-03_i, -1.6589627256511550e-05+2.9242279694184924e-04_i,  1.8648611778555849e-05+1.8743227668636150e-03_i,
-2.1048415661084618e-05+3.5443044687077929e-04_i,  2.3858419174042930e-05+2.2823299156165019e-03_i, -2.7165120271756918e-05+4.3379861797386853e-04_i,
3.1077136080027198e-05+2.8091790154387134e-03_i, -3.5731912220102692e-05+5.3724019452031145e-04_i,  4.1304813807290040e-05+3.5026158226045944e-03_i,
-4.8021576809661253e-05+6.7482214710186720e-04_i,  5.6175560035149437e-05+4.4352883596771950e-03_i, -6.6151948093105296e-05+8.6210226491269801e-04_i,
7.8462165903153691e-05+5.7213662882791838e-03_i, -9.3793532354012978e-05+1.1239769490144430e-03_i,  1.1308204765568606e-04+7.5471708958427270e-03_i,
-1.3762095926046699e-04+1.5019534504385175e-03_i,  1.6922580462649332e-04+1.0230672621969866e-02_i, -2.1049063069586676e-04+2.0687054279043132e-03_i,
2.6519512008629396e-04+1.4344561109506552e-02_i, -3.3896850323394902e-04+2.9592955170763386e-03_i,  4.4040421377794254e-04+2.0991118159805599e-02_i,
-5.8299401290046941e-04+4.4441326323197833e-03_i,  7.8861275666891432e-04+3.2478070314224949e-02_i, -1.0940757405736394e-03+7.1190598692697634e-03_i,
1.5641203764317456e-03+5.4200849498214902e-02_i, -2.3187022659704533e-03+1.2477497803908919e-02_i,  3.5947282825059572e-03+1.0085361313772158e-01_i,
-5.8988673497370608e-03+2.5020334564877532e-02_i,  1.0432151910685729e-02+2.2272525677242719e-01_i, -2.0467246865977388e-02+6.2932090485856323e-02_i,
4.6926317974182244e-02+6.7554505650337715e-01_i, -1.4072334780537626e-01+2.5796064625939580e-01_i,  7.9497110360140932e-01+5.1803545039392525e+00_i,
1.4805873973945680e+01-1.1409854850790162e+00_i, -8.5960253677296583e-01-2.4903175966776391e+01_i},
{-2.2400317936284612e+01-0.0000000000000000e+00_i, -1.5883383796386481e+00+1.9060065711515605e+01_i,  1.1331925343028971e+01+2.1082662731139301e+00_i,
1.4689150631105030e+00-3.9648716848708512e+00_i, -1.0770498750413550e-01-4.7664912254487907e-01_i,  8.6708529424927377e-02-5.1703980187996201e-01_i,
-1.5664952563471609e-02-1.1628334067606187e-01_i,  1.9276103260662671e-02-1.7046653147213686e-01_i, -4.5147976089275291e-03-4.6231550002150221e-02_i,
6.6421917702927788e-03-7.7190013683928124e-02_i, -1.7746578835485275e-03-2.3055409675972083e-02_i,  2.8901176043391145e-03-4.1483534246161337e-02_i,
-8.3736931934025576e-04-1.3154307407881607e-02_i,  1.4571663699281560e-03-2.4857638848884579e-02_i, -4.4620429981055481e-04-8.2116863575316876e-03_i,
8.1376087828155226e-04-1.6065906293151213e-02_i, -2.5943526056237715e-04-5.4680651177597128e-03_i,  4.9001668713834256e-04-1.0978851762313242e-02_i,
-1.6110255407050247e-04-3.8224692072726572e-03_i,  3.1268851453507729e-04-7.8302178287583193e-03_i, -1.0533052211045670e-04-2.7752481032900958e-03_i,
2.0894837864611059e-04-5.7763545261338544e-03_i, -7.1786461789879942e-05-2.0768385964168787e-03_i,  1.4497917830941329e-04-4.3789441780335871e-03_i,
-5.0630509107394014e-05-1.5929572749685003e-03_i,  1.0379890028864821e-04-3.3946227460200607e-03_i, -3.6754123682173180e-05-1.2469087395867754e-03_i,
7.6321344140128466e-05-2.6807860905011128e-03_i, -2.7348021626715619e-05-9.9269043951812790e-04_i,  5.7423059907002098e-05-2.1500525355121697e-03_i,
-2.0791282931922016e-05-8.0155532875639181e-04_i,  4.4084610306940300e-05-1.7468196917954431e-03_i, -1.6109759901183907e-05-6.5490207112617327e-04_i,
3.4458141462054867e-05-1.4345445395663519e-03_i, -1.2697150998884122e-05-5.4032687386798485e-04_i,  2.7375356108494772e-05-1.1885267238428635e-03_i,
-1.0164461466006418e-05-4.4933843359059472e-04_i,  2.2076336842038091e-05-9.9165366057242716e-04_i, -8.2553971729553127e-06-3.7598718121851734e-04_i,
1.8054329480230954e-05-8.3180644799571001e-04_i, -6.7971016822671081e-06-3.1601677500676440e-04_i,  1.4963902740858302e-05-7.0023098735222042e-04_i,
-5.6705641803785132e-06-2.6632407431420092e-04_i,  1.2564989804492339e-05-5.9048799409189452e-04_i, -4.7923455444190237e-06-2.2460680165279799e-04_i,
1.0687892266001862e-05-4.9776001905233597e-04_i, -4.1030125611851492e-06-1.8912844120818261e-04_i,  9.2108794779023444e-06-4.1838417205715753e-04_i,
-3.5596555793017664e-06-1.5855810169368434e-04_i,  8.0455506942275594e-06-3.4953072118906400e-04_i, -3.1309528978500957e-06-1.3185932140771101e-04_i,
7.1270943139386048e-06-2.8897783683250536e-04_i, -2.7938595221670484e-06-1.0821141709378007e-04_i,  6.4077039911118738e-06-2.3495079238513433e-04_i,
-2.5313540561655232e-06-8.6952814721472094e-05_i,  5.8520726551594335e-06-1.8600499675459903e-04_i, -2.3308893205963716e-06-6.7539416121639023e-05_i,
5.4342830454086333e-06-1.4093915253653166e-04_i, -2.1833211410471343e-06-4.9513336330395119e-05_i,  5.1356583083903007e-06-9.8729233042271769e-05_i,
-2.0821700458528714e-06-3.2478806777092029e-05_i,  4.9432904672963573e-06-5.8476802661466820e-05_i, -2.0231218940125473e-06-1.6082982452703121e-05_i,
4.8490647841514800e-06-1.9367035983819392e-05_i, -2.0037071699618991e-06-0.0000000000000000e+00_i,  4.8490647841514800e-06+1.9367035983819392e-05_i,
-2.0231218940125473e-06+1.6082982452703121e-05_i,  4.9432904672963573e-06+5.8476802661466820e-05_i, -2.0821700458528714e-06+3.2478806777092029e-05_i,
5.1356583083903007e-06+9.8729233042271769e-05_i, -2.1833211410471343e-06+4.9513336330395119e-05_i,  5.4342830454086333e-06+1.4093915253653166e-04_i,
-2.3308893205963716e-06+6.7539416121639023e-05_i,  5.8520726551594335e-06+1.8600499675459903e-04_i, -2.5313540561655232e-06+8.6952814721472094e-05_i,
6.4077039911118738e-06+2.3495079238513433e-04_i, -2.7938595221670484e-06+1.0821141709378007e-04_i,  7.1270943139386048e-06+2.8897783683250536e-04_i,
-3.1309528978500957e-06+1.3185932140771101e-04_i,  8.0455506942275594e-06+3.4953072118906400e-04_i, -3.5596555793017664e-06+1.5855810169368434e-04_i,
9.2108794779023444e-06+4.1838417205715753e-04_i, -4.1030125611851492e-06+1.8912844120818261e-04_i,  1.0687892266001862e-05+4.9776001905233597e-04_i,
-4.7923455444190237e-06+2.2460680165279799e-04_i,  1.2564989804492339e-05+5.9048799409189452e-04_i, -5.6705641803785132e-06+2.6632407431420092e-04_i,
1.4963902740858302e-05+7.0023098735222042e-04_i, -6.7971016822671081e-06+3.1601677500676440e-04_i,  1.8054329480230954e-05+8.3180644799571001e-04_i,
-8.2553971729553127e-06+3.7598718121851734e-04_i,  2.2076336842038091e-05+9.9165366057242716e-04_i, -1.0164461466006418e-05+4.4933843359059472e-04_i,
2.7375356108494772e-05+1.1885267238428635e-03_i, -1.2697150998884122e-05+5.4032687386798485e-04_i,  3.4458141462054867e-05+1.4345445395663519e-03_i,
-1.6109759901183907e-05+6.5490207112617327e-04_i,  4.4084610306940300e-05+1.7468196917954431e-03_i, -2.0791282931922016e-05+8.0155532875639181e-04_i,
5.7423059907002098e-05+2.1500525355121697e-03_i, -2.7348021626715619e-05+9.9269043951812790e-04_i,  7.6321344140128466e-05+2.6807860905011128e-03_i,
-3.6754123682173180e-05+1.2469087395867754e-03_i,  1.0379890028864821e-04+3.3946227460200607e-03_i, -5.0630509107394014e-05+1.5929572749685003e-03_i,
1.4497917830941329e-04+4.3789441780335871e-03_i, -7.1786461789879942e-05+2.0768385964168787e-03_i,  2.0894837864611059e-04+5.7763545261338544e-03_i,
-1.0533052211045670e-04+2.7752481032900958e-03_i,  3.1268851453507729e-04+7.8302178287583193e-03_i, -1.6110255407050247e-04+3.8224692072726572e-03_i,
4.9001668713834256e-04+1.0978851762313242e-02_i, -2.5943526056237715e-04+5.4680651177597128e-03_i,  8.1376087828155226e-04+1.6065906293151213e-02_i,
-4.4620429981055481e-04+8.2116863575316876e-03_i,  1.4571663699281560e-03+2.4857638848884579e-02_i, -8.3736931934025576e-04+1.3154307407881607e-02_i,
2.8901176043391145e-03+4.1483534246161337e-02_i, -1.7746578835485275e-03+2.3055409675972083e-02_i,  6.6421917702927788e-03+7.7190013683928124e-02_i,
-4.5147976089275291e-03+4.6231550002150221e-02_i,  1.9276103260662671e-02+1.7046653147213686e-01_i, -1.5664952563471609e-02+1.1628334067606187e-01_i,
8.6708529424927377e-02+5.1703980187996201e-01_i, -1.0770498750413550e-01+4.7664912254487907e-01_i,  1.4689150631105030e+00+3.9648716848708512e+00_i,
1.1331925343028971e+01-2.1082662731139301e+00_i, -1.5883383796386481e+00-1.9060065711515605e+01_i},
{-1.2122964709152143e+01-0.0000000000000000e+00_i, -2.0752641025276124e+00+1.0315233231602498e+01_i,  6.1327938027951365e+00+2.7545826325486065e+00_i,
1.9192300200092296e+00-2.1457730934321870e+00_i, -5.8289519203533320e-02-6.2277209075796147e-01_i,  1.1329015328550315e-01-2.7981992439779746e-01_i,
-8.4778112363256240e-03-1.5193150635944491e-01_i,  2.5185442627514730e-02-9.2255822038170107e-02_i, -2.4433908589010932e-03-6.0404431041639145e-02_i,
8.6784417726721748e-03-4.1774934375972884e-02_i, -9.6043792566975848e-04-3.0123344422678056e-02_i,  3.7761206259663846e-03-2.2450726954304241e-02_i,
-4.5318101000938957e-04-1.7186930887736679e-02_i,  1.9038796126108998e-03-1.3452857203858483e-02_i, -2.4148402692531412e-04-1.0729085273930542e-02_i,
1.0632298258291904e-03-8.6948058311684916e-03_i, -1.4040535125707672e-04-7.1443713723974098e-03_i,  6.4023765558542350e-04-5.9417117578464034e-03_i,
-8.7188073986294717e-05-4.9942967006023388e-03_i,  4.0854723263338531e-04-4.2376833522194644e-03_i, -5.7004467792813097e-05-3.6260363901015114e-03_i,
2.7300421311092515e-04-3.1261405426051478e-03_i, -3.8850553163546865e-05-2.7135203941052411e-03_i,  1.8942442505683917e-04-2.3698675119098920e-03_i,
-2.7401034078146586e-05-2.0812989801046048e-03_i,  1.3561979891176082e-04-1.8371565916129456e-03_i, -1.9891188400432750e-05-1.6291647797239469e-03_i,
9.9718641683671662e-05-1.4508309775061023e-03_i, -1.4800642651674017e-05-1.2970125638610409e-03_i,  7.5026843404580966e-05-1.1636000473278702e-03_i,
-1.1252161239257331e-05-1.0472825068529100e-03_i,  5.7599319147783710e-05-9.4537200485866890e-04_i, -8.7185392336763812e-06-8.5567079175552507e-04_i,
4.5021731474081286e-05-7.7636991029894631e-04_i, -6.8716486034165901e-06-7.0597108232548595e-04_i,  3.5767626439940826e-05-6.4322602786350555e-04_i,
-5.5009669056753699e-06-5.8708895602392586e-04_i,  2.8844124117811796e-05-5.3667909379735527e-04_i, -4.4677887547750592e-06-4.9125092624827745e-04_i,
2.3589118254494542e-05-4.5017040573174655e-04_i, -3.6785649228780228e-06-4.1289581450431180e-04_i,  1.9551281131317529e-05-3.7896227955637940e-04_i,
-3.0688872196028578e-06-3.4796917215421715e-04_i,  1.6416950331797522e-05-3.1956979958620152e-04_i, -2.5935987198374413e-06-2.9346292869691102e-04_i,
1.3964404210539921e-05-2.6938578111928776e-04_i, -2.2205343963958846e-06-2.4710821689008814e-04_i,  1.2034593998217326e-05-2.2642788228010181e-04_i,
-1.9264717176627286e-06-2.0716614345315555e-04_i,  1.0512018569266632e-05-1.8916466318868574e-04_i, -1.6944594978759477e-06-1.7228250592490717e-04_i,
9.3119974778383026e-06-1.5639367832277588e-04_i, -1.5120258781038676e-06-1.4138502995154900e-04_i,  8.3720687245108713e-06-1.2715445256475735e-04_i,
-1.3699589429131201e-06-1.1360932740680715e-04_i,  7.6461013974711903e-06-1.0066517885167631e-04_i, -1.2614682109381332e-06-8.8244499769987023e-05_i,
7.1002329743996562e-06-7.6275719710042211e-05_i, -1.1816048864745146e-06-6.4692291525059176e-05_i,  6.7100609389414423e-06-5.3431875892989700e-05_i,
-1.1268623086244101e-06-4.2435606074420207e-05_i,  6.4587202497357055e-06-3.1647417549240942e-05_i, -1.0949056790110490e-06-2.1013429236660386e-05_i,
6.3356084614163422e-06-1.0481364344450128e-05_i, -1.0843985061015360e-06-0.0000000000000000e+00_i,  6.3356084614163422e-06+1.0481364344450128e-05_i,
-1.0949056790110490e-06+2.1013429236660386e-05_i,  6.4587202497357055e-06+3.1647417549240942e-05_i, -1.1268623086244101e-06+4.2435606074420207e-05_i,
6.7100609389414423e-06+5.3431875892989700e-05_i, -1.1816048864745146e-06+6.4692291525059176e-05_i,  7.1002329743996562e-06+7.6275719710042211e-05_i,
-1.2614682109381332e-06+8.8244499769987023e-05_i,  7.6461013974711903e-06+1.0066517885167631e-04_i, -1.3699589429131201e-06+1.1360932740680715e-04_i,
8.3720687245108713e-06+1.2715445256475735e-04_i, -1.5120258781038676e-06+1.4138502995154900e-04_i,  9.3119974778383026e-06+1.5639367832277588e-04_i,
-1.6944594978759477e-06+1.7228250592490717e-04_i,  1.0512018569266632e-05+1.8916466318868574e-04_i, -1.9264717176627286e-06+2.0716614345315555e-04_i,
1.2034593998217326e-05+2.2642788228010181e-04_i, -2.2205343963958846e-06+2.4710821689008814e-04_i,  1.3964404210539921e-05+2.6938578111928776e-04_i,
-2.5935987198374413e-06+2.9346292869691102e-04_i,  1.6416950331797522e-05+3.1956979958620152e-04_i, -3.0688872196028578e-06+3.4796917215421715e-04_i,
1.9551281131317529e-05+3.7896227955637940e-04_i, -3.6785649228780228e-06+4.1289581450431180e-04_i,  2.3589118254494542e-05+4.5017040573174655e-04_i,
-4.4677887547750592e-06+4.9125092624827745e-04_i,  2.8844124117811796e-05+5.3667909379735527e-04_i, -5.5009669056753699e-06+5.8708895602392586e-04_i,
3.5767626439940826e-05+6.4322602786350555e-04_i, -6.8716486034165901e-06+7.0597108232548595e-04_i,  4.5021731474081286e-05+7.7636991029894631e-04_i,
-8.7185392336763812e-06+8.5567079175552507e-04_i,  5.7599319147783710e-05+9.4537200485866890e-04_i, -1.1252161239257331e-05+1.0472825068529100e-03_i,
7.5026843404580966e-05+1.1636000473278702e-03_i, -1.4800642651674017e-05+1.2970125638610409e-03_i,  9.9718641683671662e-05+1.4508309775061023e-03_i,
-1.9891188400432750e-05+1.6291647797239469e-03_i,  1.3561979891176082e-04+1.8371565916129456e-03_i, -2.7401034078146586e-05+2.0812989801046048e-03_i,
1.8942442505683917e-04+2.3698675119098920e-03_i, -3.8850553163546865e-05+2.7135203941052411e-03_i,  2.7300421311092515e-04+3.1261405426051478e-03_i,
-5.7004467792813097e-05+3.6260363901015114e-03_i,  4.0854723263338531e-04+4.2376833522194644e-03_i, -8.7188073986294717e-05+4.9942967006023388e-03_i,
6.4023765558542350e-04+5.9417117578464034e-03_i, -1.4040535125707672e-04+7.1443713723974098e-03_i,  1.0632298258291904e-03+8.6948058311684916e-03_i,
-2.4148402692531412e-04+1.0729085273930542e-02_i,  1.9038796126108998e-03+1.3452857203858483e-02_i, -4.5318101000938957e-04+1.7186930887736679e-02_i,
3.7761206259663846e-03+2.2450726954304241e-02_i, -9.6043792566975848e-04+3.0123344422678056e-02_i,  8.6784417726721748e-03+4.1774934375972884e-02_i,
-2.4433908589010932e-03+6.0404431041639145e-02_i,  2.5185442627514730e-02+9.2255822038170107e-02_i, -8.4778112363256240e-03+1.5193150635944491e-01_i,
1.1329015328550315e-01+2.7981992439779746e-01_i, -5.8289519203533320e-02+6.2277209075796147e-01_i,  1.9192300200092296e+00+2.1457730934321870e+00_i,
6.1327938027951365e+00-2.7545826325486065e+00_i, -2.0752641025276124e+00-1.0315233231602498e+01_i},
{-1.8561830307012563e+00-0.0000000000000000e+00_i, -1.1231248390613429e+00+1.3647140589464519e+00_i,  2.9938833406609444e-01+1.4907693782657527e+00_i,
1.0386798021125048e+00+5.3384012727501462e-01_i,  7.0345957258712111e-01-3.3704182679810374e-01_i,  6.1312189143078094e-02-4.2305593058264412e-01_i,
-1.4494751578417836e-01-8.2224738731067892e-02_i,  1.3630263330467641e-02+6.9924186916001080e-02_i,  9.6735350049332092e-02-3.2690642511285867e-02_i,
4.6967388427150061e-03-9.1649439085612211e-02_i, -5.2246138542576127e-02-1.6302636524913164e-02_i,  2.0436217564553357e-03+3.3211809089682819e-02_i,
3.9400503850194057e-02-9.3014999699258284e-03_i,  1.0303722214927789e-03-3.9586717107854857e-02_i, -2.7102794310345214e-02-5.8065391083871234e-03_i,
5.7541583529800884e-04+1.9609876109919527e-02_i,  2.1876470763211843e-02-3.8665059247379040e-03_i,  3.4649412236923468e-04-2.2276450828333023e-02_i,
-1.6808546635877622e-02-2.7028938973387515e-03_i,  2.2110416902709025e-04+1.3124982483492724e-02_i,  1.4182711865637011e-02-1.9623967533117341e-03_i,
1.4774881545848162e-04-1.4481443405078594e-02_i, -1.1619263424342607e-02-1.4685466549561751e-03_i,  1.0251576011400848e-04+9.5483414452172347e-03_i,
1.0120618008742035e-02-1.1263908912712214e-03_i,  7.3396906273326878e-05-1.0327773571518067e-02_i, -8.6514816678522309e-03-8.8169762528221964e-04_i,
5.3967339991024138e-05+7.3791991748173415e-03_i,  7.7208605005763212e-03-7.0193814140222623e-04_i,  4.0604235056775957e-05-7.8652123045484003e-03_i,
-6.8060263495654484e-03-5.6678520845989853e-04_i,  3.1172526894241662e-05+5.9749922520766427e-03_i,  6.1934992250487486e-03-4.6308569550757017e-04_i,
2.4365585494989014e-05-6.2957442864742877e-03_i, -5.5902140742915595e-03-3.8206879656934765e-04_i,  1.9357299941544730e-05+5.0238558552535527e-03_i,
5.1706035925756285e-03-3.1773025343958085e-04_i,  1.5610327484531078e-05-5.2439812572936005e-03_i, -4.7567994440500007e-03-2.6586308547877204e-04_i,
1.2766338805258364e-05+4.3600308778225724e-03_i,  4.4619850359670209e-03-2.2345760457637847e-04_i,  1.0581077100939090e-05-4.5148687178463136e-03_i,
-4.1710414019031144e-03-1.8831955894076937e-04_i,  8.8847894962884588e-06+3.8892811080797531e-03_i,  3.9616128878299768e-03-1.5882099254900181e-04_i,
7.5574810982988357e-06-3.9993434864332173e-03_i, -3.7549071583389011e-03-1.3373400329353725e-04_i,  6.5130753396308930e-06+3.5553598263079970e-03_i,
3.6070629908332066e-03-1.1211750891974891e-04_i,  5.6890634542294122e-06-3.6330630124700898e-03_i, -3.4612376444636082e-03-9.3238620330218444e-05_i,
5.0396167195123492e-06+3.3236544465722329e-03_i,  3.3602584269727537e-03-7.6517026828734451e-05_i,  4.5309309438272202e-06-3.3766875759968073e-03_i,
-3.2608615513316924e-03-6.1484924932582452e-05_i,  4.1380402585737032e-06+3.1727127796787047e-03_i,  3.1976797184812139e-03-4.7757579137403666e-05_i,
3.8426183919994944e-06-3.2058142331704842e-03_i, -3.1358167052793068e-03-3.5011215878316682e-05_i,  3.6314588156872996e-06+3.0896868030579894e-03_i,
3.1051643536079698e-03-2.2965984517064886e-05_i,  3.4954342105208980e-06-3.1056051759539849e-03_i, -3.0756893716986644e-03-1.1372385953722919e-05_i,
3.4288065913212762e-06+3.0678878722286695e-03_i,  3.0751229889940390e-03+0.0000000000000000e+00_i,  3.4288065913212762e-06-3.0678878722286695e-03_i,
-3.0756893716986644e-03+1.1372385953722919e-05_i,  3.4954342105208980e-06+3.1056051759539849e-03_i,  3.1051643536079698e-03+2.2965984517064886e-05_i,
3.6314588156872996e-06-3.0896868030579894e-03_i, -3.1358167052793068e-03+3.5011215878316682e-05_i,  3.8426183919994944e-06+3.2058142331704842e-03_i,
3.1976797184812139e-03+4.7757579137403666e-05_i,  4.1380402585737032e-06-3.1727127796787047e-03_i, -3.2608615513316924e-03+6.1484924932582452e-05_i,
4.5309309438272202e-06+3.3766875759968073e-03_i,  3.3602584269727537e-03+7.6517026828734451e-05_i,  5.0396167195123492e-06-3.3236544465722329e-03_i,
-3.4612376444636082e-03+9.3238620330218444e-05_i,  5.6890634542294122e-06+3.6330630124700898e-03_i,  3.6070629908332066e-03+1.1211750891974891e-04_i,
6.5130753396308930e-06-3.5553598263079970e-03_i, -3.7549071583389011e-03+1.3373400329353725e-04_i,  7.5574810982988357e-06+3.9993434864332173e-03_i,
3.9616128878299768e-03+1.5882099254900181e-04_i,  8.8847894962884588e-06-3.8892811080797531e-03_i, -4.1710414019031144e-03+1.8831955894076937e-04_i,
1.0581077100939090e-05+4.5148687178463136e-03_i,  4.4619850359670209e-03+2.2345760457637847e-04_i,  1.2766338805258364e-05-4.3600308778225724e-03_i,
-4.7567994440500007e-03+2.6586308547877204e-04_i,  1.5610327484531078e-05+5.2439812572936005e-03_i,  5.1706035925756285e-03+3.1773025343958085e-04_i,
1.9357299941544730e-05-5.0238558552535527e-03_i, -5.5902140742915595e-03+3.8206879656934765e-04_i,  2.4365585494989014e-05+6.2957442864742877e-03_i,
6.1934992250487486e-03+4.6308569550757017e-04_i,  3.1172526894241662e-05-5.9749922520766427e-03_i, -6.8060263495654484e-03+5.6678520845989853e-04_i,
4.0604235056775957e-05+7.8652123045484003e-03_i,  7.7208605005763212e-03+7.0193814140222623e-04_i,  5.3967339991024138e-05-7.3791991748173415e-03_i,
-8.6514816678522309e-03+8.8169762528221964e-04_i,  7.3396906273326878e-05+1.0327773571518067e-02_i,  1.0120618008742035e-02+1.1263908912712214e-03_i,
1.0251576011400848e-04-9.5483414452172347e-03_i, -1.1619263424342607e-02+1.4685466549561751e-03_i,  1.4774881545848162e-04+1.4481443405078594e-02_i,
1.4182711865637011e-02+1.9623967533117341e-03_i,  2.2110416902709025e-04-1.3124982483492724e-02_i, -1.6808546635877622e-02+2.7028938973387515e-03_i,
3.4649412236923468e-04+2.2276450828333023e-02_i,  2.1876470763211843e-02+3.8665059247379040e-03_i,  5.7541583529800884e-04-1.9609876109919527e-02_i,
-2.7102794310345214e-02+5.8065391083871234e-03_i,  1.0303722214927789e-03+3.9586717107854857e-02_i,  3.9400503850194057e-02+9.3014999699258284e-03_i,
2.0436217564553357e-03-3.3211809089682819e-02_i, -5.2246138542576127e-02+1.6302636524913164e-02_i,  4.6967388427150061e-03+9.1649439085612211e-02_i,
9.6735350049332092e-02+3.2690642511285867e-02_i,  1.3630263330467641e-02-6.9924186916001080e-02_i, -1.4494751578417836e-01+8.2224738731067892e-02_i,
6.1312189143078094e-02+4.2305593058264412e-01_i,  7.0345957258712111e-01+3.3704182679810374e-01_i,  1.0386798021125048e+00-5.3384012727501462e-01_i,
2.9938833406609444e-01-1.4907693782657527e+00_i, -1.1231248390613429e+00-1.3647140589464519e+00_i}};

/*!
    Data generated using:

        import scipy.signal
        import numpy as np
        np.set_printoptions(threshold=10_000, precision=12, linewidth=200)

        npoints     = 512
        fftsize     = 64
        wlen        = 64
        onesided    = False

        tone    = np.sin(2 * np.pi * np.arange(npoints) / npoints)
        win     = scipy.signal.get_window('hann', wlen)
        scale   = win.sum()

        f, t, Z = scipy.signal.stft(tone,
                        nfft=fftsize, nperseg=wlen, noverlap=wlen//2,
                        window=win, return_onesided=onesided)

        print(repr(Z.T * scale)
!*/
static const std::complex<double> STFT_FFT_64_WLEN_64_TONE_512[17][64] =
{{1.8561830307012523e+00+0.0000000000000000e+00_i, -2.9938833406609217e-01+1.4907693782657505e+00_i, -7.0345957258712155e-01-3.3704182679810224e-01_i,
1.4494751578417886e-01-8.2224738731068614e-02_i, -9.6735350049332869e-02-3.2690642511285041e-02_i,  5.2246138542576814e-02-1.6302636524914406e-02_i,
-3.9400503850194168e-02-9.3014999699242377e-03_i,  2.7102794310344420e-02-5.8065391083880358e-03_i, -2.1876470763211614e-02-3.8665059247377283e-03_i,
1.6808546635877716e-02-2.7028938973390130e-03_i, -1.4182711865637217e-02-1.9623967533114153e-03_i,  1.1619263424342812e-02-1.4685466549569223e-03_i,
-1.0120618008741776e-02-1.1263908912705067e-03_i,  8.6514816678518805e-03-8.8169762528257992e-04_i, -7.7208605005761877e-03-7.0193814140197264e-04_i,
6.8060263495653556e-03-5.6678520846005498e-04_i, -6.1934992250489707e-03-4.6308569550751466e-04_i,  5.5902140742917911e-03-3.8206879657005921e-04_i,
-5.1706035925751515e-03-3.1773025343882766e-04_i,  4.7567994440496850e-03-2.6586308547897272e-04_i, -4.4619850359668006e-03-2.2345760457617714e-04_i,
4.1710414019030563e-03-1.8831955894073066e-04_i, -3.9616128878303732e-03-1.5882099254863015e-04_i,  3.7549071583386816e-03-1.3373400329447964e-04_i,
-3.6070629908326619e-03-1.1211750891929571e-04_i,  3.4612376444633185e-03-9.3238620330368931e-05_i, -3.3602584269725420e-03-7.6517026828646847e-05_i,
3.2608615513319561e-03-6.1484924932582452e-05_i, -3.1976797184820951e-03-4.7757579136532835e-05_i,  3.1358167052790153e-03-3.5011215880655089e-05_i,
-3.1051643536058604e-03-2.2965984515677107e-05_i,  3.0756893716968325e-03-1.1372385953722919e-05_i, -3.0751229889927068e-03-0.0000000000000000e+00_i,
3.0756893716968325e-03+1.1372385953722919e-05_i, -3.1051643536058604e-03+2.2965984515677107e-05_i,  3.1358167052790153e-03+3.5011215880655089e-05_i,
-3.1976797184820951e-03+4.7757579136532835e-05_i,  3.2608615513319561e-03+6.1484924932582452e-05_i, -3.3602584269725420e-03+7.6517026828646847e-05_i,
3.4612376444633185e-03+9.3238620330368931e-05_i, -3.6070629908326619e-03+1.1211750891929571e-04_i,  3.7549071583386816e-03+1.3373400329447964e-04_i,
-3.9616128878303732e-03+1.5882099254863015e-04_i,  4.1710414019030563e-03+1.8831955894073066e-04_i, -4.4619850359668006e-03+2.2345760457617714e-04_i,
4.7567994440496850e-03+2.6586308547897272e-04_i, -5.1706035925751515e-03+3.1773025343882766e-04_i,  5.5902140742917911e-03+3.8206879657005921e-04_i,
-6.1934992250489707e-03+4.6308569550751466e-04_i,  6.8060263495653556e-03+5.6678520846005498e-04_i, -7.7208605005761877e-03+7.0193814140197264e-04_i,
8.6514816678518805e-03+8.8169762528257992e-04_i, -1.0120618008741776e-02+1.1263908912705067e-03_i,  1.1619263424342812e-02+1.4685466549569223e-03_i,
-1.4182711865637217e-02+1.9623967533114153e-03_i,  1.6808546635877716e-02+2.7028938973390130e-03_i, -2.1876470763211614e-02+3.8665059247377283e-03_i,
2.7102794310344420e-02+5.8065391083880358e-03_i, -3.9400503850194168e-02+9.3014999699242377e-03_i,  5.2246138542576814e-02+1.6302636524914406e-02_i,
-9.6735350049332869e-02+3.2690642511285041e-02_i,  1.4494751578417886e-01+8.2224738731068614e-02_i, -7.0345957258712155e-01+3.3704182679810224e-01_i,
-2.9938833406609217e-01-1.4907693782657505e+00_i},
{ 1.2122964709152136e+01+0.0000000000000000e+00_i, -6.1327938027951330e+00+2.7545826325486065e+00_i,  5.8289519203533022e-02-6.2277209075796147e-01_i,
8.4778112363261080e-03-1.5193150635944483e-01_i,  2.4433908589009817e-03-6.0404431041638958e-02_i,  9.6043792566995179e-04-3.0123344422678951e-02_i,
4.5318101000885549e-04-1.7186930887734618e-02_i,  2.4148402692583909e-04-1.0729085273931575e-02_i,  1.4040535125631865e-04-7.1443713723977238e-03_i,
8.7188073986975163e-05-4.9942967006020378e-03_i,  5.7004467792664886e-05-3.6260363901020561e-03_i,  3.8850553163818132e-05-2.7135203941052043e-03_i,
2.7401034077477633e-05-2.0812989801038016e-03_i,  1.9891188400941728e-05-1.6291647797248999e-03_i,  1.4800642651116927e-05-1.2970125638603093e-03_i,
1.1252161239538871e-05-1.0472825068521541e-03_i,  8.7185392332322920e-06-8.5567079175641325e-04_i,  6.8716486040232283e-06-7.0597108232406391e-04_i,
5.5009669056946958e-06-5.8708895602358271e-04_i,  4.4677887549443573e-06-4.9125092624864759e-04_i,  3.6785649226581466e-06-4.1289581450383844e-04_i,
3.0688872199864485e-06-3.4796917215382988e-04_i,  2.5935987190779577e-06-2.9346292869759797e-04_i,  2.2205343966391795e-06-2.4710821688944933e-04_i,
1.9264717175326244e-06-2.0716614345346910e-04_i,  1.6944594982035936e-06-1.7228250592560106e-04_i,  1.5120258777345883e-06-1.4138502995107542e-04_i,
1.3699589433382358e-06-1.1360932740644980e-04_i,  1.2614682101622782e-06-8.8244499773244833e-05_i,  1.1816048867043655e-06-6.4692291520840328e-05_i,
1.1268623086070628e-06-4.2435606076196564e-05_i,  1.0949056785669598e-06-2.1013429236660386e-05_i,  1.0843985052133576e-06+0.0000000000000000e+00_i,
1.0949056785669598e-06+2.1013429236660386e-05_i,  1.1268623086070628e-06+4.2435606076196564e-05_i,  1.1816048867043655e-06+6.4692291520840328e-05_i,
1.2614682101622782e-06+8.8244499773244833e-05_i,  1.3699589433382358e-06+1.1360932740644980e-04_i,  1.5120258777345883e-06+1.4138502995107542e-04_i,
1.6944594982035936e-06+1.7228250592560106e-04_i,  1.9264717175326244e-06+2.0716614345346910e-04_i,  2.2205343966391795e-06+2.4710821688944933e-04_i,
2.5935987190779577e-06+2.9346292869759797e-04_i,  3.0688872199864485e-06+3.4796917215382988e-04_i,  3.6785649226581466e-06+4.1289581450383844e-04_i,
4.4677887549443573e-06+4.9125092624864759e-04_i,  5.5009669056946958e-06+5.8708895602358271e-04_i,  6.8716486040232283e-06+7.0597108232406391e-04_i,
8.7185392332322920e-06+8.5567079175641325e-04_i,  1.1252161239538871e-05+1.0472825068521541e-03_i,  1.4800642651116927e-05+1.2970125638603093e-03_i,
1.9891188400941728e-05+1.6291647797248999e-03_i,  2.7401034077477633e-05+2.0812989801038016e-03_i,  3.8850553163818132e-05+2.7135203941052043e-03_i,
5.7004467792664886e-05+3.6260363901020561e-03_i,  8.7188073986975163e-05+4.9942967006020378e-03_i,  1.4040535125631865e-04+7.1443713723977238e-03_i,
2.4148402692583909e-04+1.0729085273931575e-02_i,  4.5318101000885549e-04+1.7186930887734618e-02_i,  9.6043792566995179e-04+3.0123344422678951e-02_i,
2.4433908589009817e-03+6.0404431041638958e-02_i,  8.4778112363261080e-03+1.5193150635944483e-01_i,  5.8289519203533022e-02+6.2277209075796147e-01_i,
-6.1327938027951330e+00-2.7545826325486065e+00_i},
{ 2.2400317936284608e+01+0.0000000000000000e+00_i, -1.1331925343028967e+01+2.1082662731139310e+00_i,  1.0770498750413532e-01-4.7664912254487979e-01_i,
1.5664952563471703e-02-1.1628334067606182e-01_i,  4.5147976089274624e-03-4.6231550002149430e-02_i,  1.7746578835483927e-03-2.3055409675972874e-02_i,
8.3736931933972830e-04-1.3154307407879831e-02_i,  4.4620429981049366e-04-8.2116863575322878e-03_i,  2.5943526056193306e-04-5.4680651177607849e-03_i,
1.6110255407045292e-04-3.8224692072726554e-03_i,  1.0533052211046385e-04-2.7752481032899167e-03_i,  7.1786461789753470e-05-2.0768385964179889e-03_i,
5.0630509107736296e-05-1.5929572749680144e-03_i,  3.6754123681629128e-05-1.2469087395869357e-03_i,  2.7348021626939778e-05-9.9269043951715277e-04_i,
2.0791282931864608e-05-8.0155532875691993e-04_i,  1.6109759902072085e-05-6.5490207112794963e-04_i,  1.2697150998941531e-05-5.4032687386873501e-04_i,
1.0164461466073693e-05-4.4933843358934203e-04_i,  8.2553971730379289e-06-3.7598718121875391e-04_i,  6.7971016823684810e-06-3.1601677500735052e-04_i,
5.6705641805752417e-06-2.6632407431402202e-04_i,  4.7923455438381082e-06-2.2460680165160233e-04_i,  4.1030125605971864e-06-1.8912844120970570e-04_i,
3.5596555788576772e-06-1.5855810169386779e-04_i,  3.1309528985487556e-06-1.3185932140828260e-04_i,  2.7938595217555937e-06-1.0821141709320761e-04_i,
2.5313540562891222e-06-8.6952814720991575e-05_i,  2.3308893211072476e-06-6.7539416121031870e-05_i,  2.1833211413576498e-06-4.9513336333531499e-05_i,
2.0821700454712322e-06-3.2478806775287916e-05_i,  2.0231218940125473e-06-1.6082982452481076e-05_i,  2.0037071681855423e-06+0.0000000000000000e+00_i,
2.0231218940125473e-06+1.6082982452481076e-05_i,  2.0821700454712322e-06+3.2478806775287916e-05_i,  2.1833211413576498e-06+4.9513336333531499e-05_i,
2.3308893211072476e-06+6.7539416121031870e-05_i,  2.5313540562891222e-06+8.6952814720991575e-05_i,  2.7938595217555937e-06+1.0821141709320761e-04_i,
3.1309528985487556e-06+1.3185932140828260e-04_i,  3.5596555788576772e-06+1.5855810169386779e-04_i,  4.1030125605971864e-06+1.8912844120970570e-04_i,
4.7923455438381082e-06+2.2460680165160233e-04_i,  5.6705641805752417e-06+2.6632407431402202e-04_i,  6.7971016823684810e-06+3.1601677500735052e-04_i,
8.2553971730379289e-06+3.7598718121875391e-04_i,  1.0164461466073693e-05+4.4933843358934203e-04_i,  1.2697150998941531e-05+5.4032687386873501e-04_i,
1.6109759902072085e-05+6.5490207112794963e-04_i,  2.0791282931864608e-05+8.0155532875691993e-04_i,  2.7348021626939778e-05+9.9269043951715277e-04_i,
3.6754123681629128e-05+1.2469087395869357e-03_i,  5.0630509107736296e-05+1.5929572749680144e-03_i,  7.1786461789753470e-05+2.0768385964179889e-03_i,
1.0533052211046385e-04+2.7752481032899167e-03_i,  1.6110255407045292e-04+3.8224692072726554e-03_i,  2.5943526056193306e-04+5.4680651177607849e-03_i,
4.4620429981049366e-04+8.2116863575322878e-03_i,  8.3736931933972830e-04+1.3154307407879831e-02_i,  1.7746578835483927e-03+2.3055409675972874e-02_i,
4.5147976089274624e-03+4.6231550002149430e-02_i,  1.5664952563471703e-02+1.1628334067606182e-01_i,  1.0770498750413532e-01+4.7664912254487979e-01_i,
-1.1331925343028967e+01-2.1082662731139310e+00_i},
{ 2.9267425817005492e+01+0.0000000000000000e+00_i, -1.4805873973945676e+01+1.1409854850790166e+00_i,  1.4072334780537618e-01-2.5796064625939591e-01_i,
2.0467246865977392e-02-6.2932090485855519e-02_i,  5.8988673497371450e-03-2.5020334564877351e-02_i,  2.3187022659700999e-03-1.2477497803909592e-02_i,
1.0940757405736338e-03-7.1190598692685899e-03_i,  5.8299401290090168e-04-4.4441326323201033e-03_i,  3.3896850323407917e-04-2.9592955170785369e-03_i,
2.1049063069564550e-04-2.0687054279049490e-03_i,  1.3762095926060332e-04-1.5019534504383363e-03_i,  9.3793532353176516e-05-1.1239769490149158e-03_i,
6.6151948092898864e-05-8.6210226491284709e-04_i,  4.8021576809726373e-05-6.7482214710280070e-04_i,  3.5731912219998108e-05-5.3724019451875660e-04_i,
2.7165120271399612e-05-4.3379861797487873e-04_i,  2.1048415661084618e-05-3.5443044687255565e-04_i,  1.6589627255980677e-05-2.9242279694197126e-04_i,
1.3280508908773692e-05-2.4318020790417560e-04_i,  1.0786196206521124e-05-2.0348279618045279e-04_i,  8.8808413266300637e-06-1.7102704621416707e-04_i,
7.4089491477558143e-06-1.4413355039362950e-04_i,  6.2615012029171952e-06-1.2155632512006302e-04_i,  5.3608442566221483e-06-1.0235557480938427e-04_i,
4.6509141472897441e-06-8.5811026284261099e-05_i,  4.0907871023653777e-06-7.1361750514068248e-05_i,  3.6503533803627398e-06-5.8563596922155126e-05_i,
3.3073734611846397e-06-4.7058524223632381e-05_i,  3.0454536620204073e-06-3.6552068611512764e-05_i,  2.8526465438265258e-06-2.6796424528928442e-05_i,
2.7204862683427589e-06-1.7577403563395766e-05_i,  2.6433361393074506e-06-8.7040473819577358e-06_i,  2.6179695815642390e-06+0.0000000000000000e+00_i,
2.6433361393074506e-06+8.7040473819577358e-06_i,  2.7204862683427589e-06+1.7577403563395766e-05_i,  2.8526465438265258e-06+2.6796424528928442e-05_i,
3.0454536620204073e-06+3.6552068611512764e-05_i,  3.3073734611846397e-06+4.7058524223632381e-05_i,  3.6503533803627398e-06+5.8563596922155126e-05_i,
4.0907871023653777e-06+7.1361750514068248e-05_i,  4.6509141472897441e-06+8.5811026284261099e-05_i,  5.3608442566221483e-06+1.0235557480938427e-04_i,
6.2615012029171952e-06+1.2155632512006302e-04_i,  7.4089491477558143e-06+1.4413355039362950e-04_i,  8.8808413266300637e-06+1.7102704621416707e-04_i,
1.0786196206521124e-05+2.0348279618045279e-04_i,  1.3280508908773692e-05+2.4318020790417560e-04_i,  1.6589627255980677e-05+2.9242279694197126e-04_i,
2.1048415661084618e-05+3.5443044687255565e-04_i,  2.7165120271399612e-05+4.3379861797487873e-04_i,  3.5731912219998108e-05+5.3724019451875660e-04_i,
4.8021576809726373e-05+6.7482214710280070e-04_i,  6.6151948092898864e-05+8.6210226491284709e-04_i,  9.3793532353176516e-05+1.1239769490149158e-03_i,
1.3762095926060332e-04+1.5019534504383363e-03_i,  2.1049063069564550e-04+2.0687054279049490e-03_i,  3.3896850323407917e-04+2.9592955170785369e-03_i,
5.8299401290090168e-04+4.4441326323201033e-03_i,  1.0940757405736338e-03+7.1190598692685899e-03_i,  2.3187022659700999e-03+1.2477497803909592e-02_i,
5.8988673497371450e-03+2.5020334564877351e-02_i,  2.0467246865977392e-02+6.2932090485855519e-02_i,  1.4072334780537618e-01+2.5796064625939591e-01_i,
-1.4805873973945676e+01-1.1409854850790166e+00_i},
{ 3.1678833426962989e+01+0.0000000000000000e+00_i, -1.6025762507910954e+01+2.2204460492503131e-16_i,  1.5231785406357318e-01-2.8408612612991689e-16_i,
2.2153588369192963e-02+2.2551405187698492e-16_i,  6.3848880099149656e-03+6.6613381477509392e-16_i,  2.5097452474863542e-03-2.0252218955671419e-16_i,
1.1842190481263270e-03+6.1320357233230723e-17_i,  6.3102817238083807e-04+6.5134800764782286e-16_i,  3.6689686404442901e-04-6.2803698347351007e-16_i,
2.2783341689974941e-04+2.8566016739561828e-16_i,  1.4895985290057080e-04+4.0166785173596984e-16_i,  1.0152138785749046e-04-3.1983964088322381e-17_i,
7.1602352649819756e-05-6.6613381477509392e-16_i,  5.1978180184440118e-05-3.1658703436576729e-17_i,  3.8675943088426964e-05+6.7786835517665210e-16_i,
2.9403314301235106e-05-3.3306690738754696e-16_i,  2.2782640940022247e-05-8.8817841970012523e-16_i,  1.7956483146064350e-05+3.3306690738754696e-16_i,
1.4374719259207039e-05-1.1734540401558172e-17_i,  1.1674894644607674e-05-2.7321894746634712e-17_i,  9.6125533845410876e-06+6.6613381477509392e-16_i,
8.0193887697388398e-06+1.7466496998741476e-16_i,  6.7774000643323223e-06+1.5322682014211133e-16_i,  5.8025360100458337e-06-2.2863113312288075e-16_i,
5.0341131978783775e-06-6.2803698347351007e-16_i,  4.4278360525824064e-06-3.0703250272073879e-16_i,  3.9511140268818799e-06+3.1084559099957474e-16_i,
3.5798752378729277e-06+4.1267445190229513e-18_i,  3.2963752899153181e-06-6.6613381477509392e-16_i,  3.0876823684782700e-06+2.2985086056692694e-16_i,
2.9446331176186114e-06+4.1245566335220061e-16_i,  2.8611264184519314e-06+4.4408920985006262e-16_i,  2.8336698534303650e-06+0.0000000000000000e+00_i,
2.8611264184519314e-06-4.4408920985006262e-16_i,  2.9446331176186114e-06-4.1245566335220061e-16_i,  3.0876823684782700e-06-2.2985086056692694e-16_i,
3.2963752899153181e-06+6.6613381477509392e-16_i,  3.5798752378729277e-06-4.1267445190229513e-18_i,  3.9511140268818799e-06-3.1084559099957474e-16_i,
4.4278360525824064e-06+3.0703250272073879e-16_i,  5.0341131978783775e-06+6.2803698347351007e-16_i,  5.8025360100458337e-06+2.2863113312288075e-16_i,
6.7774000643323223e-06-1.5322682014211133e-16_i,  8.0193887697388398e-06-1.7466496998741476e-16_i,  9.6125533845410876e-06-6.6613381477509392e-16_i,
1.1674894644607674e-05+2.7321894746634712e-17_i,  1.4374719259207039e-05+1.1734540401558172e-17_i,  1.7956483146064350e-05-3.3306690738754696e-16_i,
2.2782640940022247e-05+8.8817841970012523e-16_i,  2.9403314301235106e-05+3.3306690738754696e-16_i,  3.8675943088426964e-05-6.7786835517665210e-16_i,
5.1978180184440118e-05+3.1658703436576729e-17_i,  7.1602352649819756e-05+6.6613381477509392e-16_i,  1.0152138785749046e-04+3.1983964088322381e-17_i,
1.4895985290057080e-04-4.0166785173596984e-16_i,  2.2783341689974941e-04-2.8566016739561828e-16_i,  3.6689686404442901e-04+6.2803698347351007e-16_i,
6.3102817238083807e-04-6.5134800764782286e-16_i,  1.1842190481263270e-03-6.1320357233230723e-17_i,  2.5097452474863542e-03+2.0252218955671419e-16_i,
6.3848880099149656e-03-6.6613381477509392e-16_i,  2.2153588369192963e-02-2.2551405187698492e-16_i,  1.5231785406357318e-01+2.8408612612991689e-16_i,
-1.6025762507910954e+01-2.2204460492503131e-16_i},
{ 2.9267425817005496e+01+0.0000000000000000e+00_i, -1.4805873973945678e+01-1.1409854850790158e+00_i,  1.4072334780537615e-01+2.5796064625939552e-01_i,
2.0467246865977479e-02+6.2932090485856046e-02_i,  5.8988673497371077e-03+2.5020334564877886e-02_i,  2.3187022659700600e-03+1.2477497803909495e-02_i,
1.0940757405738257e-03+7.1190598692683661e-03_i,  5.8299401290154472e-04+4.4441326323207643e-03_i,  3.3896850323376514e-04+2.9592955170758945e-03_i,
2.1049063069564922e-04+2.0687054279049564e-03_i,  1.3762095926041684e-04+1.5019534504389766e-03_i,  9.3793532354104810e-05+1.1239769490139359e-03_i,
6.6151948093346423e-05+8.6210226491221782e-04_i,  4.8021576809028146e-05+6.7482214710290044e-04_i,  3.5731912220289541e-05+5.3724019451993621e-04_i,
2.7165120271344101e-05+4.3379861797393504e-04_i,  2.1048415661084618e-05+3.5443044687077929e-04_i,  1.6589627255148010e-05+2.9242279694202677e-04_i,
1.3280508909009614e-05+2.4318020790441153e-04_i,  1.0786196206697199e-05+2.0348279618092723e-04_i,  8.8808413264964900e-06+1.7102704621455608e-04_i,
7.4089491480396584e-06+1.4413355039402339e-04_i,  6.2615012026127513e-06+1.2155632512097646e-04_i,  5.3608442563305250e-06+1.0235557480893281e-04_i,
4.6509141476037562e-06+8.5811026282506860e-05_i,  4.0907871020101389e-06+7.1361750513382165e-05_i,  3.6503533803563431e-06+5.8563596922871133e-05_i,
3.3073734609001450e-06+4.7058524223576870e-05_i,  3.0454536617437189e-06+3.6552068610399072e-05_i,  2.8526465433772324e-06+2.6796424529379470e-05_i,
2.7204862681345920e-06+1.7577403564533745e-05_i,  2.6433361401956290e-06+8.7040473818467134e-06_i,  2.6179695797878821e-06+0.0000000000000000e+00_i,
2.6433361401956290e-06-8.7040473818467134e-06_i,  2.7204862681345920e-06-1.7577403564533745e-05_i,  2.8526465433772324e-06-2.6796424529379470e-05_i,
3.0454536617437189e-06-3.6552068610399072e-05_i,  3.3073734609001450e-06-4.7058524223576870e-05_i,  3.6503533803563431e-06-5.8563596922871133e-05_i,
4.0907871020101389e-06-7.1361750513382165e-05_i,  4.6509141476037562e-06-8.5811026282506860e-05_i,  5.3608442563305250e-06-1.0235557480893281e-04_i,
6.2615012026127513e-06-1.2155632512097646e-04_i,  7.4089491480396584e-06-1.4413355039402339e-04_i,  8.8808413264964900e-06-1.7102704621455608e-04_i,
1.0786196206697199e-05-2.0348279618092723e-04_i,  1.3280508909009614e-05-2.4318020790441153e-04_i,  1.6589627255148010e-05-2.9242279694202677e-04_i,
2.1048415661084618e-05-3.5443044687077929e-04_i,  2.7165120271344101e-05-4.3379861797393504e-04_i,  3.5731912220289541e-05-5.3724019451993621e-04_i,
4.8021576809028146e-05-6.7482214710290044e-04_i,  6.6151948093346423e-05-8.6210226491221782e-04_i,  9.3793532354104810e-05-1.1239769490139359e-03_i,
1.3762095926041684e-04-1.5019534504389766e-03_i,  2.1049063069564922e-04-2.0687054279049564e-03_i,  3.3896850323376514e-04-2.9592955170758945e-03_i,
5.8299401290154472e-04-4.4441326323207643e-03_i,  1.0940757405738257e-03-7.1190598692683661e-03_i,  2.3187022659700600e-03-1.2477497803909495e-02_i,
5.8988673497371077e-03-2.5020334564877886e-02_i,  2.0467246865977479e-02-6.2932090485856046e-02_i,  1.4072334780537615e-01-2.5796064625939552e-01_i,
-1.4805873973945678e+01+1.1409854850790158e+00_i},
{ 2.2400317936284612e+01+0.0000000000000000e+00_i, -1.1331925343028967e+01-2.1082662731139310e+00_i,  1.0770498750413518e-01+4.7664912254487979e-01_i,
1.5664952563471730e-02+1.1628334067606227e-01_i,  4.5147976089277321e-03+4.6231550002150380e-02_i,  1.7746578835482810e-03+2.3055409675973595e-02_i,
8.3736931934044268e-04+1.3154307407878079e-02_i,  4.4620429981040942e-04+8.2116863575329921e-03_i,  2.5943526056161907e-04+5.4680651177598429e-03_i,
1.6110255407056286e-04+3.8224692072720977e-03_i,  1.0533052211059461e-04+2.7752481032902224e-03_i,  7.1786461789775154e-05+2.0768385964173757e-03_i,
5.0630509108200335e-05+1.5929572749688778e-03_i,  3.6754123681943113e-05+1.2469087395868047e-03_i,  2.7348021626644875e-05+9.9269043951805830e-04_i,
2.0791282932419719e-05+8.0155532875714197e-04_i,  1.6109759902072085e-05+6.5490207112794963e-04_i,  1.2697150999274598e-05+5.4032687386940115e-04_i,
1.0164461465667768e-05+4.4933843358902631e-04_i,  8.2553971730535414e-06+3.7598718121869233e-04_i,  6.7971016826625166e-06+3.1601677500771785e-04_i,
5.6705641803176353e-06+2.6632407431400229e-04_i,  4.7923455443973396e-06+2.2460680165234349e-04_i,  4.1030125609135566e-06+1.8912844120906732e-04_i,
3.5596555791716892e-06+1.5855810169292583e-04_i,  3.1309528982066898e-06+1.3185932140712901e-04_i,  2.7938595223635058e-06+1.0821141709391798e-04_i,
2.5313540563036505e-06+8.6952814721414848e-05_i,  2.3308893209671687e-06+6.7539416120594720e-05_i,  2.1833211413298942e-06+4.9513336333163738e-05_i,
2.0821700451936764e-06+3.2478806775260161e-05_i,  2.0231218931243689e-06+1.6082982452703121e-05_i,  2.0037071699618991e-06+0.0000000000000000e+00_i,
2.0231218931243689e-06-1.6082982452703121e-05_i,  2.0821700451936764e-06-3.2478806775260161e-05_i,  2.1833211413298942e-06-4.9513336333163738e-05_i,
2.3308893209671687e-06-6.7539416120594720e-05_i,  2.5313540563036505e-06-8.6952814721414848e-05_i,  2.7938595223635058e-06-1.0821141709391798e-04_i,
3.1309528982066898e-06-1.3185932140712901e-04_i,  3.5596555791716892e-06-1.5855810169292583e-04_i,  4.1030125609135566e-06-1.8912844120906732e-04_i,
4.7923455443973396e-06-2.2460680165234349e-04_i,  5.6705641803176353e-06-2.6632407431400229e-04_i,  6.7971016826625166e-06-3.1601677500771785e-04_i,
8.2553971730535414e-06-3.7598718121869233e-04_i,  1.0164461465667768e-05-4.4933843358902631e-04_i,  1.2697150999274598e-05-5.4032687386940115e-04_i,
1.6109759902072085e-05-6.5490207112794963e-04_i,  2.0791282932419719e-05-8.0155532875714197e-04_i,  2.7348021626644875e-05-9.9269043951805830e-04_i,
3.6754123681943113e-05-1.2469087395868047e-03_i,  5.0630509108200335e-05-1.5929572749688778e-03_i,  7.1786461789775154e-05-2.0768385964173757e-03_i,
1.0533052211059461e-04-2.7752481032902224e-03_i,  1.6110255407056286e-04-3.8224692072720977e-03_i,  2.5943526056161907e-04-5.4680651177598429e-03_i,
4.4620429981040942e-04-8.2116863575329921e-03_i,  8.3736931934044268e-04-1.3154307407878079e-02_i,  1.7746578835482810e-03-2.3055409675973595e-02_i,
4.5147976089277321e-03-4.6231550002150380e-02_i,  1.5664952563471730e-02-1.1628334067606227e-01_i,  1.0770498750413518e-01-4.7664912254487979e-01_i,
-1.1331925343028967e+01+2.1082662731139310e+00_i},
{ 1.2122964709152141e+01+0.0000000000000000e+00_i, -6.1327938027951348e+00-2.7545826325486056e+00_i,  5.8289519203532855e-02+6.2277209075796103e-01_i,
8.4778112363261184e-03+1.5193150635944469e-01_i,  2.4433908589010581e-03+6.0404431041639353e-02_i,  9.6043792567016961e-04+3.0123344422679936e-02_i,
4.5318101000912372e-04+1.7186930887732835e-02_i,  2.4148402692616489e-04+1.0729085273932694e-02_i,  1.4040535125669769e-04+7.1443713723973439e-03_i,
8.7188073986422111e-05+4.9942967006022399e-03_i,  5.7004467792445878e-05+3.6260363901022578e-03_i,  3.8850553163986400e-05+2.7135203941044063e-03_i,
2.7401034077760393e-05+2.0812989801047765e-03_i,  1.9891188400719684e-05+1.6291647797244350e-03_i,  1.4800642651024987e-05+1.2970125638607118e-03_i,
1.1252161240316028e-05+1.0472825068525982e-03_i,  8.7185392336763812e-06+8.5567079175508098e-04_i,  6.8716486036901614e-06+7.0597108232539618e-04_i,
5.5009669052141774e-06+5.8708895602304148e-04_i,  4.4677887552080353e-06+4.9125092624890433e-04_i,  3.6785649228194758e-06+4.1289581450431375e-04_i,
3.0688872200506333e-06+3.4796917215399641e-04_i,  2.5935987191260963e-06+2.9346292869785514e-04_i,  2.2205343967175674e-06+2.4710821688879144e-04_i,
1.9264717175976630e-06+2.0716614345353416e-04_i,  1.6944594983524545e-06+1.7228250592541718e-04_i,  1.5120258775053880e-06+1.4138502995156808e-04_i,
1.3699589433320558e-06+1.1360932740651225e-04_i,  1.2614682105300395e-06+8.8244499772249102e-05_i,  1.1816048866523238e-06+6.4692291521548095e-05_i,
1.1268623085931850e-06+4.2435606075696963e-05_i,  1.0949056785669598e-06+2.1013429237104475e-05_i,  1.0843985052133576e-06+0.0000000000000000e+00_i,
1.0949056785669598e-06-2.1013429237104475e-05_i,  1.1268623085931850e-06-4.2435606075696963e-05_i,  1.1816048866523238e-06-6.4692291521548095e-05_i,
1.2614682105300395e-06-8.8244499772249102e-05_i,  1.3699589433320558e-06-1.1360932740651225e-04_i,  1.5120258775053880e-06-1.4138502995156808e-04_i,
1.6944594983524545e-06-1.7228250592541718e-04_i,  1.9264717175976630e-06-2.0716614345353416e-04_i,  2.2205343967175674e-06-2.4710821688879144e-04_i,
2.5935987191260963e-06-2.9346292869785514e-04_i,  3.0688872200506333e-06-3.4796917215399641e-04_i,  3.6785649228194758e-06-4.1289581450431375e-04_i,
4.4677887552080353e-06-4.9125092624890433e-04_i,  5.5009669052141774e-06-5.8708895602304148e-04_i,  6.8716486036901614e-06-7.0597108232539618e-04_i,
8.7185392336763812e-06-8.5567079175508098e-04_i,  1.1252161240316028e-05-1.0472825068525982e-03_i,  1.4800642651024987e-05-1.2970125638607118e-03_i,
1.9891188400719684e-05-1.6291647797244350e-03_i,  2.7401034077760393e-05-2.0812989801047765e-03_i,  3.8850553163986400e-05-2.7135203941044063e-03_i,
5.7004467792445878e-05-3.6260363901022578e-03_i,  8.7188073986422111e-05-4.9942967006022399e-03_i,  1.4040535125669769e-04-7.1443713723973439e-03_i,
2.4148402692616489e-04-1.0729085273932694e-02_i,  4.5318101000912372e-04-1.7186930887732835e-02_i,  9.6043792567016961e-04-3.0123344422679936e-02_i,
2.4433908589010581e-03-6.0404431041639353e-02_i,  8.4778112363261184e-03-1.5193150635944469e-01_i,  5.8289519203532855e-02-6.2277209075796103e-01_i,
-6.1327938027951348e+00+2.7545826325486056e+00_i},
{ 3.8694673880246385e-15+0.0000000000000000e+00_i, -1.9612715644501507e-15-2.9815387565315010e+00_i, -4.0004063837997297e-17+6.7408365359620415e-01_i,
-1.5542442595829466e-16+1.6444947746213784e-01_i,  5.9147273041581886e-17+6.5381285022569971e-02_i,  1.8761714141611895e-16+3.2605273049829811e-02_i,
8.6090745233626431e-17+1.8602999939846258e-02_i,  9.6544158927336550e-17+1.1613078216777076e-02_i,  4.4874103483236926e-17+7.7330118494754723e-03_i,
-8.2566039967268782e-17+5.4057877946776939e-03_i, -1.9076865497467505e-17+3.9247935066230614e-03_i,  1.4728482059965818e-16+2.9370933099133818e-03_i,
-5.7946561586852592e-17+2.2527817825419320e-03_i, -1.6756749029013231e-16+1.7633952505644399e-03_i,  1.5428496547140509e-16+1.4038762828040008e-03_i,
2.4529669699234776e-16+1.1335704169204014e-03_i, -8.2232690250527892e-17+9.2617139101460605e-04_i, -1.9879251285771486e-16+7.6413759314040985e-04_i,
-1.2327079068488404e-16+6.3546050687710021e-04_i, -1.1205633905887448e-16+5.3172617095844678e-04_i, -2.4354103555947654e-18+4.4691520915252341e-04_i,
3.9731965089096143e-17+3.7663911788121023e-04_i,  6.7659308301372856e-17+3.1764198509756387e-04_i,  2.3228627092252171e-16+2.6746800658876413e-04_i,
8.9032953883718111e-17+2.2423501783898868e-04_i, -2.0876717284458150e-16+1.8647724066010209e-04_i, -1.0516251799281655e-16+1.5303405365736829e-04_i,
2.3685772936310677e-17+1.2296984986556736e-04_i, -8.5069425215135903e-18+9.5515158272593825e-05_i, -5.4810464351639844e-17+7.0022431761393444e-05_i,
-4.0004063837997297e-17+4.5931969031021147e-05_i,  9.3743042595019938e-18+2.2744771908111971e-05_i,  4.5034835483122657e-18+0.0000000000000000e+00_i,
9.3743042595019938e-18-2.2744771908111971e-05_i, -4.0004063837997297e-17-4.5931969031021147e-05_i, -5.4810464351639844e-17-7.0022431761393444e-05_i,
-8.5069425215135903e-18-9.5515158272593825e-05_i,  2.3685772936310677e-17-1.2296984986556736e-04_i, -1.0516251799281655e-16-1.5303405365736829e-04_i,
-2.0876717284458150e-16-1.8647724066010209e-04_i,  8.9032953883718111e-17-2.2423501783898868e-04_i,  2.3228627092252171e-16-2.6746800658876413e-04_i,
6.7659308301372856e-17-3.1764198509756387e-04_i,  3.9731965089096143e-17-3.7663911788121023e-04_i, -2.4354103555947654e-18-4.4691520915252341e-04_i,
-1.1205633905887448e-16-5.3172617095844678e-04_i, -1.2327079068488404e-16-6.3546050687710021e-04_i, -1.9879251285771486e-16-7.6413759314040985e-04_i,
-8.2232690250527892e-17-9.2617139101460605e-04_i,  2.4529669699234776e-16-1.1335704169204014e-03_i,  1.5428496547140509e-16-1.4038762828040008e-03_i,
-1.6756749029013231e-16-1.7633952505644399e-03_i, -5.7946561586852592e-17-2.2527817825419320e-03_i,  1.4728482059965818e-16-2.9370933099133818e-03_i,
-1.9076865497467505e-17-3.9247935066230614e-03_i, -8.2566039967268782e-17-5.4057877946776939e-03_i,  4.4874103483236926e-17-7.7330118494754723e-03_i,
9.6544158927336550e-17-1.1613078216777076e-02_i,  8.6090745233626431e-17-1.8602999939846258e-02_i,  1.8761714141611895e-16-3.2605273049829811e-02_i,
5.9147273041581886e-17-6.5381285022569971e-02_i, -1.5542442595829466e-16-1.6444947746213784e-01_i, -4.0004063837997297e-17-6.7408365359620415e-01_i,
-1.9612715644501507e-15+2.9815387565315010e+00_i},
{-1.2122964709152132e+01-0.0000000000000000e+00_i,  6.1327938027951303e+00-2.7545826325486060e+00_i, -5.8289519203532925e-02+6.2277209075796125e-01_i,
-8.4778112363261601e-03+1.5193150635944500e-01_i, -2.4433908589009982e-03+6.0404431041638854e-02_i, -9.6043792566995678e-04+3.0123344422680144e-02_i,
-4.5318101000893594e-04+1.7186930887732509e-02_i, -2.4148402692572492e-04+1.0729085273932504e-02_i, -1.4040535125685471e-04+7.1443713723978158e-03_i,
-8.7188073986890053e-05+4.9942967006022781e-03_i, -5.7004467792525675e-05+3.6260363901020366e-03_i, -3.8850553163740070e-05+2.7135203941043694e-03_i,
-2.7401034077614676e-05+2.0812989801051868e-03_i, -1.9891188401306020e-05+1.6291647797242615e-03_i, -1.4800642650778656e-05+1.2970125638602573e-03_i,
-1.1252161239760916e-05+1.0472825068524871e-03_i, -8.7185392336763812e-06+8.5567079175641325e-04_i, -6.8716486038011837e-06+7.0597108232506312e-04_i,
-5.5009669055229582e-06+5.8708895602297556e-04_i, -4.4677887550727269e-06+4.9125092624874474e-04_i, -3.6785649229651926e-06+4.1289581450361379e-04_i,
-3.0688872199378762e-06+3.4796917215388322e-04_i, -2.5935987189838489e-06+2.9346292869774325e-04_i, -2.2205343966598878e-06+2.4710821688879708e-04_i,
-1.9264717174406569e-06+2.0716614345400557e-04_i, -1.6944594983821617e-06+1.7228250592563662e-04_i, -1.5120258775994967e-06+1.4138502995112573e-04_i,
-1.3699589432378386e-06+1.1360932740631102e-04_i, -1.2614682105898875e-06+8.8244499772863194e-05_i, -1.1816048868257961e-06+6.4692291521312173e-05_i,
-1.1268623086140017e-06+4.2435606075696963e-05_i, -1.0949056781228705e-06+2.1013429237104475e-05_i, -1.0843985061015360e-06-0.0000000000000000e+00_i,
-1.0949056781228705e-06-2.1013429237104475e-05_i, -1.1268623086140017e-06-4.2435606075696963e-05_i, -1.1816048868257961e-06-6.4692291521312173e-05_i,
-1.2614682105898875e-06-8.8244499772863194e-05_i, -1.3699589432378386e-06-1.1360932740631102e-04_i, -1.5120258775994967e-06-1.4138502995112573e-04_i,
-1.6944594983821617e-06-1.7228250592563662e-04_i, -1.9264717174406569e-06-2.0716614345400557e-04_i, -2.2205343966598878e-06-2.4710821688879708e-04_i,
-2.5935987189838489e-06-2.9346292869774325e-04_i, -3.0688872199378762e-06-3.4796917215388322e-04_i, -3.6785649229651926e-06-4.1289581450361379e-04_i,
-4.4677887550727269e-06-4.9125092624874474e-04_i, -5.5009669055229582e-06-5.8708895602297556e-04_i, -6.8716486038011837e-06-7.0597108232506312e-04_i,
-8.7185392336763812e-06-8.5567079175641325e-04_i, -1.1252161239760916e-05-1.0472825068524871e-03_i, -1.4800642650778656e-05-1.2970125638602573e-03_i,
-1.9891188401306020e-05-1.6291647797242615e-03_i, -2.7401034077614676e-05-2.0812989801051868e-03_i, -3.8850553163740070e-05-2.7135203941043694e-03_i,
-5.7004467792525675e-05-3.6260363901020366e-03_i, -8.7188073986890053e-05-4.9942967006022781e-03_i, -1.4040535125685471e-04-7.1443713723978158e-03_i,
-2.4148402692572492e-04-1.0729085273932504e-02_i, -4.5318101000893594e-04-1.7186930887732509e-02_i, -9.6043792566995678e-04-3.0123344422680144e-02_i,
-2.4433908589009982e-03-6.0404431041638854e-02_i, -8.4778112363261601e-03-1.5193150635944500e-01_i, -5.8289519203532925e-02-6.2277209075796125e-01_i,
6.1327938027951303e+00+2.7545826325486060e+00_i},
{-2.2400317936284605e+01-0.0000000000000000e+00_i,  1.1331925343028963e+01-2.1082662731139310e+00_i, -1.0770498750413526e-01+4.7664912254487929e-01_i,
-1.5664952563471952e-02+1.1628334067606239e-01_i, -4.5147976089274545e-03+4.6231550002148805e-02_i, -1.7746578835468626e-03+2.3055409675973644e-02_i,
-8.3736931934050469e-04+1.3154307407879075e-02_i, -4.4620429981092388e-04+8.2116863575327735e-03_i, -2.5943526056193306e-04+5.4680651177601569e-03_i,
-1.6110255407026514e-04+3.8224692072727803e-03_i, -1.0533052211078456e-04+2.7752481032898937e-03_i, -7.1786461789538365e-05+2.0768385964176350e-03_i,
-5.0630509107437056e-05+1.5929572749691281e-03_i, -3.6754123681889336e-05+1.2469087395864014e-03_i, -2.7348021627002228e-05+9.9269043951669828e-04_i,
-2.0791282931309496e-05+8.0155532875747504e-04_i, -1.6109759902072085e-05+6.5490207112883780e-04_i, -1.2697150998608464e-05+5.4032687386906808e-04_i,
-1.0164461466219410e-05+4.4933843358935938e-04_i, -8.2553971731801762e-06+3.7598718121859432e-04_i, -6.7971016821698552e-06+3.1601677500681623e-04_i,
-5.6705641805353431e-06+2.6632407431399253e-04_i, -4.7923455442576944e-06+2.2460680165161230e-04_i, -4.1030125609246154e-06+1.8912844120932753e-04_i,
-3.5596555788576772e-06+1.5855810169323982e-04_i, -3.1309528988670773e-06+1.3185932140848556e-04_i, -2.7938595223830215e-06+1.0821141709248337e-04_i,
-2.5313540554097258e-06+8.6952814721860672e-05_i, -2.3308893225010979e-06+6.7539416121420448e-05_i, -2.1833211407053937e-06+4.9513336332913938e-05_i,
-2.0821700449716318e-06+3.2478806775315672e-05_i, -2.0231218931243689e-06+1.6082982452703121e-05_i, -2.0037071699618991e-06-0.0000000000000000e+00_i,
-2.0231218931243689e-06-1.6082982452703121e-05_i, -2.0821700449716318e-06-3.2478806775315672e-05_i, -2.1833211407053937e-06-4.9513336332913938e-05_i,
-2.3308893225010979e-06-6.7539416121420448e-05_i, -2.5313540554097258e-06-8.6952814721860672e-05_i, -2.7938595223830215e-06-1.0821141709248337e-04_i,
-3.1309528988670773e-06-1.3185932140848556e-04_i, -3.5596555788576772e-06-1.5855810169323982e-04_i, -4.1030125609246154e-06-1.8912844120932753e-04_i,
-4.7923455442576944e-06-2.2460680165161230e-04_i, -5.6705641805353431e-06-2.6632407431399253e-04_i, -6.7971016821698552e-06-3.1601677500681623e-04_i,
-8.2553971731801762e-06-3.7598718121859432e-04_i, -1.0164461466219410e-05-4.4933843358935938e-04_i, -1.2697150998608464e-05-5.4032687386906808e-04_i,
-1.6109759902072085e-05-6.5490207112883780e-04_i, -2.0791282931309496e-05-8.0155532875747504e-04_i, -2.7348021627002228e-05-9.9269043951669828e-04_i,
-3.6754123681889336e-05-1.2469087395864014e-03_i, -5.0630509107437056e-05-1.5929572749691281e-03_i, -7.1786461789538365e-05-2.0768385964176350e-03_i,
-1.0533052211078456e-04-2.7752481032898937e-03_i, -1.6110255407026514e-04-3.8224692072727803e-03_i, -2.5943526056193306e-04-5.4680651177601569e-03_i,
-4.4620429981092388e-04-8.2116863575327735e-03_i, -8.3736931934050469e-04-1.3154307407879075e-02_i, -1.7746578835468626e-03-2.3055409675973644e-02_i,
-4.5147976089274545e-03-4.6231550002148805e-02_i, -1.5664952563471952e-02-1.1628334067606239e-01_i, -1.0770498750413526e-01-4.7664912254487929e-01_i,
1.1331925343028963e+01+2.1082662731139310e+00_i},
{-2.9267425817005488e+01-0.0000000000000000e+00_i,  1.4805873973945676e+01-1.1409854850790171e+00_i, -1.4072334780537604e-01+2.5796064625939608e-01_i,
-2.0467246865977829e-02+6.2932090485855643e-02_i, -5.8988673497367824e-03+2.5020334564876606e-02_i, -2.3187022659705439e-03+1.2477497803909415e-02_i,
-1.0940757405736312e-03+7.1190598692702067e-03_i, -5.8299401290034169e-04+4.4441326323193869e-03_i, -3.3896850323287696e-04+2.9592955170775950e-03_i,
-2.1049063069535740e-04+2.0687054279049945e-03_i, -1.3762095926053816e-04+1.5019534504382460e-03_i, -9.3793532353855227e-05+1.1239769490143563e-03_i,
-6.6151948092970855e-05+8.6210226491313896e-04_i, -4.8021576809932805e-05+6.7482214710196282e-04_i, -3.5731912219560957e-05+5.3724019451896476e-04_i,
-2.7165120271233079e-05+4.3379861797449015e-04_i, -2.1048415659308262e-05+3.5443044687255565e-04_i, -1.6589627255259032e-05+2.9242279694147166e-04_i,
-1.3280508908836142e-05+2.4318020790407846e-04_i, -1.0786196206366734e-05+2.0348279618023248e-04_i, -8.8808413268720576e-06+1.7102704621414495e-04_i,
-7.4089491480207933e-06+1.4413355039337450e-04_i, -6.2615012029162195e-06+1.2155632512006378e-04_i, -5.3608442567161215e-06+1.0235557480950007e-04_i,
-4.6509141467155778e-06+8.5811026283318928e-05_i, -4.0907871022312619e-06+7.1361750514088197e-05_i, -3.6503533803065782e-06+5.8563596922030659e-05_i,
-3.3073734611286949e-06+4.7058524223981928e-05_i, -3.0454536620689796e-06+3.6552068610449379e-05_i, -2.8526465437848925e-06+2.6796424530337037e-05_i,
-2.7204862680790809e-06+1.7577403563229232e-05_i, -2.6433361384192722e-06+8.7040473815136465e-06_i, -2.6179695797878821e-06-0.0000000000000000e+00_i,
-2.6433361384192722e-06-8.7040473815136465e-06_i, -2.7204862680790809e-06-1.7577403563229232e-05_i, -2.8526465437848925e-06-2.6796424530337037e-05_i,
-3.0454536620689796e-06-3.6552068610449379e-05_i, -3.3073734611286949e-06-4.7058524223981928e-05_i, -3.6503533803065782e-06-5.8563596922030659e-05_i,
-4.0907871022312619e-06-7.1361750514088197e-05_i, -4.6509141467155778e-06-8.5811026283318928e-05_i, -5.3608442567161215e-06-1.0235557480950007e-04_i,
-6.2615012029162195e-06-1.2155632512006378e-04_i, -7.4089491480207933e-06-1.4413355039337450e-04_i, -8.8808413268720576e-06-1.7102704621414495e-04_i,
-1.0786196206366734e-05-2.0348279618023248e-04_i, -1.3280508908836142e-05-2.4318020790407846e-04_i, -1.6589627255259032e-05-2.9242279694147166e-04_i,
-2.1048415659308262e-05-3.5443044687255565e-04_i, -2.7165120271233079e-05-4.3379861797449015e-04_i, -3.5731912219560957e-05-5.3724019451896476e-04_i,
-4.8021576809932805e-05-6.7482214710196282e-04_i, -6.6151948092970855e-05-8.6210226491313896e-04_i, -9.3793532353855227e-05-1.1239769490143563e-03_i,
-1.3762095926053816e-04-1.5019534504382460e-03_i, -2.1049063069535740e-04-2.0687054279049945e-03_i, -3.3896850323287696e-04-2.9592955170775950e-03_i,
-5.8299401290034169e-04-4.4441326323193869e-03_i, -1.0940757405736312e-03-7.1190598692702067e-03_i, -2.3187022659705439e-03-1.2477497803909415e-02_i,
-5.8988673497367824e-03-2.5020334564876606e-02_i, -2.0467246865977829e-02-6.2932090485855643e-02_i, -1.4072334780537604e-01-2.5796064625939608e-01_i,
1.4805873973945676e+01+1.1409854850790171e+00_i},
{-3.1678833426962992e+01-0.0000000000000000e+00_i,  1.6025762507910954e+01-3.8857805861880479e-16_i, -1.5231785406357318e-01+2.1598984405158893e-16_i,
-2.2153588369192699e-02+7.5677311639488209e-17_i, -6.3848880099146351e-03-3.5908775952719907e-16_i, -2.5097452474867154e-03-5.5938055836673994e-17_i,
-1.1842190481264677e-03+2.2015724615208009e-16_i, -6.3102817238066362e-04-3.3473725635953244e-16_i, -3.6689686404518709e-04+9.4205547521026515e-16_i,
-2.2783341689992388e-04-8.0071718569843520e-17_i, -1.4895985290025083e-04-3.0364758969704197e-16_i, -1.0152138785774704e-04-1.1725646495430730e-16_i,
-7.1602352649996698e-05+6.4922026088432006e-16_i, -5.1978180184076477e-05+1.3097162243624894e-16_i, -3.8675943088425230e-05-6.0109845571844011e-16_i,
-2.9403314301124084e-05+3.8857805861880479e-16_i, -2.2782640940022247e-05+1.7763568394002505e-15_i, -1.7956483146175373e-05-3.8857805861880479e-16_i,
-1.4374719259125507e-05-4.5361341767545861e-16_i, -1.1674894644310386e-05-3.9031278209478160e-17_i, -9.6125533845480265e-06-2.3895815881580518e-16_i,
-8.0193887700191738e-06-9.6331363025337069e-17_i, -6.7774000639203526e-06-3.9609649242510259e-16_i, -5.8025360102202954e-06+2.3042684297105986e-17_i,
-5.0341131980084547e-06+9.4205547521026515e-16_i, -4.4278360524079583e-06-9.5782485675516282e-18_i, -3.9511140271675672e-06-5.0032864004949051e-16_i,
-3.5798752378627362e-06+1.8703165101732755e-16_i, -3.2963752891741575e-06+5.2909066017292616e-16_i, -3.0876823685199034e-06-3.1637019393127019e-16_i,
-2.9446331180071894e-06-2.6456210137893953e-16_i, -2.8611264184519314e-06-4.9960036108132044e-16_i, -2.8336698534303650e-06-0.0000000000000000e+00_i,
-2.8611264184519314e-06+4.9960036108132044e-16_i, -2.9446331180071894e-06+2.6456210137893953e-16_i, -3.0876823685199034e-06+3.1637019393127019e-16_i,
-3.2963752891741575e-06-5.2909066017292616e-16_i, -3.5798752378627362e-06-1.8703165101732755e-16_i, -3.9511140271675672e-06+5.0032864004949051e-16_i,
-4.4278360524079583e-06+9.5782485675516282e-18_i, -5.0341131980084547e-06-9.4205547521026515e-16_i, -5.8025360102202954e-06-2.3042684297105986e-17_i,
-6.7774000639203526e-06+3.9609649242510259e-16_i, -8.0193887700191738e-06+9.6331363025337069e-17_i, -9.6125533845480265e-06+2.3895815881580518e-16_i,
-1.1674894644310386e-05+3.9031278209478160e-17_i, -1.4374719259125507e-05+4.5361341767545861e-16_i, -1.7956483146175373e-05+3.8857805861880479e-16_i,
-2.2782640940022247e-05-1.7763568394002505e-15_i, -2.9403314301124084e-05-3.8857805861880479e-16_i, -3.8675943088425230e-05+6.0109845571844011e-16_i,
-5.1978180184076477e-05-1.3097162243624894e-16_i, -7.1602352649996698e-05-6.4922026088432006e-16_i, -1.0152138785774704e-04+1.1725646495430730e-16_i,
-1.4895985290025083e-04+3.0364758969704197e-16_i, -2.2783341689992388e-04+8.0071718569843520e-17_i, -3.6689686404518709e-04-9.4205547521026515e-16_i,
-6.3102817238066362e-04+3.3473725635953244e-16_i, -1.1842190481264677e-03-2.2015724615208009e-16_i, -2.5097452474867154e-03+5.5938055836673994e-17_i,
-6.3848880099146351e-03+3.5908775952719907e-16_i, -2.2153588369192699e-02-7.5677311639488209e-17_i, -1.5231785406357318e-01-2.1598984405158893e-16_i,
1.6025762507910954e+01+3.8857805861880479e-16_i},
{-2.9267425817005496e+01-0.0000000000000000e+00_i,  1.4805873973945678e+01+1.1409854850790160e+00_i, -1.4072334780537626e-01-2.5796064625939580e-01_i,
-2.0467246865977607e-02-6.2932090485856518e-02_i, -5.8988673497370600e-03-2.5020334564877532e-02_i, -2.3187022659707781e-03-1.2477497803909270e-02_i,
-1.0940757405736453e-03-7.1190598692697608e-03_i, -5.8299401290078188e-04-4.4441326323198527e-03_i, -3.3896850323394912e-04-2.9592955170763386e-03_i,
-2.1049063069585283e-04-2.0687054279048250e-03_i, -1.3762095926046140e-04-1.5019534504385201e-03_i, -9.3793532353968201e-05-1.1239769490146666e-03_i,
-6.6151948093104429e-05-8.6210226491269791e-04_i, -4.8021576809466164e-05-6.7482214710228635e-04_i, -3.5731912220102191e-05-5.3724019452031091e-04_i,
-2.7165120271732679e-05-4.3379861797432362e-04_i, -2.1048415661084618e-05-3.5443044687077929e-04_i, -1.6589627255647610e-05-2.9242279694252638e-04_i,
-1.3280508908738997e-05-2.4318020790470296e-04_i, -1.0786196206593983e-05-2.0348279618027151e-04_i, -8.8808413264244990e-06-1.7102704621516627e-04_i,
-7.4089491475179403e-06-1.4413355039406372e-04_i, -6.2615012027119558e-06-1.2155632512025969e-04_i, -5.3608442566069965e-06-1.0235557480969132e-04_i,
-4.6509141465316429e-06-8.5811026282062771e-05_i, -4.0907871022929530e-06-7.1361750513582525e-05_i, -3.6503533802544280e-06-5.8563596923061085e-05_i,
-3.3073734610627754e-06-4.7058524223548247e-05_i, -3.0454536621054087e-06-3.6552068609285379e-05_i, -2.8526465435767256e-06-2.6796424530704799e-05_i,
-2.7204862677182584e-06-1.7577403563728833e-05_i, -2.6433361401956290e-06-8.7040473821797804e-06_i, -2.6179695797878821e-06-0.0000000000000000e+00_i,
-2.6433361401956290e-06+8.7040473821797804e-06_i, -2.7204862677182584e-06+1.7577403563728833e-05_i, -2.8526465435767256e-06+2.6796424530704799e-05_i,
-3.0454536621054087e-06+3.6552068609285379e-05_i, -3.3073734610627754e-06+4.7058524223548247e-05_i, -3.6503533802544280e-06+5.8563596923061085e-05_i,
-4.0907871022929530e-06+7.1361750513582525e-05_i, -4.6509141465316429e-06+8.5811026282062771e-05_i, -5.3608442566069965e-06+1.0235557480969132e-04_i,
-6.2615012027119558e-06+1.2155632512025969e-04_i, -7.4089491475179403e-06+1.4413355039406372e-04_i, -8.8808413264244990e-06+1.7102704621516627e-04_i,
-1.0786196206593983e-05+2.0348279618027151e-04_i, -1.3280508908738997e-05+2.4318020790470296e-04_i, -1.6589627255647610e-05+2.9242279694252638e-04_i,
-2.1048415661084618e-05+3.5443044687077929e-04_i, -2.7165120271732679e-05+4.3379861797432362e-04_i, -3.5731912220102191e-05+5.3724019452031091e-04_i,
-4.8021576809466164e-05+6.7482214710228635e-04_i, -6.6151948093104429e-05+8.6210226491269791e-04_i, -9.3793532353968201e-05+1.1239769490146666e-03_i,
-1.3762095926046140e-04+1.5019534504385201e-03_i, -2.1049063069585283e-04+2.0687054279048250e-03_i, -3.3896850323394912e-04+2.9592955170763386e-03_i,
-5.8299401290078188e-04+4.4441326323198527e-03_i, -1.0940757405736453e-03+7.1190598692697608e-03_i, -2.3187022659707781e-03+1.2477497803909270e-02_i,
-5.8988673497370600e-03+2.5020334564877532e-02_i, -2.0467246865977607e-02+6.2932090485856518e-02_i, -1.4072334780537626e-01+2.5796064625939580e-01_i,
1.4805873973945678e+01-1.1409854850790160e+00_i},
{-2.2400317936284612e+01-0.0000000000000000e+00_i,  1.1331925343028971e+01+2.1082662731139301e+00_i, -1.0770498750413550e-01-4.7664912254487912e-01_i,
-1.5664952563471585e-02-1.1628334067606189e-01_i, -4.5147976089275309e-03-4.6231550002150221e-02_i, -1.7746578835485002e-03-2.3055409675971809e-02_i,
-8.3736931934026563e-04-1.3154307407881602e-02_i, -4.4620429981020386e-04-8.2116863575317917e-03_i, -2.5943526056237715e-04-5.4680651177597128e-03_i,
-1.6110255407008104e-04-3.8224692072728649e-03_i, -1.0533052211045301e-04-2.7752481032900993e-03_i, -7.1786461789579564e-05-2.0768385964168583e-03_i,
-5.0630509107392821e-05-1.5929572749685010e-03_i, -3.6754123682100973e-05-1.2469087395868307e-03_i, -2.7348021626721203e-05-9.9269043951811728e-04_i,
-2.0791282932641764e-05-8.0155532875680890e-04_i, -1.6109759901183907e-05-6.5490207112617327e-04_i, -1.2697150999052553e-05-5.4032687386795786e-04_i,
-1.0164461465993896e-05-4.4933843359061185e-04_i, -8.2553971728020065e-06-3.7598718121839222e-04_i, -6.7971016822678670e-06-3.1601677500676505e-04_i,
-5.6705641800682688e-06-2.6632407431434512e-04_i, -4.7923455444188068e-06-2.2460680165279712e-04_i, -4.1030125610380230e-06-1.8912844120773376e-04_i,
-3.5596555793017664e-06-1.5855810169368391e-04_i, -3.1309528978814291e-06-1.3185932140715936e-04_i, -2.7938595221610853e-06-1.0821141709377140e-04_i,
-2.5313540561963145e-06-8.6952814721600463e-05_i, -2.3308893205946368e-06-6.7539416121639023e-05_i, -2.1833211410141745e-06-4.9513336330457569e-05_i,
-2.0821700458667491e-06-3.2478806777119784e-05_i, -2.0231218940125473e-06-1.6082982452481076e-05_i, -2.0037071699618991e-06-0.0000000000000000e+00_i,
-2.0231218940125473e-06+1.6082982452481076e-05_i, -2.0821700458667491e-06+3.2478806777119784e-05_i, -2.1833211410141745e-06+4.9513336330457569e-05_i,
-2.3308893205946368e-06+6.7539416121639023e-05_i, -2.5313540561963145e-06+8.6952814721600463e-05_i, -2.7938595221610853e-06+1.0821141709377140e-04_i,
-3.1309528978814291e-06+1.3185932140715936e-04_i, -3.5596555793017664e-06+1.5855810169368391e-04_i, -4.1030125610380230e-06+1.8912844120773376e-04_i,
-4.7923455444188068e-06+2.2460680165279712e-04_i, -5.6705641800682688e-06+2.6632407431434512e-04_i, -6.7971016822678670e-06+3.1601677500676505e-04_i,
-8.2553971728020065e-06+3.7598718121839222e-04_i, -1.0164461465993896e-05+4.4933843359061185e-04_i, -1.2697150999052553e-05+5.4032687386795786e-04_i,
-1.6109759901183907e-05+6.5490207112617327e-04_i, -2.0791282932641764e-05+8.0155532875680890e-04_i, -2.7348021626721203e-05+9.9269043951811728e-04_i,
-3.6754123682100973e-05+1.2469087395868307e-03_i, -5.0630509107392821e-05+1.5929572749685010e-03_i, -7.1786461789579564e-05+2.0768385964168583e-03_i,
-1.0533052211045301e-04+2.7752481032900993e-03_i, -1.6110255407008104e-04+3.8224692072728649e-03_i, -2.5943526056237715e-04+5.4680651177597128e-03_i,
-4.4620429981020386e-04+8.2116863575317917e-03_i, -8.3736931934026563e-04+1.3154307407881602e-02_i, -1.7746578835485002e-03+2.3055409675971809e-02_i,
-4.5147976089275309e-03+4.6231550002150221e-02_i, -1.5664952563471585e-02+1.1628334067606189e-01_i, -1.0770498750413550e-01+4.7664912254487912e-01_i,
1.1331925343028971e+01-2.1082662731139301e+00_i},
{-1.2122964709152143e+01-0.0000000000000000e+00_i,  6.1327938027951365e+00+2.7545826325486065e+00_i, -5.8289519203533320e-02-6.2277209075796147e-01_i,
-8.4778112363256466e-03-1.5193150635944508e-01_i, -2.4433908589010919e-03-6.0404431041639131e-02_i, -9.6043792566986473e-04-3.0123344422678118e-02_i,
-4.5318101000937699e-04-1.7186930887736686e-02_i, -2.4148402692546124e-04-1.0729085273930645e-02_i, -1.4040535125707675e-04-7.1443713723974098e-03_i,
-8.7188073986025510e-05-4.9942967006026684e-03_i, -5.7004467792819494e-05-3.6260363901015175e-03_i, -3.8850553163542745e-05-2.7135203941051609e-03_i,
-2.7401034078144634e-05-2.0812989801046013e-03_i, -1.9891188400473353e-05-1.6291647797240360e-03_i, -1.4800642651685916e-05-1.2970125638610275e-03_i,
-1.1252161239427849e-05-1.0472825068524871e-03_i, -8.7185392336763812e-06-8.5567079175552507e-04_i, -6.8716486028019830e-06-7.0597108232506312e-04_i,
-5.5009669056530625e-06-5.8708895602391231e-04_i, -4.4677887546980266e-06-4.9125092624831106e-04_i, -3.6785649228810585e-06-4.1289581450431202e-04_i,
-3.0688872195861611e-06-3.4796917215435203e-04_i, -2.5935987198431876e-06-2.9346292869691015e-04_i, -2.2205343964449989e-06-2.4710821689017488e-04_i,
-1.9264717176627015e-06-2.0716614345315512e-04_i, -1.6944594979489145e-06-1.7228250592476665e-04_i, -1.5120258780973624e-06-1.4138502995154900e-04_i,
-1.3699589431017713e-06-1.1360932740685052e-04_i, -1.2614682109385669e-06-8.8244499769987023e-05_i, -1.1816048862151735e-06-6.4692291524948153e-05_i,
-1.1268623086382878e-06-4.2435606074420207e-05_i, -1.0949056790110490e-06-2.1013429236882430e-05_i, -1.0843985061015360e-06-0.0000000000000000e+00_i,
-1.0949056790110490e-06+2.1013429236882430e-05_i, -1.1268623086382878e-06+4.2435606074420207e-05_i, -1.1816048862151735e-06+6.4692291524948153e-05_i,
-1.2614682109385669e-06+8.8244499769987023e-05_i, -1.3699589431017713e-06+1.1360932740685052e-04_i, -1.5120258780973624e-06+1.4138502995154900e-04_i,
-1.6944594979489145e-06+1.7228250592476665e-04_i, -1.9264717176627015e-06+2.0716614345315512e-04_i, -2.2205343964449989e-06+2.4710821689017488e-04_i,
-2.5935987198431876e-06+2.9346292869691015e-04_i, -3.0688872195861611e-06+3.4796917215435203e-04_i, -3.6785649228810585e-06+4.1289581450431202e-04_i,
-4.4677887546980266e-06+4.9125092624831106e-04_i, -5.5009669056530625e-06+5.8708895602391231e-04_i, -6.8716486028019830e-06+7.0597108232506312e-04_i,
-8.7185392336763812e-06+8.5567079175552507e-04_i, -1.1252161239427849e-05+1.0472825068524871e-03_i, -1.4800642651685916e-05+1.2970125638610275e-03_i,
-1.9891188400473353e-05+1.6291647797240360e-03_i, -2.7401034078144634e-05+2.0812989801046013e-03_i, -3.8850553163542745e-05+2.7135203941051609e-03_i,
-5.7004467792819494e-05+3.6260363901015175e-03_i, -8.7188073986025510e-05+4.9942967006026684e-03_i, -1.4040535125707675e-04+7.1443713723974098e-03_i,
-2.4148402692546124e-04+1.0729085273930645e-02_i, -4.5318101000937699e-04+1.7186930887736686e-02_i, -9.6043792566986473e-04+3.0123344422678118e-02_i,
-2.4433908589010919e-03+6.0404431041639131e-02_i, -8.4778112363256466e-03+1.5193150635944508e-01_i, -5.8289519203533320e-02+6.2277209075796147e-01_i,
6.1327938027951365e+00-2.7545826325486065e+00_i},
{-1.8561830307012563e+00-0.0000000000000000e+00_i,  2.9938833406609444e-01+1.4907693782657527e+00_i,  7.0345957258712111e-01-3.3704182679810379e-01_i,
-1.4494751578417842e-01-8.2224738731067948e-02_i,  9.6735350049332092e-02-3.2690642511285867e-02_i, -5.2246138542576114e-02-1.6302636524913164e-02_i,
3.9400503850194057e-02-9.3014999699258024e-03_i, -2.7102794310345200e-02-5.8065391083871112e-03_i,  2.1876470763211843e-02-3.8665059247379036e-03_i,
-1.6808546635877601e-02-2.7028938973387667e-03_i,  1.4182711865637047e-02-1.9623967533117363e-03_i, -1.1619263424342583e-02-1.4685466549561686e-03_i,
1.0120618008742033e-02-1.1263908912712249e-03_i, -8.6514816678522309e-03-8.8169762528223297e-04_i,  7.7208605005762848e-03-7.0193814140220856e-04_i,
-6.8060263495656192e-03-5.6678520845984681e-04_i,  6.1934992250487486e-03-4.6308569550757017e-04_i, -5.5902140742913886e-03-3.8206879656940695e-04_i,
5.1706035925756094e-03-3.1773025343947991e-04_i, -4.7567994440500146e-03-2.6586308547879231e-04_i,  4.4619850359670218e-03-2.2345760457638184e-04_i,
-4.1710414019030953e-03-1.8831955894072459e-04_i,  3.9616128878299951e-03-1.5882099254900658e-04_i, -3.7549071583388369e-03-1.3373400329354810e-04_i,
3.6070629908332066e-03-1.1211750891974848e-04_i, -3.4612376444636481e-03-9.3238620330201964e-05_i,  3.3602584269727571e-03-7.6517026828740523e-05_i,
-3.2608615513316994e-03-6.1484924932601534e-05_i,  3.1976797184812208e-03-4.7757579137403666e-05_i, -3.1358167052793068e-03-3.5011215878316682e-05_i,
3.1051643536079143e-03-2.2965984517037130e-05_i, -3.0756893716986367e-03-1.1372385953722919e-05_i,  3.0751229889940390e-03+0.0000000000000000e+00_i,
-3.0756893716986367e-03+1.1372385953722919e-05_i,  3.1051643536079143e-03+2.2965984517037130e-05_i, -3.1358167052793068e-03+3.5011215878316682e-05_i,
3.1976797184812208e-03+4.7757579137403666e-05_i, -3.2608615513316994e-03+6.1484924932601534e-05_i,  3.3602584269727571e-03+7.6517026828740523e-05_i,
-3.4612376444636481e-03+9.3238620330201964e-05_i,  3.6070629908332066e-03+1.1211750891974848e-04_i, -3.7549071583388369e-03+1.3373400329354810e-04_i,
3.9616128878299951e-03+1.5882099254900658e-04_i, -4.1710414019030953e-03+1.8831955894072459e-04_i,  4.4619850359670218e-03+2.2345760457638184e-04_i,
-4.7567994440500146e-03+2.6586308547879231e-04_i,  5.1706035925756094e-03+3.1773025343947991e-04_i, -5.5902140742913886e-03+3.8206879656940695e-04_i,
6.1934992250487486e-03+4.6308569550757017e-04_i, -6.8060263495656192e-03+5.6678520845984681e-04_i,  7.7208605005762848e-03+7.0193814140220856e-04_i,
-8.6514816678522309e-03+8.8169762528223297e-04_i,  1.0120618008742033e-02+1.1263908912712249e-03_i, -1.1619263424342583e-02+1.4685466549561686e-03_i,
1.4182711865637047e-02+1.9623967533117363e-03_i, -1.6808546635877601e-02+2.7028938973387667e-03_i,  2.1876470763211843e-02+3.8665059247379036e-03_i,
-2.7102794310345200e-02+5.8065391083871112e-03_i,  3.9400503850194057e-02+9.3014999699258024e-03_i, -5.2246138542576114e-02+1.6302636524913164e-02_i,
9.6735350049332092e-02+3.2690642511285867e-02_i, -1.4494751578417842e-01+8.2224738731067948e-02_i,  7.0345957258712111e-01+3.3704182679810379e-01_i,
2.9938833406609444e-01-1.4907693782657527e+00_i}};

#endif //STFT_GOOD_DATA_H
