/*
 * Decompiled with CFR 0.152.
 */
package jadx.gui.jobs;

import jadx.gui.jobs.Cancelable;
import jadx.gui.jobs.IBackgroundTask;
import jadx.gui.jobs.ITaskInfo;
import jadx.gui.jobs.ITaskProgress;
import jadx.gui.jobs.SimpleTask;
import jadx.gui.jobs.TaskProgress;
import jadx.gui.jobs.TaskStatus;
import jadx.gui.settings.JadxSettings;
import jadx.gui.ui.panel.ProgressPanel;
import jadx.gui.utils.NLS;
import jadx.gui.utils.UiUtils;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackgroundExecutor {
    private static final Logger LOG = LoggerFactory.getLogger(BackgroundExecutor.class);
    private final JadxSettings settings;
    private final ProgressPanel progressPane;
    private ThreadPoolExecutor taskQueueExecutor;
    private final Map<Long, IBackgroundTask> taskRunning = new ConcurrentHashMap<Long, IBackgroundTask>();
    private final AtomicLong idSupplier = new AtomicLong(0L);

    public BackgroundExecutor(JadxSettings settings, ProgressPanel progressPane) {
        this.settings = Objects.requireNonNull(settings);
        this.progressPane = Objects.requireNonNull(progressPane);
        this.reset();
    }

    public synchronized Future<TaskStatus> execute(IBackgroundTask task) {
        long id = this.idSupplier.incrementAndGet();
        TaskWorker taskWorker = new TaskWorker(id, task);
        this.taskRunning.put(id, task);
        this.taskQueueExecutor.execute(() -> {
            taskWorker.init();
            taskWorker.run();
        });
        return taskWorker;
    }

    public synchronized void cancelAll() {
        try {
            this.taskRunning.values().forEach(Cancelable::cancel);
            this.taskQueueExecutor.shutdownNow();
            boolean complete = this.taskQueueExecutor.awaitTermination(3L, TimeUnit.SECONDS);
            if (complete) {
                LOG.debug("Background task executor canceled successfully");
            } else {
                String taskNames = this.taskRunning.values().stream().map(IBackgroundTask::getTitle).collect(Collectors.joining(", "));
                LOG.debug("Background task executor cancel failed. Running tasks: {}", (Object)taskNames);
            }
        }
        catch (Exception e) {
            LOG.error("Error terminating task executor", (Throwable)e);
        }
        finally {
            this.reset();
        }
    }

    public void execute(String title, List<Runnable> backgroundJobs, Consumer<TaskStatus> onFinishUiRunnable) {
        this.execute(new SimpleTask(title, backgroundJobs, onFinishUiRunnable));
    }

    public void execute(String title, Runnable backgroundRunnable, Consumer<TaskStatus> onFinishUiRunnable) {
        this.execute(new SimpleTask(title, Collections.singletonList(backgroundRunnable), onFinishUiRunnable));
    }

    public Future<TaskStatus> execute(String title, Runnable backgroundRunnable) {
        return this.execute(new SimpleTask(title, Collections.singletonList(backgroundRunnable)));
    }

    private synchronized void reset() {
        this.taskQueueExecutor = (ThreadPoolExecutor)Executors.newFixedThreadPool(1);
        this.taskRunning.clear();
        this.idSupplier.set(0L);
    }

    private void taskComplete(long id) {
        this.taskRunning.remove(id);
    }

    private final class TaskWorker
    extends SwingWorker<TaskStatus, Void>
    implements ITaskInfo {
        private final long id;
        private final IBackgroundTask task;
        private ThreadPoolExecutor executor;
        private TaskStatus status = TaskStatus.WAIT;
        private long jobsCount;
        private long jobsComplete;
        private long time;

        public TaskWorker(long id, IBackgroundTask task) {
            this.id = id;
            this.task = task;
        }

        public void init() {
            this.addPropertyChangeListener(BackgroundExecutor.this.progressPane);
            SwingUtilities.invokeLater(() -> {
                BackgroundExecutor.this.progressPane.reset();
                if (this.task.getTaskProgress() != null) {
                    BackgroundExecutor.this.progressPane.setIndeterminate(false);
                }
            });
        }

        @Override
        protected TaskStatus doInBackground() throws Exception {
            BackgroundExecutor.this.progressPane.changeLabel(this, this.task.getTitle() + "\u2026 ");
            BackgroundExecutor.this.progressPane.changeCancelBtnVisible(this, this.task.canBeCanceled());
            try {
                this.runJobs();
            }
            finally {
                try {
                    this.task.onDone(this);
                    UiUtils.uiRunAndWait(() -> {
                        BackgroundExecutor.this.progressPane.setVisible(false);
                        this.task.onFinish(this);
                    });
                }
                finally {
                    BackgroundExecutor.this.taskComplete(this.id);
                    BackgroundExecutor.this.progressPane.changeVisibility(this, false);
                }
            }
            return this.status;
        }

        private void runJobs() throws InterruptedException {
            List<? extends Runnable> jobs = this.task.scheduleJobs();
            this.jobsCount = jobs.size();
            LOG.debug("Starting background task '{}', jobs count: {}, time limit: {} ms, memory check: {}", new Object[]{this.task.getTitle(), this.jobsCount, this.task.timeLimit(), this.task.checkMemoryUsage()});
            if (this.jobsCount != 1L) {
                BackgroundExecutor.this.progressPane.changeVisibility(this, true);
            }
            this.status = TaskStatus.STARTED;
            int threadsCount = BackgroundExecutor.this.settings.getThreadsCount();
            this.executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(threadsCount);
            for (Runnable runnable : jobs) {
                this.executor.execute(runnable);
            }
            this.executor.shutdown();
            long startTime = System.currentTimeMillis();
            this.status = this.waitTermination(this.executor, this.buildCancelCheck(startTime));
            this.time = System.currentTimeMillis() - startTime;
            this.jobsComplete = this.executor.getCompletedTaskCount();
        }

        private TaskStatus waitTermination(ThreadPoolExecutor executor, Supplier<TaskStatus> cancelCheck) throws InterruptedException {
            try {
                int k = 0;
                while (true) {
                    if (executor.isTerminated()) {
                        return TaskStatus.COMPLETE;
                    }
                    TaskStatus cancelStatus = cancelCheck.get();
                    if (cancelStatus != null) {
                        this.performCancel(executor);
                        return cancelStatus;
                    }
                    if (k < 10) {
                        Thread.sleep(200L);
                        if (k == 5) {
                            this.updateProgress(executor);
                        }
                    } else {
                        this.updateProgress(executor);
                        Thread.sleep(1000L);
                    }
                    if (this.jobsCount == 1L && k == 5) {
                        BackgroundExecutor.this.progressPane.changeVisibility(this, true);
                    }
                    ++k;
                }
            }
            catch (InterruptedException e) {
                LOG.debug("Task wait interrupted");
                this.performCancel(executor);
                return TaskStatus.CANCEL_BY_USER;
            }
            catch (Exception e) {
                LOG.error("Task wait aborted by exception", (Throwable)e);
                this.performCancel(executor);
                return TaskStatus.ERROR;
            }
        }

        private void updateProgress(ThreadPoolExecutor executor) {
            Consumer<ITaskProgress> onProgressListener = this.task.getProgressListener();
            ITaskProgress taskProgress = this.task.getTaskProgress();
            if (taskProgress == null) {
                this.setProgress(UiUtils.calcProgress(executor.getCompletedTaskCount(), this.jobsCount));
                if (onProgressListener != null) {
                    onProgressListener.accept(new TaskProgress(executor.getCompletedTaskCount(), this.jobsCount));
                }
            } else {
                this.setProgress(UiUtils.calcProgress(taskProgress));
                if (onProgressListener != null) {
                    onProgressListener.accept(taskProgress);
                }
            }
        }

        private void performCancel(ThreadPoolExecutor executor) throws InterruptedException {
            BackgroundExecutor.this.progressPane.changeLabel(this, this.task.getTitle() + " (" + NLS.str("progress.canceling") + ")\u2026 ");
            BackgroundExecutor.this.progressPane.changeIndeterminate(this, true);
            this.task.cancel();
            executor.shutdown();
            int cancelTimeout = this.task.getCancelTimeoutMS();
            if (cancelTimeout != 0 && executor.awaitTermination(cancelTimeout, TimeUnit.MILLISECONDS)) {
                LOG.debug("Task cancel complete");
                return;
            }
            LOG.debug("Forcing tasks cancel");
            executor.shutdownNow();
            boolean complete = executor.awaitTermination(this.task.getShutdownTimeoutMS(), TimeUnit.MILLISECONDS);
            LOG.debug("Forced task cancel status: {}", (Object)(complete ? "success" : "fail, still active: " + executor.getActiveCount()));
        }

        private Supplier<TaskStatus> buildCancelCheck(long startTime) {
            long waitUntilTime = this.task.timeLimit() == 0 ? 0L : startTime + (long)this.task.timeLimit();
            boolean checkMemoryUsage = this.task.checkMemoryUsage();
            return () -> {
                if (this.task.isCanceled()) {
                    return TaskStatus.CANCEL_BY_USER;
                }
                if (waitUntilTime != 0L && waitUntilTime < System.currentTimeMillis()) {
                    LOG.error("Task '{}' execution timeout, force cancel", (Object)this.task.getTitle());
                    return TaskStatus.CANCEL_BY_TIMEOUT;
                }
                if (this.isCancelled() || Thread.currentThread().isInterrupted()) {
                    LOG.warn("Task '{}' canceled", (Object)this.task.getTitle());
                    return TaskStatus.CANCEL_BY_USER;
                }
                if (checkMemoryUsage && !UiUtils.isFreeMemoryAvailable()) {
                    LOG.info("Memory usage: {}", (Object)UiUtils.memoryInfo());
                    if (this.executor.getCorePoolSize() == 1) {
                        LOG.error("Task '{}' memory limit reached, force cancel", (Object)this.task.getTitle());
                        return TaskStatus.CANCEL_BY_MEMORY;
                    }
                    LOG.warn("Low memory, reduce processing threads count to 1");
                    this.executor.setCorePoolSize(1);
                    System.gc();
                    UiUtils.sleep(1000);
                    if (!UiUtils.isFreeMemoryAvailable()) {
                        LOG.error("Task '{}' memory limit reached (after GC), force cancel", (Object)this.task.getTitle());
                        return TaskStatus.CANCEL_BY_MEMORY;
                    }
                }
                return null;
            };
        }

        @Override
        public TaskStatus getStatus() {
            return this.status;
        }

        @Override
        public long getJobsCount() {
            return this.jobsCount;
        }

        @Override
        public long getJobsComplete() {
            return this.jobsComplete;
        }

        @Override
        public long getJobsSkipped() {
            return this.jobsCount - this.jobsComplete;
        }

        @Override
        public long getTime() {
            return this.time;
        }

        public String toString() {
            return "TaskWorker{status=" + (Object)((Object)this.status) + ", jobsCount=" + this.jobsCount + ", jobsComplete=" + this.jobsComplete + ", time=" + this.time + '}';
        }
    }
}

