"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSearchRelevanceRoutes = registerSearchRelevanceRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerSearchRelevanceRoutes(router) {
  router.post({
    path: _common.ServiceEndpoints.QuerySets,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        sampling: _configSchema.schema.string(),
        querySetSize: _configSchema.schema.number()
      })
    }
  }, backendAction('POST', _common.BackendEndpoints.QuerySets));
  router.put({
    path: _common.ServiceEndpoints.QuerySets,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        sampling: _configSchema.schema.string(),
        querySetQueries: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.object({
          queryText: _configSchema.schema.string(),
          referenceAnswer: _configSchema.schema.maybe(_configSchema.schema.string())
        }), {
          minSize: 1
        }), _configSchema.schema.string()])
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.QuerySets));
  router.get({
    path: _common.ServiceEndpoints.QuerySets,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.QuerySets));
  router.delete({
    path: `${_common.ServiceEndpoints.QuerySets}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.QuerySets));
  router.put({
    path: _common.ServiceEndpoints.SearchConfigurations,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        index: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        searchPipeline: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.SearchConfigurations));
  router.get({
    path: _common.ServiceEndpoints.SearchConfigurations,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.SearchConfigurations));
  router.delete({
    path: `${_common.ServiceEndpoints.SearchConfigurations}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.SearchConfigurations));
  router.post({
    path: _common.ServiceEndpoints.Experiments,
    validate: {
      body: _configSchema.schema.object({
        querySetId: _configSchema.schema.string(),
        searchConfigurationList: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        size: _configSchema.schema.number(),
        type: _configSchema.schema.string(),
        // TODO: make mandatory conditional on experiment type
        judgmentList: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.Experiments));
  router.get({
    path: _common.ServiceEndpoints.Experiments,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.Experiments));
  router.get({
    path: `${_common.ServiceEndpoints.Experiments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.Experiments));
  router.get({
    path: `${_common.ServiceEndpoints.SearchConfigurations}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.SearchConfigurations));
  router.get({
    path: `${_common.ServiceEndpoints.QuerySets}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.QuerySets));
  router.delete({
    path: `${_common.ServiceEndpoints.Experiments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.Experiments));
  router.put({
    path: _common.ServiceEndpoints.Judgments,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        type: _configSchema.schema.string(),
        querySetId: _configSchema.schema.maybe(_configSchema.schema.string()),
        searchConfigurationList: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        size: _configSchema.schema.maybe(_configSchema.schema.number()),
        modelId: _configSchema.schema.maybe(_configSchema.schema.string()),
        tokenLimit: _configSchema.schema.maybe(_configSchema.schema.string()),
        ignoreFailure: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        contextFields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        clickModel: _configSchema.schema.maybe(_configSchema.schema.string()),
        maxRank: _configSchema.schema.maybe(_configSchema.schema.number()),
        startDate: _configSchema.schema.maybe(_configSchema.schema.string()),
        endDate: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, backendAction('PUT', _common.BackendEndpoints.Judgments));
  router.get({
    path: _common.ServiceEndpoints.Judgments,
    validate: false
  }, backendAction('GET', _common.BackendEndpoints.Judgments));
  router.get({
    path: `${_common.ServiceEndpoints.Judgments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('GET', _common.BackendEndpoints.Judgments));
  router.delete({
    path: `${_common.ServiceEndpoints.Judgments}/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, backendAction('DELETE', _common.BackendEndpoints.Judgments));
}
const backendAction = (method, path) => {
  return async (context, req, res) => {
    const dataSourceId = req.query.data_source;
    const caller = dataSourceId ? context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI : context.core.opensearch.legacy.client.callAsCurrentUser;
    try {
      let response;
      if (method === 'DELETE') {
        const {
          id
        } = req.params;
        const deletePath = `${path}/${id}`;
        response = await caller('transport.request', {
          method,
          path: deletePath
        });
      } else if (method === 'GET' && req.params.id) {
        // Handle GET request for individual experiment
        const getPath = `${path}/${req.params.id}`;
        response = await caller('transport.request', {
          method,
          path: getPath
        });
      } else {
        // Handle PUT, POST, GET as before
        response = await caller('transport.request', {
          method,
          path,
          ...(method === 'POST' || method === 'PUT' ? {
            body: req.body
          } : {})
        });
      }
      return res.ok({
        body: response
      });
    } catch (err) {
      var _err$body;
      console.error('Failed to call search-relevance APIs', err); // Keep for full server-side logging

      let clientMessage = err.message; // Default to the err.message from transport.request
      let clientAttributesError = ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message; // Default attributes error

      // Check if the backend error body contains the specific message
      if (err.body && typeof err.body === 'string' && err.body.includes(_common.DISABLED_BACKEND_PLUGIN_MESSAGE)) {
        clientMessage = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
        clientAttributesError = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
      }
      // If the backend error body is a JSON object with a message/reason
      else if (err.body && typeof err.body === 'object') {
        // Check for common backend error formats
        if (err.body.message && typeof err.body.message === 'string' && err.body.message.includes(_common.DISABLED_BACKEND_PLUGIN_MESSAGE)) {
          clientMessage = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
          clientAttributesError = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
        } else if (err.body.reason && typeof err.body.reason === 'string' && err.body.reason.includes(_common.DISABLED_BACKEND_PLUGIN_MESSAGE)) {
          clientMessage = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
          clientAttributesError = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
        } else if (err.body.error && typeof err.body.error === 'object' && err.body.error.reason && typeof err.body.error.reason === 'string' && err.body.error.reason.includes(_common.DISABLED_BACKEND_PLUGIN_MESSAGE)) {
          clientMessage = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
          clientAttributesError = _common.DISABLED_BACKEND_PLUGIN_MESSAGE;
        }
        // Fallback if specific message not found in complex body, but body has a message
        else if (err.body.message && typeof err.body.message === 'string') {
          clientMessage = err.body.message;
          clientAttributesError = err.body.message;
        }
      }
      return res.customError({
        statusCode: err.statusCode || 500,
        body: {
          message: clientMessage,
          // Use the determined message
          attributes: {
            error: clientAttributesError // Use the determined attributes error
          }
        }
      });
    }
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9jb21tb24iLCJyZWdpc3RlclNlYXJjaFJlbGV2YW5jZVJvdXRlcyIsInJvdXRlciIsInBvc3QiLCJwYXRoIiwiU2VydmljZUVuZHBvaW50cyIsIlF1ZXJ5U2V0cyIsInZhbGlkYXRlIiwiYm9keSIsInNjaGVtYSIsIm9iamVjdCIsIm5hbWUiLCJzdHJpbmciLCJkZXNjcmlwdGlvbiIsInNhbXBsaW5nIiwicXVlcnlTZXRTaXplIiwibnVtYmVyIiwiYmFja2VuZEFjdGlvbiIsIkJhY2tlbmRFbmRwb2ludHMiLCJwdXQiLCJxdWVyeVNldFF1ZXJpZXMiLCJvbmVPZiIsImFycmF5T2YiLCJxdWVyeVRleHQiLCJyZWZlcmVuY2VBbnN3ZXIiLCJtYXliZSIsIm1pblNpemUiLCJnZXQiLCJkZWxldGUiLCJwYXJhbXMiLCJpZCIsIlNlYXJjaENvbmZpZ3VyYXRpb25zIiwiaW5kZXgiLCJxdWVyeSIsInNlYXJjaFBpcGVsaW5lIiwiRXhwZXJpbWVudHMiLCJxdWVyeVNldElkIiwic2VhcmNoQ29uZmlndXJhdGlvbkxpc3QiLCJzaXplIiwidHlwZSIsImp1ZGdtZW50TGlzdCIsIkp1ZGdtZW50cyIsIm1vZGVsSWQiLCJ0b2tlbkxpbWl0IiwiaWdub3JlRmFpbHVyZSIsImJvb2xlYW4iLCJjb250ZXh0RmllbGRzIiwiY2xpY2tNb2RlbCIsIm1heFJhbmsiLCJzdGFydERhdGUiLCJlbmREYXRlIiwibWV0aG9kIiwiY29udGV4dCIsInJlcSIsInJlcyIsImRhdGFTb3VyY2VJZCIsImRhdGFfc291cmNlIiwiY2FsbGVyIiwiZGF0YVNvdXJjZSIsIm9wZW5zZWFyY2giLCJsZWdhY3kiLCJnZXRDbGllbnQiLCJjYWxsQVBJIiwiY29yZSIsImNsaWVudCIsImNhbGxBc0N1cnJlbnRVc2VyIiwicmVzcG9uc2UiLCJkZWxldGVQYXRoIiwiZ2V0UGF0aCIsIm9rIiwiZXJyIiwiX2VyciRib2R5IiwiY29uc29sZSIsImVycm9yIiwiY2xpZW50TWVzc2FnZSIsIm1lc3NhZ2UiLCJjbGllbnRBdHRyaWJ1dGVzRXJyb3IiLCJpbmNsdWRlcyIsIkRJU0FCTEVEX0JBQ0tFTkRfUExVR0lOX01FU1NBR0UiLCJyZWFzb24iLCJjdXN0b21FcnJvciIsInN0YXR1c0NvZGUiLCJhdHRyaWJ1dGVzIl0sInNvdXJjZXMiOlsic2VhcmNoX3JlbGV2YW5jZV9yb3V0ZV9zZXJ2aWNlLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQG9zZC9jb25maWctc2NoZW1hJztcbmltcG9ydCB7XG4gIElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLFxuICBJUm91dGVyLFxuICBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gIE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5LFxuICBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG59IGZyb20gJy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyBTZXJ2aWNlRW5kcG9pbnRzLCBCYWNrZW5kRW5kcG9pbnRzLCBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFIH0gZnJvbSAnLi4vLi4vY29tbW9uJztcblxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVyU2VhcmNoUmVsZXZhbmNlUm91dGVzKHJvdXRlcjogSVJvdXRlcik6IHZvaWQge1xuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBTZXJ2aWNlRW5kcG9pbnRzLlF1ZXJ5U2V0cyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHNhbXBsaW5nOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcXVlcnlTZXRTaXplOiBzY2hlbWEubnVtYmVyKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGJhY2tlbmRBY3Rpb24oJ1BPU1QnLCBCYWNrZW5kRW5kcG9pbnRzLlF1ZXJ5U2V0cylcbiAgKTtcbiAgcm91dGVyLnB1dChcbiAgICB7XG4gICAgICBwYXRoOiBTZXJ2aWNlRW5kcG9pbnRzLlF1ZXJ5U2V0cyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIG5hbWU6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBkZXNjcmlwdGlvbjogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHNhbXBsaW5nOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgcXVlcnlTZXRRdWVyaWVzOiBzY2hlbWEub25lT2YoW1xuICAgICAgICAgICAgc2NoZW1hLmFycmF5T2YoXG4gICAgICAgICAgICAgIHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgICAgICAgIHF1ZXJ5VGV4dDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgICAgICAgIHJlZmVyZW5jZUFuc3dlcjogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICB7IG1pblNpemU6IDEgfVxuICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBdKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignUFVUJywgQmFja2VuZEVuZHBvaW50cy5RdWVyeVNldHMpXG4gICk7XG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogU2VydmljZUVuZHBvaW50cy5RdWVyeVNldHMsXG4gICAgICB2YWxpZGF0ZTogZmFsc2UsXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdHRVQnLCBCYWNrZW5kRW5kcG9pbnRzLlF1ZXJ5U2V0cylcbiAgKTtcbiAgcm91dGVyLmRlbGV0ZShcbiAgICB7XG4gICAgICBwYXRoOiBgJHtTZXJ2aWNlRW5kcG9pbnRzLlF1ZXJ5U2V0c30ve2lkfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGJhY2tlbmRBY3Rpb24oJ0RFTEVURScsIEJhY2tlbmRFbmRwb2ludHMuUXVlcnlTZXRzKVxuICApO1xuICByb3V0ZXIucHV0KFxuICAgIHtcbiAgICAgIHBhdGg6IFNlcnZpY2VFbmRwb2ludHMuU2VhcmNoQ29uZmlndXJhdGlvbnMsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgaW5kZXg6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBxdWVyeTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHNlYXJjaFBpcGVsaW5lOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignUFVUJywgQmFja2VuZEVuZHBvaW50cy5TZWFyY2hDb25maWd1cmF0aW9ucylcbiAgKTtcbiAgcm91dGVyLmdldChcbiAgICB7XG4gICAgICBwYXRoOiBTZXJ2aWNlRW5kcG9pbnRzLlNlYXJjaENvbmZpZ3VyYXRpb25zLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignR0VUJywgQmFja2VuZEVuZHBvaW50cy5TZWFyY2hDb25maWd1cmF0aW9ucylcbiAgKTtcbiAgcm91dGVyLmRlbGV0ZShcbiAgICB7XG4gICAgICBwYXRoOiBgJHtTZXJ2aWNlRW5kcG9pbnRzLlNlYXJjaENvbmZpZ3VyYXRpb25zfS97aWR9YCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgaWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignREVMRVRFJywgQmFja2VuZEVuZHBvaW50cy5TZWFyY2hDb25maWd1cmF0aW9ucylcbiAgKTtcbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogU2VydmljZUVuZHBvaW50cy5FeHBlcmltZW50cyxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIHF1ZXJ5U2V0SWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgICBzZWFyY2hDb25maWd1cmF0aW9uTGlzdDogc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICBzaXplOiBzY2hlbWEubnVtYmVyKCksXG4gICAgICAgICAgdHlwZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIC8vIFRPRE86IG1ha2UgbWFuZGF0b3J5IGNvbmRpdGlvbmFsIG9uIGV4cGVyaW1lbnQgdHlwZVxuICAgICAgICAgIGp1ZGdtZW50TGlzdDogc2NoZW1hLm1heWJlKHNjaGVtYS5hcnJheU9mKHNjaGVtYS5zdHJpbmcoKSkpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdQVVQnLCBCYWNrZW5kRW5kcG9pbnRzLkV4cGVyaW1lbnRzKVxuICApO1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IFNlcnZpY2VFbmRwb2ludHMuRXhwZXJpbWVudHMsXG4gICAgICB2YWxpZGF0ZTogZmFsc2UsXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdHRVQnLCBCYWNrZW5kRW5kcG9pbnRzLkV4cGVyaW1lbnRzKVxuICApO1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke1NlcnZpY2VFbmRwb2ludHMuRXhwZXJpbWVudHN9L3tpZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdHRVQnLCBCYWNrZW5kRW5kcG9pbnRzLkV4cGVyaW1lbnRzKVxuICApO1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke1NlcnZpY2VFbmRwb2ludHMuU2VhcmNoQ29uZmlndXJhdGlvbnN9L3tpZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdHRVQnLCBCYWNrZW5kRW5kcG9pbnRzLlNlYXJjaENvbmZpZ3VyYXRpb25zKVxuICApO1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IGAke1NlcnZpY2VFbmRwb2ludHMuUXVlcnlTZXRzfS97aWR9YCxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIHBhcmFtczogc2NoZW1hLm9iamVjdCh7XG4gICAgICAgICAgaWQ6IHNjaGVtYS5zdHJpbmcoKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignR0VUJywgQmFja2VuZEVuZHBvaW50cy5RdWVyeVNldHMpXG4gICk7XG4gIHJvdXRlci5kZWxldGUoXG4gICAge1xuICAgICAgcGF0aDogYCR7U2VydmljZUVuZHBvaW50cy5FeHBlcmltZW50c30ve2lkfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGJhY2tlbmRBY3Rpb24oJ0RFTEVURScsIEJhY2tlbmRFbmRwb2ludHMuRXhwZXJpbWVudHMpXG4gICk7XG4gIHJvdXRlci5wdXQoXG4gICAge1xuICAgICAgcGF0aDogU2VydmljZUVuZHBvaW50cy5KdWRnbWVudHMsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBuYW1lOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgICAgdHlwZTogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICAgIHF1ZXJ5U2V0SWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxuICAgICAgICAgIHNlYXJjaENvbmZpZ3VyYXRpb25MaXN0OiBzY2hlbWEubWF5YmUoc2NoZW1hLmFycmF5T2Yoc2NoZW1hLnN0cmluZygpKSksXG4gICAgICAgICAgc2l6ZTogc2NoZW1hLm1heWJlKHNjaGVtYS5udW1iZXIoKSksXG4gICAgICAgICAgbW9kZWxJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgdG9rZW5MaW1pdDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgaWdub3JlRmFpbHVyZTogc2NoZW1hLm1heWJlKHNjaGVtYS5ib29sZWFuKCkpLFxuICAgICAgICAgIGNvbnRleHRGaWVsZHM6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEuc3RyaW5nKCkpKSxcbiAgICAgICAgICBjbGlja01vZGVsOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICBtYXhSYW5rOiBzY2hlbWEubWF5YmUoc2NoZW1hLm51bWJlcigpKSxcbiAgICAgICAgICBzdGFydERhdGU6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxuICAgICAgICAgIGVuZERhdGU6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGJhY2tlbmRBY3Rpb24oJ1BVVCcsIEJhY2tlbmRFbmRwb2ludHMuSnVkZ21lbnRzKVxuICApO1xuICByb3V0ZXIuZ2V0KFxuICAgIHtcbiAgICAgIHBhdGg6IFNlcnZpY2VFbmRwb2ludHMuSnVkZ21lbnRzLFxuICAgICAgdmFsaWRhdGU6IGZhbHNlLFxuICAgIH0sXG4gICAgYmFja2VuZEFjdGlvbignR0VUJywgQmFja2VuZEVuZHBvaW50cy5KdWRnbWVudHMpXG4gICk7XG4gIHJvdXRlci5nZXQoXG4gICAge1xuICAgICAgcGF0aDogYCR7U2VydmljZUVuZHBvaW50cy5KdWRnbWVudHN9L3tpZH1gLFxuICAgICAgdmFsaWRhdGU6IHtcbiAgICAgICAgcGFyYW1zOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBpZDogc2NoZW1hLnN0cmluZygpLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBiYWNrZW5kQWN0aW9uKCdHRVQnLCBCYWNrZW5kRW5kcG9pbnRzLkp1ZGdtZW50cylcbiAgKTtcbiAgcm91dGVyLmRlbGV0ZShcbiAgICB7XG4gICAgICBwYXRoOiBgJHtTZXJ2aWNlRW5kcG9pbnRzLkp1ZGdtZW50c30ve2lkfWAsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBwYXJhbXM6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGlkOiBzY2hlbWEuc3RyaW5nKCksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGJhY2tlbmRBY3Rpb24oJ0RFTEVURScsIEJhY2tlbmRFbmRwb2ludHMuSnVkZ21lbnRzKVxuICApO1xufVxuXG5jb25zdCBiYWNrZW5kQWN0aW9uID0gKG1ldGhvZCwgcGF0aCkgPT4ge1xuICByZXR1cm4gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXE6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICByZXM6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIGNvbnN0IGRhdGFTb3VyY2VJZCA9IHJlcS5xdWVyeS5kYXRhX3NvdXJjZTtcbiAgICBjb25zdCBjYWxsZXIgPSBkYXRhU291cmNlSWRcbiAgICAgID8gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChkYXRhU291cmNlSWQpLmNhbGxBUElcbiAgICAgIDogY29udGV4dC5jb3JlLm9wZW5zZWFyY2gubGVnYWN5LmNsaWVudC5jYWxsQXNDdXJyZW50VXNlcjtcblxuICAgIHRyeSB7XG4gICAgICBsZXQgcmVzcG9uc2U7XG4gICAgICBpZiAobWV0aG9kID09PSAnREVMRVRFJykge1xuICAgICAgICBjb25zdCB7IGlkIH0gPSByZXEucGFyYW1zO1xuICAgICAgICBjb25zdCBkZWxldGVQYXRoID0gYCR7cGF0aH0vJHtpZH1gO1xuICAgICAgICByZXNwb25zZSA9IGF3YWl0IGNhbGxlcigndHJhbnNwb3J0LnJlcXVlc3QnLCB7XG4gICAgICAgICAgbWV0aG9kLFxuICAgICAgICAgIHBhdGg6IGRlbGV0ZVBhdGgsXG4gICAgICAgIH0pO1xuICAgICAgfSBlbHNlIGlmIChtZXRob2QgPT09ICdHRVQnICYmIHJlcS5wYXJhbXMuaWQpIHtcbiAgICAgICAgLy8gSGFuZGxlIEdFVCByZXF1ZXN0IGZvciBpbmRpdmlkdWFsIGV4cGVyaW1lbnRcbiAgICAgICAgY29uc3QgZ2V0UGF0aCA9IGAke3BhdGh9LyR7cmVxLnBhcmFtcy5pZH1gO1xuICAgICAgICByZXNwb25zZSA9IGF3YWl0IGNhbGxlcigndHJhbnNwb3J0LnJlcXVlc3QnLCB7XG4gICAgICAgICAgbWV0aG9kLFxuICAgICAgICAgIHBhdGg6IGdldFBhdGgsXG4gICAgICAgIH0pO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gSGFuZGxlIFBVVCwgUE9TVCwgR0VUIGFzIGJlZm9yZVxuICAgICAgICByZXNwb25zZSA9IGF3YWl0IGNhbGxlcigndHJhbnNwb3J0LnJlcXVlc3QnLCB7XG4gICAgICAgICAgbWV0aG9kLFxuICAgICAgICAgIHBhdGgsXG4gICAgICAgICAgLi4uKG1ldGhvZCA9PT0gJ1BPU1QnIHx8IG1ldGhvZCA9PT0gJ1BVVCcgPyB7IGJvZHk6IHJlcS5ib2R5IH0gOiB7fSksXG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4gcmVzLm9rKHsgYm9keTogcmVzcG9uc2UgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG5cbiAgICAgIGNvbnNvbGUuZXJyb3IoJ0ZhaWxlZCB0byBjYWxsIHNlYXJjaC1yZWxldmFuY2UgQVBJcycsIGVycik7IC8vIEtlZXAgZm9yIGZ1bGwgc2VydmVyLXNpZGUgbG9nZ2luZ1xuXG4gICAgICBsZXQgY2xpZW50TWVzc2FnZSA9IGVyci5tZXNzYWdlOyAvLyBEZWZhdWx0IHRvIHRoZSBlcnIubWVzc2FnZSBmcm9tIHRyYW5zcG9ydC5yZXF1ZXN0XG4gICAgICBsZXQgY2xpZW50QXR0cmlidXRlc0Vycm9yID0gZXJyLmJvZHk/LmVycm9yIHx8IGVyci5tZXNzYWdlOyAvLyBEZWZhdWx0IGF0dHJpYnV0ZXMgZXJyb3JcblxuICAgICAgLy8gQ2hlY2sgaWYgdGhlIGJhY2tlbmQgZXJyb3IgYm9keSBjb250YWlucyB0aGUgc3BlY2lmaWMgbWVzc2FnZVxuICAgICAgaWYgKGVyci5ib2R5ICYmIHR5cGVvZiBlcnIuYm9keSA9PT0gJ3N0cmluZycgJiYgZXJyLmJvZHkuaW5jbHVkZXMoRElTQUJMRURfQkFDS0VORF9QTFVHSU5fTUVTU0FHRSkpIHtcbiAgICAgICAgICBjbGllbnRNZXNzYWdlID0gRElTQUJMRURfQkFDS0VORF9QTFVHSU5fTUVTU0FHRTtcbiAgICAgICAgICBjbGllbnRBdHRyaWJ1dGVzRXJyb3IgPSBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFO1xuICAgICAgfVxuICAgICAgLy8gSWYgdGhlIGJhY2tlbmQgZXJyb3IgYm9keSBpcyBhIEpTT04gb2JqZWN0IHdpdGggYSBtZXNzYWdlL3JlYXNvblxuICAgICAgZWxzZSBpZiAoZXJyLmJvZHkgJiYgdHlwZW9mIGVyci5ib2R5ID09PSAnb2JqZWN0Jykge1xuICAgICAgICAgIC8vIENoZWNrIGZvciBjb21tb24gYmFja2VuZCBlcnJvciBmb3JtYXRzXG4gICAgICAgICAgaWYgKGVyci5ib2R5Lm1lc3NhZ2UgJiYgdHlwZW9mIGVyci5ib2R5Lm1lc3NhZ2UgPT09ICdzdHJpbmcnICYmIGVyci5ib2R5Lm1lc3NhZ2UuaW5jbHVkZXMoRElTQUJMRURfQkFDS0VORF9QTFVHSU5fTUVTU0FHRSkpIHtcbiAgICAgICAgICAgICAgY2xpZW50TWVzc2FnZSA9IERJU0FCTEVEX0JBQ0tFTkRfUExVR0lOX01FU1NBR0U7XG4gICAgICAgICAgICAgIGNsaWVudEF0dHJpYnV0ZXNFcnJvciA9IERJU0FCTEVEX0JBQ0tFTkRfUExVR0lOX01FU1NBR0U7XG4gICAgICAgICAgfSBlbHNlIGlmIChlcnIuYm9keS5yZWFzb24gJiYgdHlwZW9mIGVyci5ib2R5LnJlYXNvbiA9PT0gJ3N0cmluZycgJiYgZXJyLmJvZHkucmVhc29uLmluY2x1ZGVzKERJU0FCTEVEX0JBQ0tFTkRfUExVR0lOX01FU1NBR0UpKSB7XG4gICAgICAgICAgICAgIGNsaWVudE1lc3NhZ2UgPSBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFO1xuICAgICAgICAgICAgICBjbGllbnRBdHRyaWJ1dGVzRXJyb3IgPSBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFO1xuICAgICAgICAgIH0gZWxzZSBpZiAoZXJyLmJvZHkuZXJyb3IgJiYgdHlwZW9mIGVyci5ib2R5LmVycm9yID09PSAnb2JqZWN0JyAmJiBlcnIuYm9keS5lcnJvci5yZWFzb24gJiYgdHlwZW9mIGVyci5ib2R5LmVycm9yLnJlYXNvbiA9PT0gJ3N0cmluZycgJiYgZXJyLmJvZHkuZXJyb3IucmVhc29uLmluY2x1ZGVzKERJU0FCTEVEX0JBQ0tFTkRfUExVR0lOX01FU1NBR0UpKSB7XG4gICAgICAgICAgICAgIGNsaWVudE1lc3NhZ2UgPSBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFO1xuICAgICAgICAgICAgICBjbGllbnRBdHRyaWJ1dGVzRXJyb3IgPSBESVNBQkxFRF9CQUNLRU5EX1BMVUdJTl9NRVNTQUdFO1xuICAgICAgICAgIH1cbiAgICAgICAgICAvLyBGYWxsYmFjayBpZiBzcGVjaWZpYyBtZXNzYWdlIG5vdCBmb3VuZCBpbiBjb21wbGV4IGJvZHksIGJ1dCBib2R5IGhhcyBhIG1lc3NhZ2VcbiAgICAgICAgICBlbHNlIGlmIChlcnIuYm9keS5tZXNzYWdlICYmIHR5cGVvZiBlcnIuYm9keS5tZXNzYWdlID09PSAnc3RyaW5nJykge1xuICAgICAgICAgICAgICBjbGllbnRNZXNzYWdlID0gZXJyLmJvZHkubWVzc2FnZTtcbiAgICAgICAgICAgICAgY2xpZW50QXR0cmlidXRlc0Vycm9yID0gZXJyLmJvZHkubWVzc2FnZTtcbiAgICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIHJldHVybiByZXMuY3VzdG9tRXJyb3Ioe1xuICAgICAgICBzdGF0dXNDb2RlOiBlcnIuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBtZXNzYWdlOiBjbGllbnRNZXNzYWdlLCAvLyBVc2UgdGhlIGRldGVybWluZWQgbWVzc2FnZVxuICAgICAgICAgIGF0dHJpYnV0ZXM6IHtcbiAgICAgICAgICAgIGVycm9yOiBjbGllbnRBdHRyaWJ1dGVzRXJyb3IsIC8vIFVzZSB0aGUgZGV0ZXJtaW5lZCBhdHRyaWJ1dGVzIGVycm9yXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcbn07XG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLGFBQUEsR0FBQUMsT0FBQTtBQVFBLElBQUFDLE9BQUEsR0FBQUQsT0FBQTtBQWJBO0FBQ0E7QUFDQTtBQUNBOztBQVlPLFNBQVNFLDZCQUE2QkEsQ0FBQ0MsTUFBZSxFQUFRO0VBQ25FQSxNQUFNLENBQUNDLElBQUksQ0FDVDtJQUNFQyxJQUFJLEVBQUVDLHdCQUFnQixDQUFDQyxTQUFTO0lBQ2hDQyxRQUFRLEVBQUU7TUFDUkMsSUFBSSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDbEJDLElBQUksRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDckJDLFdBQVcsRUFBRUosb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDNUJFLFFBQVEsRUFBRUwsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDekJHLFlBQVksRUFBRU4sb0JBQU0sQ0FBQ08sTUFBTSxDQUFDO01BQzlCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREMsYUFBYSxDQUFDLE1BQU0sRUFBRUMsd0JBQWdCLENBQUNaLFNBQVMsQ0FDbEQsQ0FBQztFQUNESixNQUFNLENBQUNpQixHQUFHLENBQ1I7SUFDRWYsSUFBSSxFQUFFQyx3QkFBZ0IsQ0FBQ0MsU0FBUztJQUNoQ0MsUUFBUSxFQUFFO01BQ1JDLElBQUksRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCQyxJQUFJLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3JCQyxXQUFXLEVBQUVKLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQzVCRSxRQUFRLEVBQUVMLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3pCUSxlQUFlLEVBQUVYLG9CQUFNLENBQUNZLEtBQUssQ0FBQyxDQUM1Qlosb0JBQU0sQ0FBQ2EsT0FBTyxDQUNaYixvQkFBTSxDQUFDQyxNQUFNLENBQUM7VUFDWmEsU0FBUyxFQUFFZCxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztVQUMxQlksZUFBZSxFQUFFZixvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDL0MsQ0FBQyxDQUFDLEVBQ0Y7VUFBRWMsT0FBTyxFQUFFO1FBQUUsQ0FDZixDQUFDLEVBQ0RqQixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUNoQjtNQUNILENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREssYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUNaLFNBQVMsQ0FDakQsQ0FBQztFQUNESixNQUFNLENBQUN5QixHQUFHLENBQ1I7SUFDRXZCLElBQUksRUFBRUMsd0JBQWdCLENBQUNDLFNBQVM7SUFDaENDLFFBQVEsRUFBRTtFQUNaLENBQUMsRUFDRFUsYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUNaLFNBQVMsQ0FDakQsQ0FBQztFQUNESixNQUFNLENBQUMwQixNQUFNLENBQ1g7SUFDRXhCLElBQUksRUFBRyxHQUFFQyx3QkFBZ0IsQ0FBQ0MsU0FBVSxPQUFNO0lBQzFDQyxRQUFRLEVBQUU7TUFDUnNCLE1BQU0sRUFBRXBCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNwQm9CLEVBQUUsRUFBRXJCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUNwQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0RLLGFBQWEsQ0FBQyxRQUFRLEVBQUVDLHdCQUFnQixDQUFDWixTQUFTLENBQ3BELENBQUM7RUFDREosTUFBTSxDQUFDaUIsR0FBRyxDQUNSO0lBQ0VmLElBQUksRUFBRUMsd0JBQWdCLENBQUMwQixvQkFBb0I7SUFDM0N4QixRQUFRLEVBQUU7TUFDUkMsSUFBSSxFQUFFQyxvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDbEJDLElBQUksRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDckJvQixLQUFLLEVBQUV2QixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQztRQUN0QnFCLEtBQUssRUFBRXhCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3RCc0IsY0FBYyxFQUFFekIsb0JBQU0sQ0FBQ2dCLEtBQUssQ0FBQ2hCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO01BQzlDLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREssYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUNhLG9CQUFvQixDQUM1RCxDQUFDO0VBQ0Q3QixNQUFNLENBQUN5QixHQUFHLENBQ1I7SUFDRXZCLElBQUksRUFBRUMsd0JBQWdCLENBQUMwQixvQkFBb0I7SUFDM0N4QixRQUFRLEVBQUU7RUFDWixDQUFDLEVBQ0RVLGFBQWEsQ0FBQyxLQUFLLEVBQUVDLHdCQUFnQixDQUFDYSxvQkFBb0IsQ0FDNUQsQ0FBQztFQUNEN0IsTUFBTSxDQUFDMEIsTUFBTSxDQUNYO0lBQ0V4QixJQUFJLEVBQUcsR0FBRUMsd0JBQWdCLENBQUMwQixvQkFBcUIsT0FBTTtJQUNyRHhCLFFBQVEsRUFBRTtNQUNSc0IsTUFBTSxFQUFFcEIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCb0IsRUFBRSxFQUFFckIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREssYUFBYSxDQUFDLFFBQVEsRUFBRUMsd0JBQWdCLENBQUNhLG9CQUFvQixDQUMvRCxDQUFDO0VBQ0Q3QixNQUFNLENBQUNDLElBQUksQ0FDVDtJQUNFQyxJQUFJLEVBQUVDLHdCQUFnQixDQUFDOEIsV0FBVztJQUNsQzVCLFFBQVEsRUFBRTtNQUNSQyxJQUFJLEVBQUVDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNsQjBCLFVBQVUsRUFBRTNCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQzNCeUIsdUJBQXVCLEVBQUU1QixvQkFBTSxDQUFDYSxPQUFPLENBQUNiLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDeEQwQixJQUFJLEVBQUU3QixvQkFBTSxDQUFDTyxNQUFNLENBQUMsQ0FBQztRQUNyQnVCLElBQUksRUFBRTlCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3JCO1FBQ0E0QixZQUFZLEVBQUUvQixvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ2EsT0FBTyxDQUFDYixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUFDO01BQzVELENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREssYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUNpQixXQUFXLENBQ25ELENBQUM7RUFDRGpDLE1BQU0sQ0FBQ3lCLEdBQUcsQ0FDUjtJQUNFdkIsSUFBSSxFQUFFQyx3QkFBZ0IsQ0FBQzhCLFdBQVc7SUFDbEM1QixRQUFRLEVBQUU7RUFDWixDQUFDLEVBQ0RVLGFBQWEsQ0FBQyxLQUFLLEVBQUVDLHdCQUFnQixDQUFDaUIsV0FBVyxDQUNuRCxDQUFDO0VBQ0RqQyxNQUFNLENBQUN5QixHQUFHLENBQ1I7SUFDRXZCLElBQUksRUFBRyxHQUFFQyx3QkFBZ0IsQ0FBQzhCLFdBQVksT0FBTTtJQUM1QzVCLFFBQVEsRUFBRTtNQUNSc0IsTUFBTSxFQUFFcEIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ3BCb0IsRUFBRSxFQUFFckIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO01BQ3BCLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDREssYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUNpQixXQUFXLENBQ25ELENBQUM7RUFDRGpDLE1BQU0sQ0FBQ3lCLEdBQUcsQ0FDUjtJQUNFdkIsSUFBSSxFQUFHLEdBQUVDLHdCQUFnQixDQUFDMEIsb0JBQXFCLE9BQU07SUFDckR4QixRQUFRLEVBQUU7TUFDUnNCLE1BQU0sRUFBRXBCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNwQm9CLEVBQUUsRUFBRXJCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUNwQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0RLLGFBQWEsQ0FBQyxLQUFLLEVBQUVDLHdCQUFnQixDQUFDYSxvQkFBb0IsQ0FDNUQsQ0FBQztFQUNEN0IsTUFBTSxDQUFDeUIsR0FBRyxDQUNSO0lBQ0V2QixJQUFJLEVBQUcsR0FBRUMsd0JBQWdCLENBQUNDLFNBQVUsT0FBTTtJQUMxQ0MsUUFBUSxFQUFFO01BQ1JzQixNQUFNLEVBQUVwQixvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJvQixFQUFFLEVBQUVyQixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDcEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNESyxhQUFhLENBQUMsS0FBSyxFQUFFQyx3QkFBZ0IsQ0FBQ1osU0FBUyxDQUNqRCxDQUFDO0VBQ0RKLE1BQU0sQ0FBQzBCLE1BQU0sQ0FDWDtJQUNFeEIsSUFBSSxFQUFHLEdBQUVDLHdCQUFnQixDQUFDOEIsV0FBWSxPQUFNO0lBQzVDNUIsUUFBUSxFQUFFO01BQ1JzQixNQUFNLEVBQUVwQixvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJvQixFQUFFLEVBQUVyQixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDcEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNESyxhQUFhLENBQUMsUUFBUSxFQUFFQyx3QkFBZ0IsQ0FBQ2lCLFdBQVcsQ0FDdEQsQ0FBQztFQUNEakMsTUFBTSxDQUFDaUIsR0FBRyxDQUNSO0lBQ0VmLElBQUksRUFBRUMsd0JBQWdCLENBQUNvQyxTQUFTO0lBQ2hDbEMsUUFBUSxFQUFFO01BQ1JDLElBQUksRUFBRUMsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO1FBQ2xCQyxJQUFJLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDO1FBQ3JCMkIsSUFBSSxFQUFFOUIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7UUFDckJ3QixVQUFVLEVBQUUzQixvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN6Q3lCLHVCQUF1QixFQUFFNUIsb0JBQU0sQ0FBQ2dCLEtBQUssQ0FBQ2hCLG9CQUFNLENBQUNhLE9BQU8sQ0FBQ2Isb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RFMEIsSUFBSSxFQUFFN0Isb0JBQU0sQ0FBQ2dCLEtBQUssQ0FBQ2hCLG9CQUFNLENBQUNPLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDbkMwQixPQUFPLEVBQUVqQyxvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN0QytCLFVBQVUsRUFBRWxDLG9CQUFNLENBQUNnQixLQUFLLENBQUNoQixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3pDZ0MsYUFBYSxFQUFFbkMsb0JBQU0sQ0FBQ2dCLEtBQUssQ0FBQ2hCLG9CQUFNLENBQUNvQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQzdDQyxhQUFhLEVBQUVyQyxvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ2EsT0FBTyxDQUFDYixvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNURtQyxVQUFVLEVBQUV0QyxvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN6Q29DLE9BQU8sRUFBRXZDLG9CQUFNLENBQUNnQixLQUFLLENBQUNoQixvQkFBTSxDQUFDTyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3RDaUMsU0FBUyxFQUFFeEMsb0JBQU0sQ0FBQ2dCLEtBQUssQ0FBQ2hCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDeENzQyxPQUFPLEVBQUV6QyxvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUM7TUFDdkMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNESyxhQUFhLENBQUMsS0FBSyxFQUFFQyx3QkFBZ0IsQ0FBQ3VCLFNBQVMsQ0FDakQsQ0FBQztFQUNEdkMsTUFBTSxDQUFDeUIsR0FBRyxDQUNSO0lBQ0V2QixJQUFJLEVBQUVDLHdCQUFnQixDQUFDb0MsU0FBUztJQUNoQ2xDLFFBQVEsRUFBRTtFQUNaLENBQUMsRUFDRFUsYUFBYSxDQUFDLEtBQUssRUFBRUMsd0JBQWdCLENBQUN1QixTQUFTLENBQ2pELENBQUM7RUFDRHZDLE1BQU0sQ0FBQ3lCLEdBQUcsQ0FDUjtJQUNFdkIsSUFBSSxFQUFHLEdBQUVDLHdCQUFnQixDQUFDb0MsU0FBVSxPQUFNO0lBQzFDbEMsUUFBUSxFQUFFO01BQ1JzQixNQUFNLEVBQUVwQixvQkFBTSxDQUFDQyxNQUFNLENBQUM7UUFDcEJvQixFQUFFLEVBQUVyQixvQkFBTSxDQUFDRyxNQUFNLENBQUM7TUFDcEIsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNESyxhQUFhLENBQUMsS0FBSyxFQUFFQyx3QkFBZ0IsQ0FBQ3VCLFNBQVMsQ0FDakQsQ0FBQztFQUNEdkMsTUFBTSxDQUFDMEIsTUFBTSxDQUNYO0lBQ0V4QixJQUFJLEVBQUcsR0FBRUMsd0JBQWdCLENBQUNvQyxTQUFVLE9BQU07SUFDMUNsQyxRQUFRLEVBQUU7TUFDUnNCLE1BQU0sRUFBRXBCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztRQUNwQm9CLEVBQUUsRUFBRXJCLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztNQUNwQixDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0RLLGFBQWEsQ0FBQyxRQUFRLEVBQUVDLHdCQUFnQixDQUFDdUIsU0FBUyxDQUNwRCxDQUFDO0FBQ0g7QUFFQSxNQUFNeEIsYUFBYSxHQUFHQSxDQUFDa0MsTUFBTSxFQUFFL0MsSUFBSSxLQUFLO0VBQ3RDLE9BQU8sT0FDTGdELE9BQThCLEVBQzlCQyxHQUFnQyxFQUNoQ0MsR0FBd0MsS0FDUTtJQUNoRCxNQUFNQyxZQUFZLEdBQUdGLEdBQUcsQ0FBQ3BCLEtBQUssQ0FBQ3VCLFdBQVc7SUFDMUMsTUFBTUMsTUFBTSxHQUFHRixZQUFZLEdBQ3ZCSCxPQUFPLENBQUNNLFVBQVUsQ0FBQ0MsVUFBVSxDQUFDQyxNQUFNLENBQUNDLFNBQVMsQ0FBQ04sWUFBWSxDQUFDLENBQUNPLE9BQU8sR0FDcEVWLE9BQU8sQ0FBQ1csSUFBSSxDQUFDSixVQUFVLENBQUNDLE1BQU0sQ0FBQ0ksTUFBTSxDQUFDQyxpQkFBaUI7SUFFM0QsSUFBSTtNQUNGLElBQUlDLFFBQVE7TUFDWixJQUFJZixNQUFNLEtBQUssUUFBUSxFQUFFO1FBQ3ZCLE1BQU07VUFBRXJCO1FBQUcsQ0FBQyxHQUFHdUIsR0FBRyxDQUFDeEIsTUFBTTtRQUN6QixNQUFNc0MsVUFBVSxHQUFJLEdBQUUvRCxJQUFLLElBQUcwQixFQUFHLEVBQUM7UUFDbENvQyxRQUFRLEdBQUcsTUFBTVQsTUFBTSxDQUFDLG1CQUFtQixFQUFFO1VBQzNDTixNQUFNO1VBQ04vQyxJQUFJLEVBQUUrRDtRQUNSLENBQUMsQ0FBQztNQUNKLENBQUMsTUFBTSxJQUFJaEIsTUFBTSxLQUFLLEtBQUssSUFBSUUsR0FBRyxDQUFDeEIsTUFBTSxDQUFDQyxFQUFFLEVBQUU7UUFDNUM7UUFDQSxNQUFNc0MsT0FBTyxHQUFJLEdBQUVoRSxJQUFLLElBQUdpRCxHQUFHLENBQUN4QixNQUFNLENBQUNDLEVBQUcsRUFBQztRQUMxQ29DLFFBQVEsR0FBRyxNQUFNVCxNQUFNLENBQUMsbUJBQW1CLEVBQUU7VUFDM0NOLE1BQU07VUFDTi9DLElBQUksRUFBRWdFO1FBQ1IsQ0FBQyxDQUFDO01BQ0osQ0FBQyxNQUFNO1FBQ0w7UUFDQUYsUUFBUSxHQUFHLE1BQU1ULE1BQU0sQ0FBQyxtQkFBbUIsRUFBRTtVQUMzQ04sTUFBTTtVQUNOL0MsSUFBSTtVQUNKLElBQUkrQyxNQUFNLEtBQUssTUFBTSxJQUFJQSxNQUFNLEtBQUssS0FBSyxHQUFHO1lBQUUzQyxJQUFJLEVBQUU2QyxHQUFHLENBQUM3QztVQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDckUsQ0FBQyxDQUFDO01BQ0o7TUFFQSxPQUFPOEMsR0FBRyxDQUFDZSxFQUFFLENBQUM7UUFBRTdELElBQUksRUFBRTBEO01BQVMsQ0FBQyxDQUFDO0lBQ25DLENBQUMsQ0FBQyxPQUFPSSxHQUFHLEVBQUU7TUFBQSxJQUFBQyxTQUFBO01BRVpDLE9BQU8sQ0FBQ0MsS0FBSyxDQUFDLHNDQUFzQyxFQUFFSCxHQUFHLENBQUMsQ0FBQyxDQUFDOztNQUU1RCxJQUFJSSxhQUFhLEdBQUdKLEdBQUcsQ0FBQ0ssT0FBTyxDQUFDLENBQUM7TUFDakMsSUFBSUMscUJBQXFCLEdBQUcsRUFBQUwsU0FBQSxHQUFBRCxHQUFHLENBQUM5RCxJQUFJLGNBQUErRCxTQUFBLHVCQUFSQSxTQUFBLENBQVVFLEtBQUssS0FBSUgsR0FBRyxDQUFDSyxPQUFPLENBQUMsQ0FBQzs7TUFFNUQ7TUFDQSxJQUFJTCxHQUFHLENBQUM5RCxJQUFJLElBQUksT0FBTzhELEdBQUcsQ0FBQzlELElBQUksS0FBSyxRQUFRLElBQUk4RCxHQUFHLENBQUM5RCxJQUFJLENBQUNxRSxRQUFRLENBQUNDLHVDQUErQixDQUFDLEVBQUU7UUFDaEdKLGFBQWEsR0FBR0ksdUNBQStCO1FBQy9DRixxQkFBcUIsR0FBR0UsdUNBQStCO01BQzNEO01BQ0E7TUFBQSxLQUNLLElBQUlSLEdBQUcsQ0FBQzlELElBQUksSUFBSSxPQUFPOEQsR0FBRyxDQUFDOUQsSUFBSSxLQUFLLFFBQVEsRUFBRTtRQUMvQztRQUNBLElBQUk4RCxHQUFHLENBQUM5RCxJQUFJLENBQUNtRSxPQUFPLElBQUksT0FBT0wsR0FBRyxDQUFDOUQsSUFBSSxDQUFDbUUsT0FBTyxLQUFLLFFBQVEsSUFBSUwsR0FBRyxDQUFDOUQsSUFBSSxDQUFDbUUsT0FBTyxDQUFDRSxRQUFRLENBQUNDLHVDQUErQixDQUFDLEVBQUU7VUFDeEhKLGFBQWEsR0FBR0ksdUNBQStCO1VBQy9DRixxQkFBcUIsR0FBR0UsdUNBQStCO1FBQzNELENBQUMsTUFBTSxJQUFJUixHQUFHLENBQUM5RCxJQUFJLENBQUN1RSxNQUFNLElBQUksT0FBT1QsR0FBRyxDQUFDOUQsSUFBSSxDQUFDdUUsTUFBTSxLQUFLLFFBQVEsSUFBSVQsR0FBRyxDQUFDOUQsSUFBSSxDQUFDdUUsTUFBTSxDQUFDRixRQUFRLENBQUNDLHVDQUErQixDQUFDLEVBQUU7VUFDNUhKLGFBQWEsR0FBR0ksdUNBQStCO1VBQy9DRixxQkFBcUIsR0FBR0UsdUNBQStCO1FBQzNELENBQUMsTUFBTSxJQUFJUixHQUFHLENBQUM5RCxJQUFJLENBQUNpRSxLQUFLLElBQUksT0FBT0gsR0FBRyxDQUFDOUQsSUFBSSxDQUFDaUUsS0FBSyxLQUFLLFFBQVEsSUFBSUgsR0FBRyxDQUFDOUQsSUFBSSxDQUFDaUUsS0FBSyxDQUFDTSxNQUFNLElBQUksT0FBT1QsR0FBRyxDQUFDOUQsSUFBSSxDQUFDaUUsS0FBSyxDQUFDTSxNQUFNLEtBQUssUUFBUSxJQUFJVCxHQUFHLENBQUM5RCxJQUFJLENBQUNpRSxLQUFLLENBQUNNLE1BQU0sQ0FBQ0YsUUFBUSxDQUFDQyx1Q0FBK0IsQ0FBQyxFQUFFO1VBQ3RNSixhQUFhLEdBQUdJLHVDQUErQjtVQUMvQ0YscUJBQXFCLEdBQUdFLHVDQUErQjtRQUMzRDtRQUNBO1FBQUEsS0FDSyxJQUFJUixHQUFHLENBQUM5RCxJQUFJLENBQUNtRSxPQUFPLElBQUksT0FBT0wsR0FBRyxDQUFDOUQsSUFBSSxDQUFDbUUsT0FBTyxLQUFLLFFBQVEsRUFBRTtVQUMvREQsYUFBYSxHQUFHSixHQUFHLENBQUM5RCxJQUFJLENBQUNtRSxPQUFPO1VBQ2hDQyxxQkFBcUIsR0FBR04sR0FBRyxDQUFDOUQsSUFBSSxDQUFDbUUsT0FBTztRQUM1QztNQUNKO01BRUEsT0FBT3JCLEdBQUcsQ0FBQzBCLFdBQVcsQ0FBQztRQUNyQkMsVUFBVSxFQUFFWCxHQUFHLENBQUNXLFVBQVUsSUFBSSxHQUFHO1FBQ2pDekUsSUFBSSxFQUFFO1VBQ0ptRSxPQUFPLEVBQUVELGFBQWE7VUFBRTtVQUN4QlEsVUFBVSxFQUFFO1lBQ1ZULEtBQUssRUFBRUcscUJBQXFCLENBQUU7VUFDaEM7UUFDRjtNQUNGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FBQztBQUNILENBQUMifQ==