"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSampleData = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _readline = _interopRequireDefault(require("readline"));
var _fs = _interopRequireDefault(require("fs"));
var _zlib = require("zlib");
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

//@ts-ignore

const BULK_INSERT_SIZE = 500;
const loadSampleData = (filePath, indexName, client, request, context, dataSourceEnabled) => {
  const {
    dataSourceId = ''
  } = request.params;
  return new Promise((resolve, reject) => {
    let count = 0;
    let docs = [];
    let isPaused = false;
    let offset = 0;
    const startTime = (0, _moment.default)(new Date().getTime()).subtract({
      days: 7
    });

    // Create the read stream for the file. Set a smaller buffer size here to prevent it from
    // getting too large, to prevent inserting too many docs at once into the index.
    const readStream = _fs.default.createReadStream(filePath, {
      highWaterMark: 1024 * 4
    });
    const lineStream = _readline.default.createInterface({
      input: readStream.pipe((0, _zlib.createUnzip)())
    });

    // This is only ran when the end of lineStream closes normally. It is used to
    // bulk insert the final batch of lines that are < BULK_INSERT_SIZE
    const onClose = async () => {
      if (docs.length > 0) {
        try {
          await bulkInsert(docs);
        } catch (err) {
          reject(err);
          return;
        }
      }
      resolve(count);
    };
    lineStream.on('close', onClose);
    lineStream.on('pause', async () => {
      isPaused = true;
    });
    lineStream.on('resume', async () => {
      isPaused = false;
    });
    lineStream.on('line', async doc => {
      // for the initial doc, get the timestamp to properly set an offset
      if (count === 0) {
        const initialTime = (0, _moment.default)(JSON.parse(doc).timestamp);
        offset = startTime.diff(initialTime);
      }
      count++;
      docs.push(doc);

      // If not currently paused: pause the stream to prevent concurrent bulk inserts
      // on the cluster which could cause performance issues.
      // Also, empty the current docs[] before performing the bulk insert to prevent
      // buffered docs from being dropped.
      if (docs.length >= BULK_INSERT_SIZE && !isPaused) {
        lineStream.pause();

        // save the docs to insert, and empty out current docs list
        const docsToInsert = docs.slice();
        docs = [];
        try {
          await bulkInsert(docsToInsert);
          lineStream.resume();
        } catch (err) {
          lineStream.removeListener('close', onClose);
          lineStream.close();
          reject(err);
        }
      }
    });
    const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, dataSourceEnabled, request, dataSourceId, client);
    const bulkInsert = async docs => {
      try {
        const bulkBody = prepareBody(docs, offset);
        const resp = await callWithRequest('bulk', {
          body: bulkBody
        });
        if (resp.errors) {
          const errorItems = resp.items;
          const firstErrorReason = (0, _lodash.isEmpty)(errorItems) ? 'Error while bulk inserting. Please try again.' : (0, _helpers.prettifyErrorMessage)(errorItems[0].index.error.reason);
          return Promise.reject(new Error(firstErrorReason));
        }
      } catch (err) {
        console.log('Error while bulk inserting. ', err);
        return Promise.reject(new Error('Error while bulk inserting. Please try again.'));
      }
    };
    const prepareBody = (docs, offset) => {
      const bulkBody = [];
      let docIdCount = count - docs.length;
      docs.forEach(doc => {
        bulkBody.push(getDocDetails(docIdCount));
        bulkBody.push(updateTimestamp(doc, offset));
        docIdCount++;
      });
      return bulkBody;
    };
    const updateTimestamp = (doc, offset) => {
      let docAsJSON = JSON.parse(doc);
      const updatedTimestamp = docAsJSON.timestamp + offset;
      docAsJSON.timestamp = updatedTimestamp;
      return docAsJSON;
    };
    const getDocDetails = docIdCount => {
      return `{ "index": { "_index": "${indexName}", "_id": ${docIdCount} } }`;
    };
  });
};
exports.loadSampleData = loadSampleData;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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