/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <unistd.h>
#include <jni.h>

#include <sal/types.h>
#include <vcl/event.hxx>
#include <android/log.h>

#include <osl/detail/android-bootstrap.h>

#include <LibreOfficeKit/LibreOfficeKit.h>

/* LibreOfficeKit */

namespace
{

jfieldID getHandleField(JNIEnv* pEnv, jobject aObject)
{
    jclass clazz = pEnv->GetObjectClass(aObject);
    return pEnv->GetFieldID(clazz, "handle", "Ljava/nio/ByteBuffer;");
}

template <typename T>
T* getHandle(JNIEnv* pEnv, jobject aObject)
{
    jobject aHandle = pEnv->GetObjectField(aObject, getHandleField(pEnv, aObject));
    return reinterpret_cast<T*>(pEnv->GetDirectBufferAddress(aHandle));
}

const char* copyJavaString(JNIEnv* pEnv, jstring aJavaString)
{
    const char* pTemp = pEnv->GetStringUTFChars(aJavaString, NULL);
    const char* pClone = strdup(pTemp);
    pEnv->ReleaseStringUTFChars(aJavaString, pTemp);

    return pClone;
}

} // anonymous namespace

extern "C" SAL_JNI_EXPORT jstring JNICALL Java_org_libreoffice_kit_Office_getError
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);
    char* pError = pLibreOfficeKit->pClass->getError(pLibreOfficeKit);
    return pEnv->NewStringUTF(pError);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Office_destroy
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);
    pLibreOfficeKit->pClass->destroy(pLibreOfficeKit);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Office_destroyAndExit(JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);
    pLibreOfficeKit->pClass->destroy(pLibreOfficeKit);
    // Stopgap fix: _exit() to force the OS to restart the LO activity.
    // Better than to hang.
    _exit(0);
}

namespace
{

struct CallbackData
{
    jmethodID aJavaCallbackMethod;
    jclass aClass;
    jobject aObject;
};

static CallbackData gCallbackData;
static CallbackData gCallbackDataLOKit;

/**
 * Handle retrieved callback
 */
void messageCallback(int nType, const char* pPayload, void* pData)
{
    CallbackData* pCallbackData = (CallbackData*) pData;

    JavaVM* pJavaVM = lo_get_javavm();
    JNIEnv* pEnv;
    bool bIsAttached = false;

    int status = pJavaVM->GetEnv((void **) &pEnv, JNI_VERSION_1_6);

    if(status < 0)
    {
        status = pJavaVM->AttachCurrentThread(&pEnv, NULL);
        if(status < 0)
        {
            return;
        }
        bIsAttached = true;
    }

    jstring sPayload = pEnv->NewStringUTF(pPayload);

    jvalue aParameter[2];
    aParameter[0].i = nType;
    aParameter[1].l = sPayload;

    pEnv->CallVoidMethodA(pCallbackData->aObject, pCallbackData->aJavaCallbackMethod, aParameter);

    pEnv->DeleteLocalRef(sPayload);

    if (bIsAttached)
    {
        pJavaVM->DetachCurrentThread();
    }
}

} // anonymous namespace

extern "C" SAL_JNI_EXPORT jobject JNICALL Java_org_libreoffice_kit_Office_documentLoadNative
    (JNIEnv* pEnv, jobject aObject, jstring documentPath)
{
    const char* aCloneDocumentPath = copyJavaString(pEnv, documentPath);
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);

    LibreOfficeKitDocument* pDocument = pLibreOfficeKit->pClass->documentLoad(pLibreOfficeKit, aCloneDocumentPath);

    if (pDocument == NULL)
        return NULL;

    jobject aHandle = pEnv->NewDirectByteBuffer((void*) pDocument, sizeof(LibreOfficeKitDocument));

    return aHandle;
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Office_setDocumentPassword
    (JNIEnv* pEnv, jobject aObject, jstring sUrl, jstring sPassword)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);

    char const* pUrl = copyJavaString(pEnv, sUrl);
    if (sPassword == NULL) {
        pLibreOfficeKit->pClass->setDocumentPassword(pLibreOfficeKit, pUrl, nullptr);
    } else {
        char const* pPassword = copyJavaString(pEnv, sPassword);
        pLibreOfficeKit->pClass->setDocumentPassword(pLibreOfficeKit, pUrl, pPassword);
    }
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Office_setOptionalFeatures
    (JNIEnv* pEnv, jobject aObject, jlong options)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);

    unsigned long long pOptions = (unsigned long long)options;

    pLibreOfficeKit->pClass->setOptionalFeatures(pLibreOfficeKit, pOptions);
}

/** Implementation of org.libreoffice.kit.Office.bindMessageCallback method */
extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Office_bindMessageCallback
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKit* pLibreOfficeKit = getHandle<LibreOfficeKit>(pEnv, aObject);

    gCallbackDataLOKit.aObject = (jobject) pEnv->NewGlobalRef(aObject);
    jclass aClass = pEnv->GetObjectClass(aObject);
    gCallbackDataLOKit.aClass = (jclass) pEnv->NewGlobalRef(aClass);

    gCallbackDataLOKit.aJavaCallbackMethod = pEnv->GetMethodID(aClass, "messageRetrievedLOKit", "(ILjava/lang/String;)V");

    pLibreOfficeKit->pClass->registerCallback(pLibreOfficeKit, messageCallback, (void*) &gCallbackDataLOKit);
}

/* Document */

/** Implementation of org.libreoffice.kit.Document.bindMessageCallback method */
extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_bindMessageCallback
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    gCallbackData.aObject = (jobject) pEnv->NewGlobalRef(aObject);
    jclass aClass = pEnv->GetObjectClass(aObject);
    gCallbackData.aClass = (jclass) pEnv->NewGlobalRef(aClass);

    gCallbackData.aJavaCallbackMethod = pEnv->GetMethodID(aClass, "messageRetrieved", "(ILjava/lang/String;)V");

    pDocument->pClass->registerCallback(pDocument, messageCallback, (void*) &gCallbackData);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_destroy
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->destroy(pDocument);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_setPart
    (JNIEnv* pEnv, jobject aObject, jint aPart)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->setPart(pDocument, aPart);
}

extern "C" SAL_JNI_EXPORT jint JNICALL Java_org_libreoffice_kit_Document_getPart
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    return (jint) pDocument->pClass->getPart(pDocument);
}

extern "C" SAL_JNI_EXPORT jstring JNICALL Java_org_libreoffice_kit_Document_getPartPageRectangles
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    char* pRectangles = pDocument->pClass->getPartPageRectangles(pDocument);
    return pEnv->NewStringUTF(pRectangles);
}

extern "C" SAL_JNI_EXPORT jint JNICALL Java_org_libreoffice_kit_Document_getParts
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    return (jint) pDocument->pClass->getParts(pDocument);
}

extern "C" SAL_JNI_EXPORT jstring JNICALL Java_org_libreoffice_kit_Document_getPartName
    (JNIEnv* pEnv, jobject aObject, jint nPartIndex)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    char* pPartName = pDocument->pClass->getPartName(pDocument, nPartIndex);
    return pEnv->NewStringUTF(pPartName);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_setPartMode
    (JNIEnv* pEnv, jobject aObject, jint nPartMode)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    pDocument->pClass->setPartMode(pDocument, nPartMode);
}

extern "C" SAL_JNI_EXPORT jint JNICALL Java_org_libreoffice_kit_Document_getDocumentTypeNative
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    return (jint) pDocument->pClass->getDocumentType(pDocument);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_paintTileNative
    (JNIEnv* pEnv, jobject aObject, jobject aByteBuffer,
    jint nCanvasWidth, jint nCanvasHeight, jint nTilePosX, jint nTilePosY,
    jint nTileWidth, jint nTileHeight)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    unsigned char* buffer = (unsigned char*) pEnv->GetDirectBufferAddress(aByteBuffer);
    pDocument->pClass->paintTile(pDocument, buffer, nCanvasWidth, nCanvasHeight, nTilePosX, nTilePosY, nTileWidth, nTileHeight);
}

extern "C" SAL_JNI_EXPORT jlong JNICALL Java_org_libreoffice_kit_Document_getDocumentHeight
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    long nWidth;
    long nHeight;
    pDocument->pClass->getDocumentSize(pDocument, &nWidth, &nHeight);
    return nHeight;
}

extern "C" SAL_JNI_EXPORT jlong JNICALL Java_org_libreoffice_kit_Document_getDocumentWidth
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    long nWidth;
    long nHeight;
    pDocument->pClass->getDocumentSize(pDocument, &nWidth, &nHeight);
    return nWidth;
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_initializeForRendering
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->initializeForRendering(pDocument, NULL);
}

extern "C" SAL_JNI_EXPORT jint JNICALL Java_org_libreoffice_kit_Document_saveAs
    (JNIEnv* pEnv, jobject aObject, jstring sUrl, jstring sFormat, jstring sOptions)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    const char* pUrl = pEnv->GetStringUTFChars(sUrl, NULL);
    const char* pFormat = pEnv->GetStringUTFChars(sFormat, NULL);
    const char* pOptions = pEnv->GetStringUTFChars(sOptions, NULL);

    int result = pDocument->pClass->saveAs(pDocument, pUrl, pFormat, pOptions);

    pEnv->ReleaseStringUTFChars(sUrl, pUrl);
    pEnv->ReleaseStringUTFChars(sFormat, pFormat);
    pEnv->ReleaseStringUTFChars(sOptions, pOptions);

    return result;
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_postKeyEvent
    (JNIEnv* pEnv, jobject aObject, jint nType, jint nCharCode, jint nKeyCode)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->postKeyEvent(pDocument, nType, nCharCode, nKeyCode);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_postMouseEvent
    (JNIEnv* pEnv, jobject aObject, jint type, jint x, jint y, jint count, jint button, jint modifier)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->postMouseEvent(pDocument, type, x, y, count, button, modifier);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_postUnoCommand
    (JNIEnv* pEnv, jobject aObject, jstring command, jstring arguments, jboolean bNotifyWhenFinished)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    const char* pCommand = pEnv->GetStringUTFChars(command, NULL);
    const char* pArguments = nullptr;
    if (arguments != NULL)
        pArguments = pEnv->GetStringUTFChars(arguments, NULL);

    pDocument->pClass->postUnoCommand(pDocument, pCommand, pArguments, bNotifyWhenFinished);

    pEnv->ReleaseStringUTFChars(command, pCommand);
    if (arguments != NULL)
        pEnv->ReleaseStringUTFChars(arguments, pArguments);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_setTextSelection
    (JNIEnv* pEnv, jobject aObject, jint type, jint x, jint y)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->setTextSelection(pDocument, type, x, y);
}

extern "C" SAL_JNI_EXPORT jstring JNICALL Java_org_libreoffice_kit_Document_getTextSelection
    (JNIEnv* pEnv, jobject aObject, jstring mimeType)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    const char* pMimeType = pEnv->GetStringUTFChars(mimeType, NULL);

    char* pUsedMimeType = 0;
    LibreOfficeKitDocumentClass* pcls = pDocument->pClass;
    char* pSelection = pcls->getTextSelection(pDocument, pMimeType, &pUsedMimeType);
    free(pUsedMimeType);

    pEnv->ReleaseStringUTFChars(mimeType, pMimeType);

    return pEnv->NewStringUTF(pSelection);
}

extern "C" SAL_JNI_EXPORT jboolean JNICALL Java_org_libreoffice_kit_Document_paste
    (JNIEnv* pEnv, jobject aObject, jstring mimeType, jstring data)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    const char* pMimeType = pEnv->GetStringUTFChars(mimeType, NULL);
    const char* pData = pEnv->GetStringUTFChars(data, NULL);
    const size_t nSize = pEnv->GetStringLength(data);

    LibreOfficeKitDocumentClass* pcls = pDocument->pClass;
    bool result = pcls->paste(pDocument, pMimeType, pData, nSize);
    pEnv->ReleaseStringUTFChars(mimeType, pMimeType);
    pEnv->ReleaseStringUTFChars(data, pData);

    return result;
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_setGraphicSelection
    (JNIEnv* pEnv, jobject aObject, jint type, jint x, jint y)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->setGraphicSelection(pDocument, type, x, y);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_resetSelection
    (JNIEnv* pEnv, jobject aObject)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->resetSelection(pDocument);
}

extern "C" SAL_JNI_EXPORT jstring JNICALL Java_org_libreoffice_kit_Document_getCommandValues
    (JNIEnv* pEnv, jobject aObject, jstring command)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);

    const char* pCommand = pEnv->GetStringUTFChars(command, NULL);

    char* pValue = pDocument->pClass->getCommandValues(pDocument, pCommand);

    pEnv->ReleaseStringUTFChars(command, pCommand);

    return pEnv->NewStringUTF(pValue);
}

extern "C" SAL_JNI_EXPORT void JNICALL Java_org_libreoffice_kit_Document_setClientZoom
    (JNIEnv* pEnv, jobject aObject, jint nTilePixelWidth, jint nTilePixelHeight, jint nTileTwipWidth, jint nTileTwipHeight)
{
    LibreOfficeKitDocument* pDocument = getHandle<LibreOfficeKitDocument>(pEnv, aObject);
    pDocument->pClass->setClientZoom(pDocument, nTilePixelWidth, nTilePixelHeight, nTileTwipWidth, nTileTwipHeight);

}
/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
