// Copyright 2017 Canonical Ltd.
// Licensed under the AGPLv3, see LICENCE file for details.

package testing

import (
	"math/rand"
	"time"
)

// NewCA returns a random one of the pre-generated certs to speed up
// tests. The comment on the certs are not going to match the args.
func NewCA(commonName, UUID string, expiry time.Time) (certPEM, keyPEM string, err error) {
	index := rand.Intn(len(certs))
	cert := certs[index]
	return cert.certPEM, cert.keyPEM, nil
}

// certs created 2017-01-13 with 10 year expiry, and testing.UUID
var certs = []struct {
	certPEM string
	keyPEM  string
}{
	{
		certPEM: `-----BEGIN CERTIFICATE-----
MIID1DCCArygAwIBAgIVAJYDNzOF4G4euymyNwEBcl+5EtESMA0GCSqGSIb3DQEB
CwUAMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVkIENB
IGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFiYWQt
NTAwZC05MDAwLTRiMWQwZDA2ZjAwZDAeFw0xNzAxMDYwMDQwMjBaFw0yNzAxMTMw
MDQwMTlaMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVk
IENBIGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFi
YWQtNTAwZC05MDAwLTRiMWQwZDA2ZjAwZDCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBALvqzQXnpOpYYatsysFxb5fAjD67dKb0MjI9gSfzbGT2qFqSjO3U
+/ujuQcUfJmGWhL+f9beAVm2vCvuNBvp5Om1yrZn+EC9Ssvr4yIw75/Nrz6CoHLb
byJ9ztMi327cyn2/n6urVMwIRZclZ5dgp6vr3EaU9VNynnZaP6JKvUyF/iJeXxs6
sfkeXchFbRho81GgHBqJ+AZ52CE9rDypw5kWdNRzLYzAgDbRTrKDD+swcClop3Uj
dlwOzbE/cy9VAck7XEOEHwp7Ues5BAMjFZxpGFYY357bS95CQU5EhtkbEaq6Nkfb
IgHFi7/NK2jnQ4J5rSl0TrYSs9Q0HpVKQzECAwEAAaNjMGEwDgYDVR0PAQH/BAQD
AgKkMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFKllixO1JCd/3W0wi7lFUWfN
S5ykMB8GA1UdIwQYMBaAFKllixO1JCd/3W0wi7lFUWfNS5ykMA0GCSqGSIb3DQEB
CwUAA4IBAQBEA89ZvuVgwGFVUO8yEixqkwa6/ZoFrsXpF5855pNg7a0AzKypQmrg
S14ZsIYE9s8jd6A+t7LYwEDuxLBj4DB5i+q0qDHH31p6s4Nco/qv3bRTq1o/0c5e
y+yYKj26IBasy9wTEvAmuMPpuv0fSCXxFyiIpSwSrBNFQjsfnuc8t7VO0NX3azqW
FRGuZrAvwl6wQZjqoI/a5Nk8lYy1maFZPsXs8dCzBealKBtyIR1i9UOFHrakxMTd
gvGz29JGYx9UtLEMBMAqQq76pS3w/MxEryj87J0opeoLXtETu7SBvGV18Q3ne8Vl
f3hgwIGnKdk6MBOlK5LisNhvXP/+dJm/
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAu+rNBeek6lhhq2zKwXFvl8CMPrt0pvQyMj2BJ/NsZPaoWpKM
7dT7+6O5BxR8mYZaEv5/1t4BWba8K+40G+nk6bXKtmf4QL1Ky+vjIjDvn82vPoKg
cttvIn3O0yLfbtzKfb+fq6tUzAhFlyVnl2Cnq+vcRpT1U3Kedlo/okq9TIX+Il5f
Gzqx+R5dyEVtGGjzUaAcGon4BnnYIT2sPKnDmRZ01HMtjMCANtFOsoMP6zBwKWin
dSN2XA7NsT9zL1UByTtcQ4QfCntR6zkEAyMVnGkYVhjfnttL3kJBTkSG2RsRqro2
R9siAcWLv80raOdDgnmtKXROthKz1DQelUpDMQIDAQABAoIBAESA9Rqo9lqju381
mrhoNOkaPUYFKiJOtRnRiArbV3F65aaW9B8jXHTAm4v0MPP6oU15T3sx1UYyMmjF
XmOqoVN9fRd/0WUwXFr4dAWb107nOXJNRlZ4ngvx1vrrJI6SmJK8z2VSgQgcH9pY
vWnWx2kKA3QLi19PUmsMvVPZcSa6th6JSGnFqJNqSHAUeUSi7IzKHep+gMrOqo45
eNYJkgHo/IIqorDClQKrnqs3/hMb3HB9gBM3CM2F74QESAtJZrEkYMmxcUfWlA/J
Z2V+MHa49KQ9F4FcAaWD/zmcfErqNAxCYo3tJys9MsnFc//WxHNG7wuqfTo4S0dD
vU3xeB0CgYEA2jU91kKGA4lomXV97P4Yw68JcjA00CZGUfduXndTTw4aR6ahUOV9
XIimSSAP4Z6ZIKKW+Y3KZpyf93jO9p90VwOdkV+fgkOGYIvv3InjLIPM0JGLM+Ha
8rhPiBLuscHlAYw9PdwC/Amhw2ktThXJvFO5kMr7dfmApSKvUW9pswsCgYEA3HaM
Aod7mNraqU5dk1toJMBb+qCeDJHeBcX8RoAaAnOdcA/8m18PNyA/WFmIoTsSej6x
3sIJMzApOsYOLvdZzdvcU/XAxRiQ+EpXz5DvWmaoDXatXpq/6ric2ooK4THYHIBF
ir8bCEtZ4k1MrxL8kIlgaKJkDJ+UeUXZ1QbxyDMCgYAq2QuUeCmipifRAn5AvECW
pFdhgc/NE8b5L3IPwLSuMGG+shy5Kucr9XVZpJ8Zua/t9Z9Zz5rv/U94I9PYH8WT
lJUrR7ClvqazbKckEjt2sUbVqc7A2FfLE6AiYz1V0HIslKIt9HbtkOKf4ZZQ3/xm
K7ajlQg9yUmWo7HzdsI1swKBgHVpcUF7lDig/elDQhTTjpBcHeBJoELDgMIEYknN
uYxqtINVEExTARBzzSipblmn0zxlnWcsYafwU1To2+7pbtty51GTxVlyrup89W0n
HEJB/+hDtyt2MRW5wlelQwwR9q2fJuowwx3Uu4cJcfifnQEBGOZA7hXG+1Tm/Y81
hpj/AoGACj/W9/Bn5v35DRBw9NGzru5l54JkV30RC9t0xiXi5b86AuIGjqm4kRI2
2LeNNF8KomC6/u2hhoaIYYG9w4TWJo8Nddnaz92jghs7czrNVsgz2H1EN6qFoaOp
5ANBh+VPwOZHb9B1uFt1Et8m5krAhUBA2gzKIUJ7OVnBQN1HJsQ=
-----END RSA PRIVATE KEY-----
`,
	}, {
		certPEM: `-----BEGIN CERTIFICATE-----
MIID1DCCArygAwIBAgIVANdDKrCzhGCapFECgQSFKpN5VrzJMA0GCSqGSIb3DQEB
CwUAMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVkIENB
IGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFiYWQt
NTAwZC05MDAwLTRiMWQwZDA2ZjAwZDAeFw0xNzAxMDYwMDUwMjlaFw0yNzAxMTMw
MDUwMjhaMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVk
IENBIGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFi
YWQtNTAwZC05MDAwLTRiMWQwZDA2ZjAwZDCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBALv6iRCtpAe8eBl3RArmyy96dBSAOM2VVJFBy4wL6TU2m1nkNCqw
HERWR7OXuhp9GNTDUBRYVvhr8upGohfSCBtHTtQH51xINa0vT21kzxAt2CmnYQce
Bfu0b3AWJ8O40+GonhZHSioA2GljJkxQLa/CHTClOkz/5GxspUHSFaoYsmI4u2b6
kDDAQxfrJ80nWBwGSjgN3bqZ1lQ1Z4169OuSSS9UFmH6ZNog36cbDrZnnvK4QspR
yYW60MenuczTzGTKI6iSfnXCjIzdJa6d5h84btBwgBVU01YYLEqw4//psIlMX2Fa
Xbs0xxkvpxi3rAvSFIG/VgUSg7kNhvKu6+UCAwEAAaNjMGEwDgYDVR0PAQH/BAQD
AgKkMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFD7zAs3bjiOjx0NeSfwFYwdZ
Y6vyMB8GA1UdIwQYMBaAFD7zAs3bjiOjx0NeSfwFYwdZY6vyMA0GCSqGSIb3DQEB
CwUAA4IBAQBcABnm4+zR41NWruBeY1tff/t2wNMz5eNdXyyfiYUh8tuIqSu1WvJl
RySPFXWU5ECfNhWCp68HLv9szX1PbqC476ltRFOvEfXNeVQ56bOcLAOjitFUiXMk
npuXg8o0jL6U+6wK+Rj7auYejOCy7cqOD7otmYjLdWmZb5a1xB8Sj3JsRM/nxp0W
kzGO2aRMvrgidVaRUhWb8B1W/MRdHiqb1NFY/TwMCgmRdPDG/9ZnHLhijqoRGIdE
LNSHGMEabz5qpEKvC3BJNc1c7ubeLKboszyIOq0rgic5PswexEKPSQr6InOF4Uqk
/2ApRv2bCDAJsHQKxso3oxtDIZCu6Zy1
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAu/qJEK2kB7x4GXdECubLL3p0FIA4zZVUkUHLjAvpNTabWeQ0
KrAcRFZHs5e6Gn0Y1MNQFFhW+Gvy6kaiF9IIG0dO1AfnXEg1rS9PbWTPEC3YKadh
Bx4F+7RvcBYnw7jT4aieFkdKKgDYaWMmTFAtr8IdMKU6TP/kbGylQdIVqhiyYji7
ZvqQMMBDF+snzSdYHAZKOA3dupnWVDVnjXr065JJL1QWYfpk2iDfpxsOtmee8rhC
ylHJhbrQx6e5zNPMZMojqJJ+dcKMjN0lrp3mHzhu0HCAFVTTVhgsSrDj/+mwiUxf
YVpduzTHGS+nGLesC9IUgb9WBRKDuQ2G8q7r5QIDAQABAoIBAQCVoqzDKwY85klQ
NEFk0C2EAN0wiJUvCaVE7/Qg6kZdBrgJiYABTylhOLFhA/GwLk665zLyzxOEMyIu
XcWgYp/zXwMgA56nNhAMaXG3qhyuS2AcYFH7gWnP4OUoe2pyAcoggsI9B8jAwOWE
ltmgu95yX3ovBAfxzW0+5gly6NI4EIv0srcBUHrbjNQ83SnkMIMTTyMKCZC70hWu
jsb/38Lq03e8k+xt7X9+jId/02IY/r1B7JB5W/1W0ULkHL2caAtji5/cEjAPv4kt
oOb/Sx2TdSapOTjKH8F69wGSub64OyTcbWCGnn/1OkioD1KAfpil1d8AOCEiCo0f
8zQcMG9hAoGBAMeDukEc7yg1VQgHfuMWCVPd+S3amN90+yDjBINDTp41DNhtXSgt
nzCxbI2qDXjuMbgsiWXnZIlm/eZoMDxTClz63o/3TsDM1DxJ8ONaOqnrP9SkR/Gg
wgo+VpwS9YTxTNxLzPubPAy+oDXItrsgdgI+EbdqpE8RgjswbywryX2pAoGBAPEy
t3r4n4Ml17BqLB5k0BrTbFT5DJ1l/4JcVp1yfciKbwggrcFeamYoUk4XKhsOziia
3SH1pSqN3yTogUqBMYLyNfPJC1j0w0N+BOAGL+R8OtuSp3lw+8zN0CeOC/f+Jl2Z
B3RwQ19FyyPwCt8pvU/7RHRcAsY+xFPY1/bITondAn89kWjorwUeY1baI3YEnchD
GTzL6alDS9YzT9sUprCcIcLVBuq4e1ukqxvUm6WMq/Sr4RKFX+KmhCHV+FDqTLXl
+4Sy6J5k4NCdyjOL9UwlI3OPKyMIbZNzlckkYLVjURNm5/1Zp43gYPMPpeC93kaM
apsM/tu1irGlkky8VMbZAoGBAI0RNpKres+/gEjjr+4O1E/8O63vA1vDXg8mZlor
rjluDcZZJoaj6+LXzltx3OR6YkUJXjAQdYtP8qSpf4TNzuFhPADbItHiRZnhyIQl
2jjuF5Z4TY8Pb5W3AflAcOuYL2QqrluBlUXRkh/wsssWpGx07vKHJfBAoZ7Qgm8J
dDXZAoGAQ03d1LsBoDlgB+0lf9rGga6gMyrawIGK6BQ8GWpzgspsNSa/2+0cf7DU
Ta1oXiCxA0VLMvdN7LoiyaEZVu16dObWP0iJLEzTq+h0E5cyDRwPwveyFlsXW7HA
6zIqsirw/etoh5NSevoa31fpsuXTZ9nNb1L3a/60ZfA/AdbWh9A=
-----END RSA PRIVATE KEY-----
`,
	}, {
		certPEM: `-----BEGIN CERTIFICATE-----
MIID0zCCArugAwIBAgIUMPSM1sUc+UgwvVFlh0tiRf9pG3QwDQYJKoZIhvcNAQEL
BQAwcTENMAsGA1UEChMEanVqdTExMC8GA1UEAwwoanVqdS1nZW5lcmF0ZWQgQ0Eg
Zm9yIG1vZGVsICJjb250cm9sbGVyIjEtMCsGA1UEBRMkZGVhZGJlZWYtMWJhZC01
MDBkLTkwMDAtNGIxZDBkMDZmMDBkMB4XDTE3MDEwNjAwNTE0M1oXDTI3MDExMzAw
NTE0MlowcTENMAsGA1UEChMEanVqdTExMC8GA1UEAwwoanVqdS1nZW5lcmF0ZWQg
Q0EgZm9yIG1vZGVsICJjb250cm9sbGVyIjEtMCsGA1UEBRMkZGVhZGJlZWYtMWJh
ZC01MDBkLTkwMDAtNGIxZDBkMDZmMDBkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A
MIIBCgKCAQEAle4HnHjMUALRRsGguXkmEYGWh8t0vqKcjp2D4/oK2q93dpH3XL8j
OwrmsCMWjZMzfKvQpVHYu+blR3cgN8Il9ZZ34SVVDmqSz2dOFj73Lb2TRmnXmXxw
ah3Ko+eFoDxs3bF64JYmkUlY3PLQTa/20ahPmnAD9RZxAtFb/a++Ono+ax/mSPS3
84I18vQmNv6bgzSlJq9DuI4RyH22Mz5GN0IesuTOYtWUwxuwEg4TSHG4L5/sEjwv
1p6azpAXKgXqw1tuzIM91JeJnWsDXNMbnAAyPcUUKnWEL3IJlhAJOLaOQLRelsQV
goIl8fKnICNS2LUIpa7F6+hJk29Ebiv1MwIDAQABo2MwYTAOBgNVHQ8BAf8EBAMC
AqQwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUU/6vb8zYl3bGG5BlJJbrnIfi
oH0wHwYDVR0jBBgwFoAUU/6vb8zYl3bGG5BlJJbrnIfioH0wDQYJKoZIhvcNAQEL
BQADggEBAIeO1SHVaL/Un0OJL2ru05n47dLDl9E+nRvGPfPeedkbufwdpNznROBn
E02cQ7Ruw075/U4mVprHRqOzlUMBjsUYBEDF94+9+8ePMQ3XJlUgQndQ09YAkW6I
gjJ32hnAZ3R3NlUrUuisONTYZi9UJbZv8/KNiFeOz5EH8bw98txXgdERtqrnpNkr
6KGVehrRZq4E96hi2PsjUUirnZcSiEy/LgS6qMsDv7YG9GvafkDuE4WvmCE8xWRG
Kcs23fA9rJu1nmwICge1tVcaHwc2L9mFuxvwK/LYj7fJLcKSoJgX5hhiO4VGblkF
KWOhPUkGqyUbfAEnLHBEu2Pas7LSlus=
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAle4HnHjMUALRRsGguXkmEYGWh8t0vqKcjp2D4/oK2q93dpH3
XL8jOwrmsCMWjZMzfKvQpVHYu+blR3cgN8Il9ZZ34SVVDmqSz2dOFj73Lb2TRmnX
mXxwah3Ko+eFoDxs3bF64JYmkUlY3PLQTa/20ahPmnAD9RZxAtFb/a++Ono+ax/m
SPS384I18vQmNv6bgzSlJq9DuI4RyH22Mz5GN0IesuTOYtWUwxuwEg4TSHG4L5/s
Ejwv1p6azpAXKgXqw1tuzIM91JeJnWsDXNMbnAAyPcUUKnWEL3IJlhAJOLaOQLRe
lsQVgoIl8fKnICNS2LUIpa7F6+hJk29Ebiv1MwIDAQABAoIBAEpRkUhIwM4wd1hh
hrApk2s+F0X83kPZC03OWqBZQ6QfMlJ+DGXXATm1L6DD49wNyYnVCTnJlmueCq0g
KcI55OMV0RqzjLP2tAtHI/f3AkOUIzrWIF8exgJz/gy9KQnCqhxCa6yEAV4nQ3tS
iVrZDLnkeoENYYjNtPs2mCjqQpttVtyfqUuVz5RfWRQmlb93VFR70jXd9XmdYgXT
Fp/+Dw7/J0LmgeloiJXe98mJidTXl3AMfU7T9+q1gNOA1j0T9H9f3QHGk6/xfzIq
tUrcl5iRiOhDA9Es7+4NPG2StELAqa4Cz3MVasltnQv4wKbGbbU94PGCwwn1fNPC
AOM/dSECgYEAwErW3qT2JwhLuiEalMVN/BFo/F1621f+MElaEGu42CW/M1X8TeIU
Teh+6sDjM0jeCH2c8d5s6kFDw9vUkmipoCa7DY3igcSlSkXaH2laWd5Xw7Cl1zBB
HbDoGwStgJA5pPVJaxlcUgFMn8Ufyp6T4Wu2Lt8Gjx3hN6hdF1EmgEsCgYEAx5o8
qIv/fJ4eyLQ9ALa6kcQIiYTxrrBR4aEED0nPGkbL45u3VpCHIsZQ8vS0hIE+g31X
FMe6mo82d776K0xyVAKseyRny7FXdAhSf5Mw3nKUxEZjj1dH81rlEma2QXCsGGHP
TahEdfy832pCnXefY5csmjPRTDfC9ku5dFt6XbkCgYBdY7uJdEqMWG4W3r7/I/gl
BPudHXVulATCxM02Dms+ps5+izM3TCaaU+FX+AggPzJ+oysXGPSCrOHRfLF3zdcc
DdNo2ND6P1wbf2NZQBx+awrq4p8YrLnjRqjjflkZiHJGmE5o8m8w8KqbKeyvxmaB
44MerniMB3H7SwyzJ1lwkwKBgBrf8XrgHq4wvARaADywXb5ynpOlisTyiJPQkjNc
xL3M+nbS6ISHQSq1eQC9YfDVVg1uOuNZYQ6eab8Fv7lEucAF/CM4a6FJZ3UACduT
fej1lII41/7cyZtCicVIfxGgkgtWYAsctNsOBIAwVnfNO3C1UhQRFgUlHOL8nyPD
jV5BAoGBAL6djOSFE0UxYUfGD3QuGtUyj2abT6emML6r51+FXvtiX+2PEuWcL1Zc
qxc6HoniXHhs8GDKN/F6EXwNSsurEARm61s7w9b5UFXPpOOL8nNJKtcW/ke2XWdC
OmjC1rvpj70fOVB+wMFS/yl/A+914woInTQ4ocGr5+SgS+7rUspm
-----END RSA PRIVATE KEY-----
`,
	}, {
		certPEM: `-----BEGIN CERTIFICATE-----
MIID0zCCArugAwIBAgIUW4R4HUXaepRGxap7AMhQP+TsmnwwDQYJKoZIhvcNAQEL
BQAwcTENMAsGA1UEChMEanVqdTExMC8GA1UEAwwoanVqdS1nZW5lcmF0ZWQgQ0Eg
Zm9yIG1vZGVsICJjb250cm9sbGVyIjEtMCsGA1UEBRMkZGVhZGJlZWYtMWJhZC01
MDBkLTkwMDAtNGIxZDBkMDZmMDBkMB4XDTE3MDEwNjAwNTMwMFoXDTI3MDExMzAw
NTMwMFowcTENMAsGA1UEChMEanVqdTExMC8GA1UEAwwoanVqdS1nZW5lcmF0ZWQg
Q0EgZm9yIG1vZGVsICJjb250cm9sbGVyIjEtMCsGA1UEBRMkZGVhZGJlZWYtMWJh
ZC01MDBkLTkwMDAtNGIxZDBkMDZmMDBkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A
MIIBCgKCAQEAyH7eNrJ9KX2t7y0d5Spnvhn93S79F7p4QncN0Dp7QNno5uWCMeq3
UWJYcEH64jJuSxQ9hPCwfkoFkJzchB14WeeBbVrOCHOEsFE8CBUK5GnATa6Xklj5
Wltaqhx6E3chOZhfzFn+ebSL1F1tpk+v3JAv2BFc1xU6lcU0ud75LxKPToWDerhE
YZvbB1Le1x9EbLl40b0k83jvNWRqso0JFaJlzIemcXHibdAlTONl2dADZ/llsYHN
a+CUexCzdqNBplT8LxpXuh3CgxkuinyqwCiILKPRDlEeerq8mHoUdzXC8IYPb1jh
ETF7zdCB40Nhe74a+E9uPHUfp2b0x04vWwIDAQABo2MwYTAOBgNVHQ8BAf8EBAMC
AqQwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU6OkK0ig5vTBfKDNdcbQhwmZV
tqswHwYDVR0jBBgwFoAU6OkK0ig5vTBfKDNdcbQhwmZVtqswDQYJKoZIhvcNAQEL
BQADggEBAKKEENYT5HIpr6Cz7uQy4t2hXTHP6nJ5bIRDLp+9ygVy5yh12cF4bJaA
a1y3y/i3hQZvksylddbHxV2KLUKp2oNjsOREdTT9Jm/68GRsxDHf7wlEoc/G6SHv
yLH8i8ouvNRtUOnyQr+FbJwAeJRUFr8Rx8mhkdg0gFJBW3Uppich6pb/vU8UhAdI
/Ccp5Bq6Roas8pCI/NRpypOfM+syoN8KG6qy1obxrNbnUKBQRm41Ig4BuZOCBUTz
NwImeMH5kumtx/r8L99EtvM4RepaNlpLTzVGEEfbwtPXCFngd9cZWbySyah/0FLp
Jrazq1uCx06CmaLFUdPs0UkUQsuwUZw=
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAyH7eNrJ9KX2t7y0d5Spnvhn93S79F7p4QncN0Dp7QNno5uWC
Meq3UWJYcEH64jJuSxQ9hPCwfkoFkJzchB14WeeBbVrOCHOEsFE8CBUK5GnATa6X
klj5Wltaqhx6E3chOZhfzFn+ebSL1F1tpk+v3JAv2BFc1xU6lcU0ud75LxKPToWD
erhEYZvbB1Le1x9EbLl40b0k83jvNWRqso0JFaJlzIemcXHibdAlTONl2dADZ/ll
sYHNa+CUexCzdqNBplT8LxpXuh3CgxkuinyqwCiILKPRDlEeerq8mHoUdzXC8IYP
b1jhETF7zdCB40Nhe74a+E9uPHUfp2b0x04vWwIDAQABAoIBABFnefCzk8vbPhMK
DpNLGBBXf0MJBi1eIXxbi1ZtDS/2+pG4kJG2t00YRWcRQy8Jg6t+yUxV/GwqDsub
4kSRrlXXk9OjU0lGhdksZBKriPOE/o1/v2aT1Kl0Tq1/pVU1PSwbvw0lLsLv2keI
CLGsLy7tUUnnT/DXxeskYExWP7lHMtXpn56eNTQBhzpZ6H9RIDnRtDbqO78v8mna
oYZwAtROpBQnmv0dgJLUYD+7y99SCHa0/GQlendmzbxVUenUJy0LfzZpmaIyffX6
n+LX1xqNvAdTM9T3fnl5HWAkKcjdCdpya7x56j415YgE+yQpc9s68gUjdhQGhMls
5IWig+ECgYEA4OlsjVC6LYS1lLuMOaSNA7h6SpSqUjUm0beaR2B4lIXHFkT0Co9I
3DR3vEqCbU83dIJxyqmExMkGy2dH0bDZvBf3UG09OI8DnvWgaAv9tLRKTUfhq08X
2FGqO365dO4Dz3q5ek3YkVPlRPEPiPkW8E4p/16+suIRC9YJVlr3N8sCgYEA5DV3
wDMACEYHkDwntxjZSC0pn+DlKhFboMNEKJUavkGVATIAoqVYJwNwTRrbNMo6nBfR
iOP+t1QAB04OWXas/fcA6oj+SLYwUPsErwdOOr4G4hAJgk1zhy+EkFgByQYe4lax
QoRhRlQ37mriChkUugu/hUyM2ifUfmDWWpX2VLECgYAznH0jhJXkoiKFCfCovaD6
bJfri2f2S2ooD+jgXjvtYTGJY7NYxP6kQYoEdFivutDv9IsmRXZJLhRYEtBBR0L/
zcQb+aVl2DOtg5IWrkG++OxgBZG8nP7MzhixZGgsc/3925y6eyiqWMUw59x3wylH
ZbhveI9sP0JIJAqqfV9jtwKBgQCxzEaEZDe+OnFiGrwGl0GSTj2D5sAkymlNE1Vj
GOhBGu9wgvijv5ayrqcsK6RluLl5K8WAlcG6K6PqDRJ4+VFGfM8l653SIL/8F9ZK
V7LHaY4LSttdveIojQkHPMoHSNF/HnP0kxcGqE77XiLGI0xWtxh8Tmve+/E+lUvs
P7IvwQKBgHSpW68m3WA6Tk44C8HFNrgsCWX7T9fkE8/GyGbK+ekMoEsM2eI5Mfrs
cPJ4+jgiSfw7FzYk5FzmxC2rz9E2kOr7IleSYrU30TlNJCMZEFVIm3TQULd84/7O
j4GhhUAMI8mdPnbva5MYlo4SFbXxf6uqE+wA/Ayw1H7+w4M3tjrF
-----END RSA PRIVATE KEY-----
`,
	}, {
		certPEM: `-----BEGIN CERTIFICATE-----
MIID1DCCArygAwIBAgIVAPhcqEYI7mY5xrcMbCfcFygHTziaMA0GCSqGSIb3DQEB
CwUAMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVkIENB
IGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFiYWQt
NTAwZC05MDAwLTRiMWQwZDA2ZjAwZDAeFw0xNzAxMDYwMDUzMDFaFw0yNzAxMTMw
MDUzMDFaMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVk
IENBIGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFi
YWQtNTAwZC05MDAwLTRiMWQwZDA2ZjAwZDCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBANbOg+gMLJNn4GWthqMH3UV0Xpm/EgGu0henzVvB5q4pQp5JzlEG
uyftqOPNNMRnsiv6c8eU/4riySVPddrcWUUhbQmRmSSmwbV87o1Udi7hPnkxwxib
9TTGqHLODP8dSd6I12rNDdUbb/m+pz3NHBFVaVu1RrxCfKfF+JzKn2PgY0P/vOib
eaKrKSvYm/WJywjP2f54w4dD2N/eu3SKGuaCoi7Y9QHK8p+vQ0pQivgWDLHOC0X9
CoAhpDL0Lo+3S/I5L0BYVR9evOnV91lQDcap2nLQiB3QgW4t2Wo3FZ+PVJbbTlM7
lsbDqF2lGRG2m4aL7A0eCV5PdZg3MwtPDsECAwEAAaNjMGEwDgYDVR0PAQH/BAQD
AgKkMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFK/Y8a+8J1BX6FlyArV+X4l6
tHyUMB8GA1UdIwQYMBaAFK/Y8a+8J1BX6FlyArV+X4l6tHyUMA0GCSqGSIb3DQEB
CwUAA4IBAQAiIF/VPN/C/+8C7O2qRl+hGfLw2D/WfbBPBrIAXI0jAFRCQ/nd2Pju
jao0/KBL4Be6cuPVlpkvipQo91sX4c1T4gx+F8owS21MdHJmP/cgfSb7JyOf+9dD
RCwHHpj9JM+gvChw1708Nhq0q+tgzS8LpqvR2QD0QUyc3q447+UZ3CkUu4AuDwTi
ozRefG42JgHTuZDQHFSM2XsJNH12V1IEtUcYrV7Az5Iv7H96o4Mzvm5TVZWgtWez
/A4MF1aDIvObKj5Q0AxvhbAa6O3RaNhYQw4Yvu2Ld2QQjmGV46qTHwPbw65tKrrJ
3fqBWq9DLGXacJxWNetOBpW7qIcc4C2f
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA1s6D6Awsk2fgZa2GowfdRXRemb8SAa7SF6fNW8HmrilCnknO
UQa7J+2o4800xGeyK/pzx5T/iuLJJU912txZRSFtCZGZJKbBtXzujVR2LuE+eTHD
GJv1NMaocs4M/x1J3ojXas0N1Rtv+b6nPc0cEVVpW7VGvEJ8p8X4nMqfY+BjQ/+8
6Jt5oqspK9ib9YnLCM/Z/njDh0PY3967dIoa5oKiLtj1Acryn69DSlCK+BYMsc4L
Rf0KgCGkMvQuj7dL8jkvQFhVH1686dX3WVANxqnactCIHdCBbi3ZajcVn49UlttO
UzuWxsOoXaUZEbabhovsDR4JXk91mDczC08OwQIDAQABAoIBACDCkDV0ndvBTOTU
6SK2EVYFYGQcUte03M/ExLoIwXCwVDulz/RljWeShF5xrmmB0xeH895tZ+e7nKe3
DadaUbcMWP/Ll8qSMBrIue0OA5Q0v/1lzHZVWrk4zvnrWOoTgm+Q+SGNsl1c+N/i
5sizHmO6Xf4hkG8nyLEhrT/kWxi1GXbJWQC9NHOhmaONGp2qdP44u3d+nvRCtdn8
Pz7WKbbgeRcLFE7y0qbtqOXZfKUfAquSvJEwJH4ap+h77LmX0AP70I2oael1Y4NG
i2AYlyOGhiVfXqtC/2MbkxlRAwASgX4+GDOhd2+1VUmic5kWi2eZELyGAOkKIyEx
G7Qqr20CgYEA4L4ReIuoXcydHOo23Ya3/emPiGNS0/AsqSyMv4Pn3mSV+AJFYj1/
08ICjlfnyLfF9kIfdY3OWsQHPtNdv2r5+6xPFjWZH3bQFt6qdjbiYgF4KZ5NUnQj
I2s/Debh/e9NWNTV/RDTMcHuVoujKB+Ydgl1nyTcJeJrTTUvbouWmTcCgYEA9K6v
ftW7/C7T2UHk/7NBTXNRy3C7K5QTrKxTT2JlNmkgYLqe6Sh2NieCJI3yKKlIb0w5
MqAdH89iwHl8iMTHuO6n7RqaIh5/GjTQnQ/IfOvnWF27lRR3LfWJzGL7B6CLTGS5
UBaXwD9JoKG773jWZHD10JfdjVjTer/LOLkIM8cCgYEAjGu1ltAL3q+d6xccOMWX
6HoqSFlmwy1pejpvpqnQElBes5sXpw4t9C1gU3fxRqQseHVQGq8DpAqfTE3cYWGB
2BINSB8tEU+3LNLrRvl78Shh3/8c6MvPuK2ltBXyVitS9JHnMjt0+b3mIQUWaFd1
dyrp5wms067oxFDa1gM5kXkCgYBODmgn8zFMKvgkR9QxClNbzN0cJHNhXJTGMhdF
rOMS74iHsyE8yk6jygjHdoz5Eb/s2YkpsFB6NGOReXAUBYqdYW0qw6DGH6p2hsDC
SiWwTGV6GDTeQzAd9kWmcTEZ5sppbXx8oyh4CPiJCO92XrsuQab5i2j/TYXxk7vV
9Zk7QwKBgHJpd+9+ArUgsjFw7yIYKmQAh2KSQuHFatw4PC+szQ2TM52kzFjE1q9M
M9DdxHkuIaJ095X/Kf9awcUPUYaup94tJbO0nEHgM7uQSXsCL1HGDG0pmW22ygWC
1h8lZTG2rqE/VEkh6GX1HZToYMJUgS9+kDxUIxV6YerGXApfRAeF
-----END RSA PRIVATE KEY-----
`,
	}, {
		certPEM: `-----BEGIN CERTIFICATE-----
MIID1DCCArygAwIBAgIVAKfGw+sRzn3IMcXpeWnXhHMuwO9YMA0GCSqGSIb3DQEB
CwUAMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVkIENB
IGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFiYWQt
NTAwZC05MDAwLTRiMWQwZDA2ZjAwZDAeFw0xNzAxMDYwMDUzMDJaFw0yNzAxMTMw
MDUzMDJaMHExDTALBgNVBAoTBGp1anUxMTAvBgNVBAMMKGp1anUtZ2VuZXJhdGVk
IENBIGZvciBtb2RlbCAiY29udHJvbGxlciIxLTArBgNVBAUTJGRlYWRiZWVmLTFi
YWQtNTAwZC05MDAwLTRiMWQwZDA2ZjAwZDCCASIwDQYJKoZIhvcNAQEBBQADggEP
ADCCAQoCggEBALbheQae7R4YTER9MfM+0FiAJRDc/cWLsNB0kvnDB9MZTVDrTrfE
OtfcqhLqMpP8X0u6AdaedGN4T8WQMNVTGfDzrm09J+dYaT34Rf1MP9xYs2rBf28F
nau0cqq35clYh5kYUfFJwUIPcuCU8W0Dh43DnsVW/szaWDjlHaNdSN3xZT+CMkrM
2v8eqXXQsJsjTk8Swi4ZV96dNKXCZ5evEDsxosaLFyrSoGHHFj+WNud8muEiRYEa
cfxZrKGhUkExB1plVrYWtZCn9gUKc1FA1d/Z4C/hoyKO0dke5KTAQHCgMG/4U/8z
+tlcjZSUG4gimvwhoybsLV7bFpXwQIdTEZMCAwEAAaNjMGEwDgYDVR0PAQH/BAQD
AgKkMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFKErcoGzxcnrb5k8mdOqIsAX
YXfPMB8GA1UdIwQYMBaAFKErcoGzxcnrb5k8mdOqIsAXYXfPMA0GCSqGSIb3DQEB
CwUAA4IBAQAtjacoyIkCerVCZLtSxEZ6QxW6iT/hHK2iaZ0tO9i6yw5/8huFk+Qn
s+DR2ApmWQIEmLEryB3uMuCEC7Qx/9C7DaJaSXcoltOzQDu0dOdTGFQtpQJMTDMG
wC/yea+wiPW4vv8HGxYMo/8ImGO7UWcl4T9akyejU3vc15oNR8HPomxNnExTtLyf
JemqUi5uuPItZFcSDOxayJSP2UcsBI7SqJDZeWjyjS8bH6YdPWunfJub4eIsejOK
GTWgS8bGxxH00sp2ZSg9CoyAEWPbI2o+Fl/r+zkNdXHSCPYlrbkVSDHRqlzr0TQh
M2BN8m2haQiMDhbo4CZHyoIjGKlXIUK1
-----END CERTIFICATE-----
`,
		keyPEM: `-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAtuF5Bp7tHhhMRH0x8z7QWIAlENz9xYuw0HSS+cMH0xlNUOtO
t8Q619yqEuoyk/xfS7oB1p50Y3hPxZAw1VMZ8POubT0n51hpPfhF/Uw/3FizasF/
bwWdq7RyqrflyViHmRhR8UnBQg9y4JTxbQOHjcOexVb+zNpYOOUdo11I3fFlP4Iy
Ssza/x6pddCwmyNOTxLCLhlX3p00pcJnl68QOzGixosXKtKgYccWP5Y253ya4SJF
gRpx/FmsoaFSQTEHWmVWtha1kKf2BQpzUUDV39ngL+GjIo7R2R7kpMBAcKAwb/hT
/zP62VyNlJQbiCKa/CGjJuwtXtsWlfBAh1MRkwIDAQABAoIBACybKDreLrXFdWzL
IJDVBFphvyiyzk8NuG3P0T2MPW9W343aJxQZj8Sw9OayKu3/g3rj/qjLc1YDtWlZ
tx+BKnWj6iqI3R/HZ7wOv3t9yqHDHNfwJFX/B6sm00Uin7CRoFlOHkZ9F982B24R
xiUx1DFWwnYdtv5eT2BpOeZrzWccnRXffbK1CZM5MnPRdc4Fda10oC/8X3fhoVF5
uDub8598ZketEGFtENeFtI7G+3OIA5RZqvJyseHhpbVfoH/w4BYNAUzlV/aQ+b8Z
AsuL4FsHPIa3MUyqPtPwzBywqaqVBTExbBFxXBKppEPdFuxPPKtwCT/XF9TeBK+w
cfLsjIkCgYEA6gn5m4ks4o/u4mkufgPtmzcKB+EkNMQXg6izFuRzYDSFP0UiU40p
L+0g06OjBEYQ5/eb9m9rLI9AyphM+CHkiLLgH7YJD0fJZ6YjsB8KCHdrTOJ7YYDR
qTEq/7hlb9ZvMvwBB+5rMaLx39zfa1z5dAPqJkZ79ho/cRRwmAtvGp0CgYEAyAqW
i3qFv9VFHV66WrNYCmShCUbg/70GZpzz/5ATxkOP5tfqtYPKXKK7XeABUcFpR2Gu
a/VT6pBSXg5u067iJ5T5ONPXcORcMFyE4WQ79fWgeCRC6TcWEPeFdC+OrojR8c6B
S1jYVAvekHtrbEIwEi/0JlFKaOIWBj3Ax+54Te8CgYBcsBSlz57yH3TJQETEbYPV
m+eiFT1ElICHiPFcWJ+E82fuFdAxbM9qBqyghMcT8zfQ1pRy9c9pN1a+pbCd+LZM
mSMADBTsc07FeNvbbX+IuNrK+UtJ4z2lly5jVkD+GNuzQnCaFodnghPux8fH+mtk
0gQSoomHfv+vJhwtayHWVQKBgQCdHDPAuCnz0HzH5G5YbiLnhsEzmAd+0+cDEKJS
RcZCIS4UD+C3TxrQRr8Bq71diJo3BfEHuVZ/E3qdhJhmiSP2EnrqtvauI6yEDyYO
H7s4IgI2WcRzhmA+UyjtZ6yYR70cPSQvGUhOOkRDFav5CsLWhfMrZQ0b8vV5UWbn
n9O84QKBgQCkK1CAX5MQ0EqqunbXHacZHT4N0N4APuBJD7STAptAHJYzsjsAw6YM
W0EusVEEOz/oLcMapDQG5eK+0CIGaS94UJMb7pQ8Hr169xm3ihor3MEx4JUh8Adb
s8X/Qi2AOFTcy2zQRQ7Nbon8crpIY98Y36Tpflwb0VW8SeuMK5SWzA==
-----END RSA PRIVATE KEY-----
`,
	},
}
