/*
 * unity-webapps-notification-context.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <libnotify/notify.h>

#include "unity-webapps-notification-context.h"
#include "unity-webapps-resource-cache.h"

#include "unity-webapps-dbus-defs.h"

#include "unity-webapps-gen-notification.h"

#include "unity-webapps-debug.h"

static void
real_show_notification (const gchar *summary,
			const gchar *body,
			const gchar *icon_url,
			GError **error)
{
  NotifyNotification *notification;
  
  error = NULL;
  
  notification = notify_notification_new (summary, body, icon_url);
  notify_notification_set_timeout (notification, 4);
  notify_notification_show (notification, error);
  
  if ((error != NULL) && (*error != NULL))
    {
      g_warning ("Error showing notification: %s", (*error)->message);
    }
  
  g_object_unref (G_OBJECT (notification));
  
  
}

static void
show_notification (const gchar *summary,
		   const gchar *body,
		   const gchar *icon_url,
		   const gchar *default_path,
		   GError **error)
{
  if ( (icon_url == NULL) || (icon_url[0] == '/') )
    {
      real_show_notification (summary, body, icon_url, error);
    }
  else
    {
      gchar *icon_path;
      
      icon_path = unity_webapps_resource_cache_lookup_uri (icon_url);

      if (icon_path == NULL)
	{
	  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Resource path from cache for uri %s was NULL using application icon for notification", icon_url);
	  real_show_notification (summary, body, default_path, error);
	}
      else
	{
	  real_show_notification (summary, body, icon_path, error);
	}
      
      g_free (icon_path);
    }
}

static gboolean
on_handle_show_notification (UnityWebappsGenNotification *notification,
			     GDBusMethodInvocation *invocation,
			     const gchar *summary,
			     const gchar *body,
			     const gchar *icon_url,
			     gpointer user_data)
{
  UnityWebappsNotificationContext *notification_context = (UnityWebappsNotificationContext *)user_data;
  GError *error;
  
  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Handling ShowNotification call, summary: %s", summary);
  
  if ((icon_url == NULL) || (g_strcmp0 (icon_url, "") == 0))
    {
      UNITY_WEBAPPS_NOTE (NOTIFICATION, "Notification contains empty URL, using application icon for notification");
      icon_url = notification_context->icon_file_name;
    }
  
  error = NULL;
  
  show_notification (summary, body, icon_url, notification_context->icon_file_name, &error);
  
  if (error != NULL)
    {
      g_dbus_method_invocation_return_gerror (invocation, error);
      g_error_free (error);
    }
  
  return TRUE;
}

static void
export_object (GDBusConnection *connection,
	       UnityWebappsNotificationContext *notification_context)
{
  GError *error;
  
  notification_context->notification_service_interface = unity_webapps_gen_notification_skeleton_new ();
  
  g_signal_connect (notification_context->notification_service_interface,
		    "handle-show-notification",
		    G_CALLBACK (on_handle_show_notification),
		    notification_context);
  
  error = NULL;
  
  g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (notification_context->notification_service_interface),
				    connection,
				    UNITY_WEBAPPS_NOTIFICATION_PATH,
				    &error);
  
  if (error != NULL)
    {
      g_error ("Error exporting Unity Webapps Notification object: %s", error->message);
      
      g_error_free (error);
      
      return;
    }
  
  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Exported Notification object");
		    
}

UnityWebappsNotificationContext *
unity_webapps_notification_context_new (GDBusConnection *connection, const gchar *name, const gchar *icon_file_name)
{
  UnityWebappsNotificationContext *notification_context;
  
  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Creating new Notification Context");
  
  notify_init (name);
  
  notification_context = g_malloc0 (sizeof (UnityWebappsNotificationContext));
  
  notification_context->icon_file_name = g_strdup (icon_file_name);
  
  notification_context->connection = g_object_ref (G_OBJECT (connection));
  
  export_object (connection, notification_context);
  
  return notification_context;
}

void
unity_webapps_notification_context_free (UnityWebappsNotificationContext *notification_context)
{
  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Finalizing Notification Context");
  g_object_unref (G_OBJECT (notification_context->notification_service_interface));

  g_free (notification_context->icon_file_name);

  g_object_unref (G_OBJECT (notification_context->connection));

  g_free (notification_context);
}
