# Copyright 2012-2016 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Arguments parser for `maascli`."""

__all__ = [
    'prepare_parser',
    ]

import argparse
import os
import sys

from maascli import api
from maascli.cli import register_cli_commands
from maascli.utils import parse_docstring


class ArgumentParser(argparse.ArgumentParser):
    """Specialisation of argparse's parser with better support for subparsers.

    Specifically, the one-shot `add_subparsers` call is disabled, replaced by
    a lazily evaluated `subparsers` property.
    """

    def _print_error(self, message):
        """Print the specified message to stderr.

        This method is used to isolate write to stderr, so that those writes
        can be intercepted in a unit test.
        """
        sys.stderr.write(message)

    def __init__(self, *args, **kwargs):
        kwargs.setdefault(
            "formatter_class", argparse.RawDescriptionHelpFormatter)
        super(ArgumentParser, self).__init__(*args, **kwargs)

    def add_subparsers(self):
        raise NotImplementedError(
            "add_subparsers has been disabled")

    @property
    def subparsers(self):
        try:
            return self.__subparsers
        except AttributeError:
            parent = super(ArgumentParser, self)
            self.__subparsers = parent.add_subparsers(title="drill down")
            self.__subparsers.metavar = "COMMAND"
            return self.__subparsers

    def error(self, message):
        """Make the default error messages more helpful

        Override default ArgumentParser error method to print the help menu
        generated by ArgumentParser instead of just printing out a list of
        valid arguments.
        """
        self.print_help(sys.stderr)
        self._print_error('\n' + message + '\n')
        sys.exit(2)


def prepare_parser(argv):
    """Create and populate an arguments parser for the maascli command."""
    help_title, help_body = parse_docstring(api)
    parser = ArgumentParser(
        description=help_body, prog=os.path.basename(argv[0]),
        epilog="http://maas.io/")
    register_cli_commands(parser)
    api.register_api_commands(parser)
    parser.add_argument(
        '--debug', action='store_true', default=False,
        help=argparse.SUPPRESS)
    return parser
