/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.functor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A concrete generic implementation for {@link IArgs}.
 * 
 */
public class Args implements IArgs {

	public static final IArgs EMPTY = new EmptyArgs();

	public static Args create() {
		return new Args();
	}

	public static Args createIndexed() {
		List list = new ArrayList();
		return new Args(list);
	}

	public static Args createIndexed(Object p1) {
		List list = new ArrayList();
		list.add(p1);
		return new Args(list);
	}

	public static Args createIndexed(Object p1, Object p2) {
		List list = new ArrayList();
		list.add(p1);
		list.add(p2);
		return new Args(list);
	}

	public static Args createIndexed(Object p1, Object p2, Object p3) {
		List list = new ArrayList();
		list.add(p1);
		list.add(p2);
		list.add(p3);
		return new Args(list);
	}

	public static Args createNamed() {
		Map map = new HashMap();
		return new Args(map);
	}

	public static Args createNamed(String key, Object value) {
		Map map = new HashMap();
		map.put(key, value);
		return new Args(map);
	}

	public static Args createNamed(String key1, Object value1, String key2,
			Object value2) {
		Map map = new HashMap();
		map.put(key1, value1);
		map.put(key2, value2);
		return new Args(map);
	}

	private Map valueMap;

	private List valueList;

	public Args() {
		super();
	}

	public Args(IArgs args) {
		super();
		if (args == null) {
			return;
		}
		// we can support both...
		if (args.isIndexed()) {
			valueList = new ArrayList();
			for (int i = 0; i < args.size(); i++) {
				valueList.add(args.get(i));
			}
		}
		if (args.isNamed()) {
			valueMap = new HashMap();
			for (Iterator i = args.names().iterator(); i.hasNext();) {
				String name = (String) i.next();
				valueMap.put(name, args.get(name));
			}
		}
	}

	public Args(List values) {
		super();
		this.valueList = values;
	}

	public Args(Map values) {
		super();
		this.valueMap = values;
	}

	public Args(Object[] values) {
		super();
		if (values != null) {
			// its common for values to be null, for example in java reflection
			// code
			// -> support it
			this.valueList = Arrays.asList(values);
		}
	}

	public void add(Object object) {
		if (valueList == null) {
			valueList = new ArrayList();
		}
		valueList.add(object);
	}

	public void clear() {
		if (valueList != null) {
			valueList.clear();
		}
		if (valueMap != null) {
			valueMap.clear();
		}
	}

	public IArgs declare(String name, int index, Object value) {
		put(index, value);
		put(name, value);
		return this;
	}

	public Object get(int index) {
		if ((valueList == null) || index < 0 || (valueList.size() <= index)) {
			return null;
		}
		return valueList.get(index);
	}

	public Object get(int index, Object defaultValue) {
		if ((valueList == null) || index < 0 || (valueList.size() <= index)) {
			return defaultValue;
		}
		return valueList.get(index);
	}

	public Object get(String name) {
		if (valueMap == null) {
			return null;
		}
		return valueMap.get(name);
	}

	public Object get(String name, Object defaultValue) {
		if (valueMap == null) {
			return defaultValue;
		}
		Object result = valueMap.get(name);
		if (result == null && !valueMap.containsKey(name)) {
			return defaultValue;
		}
		return result;
	}

	public boolean isDefined(int index) {
		if (valueList == null) {
			return false;
		}
		return index >= 0 && index < valueList.size();
	}

	public boolean isDefined(String name) {
		if (valueMap == null) {
			return false;
		}
		return valueMap.containsKey(name);
	}

	public boolean isIndexed() {
		return valueList != null;
	}

	public boolean isNamed() {
		return valueMap != null;
	}

	public Set names() {
		if (valueMap == null) {
			return Collections.EMPTY_SET;
		}
		return valueMap.keySet();
	}

	public void put(int index, Object value) {
		if (valueList == null) {
			valueList = new ArrayList();
		}
		while (index >= valueList.size()) {
			valueList.add(null);
		}
		valueList.set(index, value);
	}

	public void put(String name, Object value) {
		if (valueMap == null) {
			valueMap = new HashMap();
		}
		valueMap.put(name, value);
	}

	public int size() {
		if (valueList != null) {
			return valueList.size();
		}
		if (valueMap != null) {
			return valueMap.size();
		}
		return 0;
	}

	@Override
	public String toString() {
		return ArgTools.toString(this, "");
	}

}
