/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      TC08Enumerator.cpp
 * \brief     TC08 enumerator class
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: TC08Enumerator.cpp,v 1.10 2008/02/19 19:45:43 douglas Exp $
 *
 *************************************************************************** */

#include "../shared/PicoDownloader.h"
#include "../shared/PicoUsbDevice.h"
#include "../shared/PicoUsbID.h"

#include "TC08Enumerator.h"
#include "TC08Device.h"
#include <string.h>

// Embedded firmware
#include "../shared/IntelHexRecord.h"
#include "USBTC08_FX_OpenSource_hex.h"
#include "USBTC08_FX1_OpenSource_hex.h"

#define MAX_DEV 256

class TC08EnumeratorInternalData
{
	private:
		friend class TC08Enumerator;
		~TC08EnumeratorInternalData(void);
		class PicoDownloader *downloader;
		int preCount;  // Number of devices already present before we start
};


//////////////////////////////////////////////////////////////////////////
// Null function - used to dynamically determine if framework is present
//////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
extern "C" {
#endif
void TC08EnumeratorFramework()
{
}
#ifdef __cplusplus
}
#endif

//////////////////////////////////////////////////////////////////////////
// Destructor. Delete the downloaders
//////////////////////////////////////////////////////////////////////////
TC08EnumeratorInternalData::~TC08EnumeratorInternalData(void) {
	if(downloader) {
		delete downloader;
	}
}
//////////////////////////////////////////////////////////////////////////
/// Constructor. Set up a downloader for each type of TC08
//////////////////////////////////////////////////////////////////////////
TC08Enumerator::TC08Enumerator() {
	internalData = new TC08EnumeratorInternalData();
	internalData->preCount = this->Count();

	std::vector< PicoDownloader::DeviceSpec *> devicelist;
	
	devicelist.push_back(new  PicoDownloader::DeviceSpec((const PicoDownloader::DeviceSpec) 
			{VENDOR_ID_PICO_TECHNOLOGY,
		PRODUCT_ID_USB_TC08,
		DEVICE_ID_USB_TC08MK1_CLEAN,
		ANY_USB_ID,
		USBTC08_FX_OpenSource,
		PicoDownloader::FX}));
	devicelist.push_back(new  PicoDownloader::DeviceSpec((const PicoDownloader::DeviceSpec) 
			{VENDOR_ID_PICO_TECHNOLOGY,
		PRODUCT_ID_USB_TC08,
		DEVICE_ID_USB_TC08MK2_CLEAN,
		ANY_USB_ID,
		USBTC08_FX1_OpenSource,
		PicoDownloader::FX1}));

	internalData->downloader=PicoDownloader::Create(devicelist);
	
	std::vector< PicoDownloader::DeviceSpec *>::const_iterator it;
	for (it = devicelist.begin(); it != devicelist.end(); it++)
		delete *it;
		
}

//////////////////////////////////////////////////////////////////////////
/// Check whether all devices to which firmware was downloaded at startup
/// have renumerated successfully.
/// This may never return true, e.g. if a device is unplugged during startup
/// so you should put a timeout on any loops including it
//////////////////////////////////////////////////////////////////////////
int TC08Enumerator::Ready() {
	int count  = internalData->downloader->InitialDeviceCount();
	
	if (count < 0)
		return 0; // We haven't even finished downloading firmware yet
	else
		return (count + internalData->preCount) <= this->Count(); // There may be more devices present, this is OK!
}

//////////////////////////////////////////////////////////////////////////
/// Destructor. Close the downloaders
//////////////////////////////////////////////////////////////////////////
TC08Enumerator::~TC08Enumerator() {
	delete internalData;
}

//////////////////////////////////////////////////////////////////////////
/// Count the number of TC08 devices present
//////////////////////////////////////////////////////////////////////////
int TC08Enumerator::Count() {
	int tc08Count=0;
	PicoUsbDevice* devices[MAX_DEV];
	
  // Get the list of all Pico TC08 devices from PicoUsbDevice
	int deviceCount=PicoUsbDevice::Enumerate(devices,MAX_DEV,PRODUCT_ID_USB_TC08);
	
	int i=0;
	for(;i<deviceCount;i++) {
		// If the device has a correct DID it is one we can enumerate 
		int did=devices[i]->GetDID();
		if((did==DEVICE_ID_USB_TC08MK1_LOADED)||(did==DEVICE_ID_USB_TC08MK2_LOADED)) {
			// This is a device we can enumerate
			tc08Count++;
		}
		delete devices[i];
	}
	return tc08Count;
}

//////////////////////////////////////////////////////////////////////////
/// Return a list of connected TC08Devices
//////////////////////////////////////////////////////////////////////////
int TC08Enumerator::Enumerate(TC08Device **list,int length) {
	int curDevice=0;
	PicoUsbDevice **devices =  new PicoUsbDevice* [length];
	
  // Get the list of all Pico TC08 devices from PicoUsbDevice
	int deviceCount=PicoUsbDevice::Enumerate(devices,length,PRODUCT_ID_USB_TC08);
	
	int i=0;
	for(;i<deviceCount;i++) {
		// If the device has a correct DID it is one we can enumerate 
		int did=devices[i]->GetDID();
		if((did==DEVICE_ID_USB_TC08MK1_LOADED)||(did==DEVICE_ID_USB_TC08MK2_LOADED)) {
			// This is a device we can enumerate
			list[curDevice++]=new TC08Device(devices[i]);
		} else {
			// We can't enumerate this device
			delete devices[i];
		}
	}
	delete[] devices;
	return curDevice;
}

//////////////////////////////////////////////////////////////////////////
/// Open one (arbitrarily selected) TC08 unit
//////////////////////////////////////////////////////////////////////////
TC08Device *TC08Enumerator::Open(void) {
	PicoUsbDevice* devices[MAX_DEV];
  
  // Get the list of all Pico TC08 devices from PicoUsbDevice	
	int deviceCount=PicoUsbDevice::Enumerate(devices,MAX_DEV,PRODUCT_ID_USB_TC08);
	
	TC08Device *tc08=NULL;
	
	int i=0;
	for(;i<deviceCount;i++) {
		// If the device has a correct DID and we can open it then that is the one we want 
		int did=devices[i]->GetDID();
		if(((did==DEVICE_ID_USB_TC08MK1_LOADED)||(did==DEVICE_ID_USB_TC08MK2_LOADED))&&
		   (devices[i]->Open()!=PICODEVICE_STATE_LOCKED)) {
			
			// This is the device we want
			tc08=new TC08Device(devices[i]);
			devices[i]=NULL;
			break;
		} else {
			// We can't use this device
			delete devices[i];
		}
	}
	i++;
	// We already have the device we want
	// delete remaining devices
	for(;i<deviceCount;i++) {
		delete devices[i];
	}
	return tc08;
}

//////////////////////////////////////////////////////////////////////////
/// Open the TC08 with matching serial number
//////////////////////////////////////////////////////////////////////////
TC08Device *TC08Enumerator::Open(char *serialString) {
	PicoUsbDevice* devices[MAX_DEV];
  
  // Get the list of all Pico TC08 devices from PicoUsbDevice		
	int deviceCount=PicoUsbDevice::Enumerate(devices,MAX_DEV,PRODUCT_ID_USB_TC08);
	
	TC08Device *tc08=NULL;
	
	int i=0;
	for(;i<deviceCount;i++) {
		// If the device has a correct DID, the same serialString and we can open it then that is the one we want 
		int did=devices[i]->GetDID();
		if((did==DEVICE_ID_USB_TC08MK1_LOADED)||(did==DEVICE_ID_USB_TC08MK2_LOADED)) {
			// Check serial string
			if(strcmp(devices[i]->GetSerialString(),serialString)==0) {
				// Try to open unit
				if(devices[i]->Open()!=PICODEVICE_STATE_LOCKED) {
					tc08=new TC08Device(devices[i]);
				} else {
					// No need to look for more units, serial is unique
					delete devices[i];
				}
				devices[i]=NULL;
				break;
			}
		}
		// DID is not supported or wrong serial
		delete devices[i];
		devices[i]=NULL;
		continue;
	}
	i++;
	// We already have the device we want
	// delete the rest of the devices
	for(;i<deviceCount;i++) {
		delete devices[i];
	}
	return tc08;
}
