/* Vuls - Vulnerability Scanner
Copyright (C) 2016  Future Corporation , Japan.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

package scan

import (
	"bufio"
	"fmt"
	"strings"

	"github.com/future-architect/vuls/config"
	"github.com/future-architect/vuls/models"
	"github.com/future-architect/vuls/util"
)

// inherit OsTypeInterface
type alpine struct {
	base
}

// NewAlpine is constructor
func newAlpine(c config.ServerInfo) *alpine {
	d := &alpine{
		base: base{
			osPackages: osPackages{
				Packages:  models.Packages{},
				VulnInfos: models.VulnInfos{},
			},
		},
	}
	d.log = util.NewCustomLogger(c)
	d.setServerInfo(c)
	return d
}

// Alpine
// https://github.com/mizzy/specinfra/blob/master/lib/specinfra/helper/detect_os/alpine.rb
func detectAlpine(c config.ServerInfo) (itsMe bool, os osTypeInterface) {
	os = newAlpine(c)

	if r := exec(c, "ls /etc/alpine-release", noSudo); !r.isSuccess() {
		return false, os
	}

	if r := exec(c, "cat /etc/alpine-release", noSudo); r.isSuccess() {
		os.setDistro(config.Alpine, strings.TrimSpace(r.Stdout))
		return true, os
	}

	return false, os
}

func (o *alpine) checkScanMode() error {
	if o.getServerInfo().Mode.IsOffline() {
		return fmt.Errorf("Remove offline scan mode, Alpine needs internet connection")
	}
	return nil
}

func (o *alpine) checkDeps() error {
	o.log.Infof("Dependencies... No need")
	return nil
}

func (o *alpine) checkIfSudoNoPasswd() error {
	o.log.Infof("sudo ... No need")
	return nil
}

func (o *alpine) apkUpdate() error {
	r := o.exec("apk update", noSudo)
	if !r.isSuccess() {
		return fmt.Errorf("Failed to SSH: %s", r)
	}
	return nil
}

func (o *alpine) preCure() error {
	o.log.Infof("Scanning in %s", o.getServerInfo().Mode)
	if err := o.detectIPAddr(); err != nil {
		o.log.Debugf("Failed to detect IP addresses: %s", err)
	}
	// Ignore this error as it just failed to detect the IP addresses
	return nil
}

func (o *alpine) postScan() error {
	return nil
}

func (o *alpine) detectIPAddr() (err error) {
	o.ServerInfo.IPv4Addrs, o.ServerInfo.IPv6Addrs, err = o.ip()
	return err
}

func (o *alpine) scanPackages() error {
	if err := o.apkUpdate(); err != nil {
		return err
	}
	// collect the running kernel information
	release, version, err := o.runningKernel()
	if err != nil {
		o.log.Errorf("Failed to scan the running kernel version: %s", err)
		return err
	}
	o.Kernel = models.Kernel{
		Release: release,
		Version: version,
	}

	installed, err := o.scanInstalledPackages()
	if err != nil {
		o.log.Errorf("Failed to scan installed packages: %s", err)
		return err
	}

	updatable, err := o.scanUpdatablePackages()
	if err != nil {
		o.log.Errorf("Failed to scan installed packages: %s", err)
		return err
	}

	installed.MergeNewVersion(updatable)
	o.Packages = installed
	return nil
}

func (o *alpine) scanInstalledPackages() (models.Packages, error) {
	cmd := util.PrependProxyEnv("apk info -v")
	r := o.exec(cmd, noSudo)
	if !r.isSuccess() {
		return nil, fmt.Errorf("Failed to SSH: %s", r)
	}
	return o.parseApkInfo(r.Stdout)
}

func (o *alpine) parseInstalledPackages(stdout string) (models.Packages, models.SrcPackages, error) {
	installedPackages, err := o.parseApkInfo(stdout)
	return installedPackages, nil, err
}

func (o *alpine) parseApkInfo(stdout string) (models.Packages, error) {
	packs := models.Packages{}
	scanner := bufio.NewScanner(strings.NewReader(stdout))
	for scanner.Scan() {
		line := scanner.Text()
		ss := strings.Split(line, "-")
		if len(ss) < 3 {
			return nil, fmt.Errorf("Failed to parse apk info -v: %s", line)
		}
		name := strings.Join(ss[:len(ss)-2], "-")
		packs[name] = models.Package{
			Name:    name,
			Version: strings.Join(ss[len(ss)-2:], "-"),
		}
	}
	return packs, nil
}

func (o *alpine) scanUpdatablePackages() (models.Packages, error) {
	cmd := util.PrependProxyEnv("apk version")
	r := o.exec(cmd, noSudo)
	if !r.isSuccess() {
		return nil, fmt.Errorf("Failed to SSH: %s", r)
	}
	return o.parseApkVersion(r.Stdout)
}

func (o *alpine) parseApkVersion(stdout string) (models.Packages, error) {
	packs := models.Packages{}
	scanner := bufio.NewScanner(strings.NewReader(stdout))
	for scanner.Scan() {
		line := scanner.Text()
		if !strings.Contains(line, "<") {
			continue
		}
		ss := strings.Split(line, "<")
		namever := strings.TrimSpace(ss[0])
		tt := strings.Split(namever, "-")
		name := strings.Join(tt[:len(tt)-2], "-")
		packs[name] = models.Package{
			Name:       name,
			NewVersion: strings.TrimSpace(ss[1]),
		}
	}
	return packs, nil
}
