/*
 * LZIP decompressor
 *
 * Copyright (C) 2016-2025 Antonio Diaz Diaz.
 *
 * Licensed under GPLv2 or later, see file LICENSE in this source tree.
 */

#include "linux_module.h"
#include "linux_lzip.h"
#include "linux_mm.h"

/*
 * STATIC_RW_DATA is used in the pre-boot environment on some architectures.
 * See include/linux/decompress/mm.h for details.
 */
#ifndef STATIC_RW_DATA
#define STATIC_RW_DATA static
#endif

typedef int State;

enum { states = 12 };

static inline bool St_is_char(const State st) { return st < 7; }

static inline State St_set_char(const State st)
{
	STATIC_RW_DATA const State next[states] = { 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 4, 5 };
	return next[st];
}

static inline State St_set_match(const State st)
{
	return (st < 7) ? 7 : 10;
}

static inline State St_set_rep(const State st)
{
	return (st < 7) ? 8 : 11;
}

static inline State St_set_shortrep(const State st)
{
	return (st < 7) ? 9 : 11;
}


enum {
	min_dictionary_bits = 12,
	min_dictionary_size = 1 << min_dictionary_bits,	/* >= modeled_distances */
	max_dictionary_bits = 29,
	max_dictionary_size = 1 << max_dictionary_bits,
	literal_context_bits = 3,
	pos_state_bits = 2,
	pos_states = 1 << pos_state_bits,
	pos_state_mask = pos_states - 1,

	len_states = 4,
	dis_slot_bits = 6,
	start_dis_model = 4,
	end_dis_model = 14,
	modeled_distances = 1 << (end_dis_model / 2),	/* 128 */
	dis_align_bits = 4,
	dis_align_size = 1 << dis_align_bits,

	len_low_bits = 3,
	len_mid_bits = 3,
	len_high_bits = 8,
	len_low_symbols = 1 << len_low_bits,
	len_mid_symbols = 1 << len_mid_bits,
	len_high_symbols = 1 << len_high_bits,
	max_len_symbols = len_low_symbols + len_mid_symbols + len_high_symbols,

	min_match_len = 2,					/* must be 2 */
	max_match_len = min_match_len + max_len_symbols - 1,	/* 273 */
	min_match_len_limit = 5
};

static inline int get_len_state(const int len)
{
	return min(len - min_match_len, len_states - 1);
}

static inline int get_lit_state(const uint8_t prev_byte)
{
	return prev_byte >> (8 - literal_context_bits);
}


enum { bit_model_move_bits = 5,
	bit_model_total_bits = 11,
	bit_model_total = 1 << bit_model_total_bits
};

typedef int Bit_model;

static inline void Bm_init(Bit_model * const probability)
{
	*probability = bit_model_total / 2;
}

static inline void Bm_array_init(Bit_model bm[], const int size)
{
	int i;

	for (i = 0; i < size; ++i)
		Bm_init(&bm[i]);
}

struct Len_model {
	Bit_model choice1;
	Bit_model choice2;
	Bit_model bm_low[pos_states][len_low_symbols];
	Bit_model bm_mid[pos_states][len_mid_symbols];
	Bit_model bm_high[len_high_symbols];
};

static inline void Lm_init(struct Len_model * const lm)
{
	Bm_init(&lm->choice1);
	Bm_init(&lm->choice2);
	Bm_array_init(lm->bm_low[0], pos_states * len_low_symbols);
	Bm_array_init(lm->bm_mid[0], pos_states * len_mid_symbols);
	Bm_array_init(lm->bm_high, len_high_symbols);
}


/* Table of CRCs of all 8-bit messages. */
STATIC_RW_DATA const uint32_t crc32[256] =
  {
  0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F,
  0xE963A535, 0x9E6495A3, 0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988,
  0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91, 0x1DB71064, 0x6AB020F2,
  0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
  0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9,
  0xFA0F3D63, 0x8D080DF5, 0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172,
  0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B, 0x35B5A8FA, 0x42B2986C,
  0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
  0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423,
  0xCFBA9599, 0xB8BDA50F, 0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924,
  0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D, 0x76DC4190, 0x01DB7106,
  0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
  0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D,
  0x91646C97, 0xE6635C01, 0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E,
  0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457, 0x65B0D9C6, 0x12B7E950,
  0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
  0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7,
  0xA4D1C46D, 0xD3D6F4FB, 0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0,
  0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9, 0x5005713C, 0x270241AA,
  0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
  0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81,
  0xB7BD5C3B, 0xC0BA6CAD, 0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A,
  0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683, 0xE3630B12, 0x94643B84,
  0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
  0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB,
  0x196C3671, 0x6E6B06E7, 0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC,
  0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5, 0xD6D6A3E8, 0xA1D1937E,
  0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
  0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55,
  0x316E8EEF, 0x4669BE79, 0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236,
  0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F, 0xC5BA3BBE, 0xB2BD0B28,
  0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
  0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F,
  0x72076785, 0x05005713, 0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38,
  0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21, 0x86D3D2D4, 0xF1D4E242,
  0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
  0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69,
  0x616BFFD3, 0x166CCF45, 0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2,
  0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB, 0xAED16A4A, 0xD9D65ADC,
  0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
  0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693,
  0x54DE5729, 0x23D967BF, 0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94,
  0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D };


static inline void CRC32_update_buf(uint32_t * const crc,
					const uint8_t * const buffer,
					const long size)
{
	long i;
	uint32_t c = *crc;

	for (i = 0; i < size; ++i)
		c = crc32[(c^buffer[i])&0xFF] ^ (c >> 8);
	*crc = c;
}


STATIC_RW_DATA const uint8_t lzip_magic[4] = { 0x4C, 0x5A, 0x49, 0x50 }; /* "LZIP" */

enum { Lh_size = 6 };
typedef uint8_t Lzip_header[Lh_size];	/* 0-3 magic bytes */
					/*   4 version */
					/*   5 coded dictionary size */

static inline bool Lh_check_magic(const Lzip_header data)
{
	int i;

	for (i = 0; i < 4; ++i)
		if (data[i] != lzip_magic[i])
			return false;
	return true;
}

/* detect (truncated) header */
static inline bool Lh_check_prefix(const Lzip_header data, const int sz)
{
	int i;
	for (i = 0; i < sz && i < 4; ++i)
		if (data[i] != lzip_magic[i])
			return false;
	return sz > 0;
}

/* detect corrupt header */
static inline bool Lh_check_corrupt(const Lzip_header data)
{
	int matches = 0;
	int i;
	for (i = 0; i < 4; ++i)
		if (data[i] == lzip_magic[i])
			++matches;
	return matches > 1 && matches < 4;
}

static inline bool Lh_check_version(const Lzip_header data)
{
	return data[4] == 1;
}

static inline unsigned Lh_get_dictionary_size(const Lzip_header data)
{
	unsigned sz = 1 << (data[5] & 0x1F);

	if (sz > min_dictionary_size)
		sz -= (sz / 16) * ((data[5] >> 5) & 7);
	return sz;
}


enum { Lt_size = 20 };
typedef uint8_t Lzip_trailer[Lt_size];
			/*  0-3  CRC32 of the uncompressed data */
			/*  4-11 size of the uncompressed data */
			/* 12-19 member size including header and trailer */

static inline unsigned Lt_get_data_crc(const Lzip_trailer data)
{
	unsigned tmp = 0;
	int i;

	for (i = 3; i >= 0; --i) {
		tmp <<= 8;
		tmp += data[i];
	}
	return tmp;
}

static inline unsigned long long Lt_get_data_size(const Lzip_trailer data)
{
	unsigned long long tmp = 0;
	int i;

	for (i = 11; i >= 4; --i) {
		tmp <<= 8;
		tmp += data[i];
	}
	return tmp;
}

static inline unsigned long long Lt_get_member_size(const Lzip_trailer data)
{
	unsigned long long tmp = 0;
	int i;

	for (i = 19; i >= 12; --i) {
		tmp <<= 8;
		tmp += data[i];
	}
	return tmp;
}


struct Range_decoder {
	unsigned long long partial_member_pos;
	uint8_t *buffer;	/* input buffer */
	long buffer_size;
	long pos;		/* current pos in buffer */
	long stream_pos;	/* when reached, a new block must be read */
	uint32_t code;
	uint32_t range;
	long (*fill)(void*, unsigned long);
	bool at_stream_end;
	bool buffer_given;
};


static bool Rd_read_block(struct Range_decoder * const rdec)
{
	if (!rdec->at_stream_end) {
		rdec->stream_pos = rdec->fill ?
			rdec->fill(rdec->buffer, rdec->buffer_size) : 0;
		rdec->at_stream_end = (rdec->stream_pos < rdec->buffer_size);
		rdec->partial_member_pos += rdec->pos;
		rdec->pos = 0;
	}
	return rdec->pos < rdec->stream_pos;
}


static inline bool Rd_init(struct Range_decoder * const rdec,
				uint8_t * const inbuf, const long in_len,
				long (*fill)(void*, unsigned long))
{
	rdec->partial_member_pos = 0;
	rdec->buffer_given = (inbuf && in_len > 0);
	rdec->buffer_size = rdec->buffer_given ? in_len : 16384;
	rdec->buffer = rdec->buffer_given ? inbuf : malloc(rdec->buffer_size);
	if (!rdec->buffer)
		return false;
	rdec->pos = 0;
	rdec->stream_pos = rdec->buffer_given ? in_len : 0;
	rdec->code = 0;
	rdec->range = 0xFFFFFFFFU;
	rdec->fill = fill;
	rdec->at_stream_end = false;
	return true;
}

static inline void Rd_free(struct Range_decoder * const rdec)
{
	if (!rdec->buffer_given)
		free(rdec->buffer);
}

static inline bool Rd_finished(struct Range_decoder * const rdec)
{
	return rdec->pos >= rdec->stream_pos && !Rd_read_block(rdec);
}

static inline unsigned long long
Rd_member_position(const struct Range_decoder * const rdec)
{
	return rdec->partial_member_pos + rdec->pos;
}

static inline void Rd_reset_member_position(struct Range_decoder * const rdec)
{
	rdec->partial_member_pos = 0; rdec->partial_member_pos -= rdec->pos;
}

static inline uint8_t Rd_get_byte(struct Range_decoder * const rdec)
{
	/* 0xFF avoids decoder error if member is truncated at EOS marker */
	if (Rd_finished(rdec))
		return 0xFF;
	return rdec->buffer[rdec->pos++];
}

static inline bool Rd_load(struct Range_decoder * const rdec)
{
	int i;

	rdec->code = 0;
	rdec->range = 0xFFFFFFFFU;
	/* check first byte of the LZMA stream */
	if (Rd_get_byte(rdec) != 0)
		return false;
	for (i = 0; i < 4; ++i) rdec->code = (rdec->code << 8) | Rd_get_byte(rdec);
	return true;
}

static inline void Rd_normalize(struct Range_decoder * const rdec)
{
	if (rdec->range <= 0x00FFFFFFU) {
		rdec->range <<= 8;
		rdec->code = (rdec->code << 8) | Rd_get_byte(rdec);
	}
}

static inline unsigned Rd_decode(struct Range_decoder * const rdec,
				const int num_bits)
{
	unsigned symbol = 0;
	int i;

	for (i = num_bits; i > 0; --i) {
		bool bit;

		Rd_normalize(rdec);
		rdec->range >>= 1;
		/* symbol <<= 1; */
		/* if(rdec->code >= rdec->range) { rdec->code -= rdec->range; symbol |= 1; } */
		bit = (rdec->code >= rdec->range);
		symbol <<= 1; symbol += bit;
		rdec->code -= rdec->range & (0U - bit);
	}
	return symbol;
}

static inline unsigned Rd_decode_bit(struct Range_decoder * const rdec,
					Bit_model * const probability)
{
	uint32_t bound;

	Rd_normalize(rdec);
	bound = (rdec->range >> bit_model_total_bits) * *probability;
	if (rdec->code < bound) {
		rdec->range = bound;
		*probability += (bit_model_total - *probability) >> bit_model_move_bits;
		return 0;
	} else {
		rdec->range -= bound;
		rdec->code -= bound;
		*probability -= *probability >> bit_model_move_bits;
		return 1;
	}
}

static inline unsigned Rd_decode_tree3(struct Range_decoder * const rdec,
					Bit_model bm[])
{
	unsigned symbol = 2 | Rd_decode_bit(rdec, &bm[1]);

	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	return symbol & 7;
}

static inline unsigned Rd_decode_tree6(struct Range_decoder * const rdec,
					Bit_model bm[])
{
	unsigned symbol = 2 | Rd_decode_bit(rdec, &bm[1]);

	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	return symbol & 0x3F;
}

static inline unsigned Rd_decode_tree8(struct Range_decoder * const rdec,
					Bit_model bm[])
{
	unsigned symbol = 1;
	int i;

	for (i = 0; i < 8; ++i)
		symbol = (symbol << 1) | Rd_decode_bit(rdec, &bm[symbol]);
	return symbol & 0xFF;
}

static inline unsigned
Rd_decode_tree_reversed(struct Range_decoder * const rdec,
			Bit_model bm[], const int num_bits)
{
	unsigned model = 1;
	unsigned symbol = 0;
	int i;

	for (i = 0; i < num_bits; ++i) {
		const unsigned bit = Rd_decode_bit(rdec, &bm[model]);

		model <<= 1; model += bit;
		symbol |= (bit << i);
	}
	return symbol;
}

static inline unsigned
Rd_decode_tree_reversed4(struct Range_decoder * const rdec, Bit_model bm[])
{
	unsigned symbol = Rd_decode_bit(rdec, &bm[1]);

	symbol += Rd_decode_bit(rdec, &bm[2+symbol]) << 1;
	symbol += Rd_decode_bit(rdec, &bm[4+symbol]) << 2;
	symbol += Rd_decode_bit(rdec, &bm[8+symbol]) << 3;
	return symbol;
}

static inline unsigned Rd_decode_matched(struct Range_decoder * const rdec,
					Bit_model bm[], unsigned match_byte)
{
	unsigned symbol = 1;
	unsigned mask = 0x100;

	while (true) {
		const unsigned match_bit = (match_byte <<= 1) & mask;
		const unsigned bit = Rd_decode_bit(rdec, &bm[symbol+match_bit+mask]);

		symbol <<= 1; symbol += bit;
		if (symbol > 0xFF)
			return symbol & 0xFF;
		mask &= ~(match_bit ^ (bit << 8));	/* if( match_bit != bit ) mask = 0; */
	}
}

static inline unsigned Rd_decode_len(struct Range_decoder * const rdec,
						struct Len_model * const lm,
						const int pos_state)
{
	if (Rd_decode_bit(rdec, &lm->choice1) == 0)
		return min_match_len +
			Rd_decode_tree3(rdec, lm->bm_low[pos_state]);
	if (Rd_decode_bit(rdec, &lm->choice2) == 0)
		return min_match_len + len_low_symbols +
			Rd_decode_tree3(rdec, lm->bm_mid[pos_state]);
	return min_match_len + len_low_symbols + len_mid_symbols +
		Rd_decode_tree8(rdec, lm->bm_high);
}


struct LZ_decoder {
	unsigned long long partial_data_pos;
	struct Range_decoder *rdec;
	/* Don't move bm_* to LZd_decode_member; makes frame too large. */
	Bit_model bm_literal[1 << literal_context_bits][0x300];
	Bit_model bm_match[states][pos_states];
	Bit_model bm_rep[states];
	Bit_model bm_rep0[states];
	Bit_model bm_rep1[states];
	Bit_model bm_rep2[states];
	Bit_model bm_len[states][pos_states];
	Bit_model bm_dis_slot[len_states][1 << dis_slot_bits];
	Bit_model bm_dis[modeled_distances-end_dis_model+1];
	Bit_model bm_align[dis_align_size];
	struct Len_model match_len_model;
	struct Len_model rep_len_model;

	unsigned long buffer_size;
	unsigned dictionary_size;
	uint8_t *buffer;	/* output buffer */
	unsigned long pos;	/* current pos in buffer */
	unsigned long stream_pos;	/* first byte not yet written to file */
	uint32_t crc;
	long (*flush)(void*, unsigned long);
	bool pos_wrapped;
	bool buffer_given;
	bool write_error;
};

static void LZd_flush_data(struct LZ_decoder * const d)
{
	if (d->pos > d->stream_pos) {
		const long size = d->pos - d->stream_pos;

		CRC32_update_buf(&d->crc, d->buffer + d->stream_pos, size);
		if ((d->flush &&
		     d->flush(d->buffer + d->stream_pos, size) != size) ||
		    (!d->flush && d->pos_wrapped))
			d->write_error = true;
		if (d->pos >= d->buffer_size) {
			d->partial_data_pos += d->pos;
			d->pos = 0;
			d->pos_wrapped = true;
		}
		d->stream_pos = d->pos;
	}
}

static inline uint8_t LZd_peek_prev(const struct LZ_decoder * const d)
{
	if (d->pos > 0)
		return d->buffer[d->pos-1];
	if (d->pos_wrapped)
		return d->buffer[d->buffer_size-1];
	return 0;			/* prev_byte of first byte */
}

static inline uint8_t LZd_peek(const struct LZ_decoder * const d,
				const unsigned distance)
{
	const unsigned long i = ((d->pos > distance) ? 0 : d->buffer_size) +
				d->pos - distance - 1;
	return d->buffer[i];
}

static inline void LZd_put_byte(struct LZ_decoder * const d, const uint8_t b)
{
	d->buffer[d->pos] = b;
	if (++d->pos >= d->buffer_size)
		LZd_flush_data(d);
}

static inline void LZd_copy_block(struct LZ_decoder * const d,
				const unsigned distance, unsigned len)
{
	unsigned long lpos = d->pos, i = lpos - distance - 1;
	bool fast, fast2;

	if (lpos > distance) {
		fast = (len < d->buffer_size - lpos);
		fast2 = (fast && len <= lpos - i);
	} else {
		i += d->buffer_size;
		fast = (len < d->buffer_size - i);	/* (i == pos) may happen */
		fast2 = (fast && len <= i - lpos);
	}
	if (fast) {				/* no wrap */
		d->pos += len;
		if (fast2)			/* no wrap, no overlap */
			memcpy(d->buffer + lpos, d->buffer + i, len);
		else
			for (; len > 0; --len)
				d->buffer[lpos++] = d->buffer[i++];
	} else
		for (; len > 0; --len) {
			d->buffer[d->pos] = d->buffer[i];
			if (++d->pos >= d->buffer_size)
				LZd_flush_data(d);
			if (++i >= d->buffer_size)
				i = 0;
		}
}

static inline bool LZd_init(struct LZ_decoder * const d,
			struct Range_decoder * const rde,
			const unsigned dict_size, uint8_t * const outbuf,
			long out_size, long (*flush)(void*, unsigned long))
{
	d->partial_data_pos = 0;
	d->rdec = rde;
	Bm_array_init(d->bm_literal[0], (1 << literal_context_bits) * 0x300);
	Bm_array_init(d->bm_match[0], states * pos_states);
	Bm_array_init(d->bm_rep, states);
	Bm_array_init(d->bm_rep0, states);
	Bm_array_init(d->bm_rep1, states);
	Bm_array_init(d->bm_rep2, states);
	Bm_array_init(d->bm_len[0], states * pos_states);
	Bm_array_init(d->bm_dis_slot[0], len_states * (1 << dis_slot_bits));
	Bm_array_init(d->bm_dis, modeled_distances - end_dis_model + 1);
	Bm_array_init(d->bm_align, dis_align_size);
	Lm_init(&d->match_len_model);
	Lm_init(&d->rep_len_model);

	d->buffer_given = (outbuf && out_size > 0);
	d->buffer_size = d->buffer_given ? (unsigned long)out_size : dict_size;
	d->dictionary_size = min_t(unsigned long, d->buffer_size, dict_size);
	d->buffer = d->buffer_given ? outbuf : large_malloc(d->buffer_size);
	if (!d->buffer)
		return false;
	d->pos = 0;
	d->stream_pos = 0;
	d->crc = 0xFFFFFFFFU;
	d->flush = flush;
	d->pos_wrapped = false;
	d->write_error = false;
	/* prev_byte of first byte; also for LZd_peek( 0 ) on corrupt file */
	if (!d->buffer_given)		/* inbuf and outbuf may overlap */
		d->buffer[d->buffer_size-1] = 0;
	return true;
}

static inline void LZd_free(struct LZ_decoder * const d)
{
	if (!d->buffer_given)
		large_free(d->buffer);
}

static inline unsigned LZd_crc(const struct LZ_decoder * const d)
{
	return d->crc ^ 0xFFFFFFFFU;
}

static inline unsigned long long
LZd_data_position(const struct LZ_decoder * const d)
{
	return d->partial_data_pos + d->pos;
}


static bool LZd_check_trailer(struct LZ_decoder * const d)
{
	Lzip_trailer trailer;
	int i = 0;

	while (i < Lt_size)
		trailer[i++] = Rd_get_byte(d->rdec);

	return Lt_get_data_crc(trailer) == LZd_crc(d) &&
		Lt_get_data_size(trailer) == LZd_data_position(d) &&
		Lt_get_member_size(trailer) == Rd_member_position(d->rdec);
}


/* Return value: 0 = OK, < 0 = error (see include/linux/lzip.h). */
static int LZd_decode_member(struct LZ_decoder * const d)
{
	struct Range_decoder * const rdec = d->rdec;
	unsigned rep0 = 0;	/* rep[0-3] latest four distances */
	unsigned rep1 = 0;	/* used for efficient coding of */
	unsigned rep2 = 0;	/* repeated distances */
	unsigned rep3 = 0;
	State state = 0;

	if (!Rd_load(rdec))
		return LZIP_BAD_DATA;
	while (!Rd_finished(rdec)) {
		int len;
		const int pos_state = LZd_data_position(d) & pos_state_mask;

		if (Rd_decode_bit(rdec, &d->bm_match[state][pos_state]) == 0) {
			/* literal byte */
			Bit_model * const bm = d->bm_literal[get_lit_state(LZd_peek_prev(d))];

			if (St_is_char(state)) {
				state -= (state < 4) ? state : 3;
				LZd_put_byte(d, Rd_decode_tree8(rdec, bm));
			} else {
				state -= (state < 10) ? 3 : 6;
				LZd_put_byte(d, Rd_decode_matched(rdec, bm, LZd_peek(d, rep0)));
			}
		continue;
		}
		/* match or repeated match */
		if (Rd_decode_bit(rdec, &d->bm_rep[state]) != 0) {
			if (Rd_decode_bit(rdec, &d->bm_rep0[state]) == 0) {
				if (Rd_decode_bit(rdec, &d->bm_len[state][pos_state]) == 0) {
					state = St_set_shortrep(state);
					LZd_put_byte(d, LZd_peek(d, rep0));
					continue;
				}
			} else {
				unsigned distance;

				if (Rd_decode_bit(rdec, &d->bm_rep1[state]) == 0)
					distance = rep1;
				else {
					if (Rd_decode_bit(rdec, &d->bm_rep2[state]) == 0)
						distance = rep2;
					else {
						distance = rep3;
						rep3 = rep2;
					}
					rep2 = rep1;
				}
				rep1 = rep0;
				rep0 = distance;
			}
			state = St_set_rep(state);
			len = Rd_decode_len(rdec, &d->rep_len_model, pos_state);
		} else {			/* match */
			rep3 = rep2; rep2 = rep1; rep1 = rep0;
			len = Rd_decode_len(rdec, &d->match_len_model, pos_state);
			rep0 = Rd_decode_tree6(rdec, d->bm_dis_slot[get_len_state(len)]);
			if (rep0 >= start_dis_model) {
				const unsigned dis_slot = rep0;
				const int direct_bits = (dis_slot >> 1) - 1;

				rep0 = (2 | (dis_slot & 1)) << direct_bits;
				if (dis_slot < end_dis_model)
					rep0 += Rd_decode_tree_reversed(rdec,
						d->bm_dis + (rep0 - dis_slot), direct_bits);
				else {
					rep0 += Rd_decode(rdec, direct_bits - dis_align_bits) << dis_align_bits;
					rep0 += Rd_decode_tree_reversed4(rdec, d->bm_align);
					if (rep0 == 0xFFFFFFFFU) {	/* marker found */
						Rd_normalize(rdec);
						LZd_flush_data(d);
						if (d->write_error)
							return LZIP_WRITE_ERROR;
						if (len == min_match_len) {	/* End Of Stream marker */
							if (LZd_check_trailer(d))
								return 0;
							else
								return LZIP_BAD_CRC;
						}
						return LZIP_BAD_DATA;	/* unknown marker */
					}
				}
			}
			state = St_set_match(state);
			if (rep0 >= d->dictionary_size ||
			   (rep0 >= d->pos && !d->pos_wrapped)) {
				LZd_flush_data(d);
				return LZIP_BAD_DATA;
			}
		}
		LZd_copy_block(d, rep0, len);
	}
	LZd_flush_data(d);
	return LZIP_DATA_EOF;
}


int lzip_decompress(unsigned char *inbuf, long in_len,
			long (*fill)(void*, unsigned long),
			long (*flush)(void*, unsigned long),
			unsigned char *outbuf, long out_size,
			long *in_posp, long *out_posp)
{
	unsigned char *outptr = outbuf;
	struct Range_decoder rdec;
	struct LZ_decoder *decoder = 0;
	int retval = 0;
	bool first_member;
	bool empty = false, multi = false;

	if (in_posp)
		*in_posp = 0;
	if (out_posp)
		*out_posp = 0;

	if (!Rd_init(&rdec, inbuf, in_len, fill))
		return LZIP_OOM_INBUF;

	for (first_member = true;; first_member = false) {
		long data_pos;
		int size;
		unsigned dictionary_size;
		Lzip_header header;

		Rd_reset_member_position(&rdec);
		for (size = 0; size < Lh_size && !Rd_finished(&rdec); ++size)
			header[size] = Rd_get_byte(&rdec);
		if (Rd_finished(&rdec)) {	/* End Of File */
			if (first_member)
				retval = LZIP_HEADER1_EOF;
			else if (Lh_check_prefix(header, size))
				retval = LZIP_HEADER2_EOF;
			break;
		}
		if (!Lh_check_magic(header)) {
			if (first_member)
				retval = LZIP_BAD_MAGIC1;
			else if (Lh_check_corrupt(header))
				retval = LZIP_BAD_MAGIC2;
			break;
		}
		if (!Lh_check_version(header)) {
			retval = LZIP_BAD_VERSION;
			break;
		}
		dictionary_size = Lh_get_dictionary_size(header);
		if (dictionary_size < min_dictionary_size ||
		    dictionary_size > max_dictionary_size) {
			retval = LZIP_BAD_DICT_SIZE;
			break;
		}

		if (!decoder)
			decoder = malloc(sizeof *decoder);
		if (!decoder || !LZd_init(decoder, &rdec, dictionary_size,
					outptr, out_size, flush)) {
			retval = LZIP_OOM_OUTBUF;
			break;
		}
		retval = LZd_decode_member(decoder);
		if (in_posp)
			*in_posp += Rd_member_position(&rdec);
		multi = !first_member;
		data_pos = LZd_data_position(decoder);
		if (data_pos == 0) empty = true;
		if (outptr)
			outptr += data_pos;
		if (out_posp)
			*out_posp += data_pos;
		if (out_size > 0)
			out_size -= data_pos;
		LZd_free(decoder);
		if (retval != 0)
			break;
	}
	if (decoder)
		free(decoder);
	Rd_free(&rdec);
	if (empty && multi && retval == 0)
		retval = LZIP_EMPTY_MEMBER;
	return retval;
}

#ifndef STATIC
EXPORT_SYMBOL_GPL(lzip_decompress);
#endif
MODULE_DESCRIPTION("LZIP Decompressor");
MODULE_AUTHOR("Antonio Diaz Diaz <antonio@gnu.org>");
MODULE_LICENSE("GPL");
