/****************************************************************************
** libebml : parse EBML files, see http://embl.sourceforge.net/
**
** <file/class description>
**
** Copyright (C) 2002-2003 Steve Lhomme.  All rights reserved.
**
** This file is part of libebml.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding an other license may use this file in accordance with 
** the Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.matroska.org/license/qpl/ for QPL licensing information.
** See http://www.matroska.org/license/gpl/ for GPL licensing information.
**
** Contact license@matroska.org if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

/*!
	\file
	\version \$Id: EbmlCrc32.h,v 1.7 2003/11/03 22:14:27 robux4 Exp $
	\author Steve Lhomme     <robux4 @ users.sf.net>
	\author Jory Stone       <jcsston @ toughguy.net>
*/
#ifndef LIBEBML_CRC32_H
#define LIBEBML_CRC32_H

#include "EbmlConfig.h"
#include "EbmlTypes.h"
#include "EbmlBinary.h"

START_LIBEBML_NAMESPACE

const uint32 CRC32_NEGL = 0xffffffffL;

#if WORDS_BIGENDIAN
# define CRC32_INDEX(c) (c >> 24)
# define CRC32_SHIFTED(c) (c << 8)
#else
# define CRC32_INDEX(c) (c & 0xff)
# define CRC32_SHIFTED(c) (c >> 8)
#endif

class EbmlCrc32 : public EbmlBinary {
	public:
		EbmlCrc32();
		EbmlCrc32(const EbmlCrc32 & ElementToClone);
		static EbmlElement & Create() {return *(new EbmlCrc32);}
		const EbmlCallbacks & Generic() const {return ClassInfos;}
		bool ValidateSize() const {return (Size == 4);}
		uint32 RenderData(IOCallback & output, bool bForceRender, bool bSaveDefault = false);
		uint64 ReadData(IOCallback & input, ScopeMode ReadFully = SCOPE_ALL_DATA);
//		uint64 UpdateSize(bool bSaveDefault = false);
		
		static const EbmlCallbacks ClassInfos;
		bool IsDefaultValue() const {
			return false;
		}

		operator const EbmlId &() const {return ClassInfos.GlobalId;}
		bool IsYourId(const EbmlId & TestId) const;
	
		void AddElementCRC32(EbmlElement &ElementToCRC);
		bool CheckElementCRC32(EbmlElement &ElementToCRC);
		
		/*!
			CRC Checksum Calculation
		*/
		enum {DIGESTSIZE = 4};
		
		/*!
			Use this to quickly check a CRC32 with some data
			\return True if inputCRC matches CRC32 generated from input data
		*/
		static bool CheckCRC(uint32 inputCRC, const binary *input, uint32 length);
		/*!
			Calls Update() and Finalize(), use to create a CRC32 in one go
		*/
		void FillCRC32(const binary *input, uint32 length);
		/*!
			Add data to the CRC table, in other words process some data bit by bit
		*/
		void Update(const binary *input, uint32 length);
		/*!
			Use this with Update() to Finalize() or Complete the CRC32
		*/
		void Finalize();
		/*!
			Returns a uint32 that has the value of the CRC32
		*/
		uint32 GetCrc32() const {
			return m_crc_final;
		};
	
		void ForceCrc32(uint32 NewValue) { m_crc_final = NewValue; bValueIsSet = true;}

		EbmlElement * Clone() const;

	protected:
		void ResetCRC() {m_crc = CRC32_NEGL;}
		void UpdateByte(binary b) {m_crc = m_tab[CRC32_INDEX(m_crc) ^ b] ^ CRC32_SHIFTED(m_crc);}

		static const uint32 m_tab[256];
		uint32 m_crc;
		uint32 m_crc_final;
};

template <class T>
inline unsigned int GetAlignment(T *dummy=NULL) // VC60 workaround
{
#if (_MSC_VER >= 1300)
	return __alignof(T);
#elif defined(__GNUC__)
	return __alignof__(T);
#else
	return sizeof(T);
#endif
}

template <class T>
inline bool IsPowerOf2(T n)
{
	return n > 0 && (n & (n-1)) == 0;
}

template <class T1, class T2>
inline T2 ModPowerOf2(T1 a, T2 b)
{
	assert(IsPowerOf2(b));
	return T2(a) & (b-1);
}

inline bool IsAlignedOn(const void *p, unsigned int alignment)
{
	return IsPowerOf2(alignment) ? ModPowerOf2((unsigned int)p, alignment) == 0 : (unsigned int)p % alignment == 0;
}

template <class T>
inline bool IsAligned(const void *p, T *dummy=NULL)	// VC60 workaround
{
	return IsAlignedOn(p, GetAlignment<T>());
}

END_LIBEBML_NAMESPACE

#endif // LIBEBML_CRC32_H
