/*****************************************************************************
 * ps.c: PS and PES stream reader
 *****************************************************************************
 * Copyright (C) 1999, 2000 VideoLAN
 *
 * Authors:
 * Samuel Hocevar,     VIA, ECP, France <sam@via.ecp.fr>,    17/12/99
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*
 *  ps.c - PS stream reader
 *  this program is GPLed. Yes.
 */

/*
 * TODO: - rewrite it as a PS parser instead of a ps parser
 *           DONE
 *
 *       - autodetect audio and video channels
 *           DONE
 */

#include "vlms.h"
#include "ps.h"

/******************************************************************************
 * ps_thread
 ******************************************************************************
 * We use threading to allow cool non-blocking read from the disk. This
 * implicit thread is the disk (producer) thread, it reads packets from
 * the PS file on the disk, and stores them in a FIFO.
 ******************************************************************************/

void ps_thread(options_t *options)
{
    unsigned int left;
    int i;
    struct s_netthread netthread;
    pthread_t network_tid;
    struct s_ps ps;

    /* 4000 Go should be enough (FIXME: dunno how to do it nicer) */
    netthread.left = left = 0xffffffff;

    /* connect to the network to emit */
    netthread.out = open_socket( options, &netthread.remote_addr );
    /* Initialize the structures */
    own_pcr.start = own_pcr.end = 0; /* empty FIFO */
    pthread_mutex_init(&own_pcr.lock, NULL);
    in_data.start = in_data.end = 0; /* empty FIFO */
    pthread_mutex_init(&in_data.lock, NULL);
    pthread_cond_init(&in_data.notfull, NULL);
    pthread_cond_init(&in_data.notempty, NULL);
    in_data.b_die = 0;


    /* Arguments that were passed
     * in the command line */
    ps.audio_type = options->audio_type;
    ps.audio_channel = options->audio_channel;
    ps.subtitles_channel = options->subtitles_channel;

    /* Main structure initialization */
    ps.pes_type = NO_PES;
    ps.pes_id = 0;
    ps.private_id = 0;
    ps.pes_size = 0;
    ps.to_skip = 0;
    ps.pmt_counter = 0;
    ps.pat_counter = 0;
    for( i=0; i<256; i++ ) ps.association_table[i] = 0;
    ps.offset = 0;
    ps.scr = 0;
    ps.found_scr = 0;
    ps.found_streams = 0;
    ps.pcr_pid = options->pcr_pid;

    ps.ps_buffer = malloc( PS_BUFFER_SIZE );
    /* those 2 addresses are initialized so that a new packet is read */
    ps.ps_data = ps.ps_buffer + PS_BUFFER_SIZE - 1;
    ps.ps_end = ps.ps_buffer + PS_BUFFER_SIZE;

    /* set the first byte to zero because we know it was PS */
    ps.ps_data[0] = 0x00;

    /* The network (consumer) thread must be able to keep on emitting UDP
     * packets even when another process uses the hard disk. So we must fill
     * this buffer before.
     */
    ps_fill( options, &left, 0, &ps );

    /* Start the network thread */
    pthread_create( &network_tid, NULL, network_thread, &netthread );
    ps_fill( options, &left, 1, &ps );

    in_data.b_die = 1;
    /* Wait for the network thread to terminate */
    pthread_join( network_tid, NULL );
}

/******************************************************************************
 * ps_fill : Fill the data buffer with TS created from a PS file
 ******************************************************************************/

void ps_fill( options_t *options, unsigned int *left, int wait, ps_t *ps )
{
    int i, how_many;
    int pcr_flag;
    file_ts_packet *ts;

    /* How many TS packet for the next UDP packet */
    how_many = TS_IN_UDP;

    pcr_flag = 0;
    /* for every single TS packet */
    while( *left )
    {
        if( *left < TS_IN_UDP )
            how_many = *left;

        /* wait until we have one free item to store the UDP packet read */
        pthread_mutex_lock( &in_data.lock );
        while( (in_data.end+TS_BUFFER_SIZE+1-in_data.start)%(TS_BUFFER_SIZE+1) == TS_BUFFER_SIZE )
        {
            /* The buffer is full */
            if( wait )
            {
                pthread_cond_wait(&in_data.notfull, &in_data.lock);
            }
            else
            {
                pthread_mutex_unlock( &in_data.lock );
                if( !pcr_flag )
                {
                    printf( "Bad PCR PID\n" );
                    exit( 1 );
                }
                return;
            }
        }
        pthread_mutex_unlock(&in_data.lock);

        /* read a whole UDP packet from the file */
        ps->ts_to_write = how_many;
        if( ps_read( options, ps, ts = (file_ts_packet *)(in_data.buf + in_data.end) ) != how_many )
        {
            return;
        }

        /* Scan to mark TS packets containing a PCR */
        for( i=0; i<how_many; i++, ts++ )
        {
            pcr_flag |= keep_pcr( ps->pcr_pid, ts );
        }

        pthread_mutex_lock( &in_data.lock );
        in_data.end++;
        in_data.end %= TS_BUFFER_SIZE+1;
        pthread_cond_signal( &in_data.notempty );
        pthread_mutex_unlock( &in_data.lock );
        *left -= how_many;
    }
}

/******************************************************************************
 * ps_read : ps reading method
 ******************************************************************************/

ssize_t ps_read( options_t *options, ps_t *ps, void *ts )
{
    int i, pid, readbytes = 0;
    int datasize;
    ps->ts_written = 0;

    while( ps->ts_to_write )
    {

        /* Check that there is enough data to send,
         * otherwise read a few more bytes in the buffer */

        if( (datasize = ps->ps_end - ps->ps_data) <= TS_PACKET_SIZE )
        {
            /* copy the remaining bits at the beginning of the PS buffer */
            memmove ( ps->ps_buffer, ps->ps_data, datasize );

            /* read some bytes */
            readbytes = safe_read( options, ps->ps_buffer + datasize,
                                   PS_BUFFER_SIZE - datasize );

            ps->ps_data = ps->ps_buffer;
            ps->ps_end = ps->ps_data + datasize + readbytes;

            if(readbytes == 0 && ps->ps_end - ps->ps_data < TS_PACKET_SIZE)
            {
                fprintf ( stderr, "end of PS file\n" );
                return -1;
            }
        }

        /* If we are at the beginning of a new MPEG sequence, check
         * that it says 00 00 01 xx, then read its PTS if it is a PES */

        if( ps->to_skip == 0 && ps->offset == ps->pes_size )
        {
            while( ps->ps_data[0] || ps->ps_data[1] || (ps->ps_data[2] != 0x01)  || (ps->ps_data[3] < 0xb9) )
            {
                fprintf ( stderr, "Error: not a startcode: %.2x %.2x %.2x "
                          "instead of 00 00 01\n", ps->ps_data[0],
                          ps->ps_data[1], ps->ps_data[2] );
                ps->ps_data++;
                if( (datasize = ps->ps_end - ps->ps_data) <= TS_PACKET_SIZE )
                {
                    memmove( ps->ps_buffer, ps->ps_data, datasize );
                    readbytes = safe_read( options, ps->ps_buffer + datasize,
                                           PS_BUFFER_SIZE - datasize );
                    if(readbytes == 0)
                    {
                        fprintf ( stderr, "end of PS file 2\n" );
                        return -1;
                    }
                    ps->ps_data = ps->ps_buffer;
                    ps->ps_end = ps->ps_data + datasize + readbytes;
                }
            }

            if( ps->ps_data[3] == 0xba )
            {
                /* Pack header : Read the SCR */
                if( (ps->ps_data[4] & 0xC0) == 0x40 )
                {
                    /* MPEG-2 */
                    ps->scr = ((u64)(ps->ps_data[4] & 0x38) << 27) |
                         ((u64)(U32_AT(ps->ps_data + 4) & 0x03FFF800)
                                        << 4) |
                         ((( ((u32)U16_AT(ps->ps_data + 6) << 16)
                            | (u32)U16_AT(ps->ps_data + 8) ) & 0x03FFF800)
                                        >> 11);
                }
                else
                {
                    /* MPEG-1 */
                    ps->scr = ((u64)(ps->ps_data[4] & 0x0E) << 29) |
                         (((u64)U32_AT(ps->ps_data + 4) & 0xFFFE00) << 6) |
                         ((u64)ps->ps_data[7] << 7) |
                         ((u64)ps->ps_data[8] >> 1);
                }
            }

            ps->pes_type = NO_PES;
            ps->offset = 0;
            ps->pes_size = (ps->ps_data[4] << 8) + ps->ps_data[5] + 6;
        }

        /* if the actual data we have in pes_data is not a PES, then
         * we read the next one. */
        if( (ps->pes_type == NO_PES) && !ps->to_skip )
        {
            ps->pes_id = ps->ps_data[3];

            if (ps->pes_id == 0xbd)
            {
                ps->private_id = ps->ps_data[ 9 + ps->ps_data[8] ];
                if ((ps->private_id & 0xf0) == 0x80)
                {
                    /* ac3 audio stream (0x80 - 0x8f) */
                    ps->pes_type = AC3_PES;
                }
                else if ((ps->private_id & 0xe0) == 0x20)
                {
                    /* subtitles (0x20 - 0x3f) */
                    ps->pes_type = SUBTITLE_PES;
                }
                else if ((ps->private_id & 0xf0) == 0xa0)
                {
                    /* lpcm audio stream (0xa0 - 0xaf) */
                    ps->pes_type = LPCM_PES;
                }
                else
                {
                    /* unknown private data */
                    ps->pes_type = PRIVATE_PES;
                    ps->to_skip = ps->pes_size;
                }
            }
            else if ((ps->pes_id & 0xe0) == 0xc0)
            {
                /* audio stream */
                ps->pes_type = AUDIO_PES;
            }
            else if ((ps->pes_id & 0xf0) == 0xe0)
            {
                /* video stream */
                ps->pes_type = VIDEO_PES;
            }
            else if (ps->pes_id == 0xba)
            {
                /* pack header */
                int i = 4;
                ps->pes_type = NO_PES;

                if( (ps->ps_data[i] & 0xC0) == 0x40 )
                {
                    i += 10 + ( ps->ps_data[i+9] & 0x07 );
                }
                else
                {
                    i += 8;
                }
                ps->pes_size = ps->to_skip = i;
            }
            else if (ps->pes_id == 0xb9)
            {
                /* sequence end stream */
                ps->pes_size = ps->to_skip = 4;
            }
            else
            {
                ps->pes_type = UNKNOWN_PES;
                ps->to_skip = ps->pes_size;
            }
        }

        if( ps->to_skip )
        {
            if( ps->to_skip < TS_PACKET_SIZE )
            {
                ps->ps_data += ps->to_skip;
                ps->offset  += ps->to_skip;
                ps->to_skip  = 0;
            }
            else
            {
                ps->ps_data += TS_PACKET_SIZE;
                ps->offset  += TS_PACKET_SIZE;
                ps->to_skip -= TS_PACKET_SIZE;
            }
        }

        /* now that we know what we have, we can either
         * write this packet's data in the buffer, skip it,
         * or write a PMT or PAT table and wait for the next
         * turn before writing the packet. */
        switch (ps->sent_ts & 0xff)
        {
            case 0x01:
                write_pmt(ps,ts);
                ps->pmt_counter++;
                ps->ts_to_write--; ps->ts_written++; ts+=188;
                break;

            case 0x00:
                write_pat(ps,ts);
                ps->pat_counter++;
                ps->ts_to_write--; ps->ts_written++; ts+=188;
                break;
        }

        /* if there's still no found PCR_PID, and no PTS in this PES,
         * then we just trash it */
        if (!ps->found_scr)
        {
            if (ps->scr)
            {
                fprintf( stderr, "Found a PTS - entering main loop\n" );
                ps->found_scr = 1;
            }
            else
            {
                ps->pes_type = NO_PES;
            }
        }

        if (ps->ts_to_write)
        {
            switch(ps->pes_type)
            {
                case VIDEO_PES:
                case AUDIO_PES:
                case AC3_PES:
                case SUBTITLE_PES:
                case LPCM_PES:
                    pid = get_pid (ps);
                    write_media_ts(ps, ts, pid);
                    ps->ts_to_write--; ps->ts_written++; ts+=188;
                    ps->media_counter[pid]++;
                    break;
                case UNKNOWN_PES:
                default:
                    //fprintf(stderr, "setting PES type to NO_PES\n");
                    ps->pes_type = NO_PES;
                    break;
            }
        }
    }

    //ps->ps_data += TS_PACKET_SIZE;

    return ps->ts_written;
}

/******************************************************************************
 * get_pid : gets a pid from a PES type
 ******************************************************************************/

int get_pid (ps_t *ps)
{
    int i, tofind, delta;
    char* type;

    switch(ps->pes_type)
    {
        case VIDEO_PES:
            delta = 0x20; /* 0x20 - 0x2f */
            type = "MPEG video";
            tofind = ps->pes_id;
            break;
        case AUDIO_PES:
            delta = 0x40; /* 0x40 - 0x5f */
            type = "MPEG audio";
            tofind = ps->pes_id;
            break;
        /* XXX: 0x64 is for the PMT, so don't take it !!! */
        case AC3_PES:
            delta = 0x80; /* 0x80 - 0x8f */
            type = "MPEG private (AC3 audio)";
            tofind = ps->private_id;
            break;
        case LPCM_PES:
            delta = 0x90; /* 0x90 - 0x9f */
            type = "MPEG private (LPCM audio)";
            tofind = ps->private_id;
            break;
        case SUBTITLE_PES:
            delta = 0xa0; /* 0xa0 - 0xbf */
            type = "MPEG private (DVD subtitle)";
            tofind = ps->private_id;
            break;
        default:
            return(-1);
    }

    i = delta + (tofind & 0x1f);

    if( ps->association_table[i] == 0)
    {
        fprintf( stderr, "Found %s stream at 0x%.2x, allocating PID 0x%.2x\n",
                 type, tofind, i );
        ps->association_table[i] = 1;
    }

    return ( i );

}

/******************************************************************************
 * write_media_ts : writes a ts packet from a ps stream
 ******************************************************************************/

void write_media_ts(ps_t *ps, unsigned char *ts, unsigned int pid)
{
    int i,j;
    long int extclock;

    /* if offset == 0, it means we haven't examined the PS yet */
    if (ps->offset == 0)
    {
        if (ps->pes_size < 184) {

            ts[0] = 0x47;    /* sync_byte */
            ts[1] = 0x40;    /* payload_unit_start_indicator si dbut de PES */
            ts[2] = pid;

            ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
            ts[4] = 184 - ps->pes_size - 1;
            ts[5] = 0x00;
            /* needed stuffing bytes */
            for (i=6 ; i < 188 - ps->pes_size ; i++) ts[i]=0xff;
            memcpy(ts + 188 - ps->pes_size, ps->ps_data, ps->pes_size);

            /* this PS is finished, next time we'll pick a new one */
            ps->pes_type = NO_PES;
            ps->ps_data += ps->pes_size;
            ps->offset += ps->pes_size;
            return;

        }
    }

    /* now we still can have offset == 0, but size is initialized */

    /* sync_byte */
    ts[0] = 0x47;
    /* payload_unit_start_indicator si dbut de PES */
    ts[1] = (ps->offset == 0) ? 0x40 : 0x00;
    /* the PID */
    ts[2] = pid;

    if ( (ps->offset == 0) && ps->scr && (ps->pcr_pid == pid) )
    {

        ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
        ts[4] = 0x07; /* taille de l'adaptation field */
        ts[5] = 0x50; /* rtfm */

        //fprintf( stderr, "writing PTS %.16llx\n", ps->scr );
        extclock = 0x000;

        /* ---111111110000000000000000000000000 */
        ts[6] = (ps->scr & 0x1fe000000) >> 25;
        /* ---000000001111111100000000000000000 */
        ts[7] = (ps->scr & 0x001fe0000) >> 17;
        /* ---000000000000000011111111000000000 */
        ts[8] = (ps->scr & 0x00001fe00) >>  9;
        /* ---000000000000000000000000111111110 */
        ts[9] = (ps->scr & 0x0000001fe) >>  1;

        ts[10] = 0x7e + ((ps->scr & 0x01) << 7) + ((extclock & 0x100) >> 8);
        ts[11] = extclock & 0xff;

        ps->scr = 0;

        memcpy( ts + 4 + 8, ps->ps_data, 184 - 8 );

        ts[15] = 0xe0; /* FIXME : we don't know how to choose program yet */

        ps->offset += 184 - 8;
        ps->ps_data += 184 - 8;
    }
    else if (ps->offset <= ps->pes_size - 184)
    {

        ts[3] = 0x10 + (ps->media_counter[pid] & 0x0f);
        memcpy(ts + 4, ps->ps_data, 184);

        ps->offset += 184;
        ps->ps_data += 184;

    }
    else
    {
        j = ps->pes_size - ps->offset;
        ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
        ts[4] = 184 - j - 1;
        ts[5] = 0x00;
        for (i=6 ; i < 188 - j ; i++) ts[i]=0xff; /* needed stuffing bytes */
        memcpy(ts + 4 + 184 - j, ps->ps_data, j);
        ps->offset += j; /* offset = size */
        ps->ps_data += j; /* offset = size */

        /* the PES is finished */
        ps->pes_type = NO_PES;
        ps->sent_ts++;
    }

    //fprintf(stderr, "[PES size: %i]\n", ps->pes_size);
}

/******************************************************************************
 * write_pat : writes a program association table
 ******************************************************************************/

void write_pat(ps_t *ps, unsigned char *ts)
{
    int i;

    ts[0] = 0x47; /* sync_byte */
    ts[1] = 0x40;
    ts[2] = 0x00; /* PID = 0x0000 */
    ts[3] = 0x10 | (ps->pat_counter & 0x0f);
    ts[4] = 0x00;
    
    ts[5] = 0x00; /* 0x00: Program association section */

    ts[6] = 0xb0; /* */
    ts[7] = 0x11; /* section_length = 0x011 */

    ts[8] = 0x00;
    ts[9] = 0xbb; /* TS id = 0x00b0 (what the vlc calls "Stream ID") */

    ts[10] = 0xc1;
    /* section # and last section # */
    ts[11] = ts[12] = 0x00;

    /* Network PID (useless) */
    ts[13] = ts[14] = 0x00; ts[15] = 0xe0; ts[16] = 0x10;

    /* Program Map PID */
    ts[17] = 0x03; ts[18] = 0xe8; ts[19] = 0xe0; ts[20] = 0x64;

    /* CRC */
    ts[21] = 0x4d; ts[22] = 0x6a; ts[23] = 0x8b; ts[24] = 0x0f;

    /* needed stuffing bytes */
    for (i=25 ; i < 188 ; i++) ts[i]=0xff;

    ps->sent_ts++;
}

/******************************************************************************
 * write_pmt : writes a program map table
 ******************************************************************************/

void write_pmt(ps_t *ps, unsigned char *ts)
{
    int i;

    ts[0] = 0x47; /* sync_byte */
    ts[1] = 0x40;
    ts[2] = 0x0064; /* PID = 0x0064 */
    ts[3] = 0x10 | (ps->pmt_counter & 0x0f);
    ts[4] = 0x00;

    ts[5] = 0x02; /* 0x02: Program map section */

    ts[6] = 0xb0; /* */
    ts[7] = 0x34; /* section_length = 0x034 */

    ts[8] = 0x03;
    ts[9] = 0xe8; /* prog number */

    ts[10] = 0xc1;
    /* section # and last section # */
    ts[11] = ts[12] = 0x00;

    /* PCR PID */
    ts[13] = 0xe0;
    ts[14] = 0x20;

    /* program_info_length == 0 */
    ts[15] = 0xf0; ts[16] = 0x00;

    /* Program Map / Video PID */
    ts[17] = 0x02; /* stream type = video */
    ts[18] = 0xe0; ts[19] = 0x20;
    ts[20] = 0xf0; ts[21] = 0x09; /* es info length */
    /* useless info */
    ts[22] = 0x07; ts[23] = 0x04; ts[24] = 0x08; ts[25] = 0x80; ts[26] = 0x24;
    ts[27] = 0x02; ts[28] = 0x11; ts[29] = 0x01; ts[30] = 0xfe;

    switch ( ps->audio_type )
    {
    case REQUESTED_AC3 :
        /* ac3 */
        ts[31] = 0x81; /* stream type = audio */
        ts[32] = 0xe0; ts[33] = 0x80 + ps->audio_channel;
        ts[34] = 0xf0; ts[35] = 0x00; /* es info length */
        break;

    case REQUESTED_MPEG :
        /* mpeg */
        ts[31] = 0x04; /* stream type = audio */
        ts[32] = 0xe0; ts[33] = 0x40 + ps->audio_channel;
        ts[34] = 0xf0; ts[35] = 0x00; /* es info length */
        break;

    case REQUESTED_LPCM :
        /* LPCM audio */
        ts[31] = 0x81;
        ts[32] = 0xe0; ts[33] = 0xa0 + ps->audio_channel;
        ts[34] = 0xf0; ts[35] = 0x00; /* es info  length */
        break;

    default :
        /* No audio */
        ts[31] = 0x00;
        ts[35] = 0x00; /* es info  length */
    }

    /* Subtitles */
    if( ps->subtitles_channel == NO_SUBTITLES )
    {
        ts[36] = 0x00;
        ts[37] = 0x00; ts[38] = 0x00;
        ts[39] = 0xf0; ts[40] = 0x00; /* es info length */
    }
    else
    {
        ts[36] = 0x82;
        ts[37] = 0xe0; ts[38] = 0xa0 + ( ps->subtitles_channel );
        ts[39] = 0xf0; ts[40] = 0x00; /* es info length */
        //fprintf(stderr, "subtitle %.2x\n", 0xa0 + ( ps->subtitles_channel ) );
    }

    /* CRC FIXME: not calculated yet*/
    ts[41] = 0x96; ts[42] = 0x70; ts[43] = 0x0b; ts[44] = 0x7c;

    /* needed stuffing bytes */
    for (i=45 ; i < 188 ; i++) ts[i]=0xff;

    ps->sent_ts++;
}


