/*****************************************************************************
 * ps.c: TS stream reader
 *****************************************************************************
 * Copyright (C) 1997, 1998, 1999, 2000 VideoLAN
 *
 * Authors:
 * Regis Duchesne,     VIA, ECP, France <regis@via.ecp.fr>,  27/01/97
 * Michel Lespinasse,  VIA, ECP, France <walken@via.ecp.fr>, 02/02/97
 * Jean-Marc Dressler, VIA, ECP, France <polux@via.ecp.fr>,  17/05/99
 * Samuel Hocevar,     VIA, ECP, France <sam@via.ecp.fr>,    17/12/99
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*
 *  ts.c - TS stream reader
 *  this program is GPLed. Yes.
 */

#include "vlms.h"
#include "ts.h"

/******************************************************************************
 * ts_thread
 ******************************************************************************
 * We use threading to allow cool non-blocking read from the disk. This
 * implicit thread is the disk (producer) thread, it reads packets from
 * the TS file on the disk, and stores them in a FIFO.
 ******************************************************************************/

void ts_thread( options_t *options )
{
    unsigned int left;
    struct stat in_stat;
    struct s_netthread netthread;
    pthread_t network_tid;

    /* open the TS file for reading */
    if( options->in == 0 )
    {
	/* 200 Go should be enough (FIXME: dunno how to do it nicer) */
        netthread.left = left = 1000000000;
    }
    else
    {
        /* get the file size */
        fstat(options->in, &in_stat);
        if( in_stat.st_size % TS_PACKET_SIZE )
        {
            fprintf( stderr,"Wrong file size, I will emit a truncated file.\n" );
        }
        netthread.left = left = in_stat.st_size / TS_PACKET_SIZE;
    }

    /* connect to the network to emit */
    netthread.out = open_socket( options, &netthread.remote_addr );
    /* Initialize the structures */
    own_pcr.start = own_pcr.end = 0; /* empty FIFO */
    pthread_mutex_init( &own_pcr.lock, NULL );
    in_data.start = in_data.end = 0; /* empty FIFO */
    pthread_mutex_init( &in_data.lock, NULL );
    pthread_cond_init( &in_data.notfull, NULL );
    pthread_cond_init( &in_data.notempty, NULL );
    in_data.b_die = 0;
    /* The network (consumer) thread must be able to keep on emitting UDP
    * packets even when another process uses the hard disk. So we must fill
    * this buffer before.
    */
    ts_fill( options, &left, 0, options->pcr_pid );
    
    /* start the network thread */
    pthread_create( &network_tid, NULL, network_thread, &netthread );
    ts_fill( options, &left, 1, options->pcr_pid );
   
    /* Wait for the network thread to terminate */
    pthread_join( network_tid, NULL );

    fprintf( stderr, "finished\n" );
}



/******************************************************************************
 * ts_fill : Fill the data buffer
 ******************************************************************************/

void ts_fill( options_t *options, unsigned int *left, int wait, int pcr_pid )
{
    int i, how_many;
    int pcr_flag;
    file_ts_packet *ts;
    int first_time = 1 - wait;
    
    /* How many TS packet for the next UDP packet */
    how_many = TS_IN_UDP;
    
    pcr_flag = 0;
    while( *left )
    {
        if( *left < TS_IN_UDP )
            how_many = *left;
        
        /* wait until we have one free item to store the UDP packet read */
        pthread_mutex_lock( &in_data.lock );
        while( (in_data.end+TS_BUFFER_SIZE+1-in_data.start)%(TS_BUFFER_SIZE+1) == TS_BUFFER_SIZE )
        {
            /* The buffer is full */
            if( wait )
            {
                pthread_cond_wait( &in_data.notfull, &in_data.lock );
            }
            else
            {
                pthread_mutex_unlock( &in_data.lock );
                if( !pcr_flag )
                {
                    fprintf (stderr,"mauvais PCR_PID\n");
                    exit (1);
                }
                return;
            }
        }
        pthread_mutex_unlock(&in_data.lock);
        
        /* read an UDP packet from the file - beware the first byte */
        if( first_time )
        {
            if( safe_read( options, ((unsigned char *)ts =
                           (file_ts_packet *)(in_data.buf + in_data.end)) + 1,
                             how_many * TS_PACKET_SIZE - 1 )
                           != how_many * TS_PACKET_SIZE - 1)
            {
                err_sys("read() error");
            }
            ((unsigned char *)ts)[0] = 0x47;
            first_time = 0;
        }
        else
        {
            if(safe_read( options, (unsigned char *)ts =
                          (file_ts_packet *)(in_data.buf + in_data.end),
                            how_many * TS_PACKET_SIZE)
                          != how_many*TS_PACKET_SIZE )
            {
                err_sys("read() error");
            }
        }
        
        /* Scan to mark TS packets containing a PCR */
        for( i=0; i<how_many; i++, ts++ )
        {
            pcr_flag |= keep_pcr( pcr_pid, ts );
        }
        
        pthread_mutex_lock( &in_data.lock );
        in_data.end++;
        in_data.end %= TS_BUFFER_SIZE+1;
        pthread_cond_signal( &in_data.notempty );
        pthread_mutex_unlock( &in_data.lock );
        *left -= how_many;
    }
}

