/* packet-dis-enums.c
 * String definitions for DIS enumerations.
 * Copyright 2005, Scientific Research Corporation
 * Initial implementation by Jeremy Ouellette <jouellet@scires.com>
 *
 * $Id$
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "packet-dis-enums.h"
#include <epan/value_string.h>

const value_string DIS_PDU_ProtocolVersion_Strings[] =
{
    { DIS_VERSION_OTHER,             "Other" },
    { DIS_VERSION_1_0,               "DIS PDU version 1.0 (May 92)" },
    { DIS_VERSION_IEEE_1278_1993,    "IEEE 1278-1993" },
    { DIS_VERSION_2_0_3RD_DRAFT,     "DIS PDU version 2.0 - third draft (May 93)" },
    { DIS_VERSION_2_0_4TH_DRAFT,     "DIS PDU version 2.0 - fourth draft (revised) March 16, 1994" },
    { DIS_VERSION_IEEE_1278_1_1995,  "IEEE 1278.1-1995" },
    { DIS_VERSION_IEEE_1278_1A_1998, "IEEE 1278.1A-1998" },
    { DIS_VERSION_IEEE_1278_1_200X,  "IEEE 1278.1-200X" },
    { 0,                             NULL }
};

const value_string DIS_PDU_ProtocolFamily_Strings[] =
{
    { DIS_PROTOCOLFAMILY_OTHER,                                  "Other" },
    { DIS_PROTOCOLFAMILY_ENTITY_INFORMATION_INTERACTION,         "Entity information / interaction" },
    { DIS_PROTOCOLFAMILY_WARFARE,                                "Warfare" },
    { DIS_PROTOCOLFAMILY_LOGISTICS,                              "Logistics" },
    { DIS_PROTOCOLFAMILY_RADIO_COMMUNICATIONS,                   "Radio communications" },
    { DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT,                  "Simulation management" },
    { DIS_PROTOCOLFAMILY_DISTRIBUTED_EMISSION_REGENERATION,      "Distributed emission regeneration" },
    { DIS_PROTOCOLFAMILY_ENTITY_MANAGEMENT,                      "Entity management" },
    { DIS_PROTOCOLFAMILY_MINEFIELD,                              "Minefield" },
    { DIS_PROTOCOLFAMILY_SYNTHETIC_ENVIRONMENT,                  "Synthetic environment" },
    { DIS_PROTOCOLFAMILY_SIMULATION_MANAGEMENT_WITH_RELIABILITY, "Simulation management with reliability" },
    { DIS_PROTOCOLFAMILY_LIVE_ENTITY,                            "Live entity" },
    { DIS_PROTOCOLFAMILY_NON_REAL_TIME,                          "Non-real time" },
    { DIS_PROTOCOLFAMILY_INFORMATION_OPERATIONS,                 "Information Operations" },
    { DIS_PROTOCOLFAMILY_EXPERIMENTAL_COMPUTER_GENERATED_FORCES, "Experimental - Computer Generated Forces" },
    { DIS_PROTOCOLFAMILY_EXPERIMENTAL_VDIS,                      "Experimental - V-DIS" },
    { DIS_PROTOCOLFAMILY_PERSISTENT_OBJECT,                      "Persistent object" },
    { DIS_PROTOCOLFAMILY_EXPERIMENTAL,                           "Experimental" },
    { 0,                                                         NULL }
};

const value_string DIS_PDU_Type_Strings[] =
{
    { DIS_PDUTYPE_OTHER,                              "Other" },
    { DIS_PDUTYPE_ENTITY_STATE,                       "Entity State" },
    { DIS_PDUTYPE_FIRE,                               "Fire" },
    { DIS_PDUTYPE_DETONATION,                         "Detonation" },
    { DIS_PDUTYPE_COLLISION,                          "Collision" },
    { DIS_PDUTYPE_SERVICE_REQUEST,                    "Service Request" },
    { DIS_PDUTYPE_RESUPPLY_OFFER,                     "Resupply Offer" },
    { DIS_PDUTYPE_RESUPPLY_RECEIVED,                  "Resupply Received" },
    { DIS_PDUTYPE_RESUPPLY_CANCEL,                    "Resupply Cancel" },
    { DIS_PDUTYPE_REPAIR_COMPLETE,                    "Repair Complete" },
    { DIS_PDUTYPE_REPAIR_RESPONSE,                    "Repair Response" },
    { DIS_PDUTYPE_CREATE_ENTITY,                      "Create Entity" },
    { DIS_PDUTYPE_REMOVE_ENTITY,                      "Remove Entity" },
    { DIS_PDUTYPE_START_RESUME,                       "Start / Resume" },
    { DIS_PDUTYPE_STOP_FREEZE,                        "Stop / Freeze" },
    { DIS_PDUTYPE_ACKNOWLEDGE,                        "Acknowledge" },
    { DIS_PDUTYPE_ACTION_REQUEST,                     "Action Request" },
    { DIS_PDUTYPE_ACTION_RESPONSE,                    "Action Response" },
    { DIS_PDUTYPE_DATA_QUERY,                         "Data Query" },
    { DIS_PDUTYPE_SET_DATA,                           "Set Data" },
    { DIS_PDUTYPE_DATA,                               "Data" },
    { DIS_PDUTYPE_EVENT_REPORT,                       "Event Report" },
    { DIS_PDUTYPE_COMMENT,                            "Comment" },
    { DIS_PDUTYPE_ELECTROMAGNETIC_EMISSION,           "Electromagnetic Emission" },
    { DIS_PDUTYPE_DESIGNATOR,                         "Designator" },
    { DIS_PDUTYPE_TRANSMITTER,                        "Transmitter" },
    { DIS_PDUTYPE_SIGNAL,                             "Signal" },
    { DIS_PDUTYPE_RECEIVER,                           "Receiver" },
    { DIS_PDUTYPE_IFF_ATC_NAVAIDS,                    "IFF / ATC / NAVAIDS" },
    { DIS_PDUTYPE_UNDERWATER_ACOUSTIC,                "Underwater Acoustic" },
    { DIS_PDUTYPE_SUPPLEMENTAL_EMISSION_ENTITY_STATE, "Supplemental Emission Entity State" },
    { DIS_PDUTYPE_INTERCOM_SIGNAL,                    "Intercom Signal" },
    { DIS_PDUTYPE_INTERCOM_CONTROL,                   "Intercom Control" },
    { DIS_PDUTYPE_AGGREGATE_STATE,                    "Aggregate State" },
    { DIS_PDUTYPE_IS_GROUP_OF,                        "IsGroupOf" },
    { DIS_PDUTYPE_TRANSFER_CONTROL,                   "Transfer Control" },
    { DIS_PDUTYPE_IS_PART_OF,                         "IsPartOf" },
    { DIS_PDUTYPE_MINEFIELD_STATE,                    "Minefield State" },
    { DIS_PDUTYPE_MINEFIELD_QUERY,                    "Minefield Query" },
    { DIS_PDUTYPE_MINEFIELD_DATA,                     "Minefield Data" },
    { DIS_PDUTYPE_MINEFIELD_RESPONSE_NAK,             "Minefield Response NAK" },
    { DIS_PDUTYPE_ENVIRONMENTAL_PROCESS,              "Environmental Process" },
    { DIS_PDUTYPE_GRIDDED_DATA,                       "Gridded Data" },
    { DIS_PDUTYPE_POINT_OBJECT_STATE,                 "Point Object State" },
    { DIS_PDUTYPE_LINEAR_OBJECT_STATE,                "Linear Object State" },
    { DIS_PDUTYPE_AREAL_OBJECT_STATE,                 "Areal Object State" },
    { DIS_PDUTYPE_TSPI,                               "TSPI" },
    { DIS_PDUTYPE_APPEARANCE,                         "Appearance" },
    { DIS_PDUTYPE_ARTICULATED_PARTS,                  "Articulated Parts" },
    { DIS_PDUTYPE_LE_FIRE,                            "LE Fire" },
    { DIS_PDUTYPE_LE_DETONATION,                      "LE Detonation" },
    { DIS_PDUTYPE_CREATE_ENTITY_R,                    "Create Entity-R" },
    { DIS_PDUTYPE_REMOVE_ENTITY_R,                    "Remove Entity-R" },
    { DIS_PDUTYPE_START_RESUME_R,                     "Start / Resume-R" },
    { DIS_PDUTYPE_STOP_FREEZE_R,                      "Stop / Freeze-R" },
    { DIS_PDUTYPE_ACKNOWLEDGE_R,                      "Acknowledge-R" },
    { DIS_PDUTYPE_ACTION_REQUEST_R,                   "Action Request-R" },
    { DIS_PDUTYPE_ACTION_RESPONSE_R,                  "Action Response-R" },
    { DIS_PDUTYPE_DATA_QUERY_R,                       "Data Query-R" },
    { DIS_PDUTYPE_SET_DATA_R,                         "Set Data-R" },
    { DIS_PDUTYPE_DATA_R,                             "Data-R" },
    { DIS_PDUTYPE_EVENT_REPORT_R,                     "Event Report-R" },
    { DIS_PDUTYPE_COMMENT_R,                          "Comment-R" },
    { DIS_PDUTYPE_RECORD_R,                           "Record-R" },
    { DIS_PDUTYPE_SET_RECORD_R,                       "Set Record-R" },
    { DIS_PDUTYPE_RECORD_QUERY_R,                     "Record Query-R" },
    { DIS_PDUTYPE_COLLISION_ELASTIC,                  "Collision Elastic" },
    { DIS_PDUTYPE_ENTITY_STATE_UPDATE,                "Entity State Update" },
    { DIS_PDUTYPE_DIRECTED_ENERGY_FIRE,               "Directed Energy Fire" },
    { DIS_PDUTYPE_ENTITY_DAMAGE_STATUS,               "Entity Damage Status" },
    { DIS_PDUTYPE_INFORMATION_OPERATIONS_ACTION,      "Info Operations Action" },
    { DIS_PDUTYPE_INFORMATION_OPERATIONS_REPORT,      "Info Operations Report" },
    { DIS_PDUTYPE_ATTRIBUTE,                          "Attribute" },
    { DIS_PDUTYPE_ANNOUNCE_OBJECT,                    "Announce Object" },
    { DIS_PDUTYPE_DELETE_OBJECT,                      "Delete Object" },
    { DIS_PDUTYPE_DESCRIBE_APPLICATION,               "Describe Application" },
    { DIS_PDUTYPE_DESCRIBE_EVENT,                     "Describe Event" },
    { DIS_PDUTYPE_DESCRIBE_OBJECT,                    "Describe Object" },
    { DIS_PDUTYPE_REQUEST_EVENT,                      "Request Event" },
    { DIS_PDUTYPE_REQUEST_OBJECT,                     "Request Object" },
    { DIS_PDUTYPE_APPLICATION_CONTROL,                "Application Control" },
    { DIS_PDUTYPE_STEALTH_STATE,                      "Stealth State" },
    { 0,                                              NULL }
};

const value_string DIS_PDU_EntityKind_Strings[] =
{
    { DIS_ENTITYKIND_OTHER,            "Other" },
    { DIS_ENTITYKIND_PLATFORM,         "Platform" },
    { DIS_ENTITYKIND_MUNITION,         "Munition" },
    { DIS_ENTITYKIND_LIFE_FORM,        "Life form" },
    { DIS_ENTITYKIND_ENVIRONMENTAL,    "Environmental" },
    { DIS_ENTITYKIND_CULTURAL_FEATURE, "Cultural feature" },
    { DIS_ENTITYKIND_SUPPLY,           "Supply" },
    { DIS_ENTITYKIND_RADIO,            "Radio" },
    { DIS_ENTITYKIND_EXPENDABLE,       "Expendable" },
    { DIS_ENTITYKIND_SENSOR_EMITTER,   "Sensor/Emitter" },
    { 0,                               NULL }
};

const value_string DIS_PDU_Domain_Strings[] =
{
    { DIS_DOMAIN_OTHER,      "Other" },
    { DIS_DOMAIN_LAND,       "Land" },
    { DIS_DOMAIN_AIR,        "Air" },
    { DIS_DOMAIN_SURFACE,    "Surface" },
    { DIS_DOMAIN_SUBSURFACE, "Subsurface" },
    { DIS_DOMAIN_SPACE,      "Space" },
    { 0,                     NULL }
};

const value_string DIS_PDU_Category_LandPlatform_Strings[] =
{
    { DIS_CATEGORY_LANDPLATFORM_OTHER,                                      "Other" },
    { DIS_CATEGORY_LANDPLATFORM_TANK,                                       "Tank" },
    { DIS_CATEGORY_LANDPLATFORM_ARMORED_FIGHTING_VEHICLE,                   "Armored fighting vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_ARMORED_UTILITY_VEHICLE,                    "Armored utility vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_SELF_PROPELLED_ARTILLERY,                   "Self-propelled artillery" },
    { DIS_CATEGORY_LANDPLATFORM_TOWED_ARTILLERY,                            "Towed artillery" },
    { DIS_CATEGORY_LANDPLATFORM_SMALL_WHEELED_UTILITY_VEHICLE,              "Small wheeled utility vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_LARGE_WHEELED_UTILITY_VEHICLE,              "Large wheeled utility vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_SMALL_TRACKED_UTILITY_VEHICLE,              "Small tracked utility vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_LARGE_TRACKED_UTILITY_VEHICLE,              "Large tracked utility vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_MORTAR,                                     "Mortar" },
    { DIS_CATEGORY_LANDPLATFORM_MINE_PLOW,                                  "Mine plow" },
    { DIS_CATEGORY_LANDPLATFORM_MINE_RAKE,                                  "Mine rake" },
    { DIS_CATEGORY_LANDPLATFORM_MINE_ROLLER,                                "Mine roller" },
    { DIS_CATEGORY_LANDPLATFORM_CARGO_TRAILER,                              "Cargo trailer" },
    { DIS_CATEGORY_LANDPLATFORM_FUEL_TRAILER,                               "Fuel trailer" },
    { DIS_CATEGORY_LANDPLATFORM_GENERATOR_TRAILER,                          "Generator trailer" },
    { DIS_CATEGORY_LANDPLATFORM_WATER_TRAILER,                              "Water trailer" },
    { DIS_CATEGORY_LANDPLATFORM_ENGINEER_EQUIPMENT,                         "Engineer equipment" },
    { DIS_CATEGORY_LANDPLATFORM_HEAVY_EQUIPMENT_TRANSPORT_TRAILER,          "Heavy equipment transport trailer" },
    { DIS_CATEGORY_LANDPLATFORM_MAINTENANCE_EQUIPMENT_TRAILER,              "Maintenance equipment trailer" },
    { DIS_CATEGORY_LANDPLATFORM_LIMBER,                                     "Limber" },
    { DIS_CATEGORY_LANDPLATFORM_CHEMICAL_DECONTAMINATION_TRAILER,           "Chemical decontamination trailer" },
    { DIS_CATEGORY_LANDPLATFORM_WARNING_SYSTEM,                             "Warning system" },
    { DIS_CATEGORY_LANDPLATFORM_TRAIN_ENGINE,                               "Train engine" },
    { DIS_CATEGORY_LANDPLATFORM_TRAIN_CAR,                                  "Train car" },
    { DIS_CATEGORY_LANDPLATFORM_TRAIN_CABOOSE,                              "Train caboose" },
    { DIS_CATEGORY_LANDPLATFORM_CIVILIAN_VEHICLE,                           "Civilian vehicle" },
    { DIS_CATEGORY_LANDPLATFORM_AIR_DEFENSE_MISSILE_DEFENSE_UNIT_EQUIPMENT, "Air defense / missile defense unit equipment" },
    { DIS_CATEGORY_LANDPLATFORM_C3I_SYSTEM,                                 "C3I system" },
    { DIS_CATEGORY_LANDPLATFORM_OPERATIONS_FACILITY,                        "Operations facility" },
    { DIS_CATEGORY_LANDPLATFORM_INTELLIGENCE_FACILITY,                      "Intelligence facility" },
    { DIS_CATEGORY_LANDPLATFORM_SURVEILLANCE_FACILITY,                      "Surveillance facility" },
    { DIS_CATEGORY_LANDPLATFORM_COMMUNICATIONS_FACILITY,                    "Communications facility" },
    { DIS_CATEGORY_LANDPLATFORM_COMMAND_FACILITY,                           "Command facility" },
    { DIS_CATEGORY_LANDPLATFORM_C4I_FACILITY,                               "C4I facility" },
    { DIS_CATEGORY_LANDPLATFORM_CONTROL_FACILITY,                           "Control facility" },
    { DIS_CATEGORY_LANDPLATFORM_FIRE_CONTROL_FACILITY,                      "Fire control facility" },
    { DIS_CATEGORY_LANDPLATFORM_MISSILE_DEFENSE_FACILITY,                   "Missile defense facility" },
    { DIS_CATEGORY_LANDPLATFORM_FIELD_COMMAND_POST,                         "Field command post" },
    { DIS_CATEGORY_LANDPLATFORM_OBSERVATION_POST,                           "Field observation post" },
    { 0,                                                                    NULL }
};

const value_string DIS_PDU_Category_AirPlatform_Strings[] =
{
    { DIS_CATEGORY_AIRPLATFORM_OTHER,                             "Other" },
    { DIS_CATEGORY_AIRPLATFORM_FIGHTER_AIR_DEFENSE,               "Fighter / air defense" },
    { DIS_CATEGORY_AIRPLATFORM_ATTACK_STRIKE,                     "Attack / strike" },
    { DIS_CATEGORY_AIRPLATFORM_BOMBER,                            "Bomber" },
    { DIS_CATEGORY_AIRPLATFORM_CARGO_TANKER,                      "Cargo tanker" },
    { DIS_CATEGORY_AIRPLATFORM_ASW_PATROL_OBSERVATION,            "ASW / patrol / observation" },
    { DIS_CATEGORY_AIRPLATFORM_ELECTRONIC_WARFARE,                "Electronic warfare" },
    { DIS_CATEGORY_AIRPLATFORM_RECONNAISSANCE,                    "Reconnaissance" },
    { DIS_CATEGORY_AIRPLATFORM_SURVEILLANCE_C2,                   "Surveillance / C2" },
    { DIS_CATEGORY_AIRPLATFORM_ATTACK_HELICOPTER,                 "Attack helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_UTILITY_HELICOPTER,                "Utility helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_ANTISUB_WARFARE_PATROL_HELICOPTER, "Antisubmarine warfare / patrol helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_CARGO_HELICOPTER,                  "Cargo helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_OBSERVATION_HELICOPTER,            "Observation helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_SPECIAL_OPERATIONS_HELICOPTER,     "Special operations helicopter" },
    { DIS_CATEGORY_AIRPLATFORM_TRAINER,                           "Trainer" },
    { DIS_CATEGORY_AIRPLATFORM_UNMANNED,                          "Unmanned" },
    { DIS_CATEGORY_AIRPLATFORM_NON_COMBATANT_COMMERCIAL_AIRCRAFT, "Non-combatant commercial aircraft" },
    { 0,                                                          NULL }
};

const value_string DIS_PDU_Category_SurfacePlatform_Strings[] =
{
    { DIS_CATEGORY_SURFACEPLATFORM_OTHER, "Other" },
    { 0,                                  NULL}
};

const value_string DIS_PDU_Category_SubsurfacePlatform_Strings[] =
{
    { DIS_CATEGORY_SUBSURFACEPLATFORM_OTHER, "Other" },
    { 0,                                     NULL }
};

const value_string DIS_PDU_Category_SpacePlatform_Strings[] =
{
    { DIS_CATEGORY_SPACEPLATFORM_OTHER, "Other" },
    { 0,                                NULL }
};

const value_string DIS_PDU_Encoding_Class_Strings[] =
{
    { DIS_ENCODING_CLASS_ENCODED_AUDIO,    "Encoded Audio" },
    { DIS_ENCODING_CLASS_RAW_BINARY_DATA,  "Raw Binary Data" },
    { DIS_ENCODING_CLASS_APPL_SPEC_DATA,   "Application-Specific Data" },
    { DIS_ENCODING_CLASS_DB_INDEX,         "Database index" },
    { 0,                                   NULL }
};

const value_string DIS_PDU_Encoding_Type_Strings[] =
{
    { DIS_ENCODING_TYPE_OTHER,               "Other" },
    { DIS_ENCODING_TYPE_MU_LAW,              "8-bit mu-law" },
    { DIS_ENCODING_TYPE_CVSD,                "CVSD per MIL-STD-188-113" },
    { DIS_ENCODING_TYPE_ADPCM_G721,          "ADPCM per CCITT G.721" },
    { DIS_ENCODING_TYPE_16BIT_LINEAR_PCM,    "16-bit linear PCM" },
    { DIS_ENCODING_TYPE_8BIT_LINEAR_PCM,     "8-bit linear PCM" },
    { DIS_ENCODING_TYPE_VECTOR_QUANTIZATION, "VQ (Vector Quantization" },
    { 0,                                NULL }
};

const value_string DIS_PDU_TDL_Type_Strings[] =
{
    {DIS_TDL_TYPE_OTHER,                     "Other" },
    {DIS_TDL_TYPE_PADIL,                     "PADIL" },
    {DIS_TDL_TYPE_NATO_LINK1,                "NATO Link-1" },
    {DIS_TDL_TYPE_ATDL1,                     "ATDL-1" },
    {DIS_TDL_TYPE_LINK_11B,                  "Link 11B (TADIL B)" },
    {DIS_TDL_TYPE_SADL,                      "Situational Awareness Data Link (SADL)" },
    {DIS_TDL_TYPE_JTIDS_TADIL_J,             "Link 16 Legacy Format  (JTIDS/TADIL-J) " },
    {DIS_TDL_TYPE_JTIDS_FDL_TADIL_J,         "Link 16 Legacy Format (JTIDS/FDL/TADIL-J)" },
    {DIS_TDL_TYPE_LINK_11A,                  "Link 11A (TADIL A)" },
    {DIS_TDL_TYPE_IJMS,                      "IJMS" },
    {DIS_TDL_TYPE_LINK_4A,                   "Link 4A (TADIL C)" },
    {DIS_TDL_TYPE_LINK_4C,                   "Link 4C" },
    {DIS_TDL_TYPE_TIBS,                      "TIBS" },
    {DIS_TDL_TYPE_ATL,                       "ATL" },
    {DIS_TDL_TYPE_CONSTANT_SRC,              "Constant Source" },
    {DIS_TDL_TYPE_ABBRV_CC,                  "Abbreviated Command and Control" },
    {DIS_TDL_TYPE_MILSTAR,                   "MILSTAR" },
    {DIS_TDL_TYPE_ATHS,                      "ATHS" },
    {DIS_TDL_TYPE_OTHGOLD,                   "OTHGOLD" },
    {DIS_TDL_TYPE_TACELINT,                  "TACELINT" },
    {DIS_TDL_TYPE_AWW13,                     "Weapons Data Link (AWW-13)" },
    {DIS_TDL_TYPE_ABBRV_CC_2,                "Abbreviated Command and Control" },
    {DIS_TDL_TYPE_EPLRS,                     "Enhanced Position Location Reporting System (EPLRS)" },
    {DIS_TDL_TYPE_PLRS,                      "Position Location Reporting System  (PLRS)" },
    {DIS_TDL_TYPE_SINCGARS,                  "SINCGARS" },
    {DIS_TDL_TYPE_HAVE_QUICK_I,              "Have Quick I" },
    {DIS_TDL_TYPE_HAVE_QUICK_II,             "Have Quick II" },
    {DIS_TDL_TYPE_HAVE_QUICK_IIA,            "Have Quick IIA (Saturn)" },
    {DIS_TDL_TYPE_IFDL1,                     "Intra-Flight Data Link 1" },
    {DIS_TDL_TYPE_IFDL2,                     "Intra-Flight Data Link 2" },
    {DIS_TDL_TYPE_IDM,                       "Improved Data Modem (IDM)" },
    {DIS_TDL_TYPE_AFAPD,                     "Air Force Application Program Development (AFAPD)" },
    {DIS_TDL_TYPE_CEC,                       "Cooperative Engagement Capability (CEC)" },
    {DIS_TDL_TYPE_FAAD_FDL,                  "Forward Area Air Defense (FAAD) Data Link (FDL)" },
    {DIS_TDL_TYPE_GBDL,                      "Ground Based Data Link (GBDL)" },
    {DIS_TDL_TYPE_IVIS,                      "Intra Vehicular Info System (IVIS)" },
    {DIS_TDL_TYPE_MTS,                       "Marine Tactical System (MTS)" },
    {DIS_TDL_TYPE_TACFIRE,                   "Tactical Fire Direction System (TACFIRE)" },
    {DIS_TDL_TYPE_IBS,                       "Integrated Broadcast Service (IBS)" },
    {DIS_TDL_TYPE_ABIT,                      "Airborne Information Transfer (ABIT)" },
    {DIS_TDL_TYPE_ATARS,                     "Advanced Tactical Airborne Reconnaissance System (ATARS) Data Link" },
    {DIS_TDL_TYPE_BGPHES,                    "Battle Group Passive Horizon Extension System (BGPHES) Data Link" },
    {DIS_TDL_TYPE_CHBDL,                     "Common High Bandwidth Data Link (CHBDL)" },
    {DIS_TDL_TYPE_GUARDRAIL_IDL,             "Guardrail Interoperable Data Link (IDL)" },
    {DIS_TDL_TYPE_GUARDRAIL_CSS1,            "Guardrail Common Sensor System One (CSS1) Data Link" },
    {DIS_TDL_TYPE_GUARDRAIL_CSS2,            "Guardrail Common Sensor System Two (CSS2) Data Link" },
    {DIS_TDL_TYPE_GUARDRAIL_CSS2_MRDL,       "Guardrail CSS2 Multi-Role Data Link (MRDL)" },
    {DIS_TDL_TYPE_GUARDRAIL_CSS2_DASR,       "Guardrail CSS2 Direct Air to Satellite Relay (DASR) Data Link" },
    {DIS_TDL_TYPE_LOS_TETHER,                "Line of Sight (LOS) Data Link Implementation (LOS tether)" },
    {DIS_TDL_TYPE_LWCDL,                     "Lightweight CDL (LWCDL)" },
    {DIS_TDL_TYPE_L_52M,                     "L-52M (SR-71)" },
    {DIS_TDL_TYPE_RR_ROWL_DL,                "Rivet Reach/Rivet Owl Data Link" },
    {DIS_TDL_TYPE_SENIOR_SPAN,               "Senior Span" },
    {DIS_TDL_TYPE_SENIOR_SPUR,               "Senior Spur" },
    {DIS_TDL_TYPE_SENIOR_STRETCH,            "Senior Stretch." },
    {DIS_TDL_TYPE_SENIOR_YEAR_IDL,           "Senior Year Interoperable Data Link (IDL)" },
    {DIS_TDL_TYPE_SPACE_CDL,                 "Space CDL" },
    {DIS_TDL_TYPE_TR_1_MIST_ADL,             "TR-1 mode MIST Airborne Data Link" },
    {DIS_TDL_TYPE_KU_BAND_SATCOM,            "Ku-band SATCOM Data Link Implementation (UAV)" },
    {DIS_TDL_TYPE_MECDL,                     "Mission Equipment Control Data link (MECDL)" },
    {DIS_TDL_TYPE_RADAR_DATA_TSDL,           "Radar Data Transmitting Set Data Link" },
    {DIS_TDL_TYPE_SCDL,                      "Surveillance and Control Data Link (SCDL)" },
    {DIS_TDL_TYPE_TACTICAL_UAV_VIDEO,        "Tactical UAV Video" },
    {DIS_TDL_TYPE_UHF_SATCOM,                "UHF SATCOM Data Link Implementation (UAV)" },
    {DIS_TDL_TYPE_TCDL,                      "Tactical Common Data Link (TCDL)" },
    {DIS_TDL_TYPE_LLAPI,                     "Low Level Air Picture Interface (LLAPI)" },
    {DIS_TDL_TYPE_WEAPONS_DL,                "Weapons Data Link (AGM-130)" },
    {DIS_TDL_TYPE_GC3,                       "GC3" },
    {DIS_TDL_TYPE_LINK16_STD,                "Link 16 Standardized Format (JTIDS/MIDS/TADIL J)" },
    {DIS_TDL_TYPE_LINK16_EDR,                "Link 16 Enhanced Data Rate (EDR JTIDS/MIDS/TADIL-J)" },
    {DIS_TDL_TYPE_JTIDS_NET_DATA_LOAD,       "JTIDS/MIDS Net Data Load (TIMS/TOMS)" },
    {DIS_TDL_TYPE_LINK22,                    "Link 22" },
    {DIS_TDL_TYPE_AFIWC_IADS,                "AFIWC IADS Communications Links" },
    { 0,                                     NULL }
};

const value_string DIS_PDU_RadioCategory_Strings[] =
{
    {0,     "Other" },
    {1,     "Voice Transmission/Reception" },
    {2,     "Data Link Transmission/Reception" },
    {3,     "Voice and Data Link Transmission/Reception" },
    {4,     "Instrumented Landing System (ILS) Glideslope Transmitter" },
    {5,     "Instrumented Landing System (ILS) Localizer Transmitter" },
    {6,     "Instrumented Landing System (ILS) Outer Marker Beacon" },
    {7,     "Instrumented Landing System (ILS) Middle Marker Beacon" },
    {8,     "Instrumented Landing System (ILS) Inner Marker Beacon" },
    {9,     "Instrumented Landing System (ILS) Receiver (Platform Radio)" },
    {10,    "Tactical Air Navigation (TACAN) Transmitter (Ground Fixed Equipment)" },
    {11,    "Tactical Air Navigation (TACAN) Receiver (Moving Platform Equipment)" },
    {12,    "Tactical Air Navigation (TACAN) Transmitter/Receiver (Moving Platform Equipment)" },
    {13,    "Variable Omni-Ranging (VOR) Transmitter (Ground Fixed Equipment)" },
    {14,    "Variable Omni-Ranging (VOR) with Distance Measuring Equipment (DME) Transmitter (Ground Fixed Equipment)" },
    {15,    "Combined VOR/ILS Receiver (Moving Platform Equipment)" },
    {16,    "Combined VOR & TACAN (VORTAC) Transmitter" },
    {17,    "Non-Directional Beacon (NDB) Transmitter" },
    {18,    "Non-Directional Beacon (NDB) Receiver" },
    {19,    "Non-Directional Beacon (NDB) with Distance Measuring Equipment (DME) Transmitter" },
    {20,    "Distance Measuring Equipment (DME)" },
    {0,     NULL }
};

const value_string DIS_PDU_NomenclatureVersion_Strings[] =
{
    {0,    "Other" },
    {1,    "Joint Electronics Type Designation System (JETDS) Nomenclature (AN/ per Mil-STD-196)" },
    {2,    "Manufacturer Designation" },
    {3,    "National Designation" },
    {0,    NULL }
};

const value_string DIS_PDU_Nomenclature_Strings[] =
{
    {0,    "Other" },
    {1,    "AN/ARN-118" },
    {2,    "AN/ARN-139" },
    {3,    "Generic Ground Fixed Transmitter" },
    {4,    "Generic Ground Mobile Transmitter" },
    {0,    NULL }
};

const value_string DIS_PDU_RadioTransmitState_Strings[] =
{
    {0,    "Off" },
    {1,    "On but not transmitting" },
    {2,    "On and transmitting" },
    {0,    NULL }
};

const value_string DIS_PDU_RadioInputSource_Strings[] =
{
    {0,    "Other" },
    {1,    "Pilot" },
    {2,    "Copilot" },
    {3,    "First Officer" },
    {4,    "Driver" },
    {5,    "Loader" },
    {6,    "Gunner" },
    {7,    "Commander" },
    {8,    "Digital Data Device" },
    {9,    "Intercom" },
    {0,    NULL }
};

const value_string DIS_PDU_AntennaPatternType_Strings[] =
{
    {0,    "Omni-directional" },
    {1,    "Beam" },
    {2,    "Spherical harmonic" },
    {0,    NULL }
};

const value_string DIS_PDU_MajorModulation_Strings[] =
{
    {DIS_MAJOR_MOD_OTHER,                    "Other" },
    {DIS_MAJOR_MOD_AMPLITUDE,                "Amplitude" },
    {DIS_MAJOR_MOD_AMPLITUDE_AND_ANGLE,      "Amplitude and Angle" },
    {DIS_MAJOR_MOD_ANGLE,                    "Angle" },
    {DIS_MAJOR_MOD_COMBINATION,              "Combination" },
    {DIS_MAJOR_MOD_PULSE,                    "Pulse" },
    {DIS_MAJOR_MOD_UNMODULATED,              "Unmodulated" },
    {DIS_MAJOR_MOD_CPSM,                     "Carrier Phase Shift Modulation (CPSM)" },
    {0,                                      NULL }
};

const value_string DIS_PDU_EmissionFunction_Strings[] =
{
    {DIS_EMISSION_FUNCTION_OTHER,                    "Other" },
    {DIS_EMISSION_FUNCTION_MULTI_FUNCTION,           "Multi-Function" },
    {DIS_EMISSION_FUNCTION_EARLY_WARNING_SURVEILLANCE,
                                                "Early Warning/Surveillance" },
    {DIS_EMISSION_FUNCTION_HEIGHT_FINDING,           "Height Finding" },
    {DIS_EMISSION_FUNCTION_FIRE_CONTROL,             "Fire Control" },
    {DIS_EMISSION_FUNCTION_ACQUISITION_DETECTION,    "Acquisition/Detection" },
    {DIS_EMISSION_FUNCTION_TRACKING,                 "Tracking" },
    {DIS_EMISSION_FUNCTION_GUIDANCE_ILLUMINATION,    "Guidance/Illumination" },
    {DIS_EMISSION_FUNCTION_FIRING_POINT_LAUNCH_POINT_LOCATION,
                                        "Firing point/launch point location" },
    {DIS_EMISSION_FUNCTION_RANGING,                  "Ranging" },
    {DIS_EMISSION_FUNCTION_RADAR_ALTIMETER,          "Radar Altimeter" },
    {DIS_EMISSION_FUNCTION_IMAGING,                  "Imaging" },
    {DIS_EMISSION_FUNCTION_MOTION_DETECTION,         "Motion Detection" },
    {DIS_EMISSION_FUNCTION_NAVIGATION,               "Navigation" },
    {DIS_EMISSION_FUNCTION_WEATHER_METEROLOGICAL,    "Weather / Meterological"},
    {DIS_EMISSION_FUNCTION_INSTRUMENTATION,          "Instrumentation" },
    {DIS_EMISSION_FUNCTION_IDENTIFICATION_CLASSIFICATION_INCLUDING_IFF,
                            "Identification/Classification (including IFF)" },
    {DIS_EMISSION_FUNCTION_AAA_FIRE_CONTROL,
                                "AAA (Anti-Aircraft Artillery) Fire Control" },
    {DIS_EMISSION_FUNCTION_AIR_SEARCH_BOMB,           "Air Search/Bomb" },
    {DIS_EMISSION_FUNCTION_AIR_INTERCEPT,             "Air Intercept" },
    {DIS_EMISSION_FUNCTION_ALTIMETER,                 "Altimeter" },
    {DIS_EMISSION_FUNCTION_AIR_MAPPING,               "Air Mapping" },
    {DIS_EMISSION_FUNCTION_AIR_TRAFFIC_CONTROL,       "Air Traffic Control" },
    {DIS_EMISSION_FUNCTION_BEACON,                    "Beacon" },
    {DIS_EMISSION_FUNCTION_BATTLEFIELD_SURVEILLANCE,
                                                "Battlefield Surveillance" },
    {DIS_EMISSION_FUNCTION_GROUND_CONTROL_APPROACH,
                                                "Ground Control Approach" },
    {DIS_EMISSION_FUNCTION_GROUND_CONTROL_INTERCEPT,
                                                "Ground Control Intercept" },
    {DIS_EMISSION_FUNCTION_COASTAL_SURVEILLANCE,      "Coastal Surveillance" },
    {DIS_EMISSION_FUNCTION_DECOY_MIMIC,               "Decoy/Mimic" },
    {DIS_EMISSION_FUNCTION_DATA_TRANSMISSION,         "Data Transmission" },
    {DIS_EMISSION_FUNCTION_EARTH_SURVEILLANCE,        "Earth Surveillance" },
    {DIS_EMISSION_FUNCTION_GUN_LAY_BEACON,            "Gun Lay Beacon" },
    {DIS_EMISSION_FUNCTION_GROUND_MAPPING,            "Ground Mapping" },
    {DIS_EMISSION_FUNCTION_HARBOR_SURVEILLANCE,       "Harbor Surveillance" },
    {DIS_EMISSION_FUNCTION_ILS,
                                          "ILS (Instrument Landing System)" },
    {DIS_EMISSION_FUNCTION_IONOSPHERIC_SOUND,         "Ionospheric Sound" },
    {DIS_EMISSION_FUNCTION_INTERROGATOR,              "Interrogator" },
    {DIS_EMISSION_FUNCTION_BARRAGE_JAMMING,           "Barrage Jamming" },
    {DIS_EMISSION_FUNCTION_CLICK_JAMMING,             "Click Jamming" },
    {DIS_EMISSION_FUNCTION_FREQUENCY_SWEPT_JAMMING,
                                                "Frequency Swept Jamming" },
    {DIS_EMISSION_FUNCTION_JAMMING,                   "Jamming" },
    {DIS_EMISSION_FUNCTION_PULSED_JAMMING,            "Pulsed Jamming" },
    {DIS_EMISSION_FUNCTION_REPEATER_JAMMING,          "Repeater Jamming" },
    {DIS_EMISSION_FUNCTION_SPOT_NOISE_JAMMING,        "Spot Noise Jamming" },
    {DIS_EMISSION_FUNCTION_MISSILE_ACQUISITION,       "Missile Acquisition" },
    {DIS_EMISSION_FUNCTION_MISSILE_DOWNLINK,          "Missile Downlink" },
    {DIS_EMISSION_FUNCTION_SPACE,                     "Space" },
    {DIS_EMISSION_FUNCTION_SURFACE_SEARCH,            "Surface Search" },
    {DIS_EMISSION_FUNCTION_SHELL_TRACKING,            "Shell Tracking" },
    {DIS_EMISSION_FUNCTION_TELEVISION,                "Television" },
    {DIS_EMISSION_FUNCTION_UNKNOWN,                   "Unknown" },
    {DIS_EMISSION_FUNCTION_VIDEO_REMOTING,            "Video Remoting" },
    {DIS_EMISSION_FUNCTION_EXPERIMENTAL_OR_TRAINING,
                                                "Experimental or training" },
    {DIS_EMISSION_FUNCTION_MISSILE_GUIDANCE,          "Missile Guidance" },
    {DIS_EMISSION_FUNCTION_MISSILE_HOMING,            "Missile Homing" },
    {DIS_EMISSION_FUNCTION_MISSILE_TRACKING,          "Missile Tracking" },
    {DIS_EMISSION_FUNCTION_JAMMING_NOISE,             "Jamming, noise" },
    {DIS_EMISSION_FUNCTION_JAMMING_DECEPTION,         "Jamming, deception" },
    {DIS_EMISSION_FUNCTION_NAVIGATION_DISTANCE_MEASURING_EQUIPMENT,
                                "Navigation/Distance Measuring Equipment" },
    {DIS_EMISSION_FUNCTION_TERRAIN_FOLLOWING,         "Terrain Following" },
    {DIS_EMISSION_FUNCTION_WEATHER_AVOIDANCE,         "Weather Avoidance" },
    {DIS_EMISSION_FUNCTION_PROXIMITY_FUSE,            "Proximity Fuse" },
    {DIS_EMISSION_FUNCTION_RADIOSONDE,                "Radiosonde" },
    {DIS_EMISSION_FUNCTION_SONOBUOY,                  "Sonobuoy" },
    {DIS_EMISSION_FUNCTION_BATHYTHERMAL_SENSOR,       "Bathythermal Sensor" },
    {DIS_EMISSION_FUNCTION_TOWED_COUNTER_MEASURE,     "Towed Counter Measure"},
    {DIS_EMISSION_FUNCTION_WEAPON_NON_LETHAL,         "Weapon, non-lethal" },
    {DIS_EMISSION_FUNCTION_WEAPON_LETHAL,             "Weapon, lethal" },
    {0,                                      NULL }
};

const value_string DIS_PDU_BeamFunction_Strings[] =
{
    {DIS_BEAM_FUNCTION_OTHER,               "Other" },
    {DIS_BEAM_FUNCTION_SEARCH,              "Search" },
    {DIS_BEAM_FUNCTION_HEIGHT_FINDER,       "Height finder" },
    {DIS_BEAM_FUNCTION_ACQUISITION,         "Acquisition" },
    {DIS_BEAM_FUNCTION_TRACKING,            "Tracking" },
    {DIS_BEAM_FUNCTION_ACQUISITION_TRACKING,"Acquisition and tracking" },
    {DIS_BEAM_FUNCTION_COMMAND_GUIDANCE,    "Command guidance" },
    {DIS_BEAM_FUNCTION_ILLUMINATION,        "Illumination" },
    {DIS_BEAM_FUNCTION_RANGE_ONLY_RADAR,    "Range only radar" },
    {DIS_BEAM_FUNCTION_MISSILE_BEACON,      "Missile beacon" },
    {DIS_BEAM_FUNCTION_MISSILE_FUZE,        "Missile fuze" },
    {DIS_BEAM_FUNCTION_ACTIVE_RADAR_MISSILE_SEEKER,
                                            "Active radar missile seeker" },
    {DIS_BEAM_FUNCTION_JAMMER,              "Jammer" },
    {DIS_BEAM_FUNCTION_IFF,                 "IFF" },
    {DIS_BEAM_FUNCTION_NAVIGATIONAL_WEATHER,"Navigational/Weather" },
    {DIS_BEAM_FUNCTION_METEOROLOGICAL,      "Meteorological" },
    {DIS_BEAM_FUNCTION_DATA_TRANSMISSION,   "Data transmission" },
    {DIS_BEAM_FUNCTION_NAVIGATIONAL_DIRECTIONAL_BEACON,
                                            "Navigational directional beacon" },
    {0,                                     NULL }
};

const value_string DIS_PDU_DetailModulationAmplitude_Strings[] =
{
    {0,    "Other" },
    {1,    "AFSK (Audio Frequency Shift Keying)" },
    {2,    "AM (Amplitude Modulation)" },
    {3,    "CW (Continuous Wave Modulation)" },
    {4,    "DSB (Double Sideband)" },
    {5,    "ISB (Independent Sideband)" },
    {6,    "LSB (Single Band Suppressed Carrier, Lower Sideband Mode)" },
    {7,    "SSB-Full (Single Sideband Full Carrier)" },
    {8,    "SSB-Reduc (Single Band Reduced Carrier)" },
    {9,    "USB (Single Band Suppressed Carrier, Upper Sideband Mode)" },
    {10,   "VSB (Vestigial Sideband)" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationAmpAndAngle_Strings[] =
{
    {0,    "Other" },
    {1,    "Amplitude and Angle" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationAngle_Strings[] =
{
    {0,    "Other" },
    {1,    "FM (Frequency Modulation)" },
    {2,    "FSK (Frequency Shift Keying)" },
    {3,    "PM (Phase Modulation)" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationCombination_Strings[] =
{
    {0,    "Other" },
    {1,    "Amplitude-Angle-Pulse" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationPulse_Strings[] =
{
    {0,    "Other" },
    {1,    "Pulse" },
    {2,    "X Band TACAN Pulse" },
    {3,    "Y Band TACAN Pulse" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationUnmodulated_Strings[] =
{
    {0,    "Other" },
    {1,    "Continuous Wave emission of an unmodulated carrier" },
    {0,    NULL }
};

const value_string DIS_PDU_DetailModulationCPSM_Strings[] =
{
    {0,    "Other" },
    {0,    NULL }
};

const value_string DIS_PDU_ModParamMsgStart_Strings[] =
{
    {0,    "Not start of message" },
    {1,    "Start of Message" },
    {0,    NULL }
};

const value_string DIS_PDU_ModParamClrChannel_Strings[] =
{
    {0,    "Not clear channel" },
    {1,    "Clear channel" },
    {0,    NULL }
};

const value_string DIS_PDU_TSAllocationFidelity_Strings[] =
{
    {0,    "Time Slot Allocation Fidelity Level 0" },
    {1,    "Time Slot Allocation Fidelity Level 1" },
    {2,    "Time Slot Allocation Fidelity Level 2" },
    {3,    "Time Slot Allocation Fidelity Level 3" },
    {4,    "Time Slot Allocation Fidelity Level 4" },
    {0,    NULL }
};

const value_string DIS_PDU_TerminalPrimaryMode_Strings[] =
{
    {0,    "NTR" },
    {1,    "JTIDS Unit Participant" },
    {0,    NULL }
};

const value_string DIS_PDU_TerminalSecondaryMode_Strings[] =
{
    {0,    "None" },
    {1,    "Net Position Reference" },
    {2,    "Primary Navigation Controller" },
    {3,    "Secondary Navigation Controller" },
    {0,    NULL }
};

const value_string DIS_PDU_ModParamSyncState_Strings[] =
{
    {0,    "Undefined" },
    {1,    "Coarse Synchronization" },
    {1,    "Fine Synchronization" },
    {0,    NULL }
};


const value_string DIS_PDU_SystemModulation_Strings[] =
{
    {0,    "Other" },
    {1,    "Generic" },
    {2,    "HQ" },
    {3,    "HQII" },
    {4,    "HQIIA" },
    {5,    "SINCGARS" },
    {6,    "CCTT SINCGARS" },
    {7,    "EPLRS (Enhanced Position Location Reporting System)" },
    {8,    "JTIDS/MIDS" },
    {0,    NULL }
};

const value_string DIS_PDU_CryptoSystem_Strings[] =
{
    {0,    "Other" },
    {1,    "KY-28" },
    {2,    "VINSON (KY-57, KY-58, SINCGARS ICOM)" },
    {3,    "Narrow Spectrum Secure Voice (NSVE)" },
    {4,    "Wide Spectrum Secure Voice (WSVE)" },
    {0,    NULL }
};

const value_string DIS_PDU_AcknowledgeFlag_Strings[] =
{
    { DIS_ACKNOWLEDGE_FLAG_CREATE_ENTITY,            "Create Entity" },
    { DIS_ACKNOWLEDGE_FLAG_REMOVE_ENTITY,            "Remove Entity" },
    { DIS_ACKNOWLEDGE_FLAG_START_RESUME,             "Start Resume" },
    { DIS_ACKNOWLEDGE_FLAG_STOP_FREEZE,              "Stop Freeze" },
    { DIS_ACKNOWLEDGE_FLAG_TRANSFER_CONTROL_REQUEST, "Transfer Control Request" },
    { 0,                                             NULL }
};

const value_string DIS_PDU_ActionId_Strings[] =
{
    { DIS_ACTION_ID_OTHER,                                        "Other" },
    { DIS_ACTION_ID_LOCAL_STORAGE_OF_THE_REQUESTED_INFORMATION,   "LocalStorageOfTheRequestedInformation" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_AMMUNITION,     "InformSMofEventRanOutOfAmmunition" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_KILLED_IN_ACTION,          "InformSMofEventKilledInAction" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_DAMAGE,                    "InformSMofEventDamage" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_MOBILITY_DISABLED,         "InformSMofEventMobilityDisabled" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_FIRE_DISABLED,             "InformSMofEventFireDisabled" },
    { DIS_ACTION_ID_INFORM_SM_OF_EVENT_RAN_OUT_OF_FUEL,           "InformSMofEventRanOutOfFuel" },
    { DIS_ACTION_ID_RECALL_CHECKPOINT_DATA,                       "RecallCheckpointData" },
    { DIS_ACTION_ID_RECALL_INITIAL_PARAMETERS,                    "RecallInitialParameters" },
    { DIS_ACTION_ID_INITIATE_TETHER_LEAD,                         "InitiateTetherLead" },
    { DIS_ACTION_ID_INITIATE_TETHER_FOLLOW,                       "InitiateTetherFollow" },
    { DIS_ACTION_ID_UNTETHER,                                     "Untether" },
    { DIS_ACTION_ID_INITIATE_SERVICE_STATION_RESUPPLY,            "InitiateServiceStationResupply" },
    { DIS_ACTION_ID_INITIATE_TAILGATE_RESUPPLY,                   "InitiateTailgateResupply" },
    { DIS_ACTION_ID_INITIATE_HITCH_LEAD,                          "InitiateHitchLead" },
    { DIS_ACTION_ID_INITIATE_HITCH_FOLLOW,                        "InitiateHitchFollow" },
    { DIS_ACTION_ID_UNHITCH,                                      "Unhitch" },
    { DIS_ACTION_ID_MOUNT,                                        "Mount" },
    { DIS_ACTION_ID_DISMOUNT,                                     "Dismount" },
    { DIS_ACTION_ID_START_DRC,                                    "StartDRC" },
    { DIS_ACTION_ID_STOP_DRC,                                     "StopDRC" },
    { DIS_ACTION_ID_DATA_QUERY,                                   "DataQuery" },
    { DIS_ACTION_ID_STATUS_REQUEST,                               "StatusRequest" },
    { DIS_ACTION_ID_SEND_OBJECT_STATE_DATA,                       "SendObjectStateData" },
    { DIS_ACTION_ID_RECONSTITUTE,                                 "Reconstitute" },
    { DIS_ACTION_ID_LOCK_SITE_CONFIGURATION,                      "LockSiteConfiguration" },
    { DIS_ACTION_ID_UNLOCK_SITE_CONFIGURATION,                    "UnlockSiteConfiguration" },
    { DIS_ACTION_ID_UPDATE_SITE_CONFIGURATION,                    "UpdateSiteConfiguration" },
    { DIS_ACTION_ID_QUERY_SITE_CONFIGURATION,                     "QuerySiteConfiguration" },
    { DIS_ACTION_ID_TETHERING_INFORMATION,                        "TetheringInformation" },
    { DIS_ACTION_ID_MOUNT_INTENT,                                 "MountIntent" },
    { DIS_ACTION_ID_ACCEPT_SUBSCRIPTION,                          "AcceptSubscription" },
    { DIS_ACTION_ID_UNSUBSCRIBE,                                  "Unsubscribe" },
    { DIS_ACTION_ID_TELEPORT_ENTITY,                              "TeleportEntity" },
    { DIS_ACTION_ID_CHANGE_AGGREGATE_STATE,                       "ChangeAggregateState" },
    { DIS_ACTION_ID_REQUEST_START_PDU,                            "RequestStartPdu" },
    { DIS_ACTION_ID_WAKEUP_GET_READY_FOR_INITIALIZATION,          "WakeupGetReadyForInitialization" },
    { DIS_ACTION_ID_INITIALIZE_INTERNAL_PARAMETERS,               "InitializeInternalParameters" },
    { DIS_ACTION_ID_SEND_PLAN_DATA,                               "SendPlanData" },
    { DIS_ACTION_ID_SYNCHRONIZE_INTERNAL_CLOCKS,                  "SynchronizeInternalClocks" },
    { DIS_ACTION_ID_RUN,                                          "Run" },
    { DIS_ACTION_ID_SAVE_INTERNAL_PARAMETERS,                     "SaveInternalParameters" },
    { DIS_ACTION_ID_SIMULATE_MALFUNCTION,                         "SimulateMalfunction" },
    { DIS_ACTION_ID_JOIN_EXERCISE,                                "JoinExercise" },
    { DIS_ACTION_ID_RESIGN_EXERCISE,                              "ResignExercise" },
    { DIS_ACTION_ID_TIME_ADVANCE,                                 "TimeAdvance" },
    { DIS_ACTION_ID_COMMAND_FROM_SIMULATOR,                       "CommandFromSimulator" },
    { DIS_ACTION_ID_SLING_LOAD_CAPABILITY_REQUEST,                "SlingLoadCapabilityRequest" },
    { DIS_ACTION_ID_SLING_ATTACH_REQUEST,                         "SlingAttachRequest" },
    { DIS_ACTION_ID_SLING_RELEASE_REQUEST,                        "SlingReleaseRequest" },
    { DIS_ACTION_ID_AIRMOUNT_MOUNT_REQUEST,                       "AirmountMountRequest" },
    { DIS_ACTION_ID_AIRMOUNT_DISMOUNT_REQUEST,                    "AirmountDismountRequest" },
    { DIS_ACTION_ID_AIRMOUNT_INFO_REQUEST,                        "AirmountInfoRequest" },
    { 0,                                                          NULL }
};

const value_string DIS_PDU_ApplicationGeneralStatus_Strings[] =
{
    { DIS_APPLICATION_GENERAL_STATUS_UNKNOWN,                  "Unknown" },
    { DIS_APPLICATION_GENERAL_STATUS_FUNCTIONAL,               "Functional" },
    { DIS_APPLICATION_GENERAL_STATUS_DEGRADED_BUT_FUNCTIONAL,  "Degraded But Functional" },
    { DIS_APPLICATION_GENERAL_STATUS_NOT_FUNCTIONAL,           "Not Functional" },
    { 0,                                                       NULL }
};

const value_string DIS_PDU_ApplicationStatusType_Strings[] =
{
    { DIS_APPLICATION_STATUS_TYPE_NOT_SPECIFIED,            "Not Specified" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_USER,           "Cpu Usage User" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_SYSTEM,         "Cpu Usage System" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IO,             "Cpu Usage Io" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_IDLE,           "Cpu Usage Idle" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_STEAL,          "Cpu Usage Steal" },
    { DIS_APPLICATION_STATUS_TYPE_CPU_USAGE_NICE,           "Cpu Usage Nice" },
    { DIS_APPLICATION_STATUS_TYPE_MEMORY_FREE,              "Memory Free" },
    { DIS_APPLICATION_STATUS_TYPE_MEMORY_USED,              "Memory Used" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_FREE,                "Swap Free" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_USED,                "Swap Used" },
    { DIS_APPLICATION_STATUS_TYPE_SWAP_CACHED,              "Swap Cached" },
    { DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_PACKETS_SEC,  "Transmitted Packets Sec" },
    { DIS_APPLICATION_STATUS_TYPE_TRANSMITTED_BYTES_SEC,    "Transmitted Bytes Sec" },
    { DIS_APPLICATION_STATUS_TYPE_RECEIVED_PACKETS_SEC,     "Received Packets Sec" },
    { DIS_APPLICATION_STATUS_TYPE_RECEIVED_BYTES_SEC,       "Received Bytes Sec" },
    { DIS_APPLICATION_STATUS_TYPE_NICE_LEVEL,               "Nice Level" },
    { 0,                                                    NULL }
};

const value_string DIS_PDU_ApplicationType_Strings[] =
{
    { DIS_APPLICATION_TYPE_OTHER,                        "Other" },
    { DIS_APPLICATION_TYPE_RESOURCE_MANAGER,             "Resource Manager" },
    { DIS_APPLICATION_TYPE_SIMULATION_MANAGER,           "Simulation Manager" },
    { DIS_APPLICATION_TYPE_GATEWAY,                      "Gateway" },
    { DIS_APPLICATION_TYPE_STEALTH,                      "Stealth" },
    { DIS_APPLICATION_TYPE_TACTICAL_INTERNET_INTERFACE,  "Tactical Internet Interface" },
    { 0,                                                 NULL }
};

const value_string DIS_PDU_DetonationResult_Strings[] =
{
    { DIS_DETONATION_RESULT_OTHER,                                   "Other" },
    { DIS_DETONATION_RESULT_ENTITY_IMPACT,                           "Entity impact" },
    { DIS_DETONATION_RESULT_ENTITY_PROXIMATE_DETONATION,             "Entity proximate detonation" },
    { DIS_DETONATION_RESULT_GROUND_IMPACT,                           "Ground impact" },
    { DIS_DETONATION_RESULT_GROUND_PROXIMATE_DETONATION,             "Ground proximate detonation" },
    { DIS_DETONATION_RESULT_DETONATION,                              "Detonation" },
    { DIS_DETONATION_RESULT_NONE_DUD,                                "None or no detonation (dud)" },
    { DIS_DETONATION_RESULT_HE_HIT_SMALL,                            "HE hit, small" },
    { DIS_DETONATION_RESULT_HE_HIT_MEDIUM,                           "HE hit, medium" },
    { DIS_DETONATION_RESULT_HE_HIT_LARGE,                            "HE hit, large" },
    { DIS_DETONATION_RESULT_ARMOR_PIERCING_HIT,                      "Armor-piercing hit" },
    { DIS_DETONATION_RESULT_DIRT_BLAST_SMALL,                        "Dirt blast, small" },
    { DIS_DETONATION_RESULT_DIRT_BLAST_MEDIUM,                       "Dirt blast, medium" },
    { DIS_DETONATION_RESULT_DIRT_BLAST_LARGE,                        "Dirt blast, large" },
    { DIS_DETONATION_RESULT_WATER_BLAST_SMALL,                       "Water blast, small" },
    { DIS_DETONATION_RESULT_WATER_BLAST_MEDIUM,                      "Water blast, medium" },
    { DIS_DETONATION_RESULT_WATER_BLAST_LARGE,                       "Water blast, large" },
    { DIS_DETONATION_RESULT_AIR_HIT,                                 "Air hit" },
    { DIS_DETONATION_RESULT_BUILDING_HIT_SMALL,                      "Building hit, small" },
    { DIS_DETONATION_RESULT_BUILDING_HIT_MEDIUM,                     "Building hit, medium" },
    { DIS_DETONATION_RESULT_BUILDING_HIT_LARGE,                      "Building hit, large" },
    { DIS_DETONATION_RESULT_MINE_CLEARING_LINE_CHARGE,               "Mine-clearing line charge" },
    { DIS_DETONATION_RESULT_ENVIRONMENT_OBJECT_IMPACT,               "Environment object impact" },
    { DIS_DETONATION_RESULT_ENVIRONMENT_OBJECT_PROXIMATE_DETONATION, "Environment object proximate detonation" },
    { DIS_DETONATION_RESULT_WATER_IMPACT,                            "Water impact" },
    { DIS_DETONATION_RESULT_AIR_BURST,                               "Air burst" },
    { DIS_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1,               "Kill with fragment type 1" },
    { DIS_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2,               "Kill with fragment type 2" },
    { DIS_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_3,               "Kill with fragment type 3" },
    { DIS_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_1_AFTER_FOF,     "Kill with fragment type 1 after fly-out failure" },
    { DIS_DETONATION_RESULT_KILL_WITH_FRAGMENT_TYPE_2_AFTER_FOF,     "Kill with fragment type 2 after fly-out failure" },
    { DIS_DETONATION_RESULT_MISS_DUE_TO_FOF,                         "Miss due to fly-out failure" },
    { DIS_DETONATION_RESULT_MISS_DUE_TO_ENDGAME_FAILURE,             "Miss due to end-game failure" },
    { DIS_DETONATION_RESULT_MISS_DUE_TO_FOF_AND_ENDGAME_FAILURE,     "Miss due to fly-out and end-game failure" },
    { 0,                                                             NULL }
};

const value_string DIS_PDU_ControlId_Strings[] =
{
    { DIS_CONTROL_ID_OTHER,                                  "Other" },
    { DIS_CONTROL_ID_SHUTDOWN,                               "Shutdown" },
    { DIS_CONTROL_ID_DATA_QUERY,                             "Data Query" },
    { DIS_CONTROL_ID_DATA,                                   "Data" },
    { DIS_CONTROL_ID_SET_DATA,                               "Set Data" },
    { DIS_CONTROL_ID_ADD_DATA,                               "Add Data" },
    { DIS_CONTROL_ID_REMOVE_DATA,                            "Remove Data" },
    { DIS_CONTROL_ID_STATUS,                                 "Status" },
    { 0,                                                     NULL }
};

const value_string DIS_PDU_FrozenBehavior_Strings[] =
{
    { DIS_FROZEN_BEHAVIOR_RUN_INTERNAL_SIMULATION_CLOCK,                "Run Internal Simulation Clock" },
    { DIS_FROZEN_BEHAVIOR_TRANSMIT_PDUS,                                "Transmit PDUs" },
    { DIS_FROZEN_BEHAVIOR_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES,          "Update Sim Models Of Other Entities" },
    { DIS_FROZEN_BEHAVIOR_CONTINUE_TRANSMIT_PDU,                        "Continue Transmit PDU" },
    { DIS_FROZEN_BEHAVIOR_CEASE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES,    "Cease Update Sim Models Of Other Entities" },
    { DIS_FROZEN_BEHAVIOR_CONTINUE_UPDATE_SIM_MODELS_OF_OTHER_ENTITIES, "Continue Update Sim Models Of Other Entities" },
    { 0,                                                                NULL }
};

const value_string DIS_PDU_ParameterTypeDesignator_Strings[] =
{
    { DIS_PARAM_TYPE_DESIG_ARTICULATED_PART,                        "Articulated Part" },
    { DIS_PARAM_TYPE_DESIG_ATTACHED_PART,                           "Attached Part" },
    { DIS_PARAM_TYPE_DESIG_SEPARATION,                              "Separation" },
    { DIS_PARAM_TYPE_DESIG_ENTITY_TYPE,                             "Entity Type" },
    { DIS_PARAM_TYPE_DESIG_ENTITY_ASSOCIATION,                      "Entity Association" },
    { DIS_PARAM_TYPE_DESIG_ANTENNA_LOCATION,                        "Antenna Location" },
    { DIS_PARAM_TYPE_DESIG_EXTENDED_PLATFORM_APPEARANCE,            "Extended Platform Appearance" },
    { DIS_PARAM_TYPE_DESIG_EXTENDED_LIFEFORM_APPEARANCE,            "Extended Lifeform Appearance" },
    { DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_LIGHTS,                    "High Fidelity Lights" },
    { DIS_PARAM_TYPE_DESIG_CHEVRON_MARKING,                         "Chevron Marking" },
    { DIS_PARAM_TYPE_DESIG_HIGH_FIDELITY_THERMAL_SENSOR,            "High Fidelity Thermal Sensor" },
    { DIS_PARAM_TYPE_DESIG_ENTITY_OFFSET,                           "Entity Offset" },
    { DIS_PARAM_TYPE_DESIG_DEAD_RECKONING,                          "Dead Reckoning" },
    { DIS_PARAM_TYPE_DESIG_ARMY_TASK_ORGANIZATION,                  "Army Task Organization" },
    { DIS_PARAM_TYPE_DESIG_HEAD_GAZING_WEAPON_AIMING,               "Head Gazing Weapon Aiming" },
    { DIS_PARAM_TYPE_DESIG_LIFEFORM_ACTION_SEQUENCE,                "Lifeform Action Sequence" },
    { DIS_PARAM_TYPE_DESIG_LEGACY_EXTENDED_LIFEFORM_APPEARANCE,     "Legacy Extended Lifeform Appearance" },
    { 0,                                                            NULL }
};

const value_string DIS_PDU_Reason_Strings[] =
{
    { DIS_REASON_OTHER,                                            "Other" },
    { DIS_REASON_RECESS,                                           "Recess" },
    { DIS_REASON_TERMINATION,                                      "Termination" },
    { DIS_REASON_SYSTEM_FAILURE,                                   "System Failure" },
    { DIS_REASON_SECURITY_VIOLATION,                               "Security Violation" },
    { DIS_REASON_ENTITY_RECONSTITUTION,                            "Entity Reconstitution" },
    { DIS_REASON_STOP_FOR_RESET,                                   "Stop For Reset" },
    { DIS_REASON_STOP_FOR_RESTART,                                 "Stop For Restart" },
    { DIS_REASON_ABORT_TRAINING_RETURN_TO_TACTICAL_OPERATIONS,     "Abort Training Return To Tactical Operations" },
    { 0,                                                           NULL }
};

const value_string DIS_PDU_RequestStatus_Strings[] =
{
    { DIS_REQUEST_STATUS_OTHER,                          "Other" },
    { DIS_REQUEST_STATUS_PENDING,                        "Pending" },
    { DIS_REQUEST_STATUS_EXECUTING,                      "Executing" },
    { DIS_REQUEST_STATUS_PARTIALLY_COMPLETE,             "Partially Complete" },
    { DIS_REQUEST_STATUS_COMPLETE,                       "Complete" },
    { DIS_REQUEST_STATUS_REQUEST_REJECTED,               "Request Rejected" },
    { DIS_REQUEST_STATUS_RETRANSMIT_REQUEST_NOW,         "Retransmit Request Now" },
    { DIS_REQUEST_STATUS_RETRANSMIT_REQUEST_LATER,       "Retransmit Request Later" },
    { DIS_REQUEST_STATUS_INVALID_TIME_PARAMETERS,        "Invalid Time Parameters" },
    { DIS_REQUEST_STATUS_SIMULATION_TIME_EXCEEDED,       "Simulation Time Exceeded" },
    { DIS_REQUEST_STATUS_REQUEST_DONE,                   "Request Done" },
    { DIS_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_1,        "TACCSF LOS Reply Type 1" },
    { DIS_REQUEST_STATUS_TACCSF_LOS_REPLY_TYPE_2,        "TACCSF LOS Reply Type 2" },
    { DIS_REQUEST_STATUS_JOIN_EXERCISE_REQUEST_REJECTED, "Join Exercise Request Rejected" },
    { 0,                                                 NULL }
};

const value_string DIS_PDU_RequiredReliabilityService_Strings[] =
{
    { DIS_REQUIRED_RELIABILITY_SERVICE_ACKNOWLEDGED,     "Acknowledged" },
    { DIS_REQUIRED_RELIABILITY_SERVICE_UNACKNOWLEDGED,   "Unacknowledged" },
    { 0,                                                 NULL }
};

const value_string DIS_PDU_DisResponseFlag_Strings[] =
{
    { DIS_RESPONSE_FLAG_OTHER,                      "Other" },
    { DIS_RESPONSE_FLAG_ABLE_TO_COMPLY,             "Able To Comply" },
    { DIS_RESPONSE_FLAG_UNABLE_TO_COMPLY,           "Unable To Comply" },
    { DIS_RESPONSE_FLAG_PENDING_OPERATOR_ACTION,    "Pending Operator Action" },
    { 0,                                            NULL }
};

const value_string DIS_PDU_PersistentObjectType_Strings[] =
{
    { DIS_PERSISTENT_OBJECT_TYPE_OTHER,             "Other" },
    { DIS_PERSISTENT_OBJECT_TYPE_SIMULATOR_PRESENT, "Simulator Present" },
    { DIS_PERSISTENT_OBJECT_TYPE_DESCRIBE_OBJECT,   "Describe Object" },
    { DIS_PERSISTENT_OBJECT_TYPE_OBJECTS_PRESENT,   "Objects Present" },
    { DIS_PERSISTENT_OBJECT_TYPE_OBJECT_REQUEST,    "Object Request" },
    { DIS_PERSISTENT_OBJECT_TYPE_DELETE_OBJECTS,    "Delete Objects" },
    { DIS_PERSISTENT_OBJECT_TYPE_SET_WORLD_STATE,   "Set World State" },
    { DIS_PERSISTENT_OBJECT_TYPE_NOMINATION,        "Nomination" },
    { 0,                                            NULL }
};

const value_string DIS_PDU_PO_ObjectClass_Strings[] =
{
    { DIS_PO_OBJECT_CLASS_OTHER,                    "Other" },
    { DIS_PO_OBJECT_CLASS_WORLD_STATE,              "World State" },
    { DIS_PO_OBJECT_CLASS_OVERLAY,                  "Overlay" },
    { DIS_PO_OBJECT_CLASS_POINT,                    "Point" },
    { DIS_PO_OBJECT_CLASS_LINE,                     "Line" },
    { DIS_PO_OBJECT_CLASS_SECTOR,                   "Sector" },
    { DIS_PO_OBJECT_CLASS_TEXT,                     "Text" },
    { DIS_PO_OBJECT_CLASS_UNIT,                     "Unit" },
    { DIS_PO_OBJECT_CLASS_UNIT_DEFINITION,          "Unit Definition" },
    { DIS_PO_OBJECT_CLASS_STEALTH_CONTROLLER,       "Stealth Controller" },
    { DIS_PO_OBJECT_CLASS_H_HOUR,                   "H Hour" },
    { DIS_PO_OBJECT_CLASS_VARIABLE,                 "Variable" },
    { DIS_PO_OBJECT_CLASS_TASK,                     "Task" },
    { DIS_PO_OBJECT_CLASS_TASK_STATE,               "Task State" },
    { DIS_PO_OBJECT_CLASS_TASK_FRAME,               "Task Frame" },
    { DIS_PO_OBJECT_CLASS_TASK_AUTHORIZATION,       "Task Authorization" },
    { DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT,         "Parametric Input" },
    { DIS_PO_OBJECT_CLASS_PARAMETRIC_INPUT_HOLDER,  "Parametric Input Holder" },
    { DIS_PO_OBJECT_CLASS_EXERCISE_INITIALIZER,     "Exercise Initializer" },
    { DIS_PO_OBJECT_CLASS_FIRE_PARAMETERS,          "Fire Parameters" },
    { DIS_PO_OBJECT_CLASS_MINEFIELD,                "Minefield" },
    { DIS_PO_OBJECT_CLASS_SIMULATION_REQUEST,       "Simulation Request" },
    { DIS_PO_OBJECT_CLASS_NET_SUBSCRIPTION,         "Net Subscription" },
    { DIS_PO_OBJECT_CLASS_LINK,                     "Link" },
    { DIS_PO_OBJECT_CLASS_MINEFIELD_PARENT,         "Minefield Parent" },
    { DIS_PO_OBJECT_CLASS_CHEMICAL,                 "Chemical" },
    { DIS_PO_OBJECT_CLASS_AlertUser,                "Alert User" },
    { DIS_PO_OBJECT_CLASS_HAND_OFF,                 "Hand Off" },
    { DIS_PO_OBJECT_CLASS_CIRCUIT,                  "Circuit" },
    { DIS_PO_OBJECT_CLASS_CARGO,                    "Cargo" },
    { DIS_PO_OBJECT_CLASS_MCM_ROUTE,                "Mcm Route" },
    { DIS_PO_OBJECT_CLASS_MESSAGE,                  "Message" },
    { 0,                                            NULL }
};

const value_string DIS_PDU_EmitterName_Strings[] =
{
    { DIS_EMITTER_NAME_10, "1RL138" },
    { DIS_EMITTER_NAME_45, "1226 DECCA MIL" },
    { DIS_EMITTER_NAME_80, "9GR400" },
    { DIS_EMITTER_NAME_90, "9GR600" },
    { DIS_EMITTER_NAME_135, "9LV 200 TA" },
    { DIS_EMITTER_NAME_180, "9LV 200 TV" },
    { DIS_EMITTER_NAME_225, "A310Z" },
    { DIS_EMITTER_NAME_270, "A325A" },
    { DIS_EMITTER_NAME_315, "A346Z" },
    { DIS_EMITTER_NAME_360, "A353B" },
    { DIS_EMITTER_NAME_405, "A372A" },
    { DIS_EMITTER_NAME_450, "A372B" },
    { DIS_EMITTER_NAME_495, "A372C" },
    { DIS_EMITTER_NAME_540, "A377A" },
    { DIS_EMITTER_NAME_585, "A377B" },
    { DIS_EMITTER_NAME_630, "A380Z" },
    { DIS_EMITTER_NAME_675, "A381Z" },
    { DIS_EMITTER_NAME_720, "A398Z" },
    { DIS_EMITTER_NAME_765, "A403Z" },
    { DIS_EMITTER_NAME_810, "A409A" },
    { DIS_EMITTER_NAME_855, "A418A" },
    { DIS_EMITTER_NAME_900, "A419Z" },
    { DIS_EMITTER_NAME_945, "A429Z" },
    { DIS_EMITTER_NAME_990, "A432Z" },
    { DIS_EMITTER_NAME_1035, "A434Z" },
    { DIS_EMITTER_NAME_1080, "A401A" },
    { DIS_EMITTER_NAME_1095, "AA-12 Seeker" },
    { DIS_EMITTER_NAME_1100, "Agave" },
    { DIS_EMITTER_NAME_1125, "AGRION 15" },
    { DIS_EMITTER_NAME_1170, "AI MK 23" },
    { DIS_EMITTER_NAME_1215, "AIDA II" },
    { DIS_EMITTER_NAME_1260, "Albatros MK2" },
    { DIS_EMITTER_NAME_1280, "1L13-3 (55G6), Box Spring" },
    { DIS_EMITTER_NAME_1282, "1L13-3 (55G6), Box Spring B" },
    { DIS_EMITTER_NAME_1305, "ANA SPS 502" },
    { DIS_EMITTER_NAME_1350, "ANRITSU Electric AR-30A" },
    { DIS_EMITTER_NAME_1395, "Antilope V" },
    { DIS_EMITTER_NAME_1400, "AN/ALE-50" },
    { DIS_EMITTER_NAME_1440, "AN/ALQ 99" },
    { DIS_EMITTER_NAME_1485, "AN/ALQ-100" },
    { DIS_EMITTER_NAME_1530, "AN/ALQ-101" },
    { DIS_EMITTER_NAME_1575, "AN/ALQ-119" },
    { DIS_EMITTER_NAME_1585, "AN/ALQ-122" },
    { DIS_EMITTER_NAME_1620, "AN/ALQ-126A" },
    { DIS_EMITTER_NAME_1626, "AN/ALQ-131" },
    { DIS_EMITTER_NAME_1628, "AN/ALQ-135C/D" },
    { DIS_EMITTER_NAME_1630, "AN/ALQ-144A(V)3" },
    { DIS_EMITTER_NAME_1632, "AN/ALQ-153" },
    { DIS_EMITTER_NAME_1634, "AN/ALQ-155" },
    { DIS_EMITTER_NAME_1636, "AN/ALQ-161/A" },
    { DIS_EMITTER_NAME_1638, "AN/ALQ-162" },
    { DIS_EMITTER_NAME_1640, "AN/ALQ-165" },
    { DIS_EMITTER_NAME_1642, "AN/ALQ-167" },
    { DIS_EMITTER_NAME_1644, "AN/ALQ-172(V)2" },
    { DIS_EMITTER_NAME_1646, "AN/ALQ-176" },
    { DIS_EMITTER_NAME_1648, "AN/ALQ-184" },
    { DIS_EMITTER_NAME_1650, "AN/ALQ-188" },
    { DIS_EMITTER_NAME_1652, "AN/ALR-56" },
    { DIS_EMITTER_NAME_1654, "AN/ALR-69" },
    { DIS_EMITTER_NAME_1656, "AN/ALT-16A" },
    { DIS_EMITTER_NAME_1658, "AN/ALT-28" },
    { DIS_EMITTER_NAME_1660, "AN/ALT-32A" },
    { DIS_EMITTER_NAME_1665, "AN/APD 10" },
    { DIS_EMITTER_NAME_1710, "AN/APG 53" },
    { DIS_EMITTER_NAME_1755, "AN/APG 59" },
    { DIS_EMITTER_NAME_1800, "AN/APG-63" },
    { DIS_EMITTER_NAME_1805, "AN/APG-63(V)1" },
    { DIS_EMITTER_NAME_1807, "AN/APG-63(V)2" },
    { DIS_EMITTER_NAME_1809, "AN/APG-63(V)3" },
    { DIS_EMITTER_NAME_1845, "AN/APG 65" },
    { DIS_EMITTER_NAME_1870, "AN/APG-66" },
    { DIS_EMITTER_NAME_1890, "AN/APG 68" },
    { DIS_EMITTER_NAME_1935, "AN/APG 70" },
    { DIS_EMITTER_NAME_1945, "AN/APG-73" },
    { DIS_EMITTER_NAME_1960, "AN/APG-77" },
    { DIS_EMITTER_NAME_1970, "AN/APG-78" },
    { DIS_EMITTER_NAME_1980, "AN/APG-502" },
    { DIS_EMITTER_NAME_2025, "AN/APN-1" },
    { DIS_EMITTER_NAME_2070, "AN/APN-22" },
    { DIS_EMITTER_NAME_2115, "AN/APN 59" },
    { DIS_EMITTER_NAME_2160, "AN/APN-69" },
    { DIS_EMITTER_NAME_2205, "AN/APN-81" },
    { DIS_EMITTER_NAME_2250, "AN/APN-117" },
    { DIS_EMITTER_NAME_2295, "AN/APN-118" },
    { DIS_EMITTER_NAME_2340, "AN/APN-130" },
    { DIS_EMITTER_NAME_2385, "AN/APN-131" },
    { DIS_EMITTER_NAME_2430, "AN/APN-133" },
    { DIS_EMITTER_NAME_2475, "AN/APN-134" },
    { DIS_EMITTER_NAME_2520, "AN/APN-147" },
    { DIS_EMITTER_NAME_2565, "AN/APN-150" },
    { DIS_EMITTER_NAME_2610, "AN/APN-153" },
    { DIS_EMITTER_NAME_2655, "AN/APN 154" },
    { DIS_EMITTER_NAME_2700, "AN/APN-155" },
    { DIS_EMITTER_NAME_2745, "AN/APN-159" },
    { DIS_EMITTER_NAME_2790, "AN/APN-182" },
    { DIS_EMITTER_NAME_2835, "AN/APN-187" },
    { DIS_EMITTER_NAME_2880, "AN/APN-190" },
    { DIS_EMITTER_NAME_2925, "AN/APN 194" },
    { DIS_EMITTER_NAME_2970, "AN/APN-195" },
    { DIS_EMITTER_NAME_3015, "AN/APN-198" },
    { DIS_EMITTER_NAME_3060, "AN/APN-200" },
    { DIS_EMITTER_NAME_3105, "AN/APN 202" },
    { DIS_EMITTER_NAME_3150, "AN/APN-217" },
    { DIS_EMITTER_NAME_3152, "AN/APN-218" },
    { DIS_EMITTER_NAME_3160, "AN/APN-238" },
    { DIS_EMITTER_NAME_3162, "AN/APN-239" },
    { DIS_EMITTER_NAME_3164, "AN/APN-241" },
    { DIS_EMITTER_NAME_3166, "AN/APN-242" },
    { DIS_EMITTER_NAME_3195, "AN/APN-506" },
    { DIS_EMITTER_NAME_3240, "AN/APQ-72" },
    { DIS_EMITTER_NAME_3285, "AN/APQ-99" },
    { DIS_EMITTER_NAME_3330, "AN/APQ 100" },
    { DIS_EMITTER_NAME_3375, "AN/APQ-102" },
    { DIS_EMITTER_NAME_3420, "AN/APQ-109" },
    { DIS_EMITTER_NAME_3465, "AN/APQ 113" },
    { DIS_EMITTER_NAME_3510, "AN/APQ 120" },
    { DIS_EMITTER_NAME_3555, "AN/APQ 126" },
    { DIS_EMITTER_NAME_3600, "AN/APQ-128" },
    { DIS_EMITTER_NAME_3645, "AN/APQ-129" },
    { DIS_EMITTER_NAME_3690, "AN/APQ 148" },
    { DIS_EMITTER_NAME_3735, "AN/APQ-153" },
    { DIS_EMITTER_NAME_3780, "AN/APQ 159" },
    { DIS_EMITTER_NAME_3785, "AN/APQ-164" },
    { DIS_EMITTER_NAME_3788, "AN/APQ-166" },
    { DIS_EMITTER_NAME_3795, "AN/APQ-181" },
    { DIS_EMITTER_NAME_3820, "AN/APS-31" },
    { DIS_EMITTER_NAME_3825, "AN/APS-42" },
    { DIS_EMITTER_NAME_3870, "AN/APS 80" },
    { DIS_EMITTER_NAME_3915, "AN/APS-88" },
    { DIS_EMITTER_NAME_3960, "AN/APS 115" },
    { DIS_EMITTER_NAME_4005, "AN/APS 116" },
    { DIS_EMITTER_NAME_4050, "AN/APS-120" },
    { DIS_EMITTER_NAME_4095, "AN/APS 121" },
    { DIS_EMITTER_NAME_4140, "AN/APS 124" },
    { DIS_EMITTER_NAME_4185, "AN/APS 125" },
    { DIS_EMITTER_NAME_4230, "AN/APS-128" },
    { DIS_EMITTER_NAME_4275, "AN/APS 130" },
    { DIS_EMITTER_NAME_4320, "AN/APS 133" },
    { DIS_EMITTER_NAME_4365, "AN/APS-134" },
    { DIS_EMITTER_NAME_4410, "AN/APS 137" },
    { DIS_EMITTER_NAME_4455, "AN/APS-138" },
    { DIS_EMITTER_NAME_4465, "AN/APS-143 (V) 1" },
    { DIS_EMITTER_NAME_4500, "AN/APW 22" },
    { DIS_EMITTER_NAME_4545, "AN/APW 23" },
    { DIS_EMITTER_NAME_4590, "AN/APX-6" },
    { DIS_EMITTER_NAME_4635, "AN/APX 7" },
    { DIS_EMITTER_NAME_4680, "AN/APX 39" },
    { DIS_EMITTER_NAME_4725, "AN/APX-72" },
    { DIS_EMITTER_NAME_4770, "AN/APX 76" },
    { DIS_EMITTER_NAME_4815, "AN/APX 78" },
    { DIS_EMITTER_NAME_4860, "AN/APX 101" },
    { DIS_EMITTER_NAME_4870, "AN/APX-113 AIFF" },
    { DIS_EMITTER_NAME_4900, "AN/APY-1" },
    { DIS_EMITTER_NAME_4905, "AN/APY 2" },
    { DIS_EMITTER_NAME_4950, "AN/APY 3" },
    { DIS_EMITTER_NAME_4953, "AN/APY-8, LYNX(tm)" },
    { DIS_EMITTER_NAME_4995, "AN/ARN 21" },
    { DIS_EMITTER_NAME_5040, "AN/ARN 52" },
    { DIS_EMITTER_NAME_5085, "AN/ARN 84" },
    { DIS_EMITTER_NAME_5130, "AN/ARN 118" },
    { DIS_EMITTER_NAME_5175, "AN/ARW 73" },
    { DIS_EMITTER_NAME_5220, "AN/ASB 1" },
    { DIS_EMITTER_NAME_5265, "AN/ASG 21" },
    { DIS_EMITTER_NAME_5280, "AN/ASQ-108" },
    { DIS_EMITTER_NAME_5310, "AN/AWG 9" },
    { DIS_EMITTER_NAME_5355, "AN/BPS-9" },
    { DIS_EMITTER_NAME_5400, "AN/BPS 15" },
    { DIS_EMITTER_NAME_5405, "AN/BPS-16" },
    { DIS_EMITTER_NAME_5420, "AN/CRM-30" },
    { DIS_EMITTER_NAME_5430, "AN/DPW-23" },
    { DIS_EMITTER_NAME_5445, "AN/DSQ 26 Phoenix MH" },
    { DIS_EMITTER_NAME_5490, "AN/DSQ 28 Harpoon MH" },
    { DIS_EMITTER_NAME_5495, "AN/FPN-40" },
    { DIS_EMITTER_NAME_5500, "AN/FPN-62" },
    { DIS_EMITTER_NAME_5505, "AN/FPS-16" },
    { DIS_EMITTER_NAME_5507, "AN/FPS-18" },
    { DIS_EMITTER_NAME_5508, "AN/FPS-89" },
    { DIS_EMITTER_NAME_5510, "AN/FPS-117" },
    { DIS_EMITTER_NAME_5515, "AN/FPS-20R" },
    { DIS_EMITTER_NAME_5520, "AN/FPS-77" },
    { DIS_EMITTER_NAME_5525, "AN/FPS-103" },
    { DIS_EMITTER_NAME_5527, "AN/GPN-12" },
    { DIS_EMITTER_NAME_5530, "AN/GPX-6" },
    { DIS_EMITTER_NAME_5535, "AN/GPX 8" },
    { DIS_EMITTER_NAME_5537, "AN/GRN-12" },
    { DIS_EMITTER_NAME_5540, "AN/MPQ-10" },
    { DIS_EMITTER_NAME_5545, "AN/MPQ-33/39/46/57/61 (HPIR) ILL" },
    { DIS_EMITTER_NAME_5550, "AN/MPQ-34/48/55/62 (CWAR) TA" },
    { DIS_EMITTER_NAME_5551, "AN/MPQ-49" },
    { DIS_EMITTER_NAME_5555, "AN/MPQ-35/50 (PAR) TA" },
    { DIS_EMITTER_NAME_5560, "AN/MPQ-37/51 (ROR) TT" },
    { DIS_EMITTER_NAME_5570, "AN/MPQ-53" },
    { DIS_EMITTER_NAME_5571, "AN/MPQ-63" },
    { DIS_EMITTER_NAME_5575, "AN/MPQ-64" },
    { DIS_EMITTER_NAME_5580, "AN/SPG-34" },
    { DIS_EMITTER_NAME_5625, "AN/SPG 50" },
    { DIS_EMITTER_NAME_5670, "AN/SPG 51" },
    { DIS_EMITTER_NAME_5715, "AN/SPG-51 CWI TI" },
    { DIS_EMITTER_NAME_5760, "AN/SPG-51 FC" },
    { DIS_EMITTER_NAME_5805, "AN/SPG 52" },
    { DIS_EMITTER_NAME_5850, "AN/SPG-53" },
    { DIS_EMITTER_NAME_5895, "AN/SPG 55B" },
    { DIS_EMITTER_NAME_5940, "AN/SPG 60" },
    { DIS_EMITTER_NAME_5985, "AN/SPG 62" },
    { DIS_EMITTER_NAME_6030, "AN/SPN 35" },
    { DIS_EMITTER_NAME_6075, "AN/SPN 43" },
    { DIS_EMITTER_NAME_6120, "AN/SPQ-2" },
    { DIS_EMITTER_NAME_6165, "AN/SPQ 9" },
    { DIS_EMITTER_NAME_6210, "AN/SPS-4" },
    { DIS_EMITTER_NAME_6255, "AN/SPS-5" },
    { DIS_EMITTER_NAME_6300, "AN/SPS-5C" },
    { DIS_EMITTER_NAME_6345, "AN/SPS-6" },
    { DIS_EMITTER_NAME_6390, "AN/SPS 10" },
    { DIS_EMITTER_NAME_6435, "AN/SPS 21" },
    { DIS_EMITTER_NAME_6480, "AN/SPS-28" },
    { DIS_EMITTER_NAME_6525, "AN/SPS-37" },
    { DIS_EMITTER_NAME_6570, "AN/SPS-39A" },
    { DIS_EMITTER_NAME_6615, "AN/SPS 40" },
    { DIS_EMITTER_NAME_6660, "AN/SPS-41" },
    { DIS_EMITTER_NAME_6705, "AN/SPS-48" },
    { DIS_EMITTER_NAME_6750, "AN/SPS-48C" },
    { DIS_EMITTER_NAME_6752, "AN/SPS-48E" },
    { DIS_EMITTER_NAME_6795, "AN/SPS-49" },
    { DIS_EMITTER_NAME_6796, "AN/SPS-49(V)1" },
    { DIS_EMITTER_NAME_6797, "AN/SPS-49(V)2" },
    { DIS_EMITTER_NAME_6798, "AN/SPS-49(V)3" },
    { DIS_EMITTER_NAME_6799, "AN/SPS-49(V)4" },
    { DIS_EMITTER_NAME_6800, "AN/SPS-49(V)5" },
    { DIS_EMITTER_NAME_6801, "AN/SPS-49(V)6" },
    { DIS_EMITTER_NAME_6802, "AN/SPS-49(V)7" },
    { DIS_EMITTER_NAME_6803, "AN/SPS-49(V)8" },
    { DIS_EMITTER_NAME_6804, "AN/SPS-49A(V)1" },
    { DIS_EMITTER_NAME_6840, "AN/SPS 52" },
    { DIS_EMITTER_NAME_6885, "AN/SPS 53" },
    { DIS_EMITTER_NAME_6930, "AN/SPS 55" },
    { DIS_EMITTER_NAME_6975, "AN/SPS-55 SS" },
    { DIS_EMITTER_NAME_7020, "AN/SPS-58" },
    { DIS_EMITTER_NAME_7065, "AN/SPS 59" },
    { DIS_EMITTER_NAME_7110, "AN/SPS 64" },
    { DIS_EMITTER_NAME_7155, "AN/SPS 65" },
    { DIS_EMITTER_NAME_7200, "AN/SPS 67" },
    { DIS_EMITTER_NAME_7245, "AN/SPY-1" },
    { DIS_EMITTER_NAME_7250, "AN/SPY-1A" },
    { DIS_EMITTER_NAME_7252, "AN/SPY-1B" },
    { DIS_EMITTER_NAME_7253, "AN/SPY-1B(V)" },
    { DIS_EMITTER_NAME_7260, "AN/SPY-1D" },
    { DIS_EMITTER_NAME_7261, "AN/SPY-1D(V)" },
    { DIS_EMITTER_NAME_7265, "AN/SPY-1F" },
    { DIS_EMITTER_NAME_7270, "AN/TPN-17" },
    { DIS_EMITTER_NAME_7275, "AN/TPN-24" },
    { DIS_EMITTER_NAME_7280, "AN/TPQ-18" },
    { DIS_EMITTER_NAME_7295, "AN/TPQ-36" },
    { DIS_EMITTER_NAME_7300, "AN/TPQ-37" },
    { DIS_EMITTER_NAME_7301, "AN/TPQ-38 (V8)" },
    { DIS_EMITTER_NAME_7303, "AN/TPQ-47" },
    { DIS_EMITTER_NAME_7305, "AN/TPS-43" },
    { DIS_EMITTER_NAME_7310, "AN/TPS-43E" },
    { DIS_EMITTER_NAME_7315, "AN/TPS-59" },
    { DIS_EMITTER_NAME_7320, "AN/TPS-63" },
    { DIS_EMITTER_NAME_7322, "AN/TPS-70 (V) 1" },
    { DIS_EMITTER_NAME_7325, "AN/TPS-75" },
    { DIS_EMITTER_NAME_7330, "AN/TPX-46(V)7" },
    { DIS_EMITTER_NAME_7335, "AN/ULQ-6A" },
    { DIS_EMITTER_NAME_7380, "AN/UPN 25" },
    { DIS_EMITTER_NAME_7425, "AN/UPS 1" },
    { DIS_EMITTER_NAME_7426, "AN/UPS-2" },
    { DIS_EMITTER_NAME_7470, "AN/UPX 1" },
    { DIS_EMITTER_NAME_7515, "AN/UPX 5" },
    { DIS_EMITTER_NAME_7560, "AN/UPX 11" },
    { DIS_EMITTER_NAME_7605, "AN/UPX 12" },
    { DIS_EMITTER_NAME_7650, "AN/UPX 17" },
    { DIS_EMITTER_NAME_7695, "AN/UPX 23" },
    { DIS_EMITTER_NAME_7740, "AN/VPS 2" },
    { DIS_EMITTER_NAME_7785, "Apelco AD 7 7" },
    { DIS_EMITTER_NAME_7830, "APG 71" },
    { DIS_EMITTER_NAME_7875, "APN 148" },
    { DIS_EMITTER_NAME_7920, "APN 227" },
    { DIS_EMITTER_NAME_7965, "(deleted)" },
    { DIS_EMITTER_NAME_8010, "(deleted)" },
    { DIS_EMITTER_NAME_8055, "(deleted)" },
    { DIS_EMITTER_NAME_8100, "APS 504 V3" },
    { DIS_EMITTER_NAME_8105, "AR 3D" },
    { DIS_EMITTER_NAME_8112, "Plessey AR-5" },
    { DIS_EMITTER_NAME_8115, "AR 320" },
    { DIS_EMITTER_NAME_8120, "AR 327" },
    { DIS_EMITTER_NAME_8145, "AR M31" },
    { DIS_EMITTER_NAME_8190, "ARI 5954" },
    { DIS_EMITTER_NAME_8235, "ARI 5955" },
    { DIS_EMITTER_NAME_8280, "ARI 5979" },
    { DIS_EMITTER_NAME_8325, "ARINC 564 BNDX/KING RDR 1E" },
    { DIS_EMITTER_NAME_8370, "ARINC 700 BNDX/KING RDR 1E" },
    { DIS_EMITTER_NAME_8375, "ARK-1" },
    { DIS_EMITTER_NAME_8380, "ARSR-3" },
    { DIS_EMITTER_NAME_8390, "ARSR-18" },
    { DIS_EMITTER_NAME_8415, "AS 2 Kipper" },
    { DIS_EMITTER_NAME_8460, "AS 2 Kipper MH" },
    { DIS_EMITTER_NAME_8505, "AS 4 Kitchen" },
    { DIS_EMITTER_NAME_8550, "AS 4 Kitchen MH" },
    { DIS_EMITTER_NAME_8595, "AS 5 Kelt MH" },
    { DIS_EMITTER_NAME_8640, "AS 6 Kingfish MH" },
    { DIS_EMITTER_NAME_8685, "AS 7 Kerry" },
    { DIS_EMITTER_NAME_8730, "AS 7 Kerry MG" },
    { DIS_EMITTER_NAME_8735, "AS 15 KENT altimeter" },
    { DIS_EMITTER_NAME_8760, "Aspide AAM/SAM ILL" },
    { DIS_EMITTER_NAME_8772, "ASR-4" },
    { DIS_EMITTER_NAME_8775, "ASR O" },
    { DIS_EMITTER_NAME_8780, "ASR-5" },
    { DIS_EMITTER_NAME_8782, "ASR-7" },
    { DIS_EMITTER_NAME_8785, "ASR-8" },
    { DIS_EMITTER_NAME_8790, "ASR-9" },
    { DIS_EMITTER_NAME_8812, "Raytheon ASR-10SS" },
    { DIS_EMITTER_NAME_8820, "AT 2 Swatter MG" },
    { DIS_EMITTER_NAME_8840, "ATCR-33" },
    { DIS_EMITTER_NAME_8845, "ATCR 33 K/M" },
    { DIS_EMITTER_NAME_8865, "Atlas Elektronk TRS N" },
    { DIS_EMITTER_NAME_8870, "ATLAS-9740 VTS" },
    { DIS_EMITTER_NAME_8910, "AVG 65" },
    { DIS_EMITTER_NAME_8955, "AVH 7" },
    { DIS_EMITTER_NAME_9000, "AVQ 20" },
    { DIS_EMITTER_NAME_9045, "AVQ30X" },
    { DIS_EMITTER_NAME_9075, "AVQ-50 (RCA)" },
    { DIS_EMITTER_NAME_9090, "AVQ 70" },
    { DIS_EMITTER_NAME_9135, "AWS 5" },
    { DIS_EMITTER_NAME_9180, "AWS 6" },
    { DIS_EMITTER_NAME_9200, "B597Z" },
    { DIS_EMITTER_NAME_9205, "B636Z" },
    { DIS_EMITTER_NAME_9225, "Back Net A B" },
    { DIS_EMITTER_NAME_9270, "Back Trap" },
    { DIS_EMITTER_NAME_9310, "BALTYK" },
    { DIS_EMITTER_NAME_9315, "Ball End" },
    { DIS_EMITTER_NAME_9360, "Ball Gun" },
    { DIS_EMITTER_NAME_9405, "Band Stand" },
    { DIS_EMITTER_NAME_9450, "P-35/37 (A); P-50 (B), Bar Lock" },
    { DIS_EMITTER_NAME_9495, "Bass Tilt" },
    { DIS_EMITTER_NAME_9540, "Beacon" },
    { DIS_EMITTER_NAME_9585, "Bean Sticks" },
    { DIS_EMITTER_NAME_9630, "Bee Hind" },
    { DIS_EMITTER_NAME_9640, "Bell Crown A" },
    { DIS_EMITTER_NAME_9642, "Bell Crown B" },
    { DIS_EMITTER_NAME_9645, "BIG BACK" },
    { DIS_EMITTER_NAME_9660, "Big Bird" },
    { DIS_EMITTER_NAME_9675, "Big Bulge" },
    { DIS_EMITTER_NAME_9720, "Big Bulge A" },
    { DIS_EMITTER_NAME_9765, "Big Bulge B" },
    { DIS_EMITTER_NAME_9780, "SNAR-10, Big Fred" },
    { DIS_EMITTER_NAME_9810, "Big Mesh" },
    { DIS_EMITTER_NAME_9855, "Big Net" },
    { DIS_EMITTER_NAME_9885, "9S15MT, Bill Board" },
    { DIS_EMITTER_NAME_9900, "Bill Fold" },
    { DIS_EMITTER_NAME_9905, "Blowpipe MG" },
    { DIS_EMITTER_NAME_9930, "Blue Fox, Sea Harrier FRS Mk 1/5" },
    { DIS_EMITTER_NAME_9935, "Blue Vixen, Sea Harrier F/A Mk 2" },
    { DIS_EMITTER_NAME_9945, "Blue Silk" },
    { DIS_EMITTER_NAME_9990, "Blue Parrot" },
    { DIS_EMITTER_NAME_10035, "Blue Orchid" },
    { DIS_EMITTER_NAME_10080, "Boat Sail" },
    { DIS_EMITTER_NAME_10125, "Bofors Electronic 9LV 331" },
    { DIS_EMITTER_NAME_10170, "Bofors Ericsson Sea Giraffe 50 HC" },
    { DIS_EMITTER_NAME_10215, "Bowl Mesh" },
    { DIS_EMITTER_NAME_10260, "Box Brick" },
    { DIS_EMITTER_NAME_10305, "Box Tail" },
    { DIS_EMITTER_NAME_10350, "BPS 11A" },
    { DIS_EMITTER_NAME_10395, "BPS 14" },
    { DIS_EMITTER_NAME_10440, "BPS 15A" },
    { DIS_EMITTER_NAME_10485, "BR-15 Tokyo KEIKI" },
    { DIS_EMITTER_NAME_10510, "BRIDGEMASTE" },
    { DIS_EMITTER_NAME_10530, "Bread Bin" },
    { DIS_EMITTER_NAME_10575, "BT 271" },
    { DIS_EMITTER_NAME_10620, "BX 732" },
    { DIS_EMITTER_NAME_10665, "Buzz Stand" },
    { DIS_EMITTER_NAME_10710, "C 5A Multi Mode Radar" },
    { DIS_EMITTER_NAME_10755, "Caiman" },
    { DIS_EMITTER_NAME_10800, "Cake Stand" },
    { DIS_EMITTER_NAME_10845, "Calypso C61" },
    { DIS_EMITTER_NAME_10890, "Calypso Ii" },
    { DIS_EMITTER_NAME_10895, "Cardion Coastal" },
    { DIS_EMITTER_NAME_10935, "Castor Ii" },
    { DIS_EMITTER_NAME_10940, "Castor 2J TT (Crotale NG)" },
    { DIS_EMITTER_NAME_10980, "Cat House" },
    { DIS_EMITTER_NAME_10985, "CDR-431" },
    { DIS_EMITTER_NAME_11000, "Chair Back TT" },
    { DIS_EMITTER_NAME_11010, "Chair Back ILL" },
    { DIS_EMITTER_NAME_11025, "Cheese Brick" },
    { DIS_EMITTER_NAME_11070, "Clam Pipe" },
    { DIS_EMITTER_NAME_11115, "Clamshell" },
    { DIS_EMITTER_NAME_11160, "Collins WXR-700X" },
    { DIS_EMITTER_NAME_11205, "Collins DN 101" },
    { DIS_EMITTER_NAME_11250, "Contraves Sea Hunter MK 4" },
    { DIS_EMITTER_NAME_11260, "Corn Can" },
    { DIS_EMITTER_NAME_11270, "CR-105 RMCA" },
    { DIS_EMITTER_NAME_11295, "Cross Bird" },
    { DIS_EMITTER_NAME_11340, "Cross Dome" },
    { DIS_EMITTER_NAME_11385, "Cross Legs" },
    { DIS_EMITTER_NAME_11430, "Cross Out" },
    { DIS_EMITTER_NAME_11475, "Cross Slot" },
    { DIS_EMITTER_NAME_11520, "Cross Sword" },
    { DIS_EMITTER_NAME_11565, "Cross Up" },
    { DIS_EMITTER_NAME_11610, "Cross Sword FC" },
    { DIS_EMITTER_NAME_11655, "Crotale Acquisition TA, THD-5000" },
    { DIS_EMITTER_NAME_11660, "Crotale NG TA, Griffon" },
    { DIS_EMITTER_NAME_11665, "Crotale TT" },
    { DIS_EMITTER_NAME_11700, "Crotale MGMissile System" },
    { DIS_EMITTER_NAME_11745, "CSS C 3C CAS 1M1 M2 MH" },
    { DIS_EMITTER_NAME_11790, "CSS C 2B HY 1A MH" },
    { DIS_EMITTER_NAME_11835, "CWS 2" },
    { DIS_EMITTER_NAME_11880, "Cylinder Head" },
    { DIS_EMITTER_NAME_11925, "Cyrano II" },
    { DIS_EMITTER_NAME_11970, "Cyrano IV" },
    { DIS_EMITTER_NAME_11975, "Cyrano IV-M" },
    { DIS_EMITTER_NAME_12010, "DA-01/00" },
    { DIS_EMITTER_NAME_12015, "DA 05 00" },
    { DIS_EMITTER_NAME_12060, "Dawn" },
    { DIS_EMITTER_NAME_12105, "Dead Duck" },
    { DIS_EMITTER_NAME_12110, "DECCA-20 V90/9" },
    { DIS_EMITTER_NAME_12111, "DECCA-20 V90S" },
    { DIS_EMITTER_NAME_12150, "DECCA 45" },
    { DIS_EMITTER_NAME_12195, "DECCA 50" },
    { DIS_EMITTER_NAME_12240, "DECCA 110" },
    { DIS_EMITTER_NAME_12285, "DECCA 170" },
    { DIS_EMITTER_NAME_12292, "DECCA HF 2" },
    { DIS_EMITTER_NAME_12330, "DECCA 202" },
    { DIS_EMITTER_NAME_12375, "DECCA D202" },
    { DIS_EMITTER_NAME_12420, "DECCA 303" },
    { DIS_EMITTER_NAME_12430, "DECCA 535" },
    { DIS_EMITTER_NAME_12465, "DECCA 626" },
    { DIS_EMITTER_NAME_12510, "DECCA 629" },
    { DIS_EMITTER_NAME_12555, "DECCA 914" },
    { DIS_EMITTER_NAME_12600, "DECCA 916" },
    { DIS_EMITTER_NAME_12610, "DECCA 926" },
    { DIS_EMITTER_NAME_12645, "DECCA 1226 Commercial" },
    { DIS_EMITTER_NAME_12690, "DECCA 1626" },
    { DIS_EMITTER_NAME_12735, "DECCA 2459" },
    { DIS_EMITTER_NAME_12780, "DECCA AWS 1" },
    { DIS_EMITTER_NAME_12782, "DECCA AWS 2" },
    { DIS_EMITTER_NAME_12785, "DECCA AWS 4" },
    { DIS_EMITTER_NAME_12787, "DECCA AWS-4 (2)" },
    { DIS_EMITTER_NAME_12800, "DECCA MAR" },
    { DIS_EMITTER_NAME_12805, "DECCA RM 326" },
    { DIS_EMITTER_NAME_12825, "DECCA RM 416" },
    { DIS_EMITTER_NAME_12870, "DECCA RM 914" },
    { DIS_EMITTER_NAME_12915, "DECCA RM 1690" },
    { DIS_EMITTER_NAME_12960, "DECCA Super 101 MK 3" },
    { DIS_EMITTER_NAME_13005, "DISS 1" },
    { DIS_EMITTER_NAME_13050, "Rapier TTDN 181, DN 181" },
    { DIS_EMITTER_NAME_13055, "Rapier 2000 TT, BLINDFIRE FSC TT" },
    { DIS_EMITTER_NAME_13095, "Dog Ear" },
    { DIS_EMITTER_NAME_13140, "Dog House" },
    { DIS_EMITTER_NAME_13185, "Don 2" },
    { DIS_EMITTER_NAME_13230, "Don A/B/2/Kay" },
    { DIS_EMITTER_NAME_13275, "Donets" },
    { DIS_EMITTER_NAME_13320, "Down Beat" },
    { DIS_EMITTER_NAME_13365, "DRAA 2A" },
    { DIS_EMITTER_NAME_13410, "DRAA 2B" },
    { DIS_EMITTER_NAME_13455, "DRAC 39" },
    { DIS_EMITTER_NAME_13500, "DRBC 30B" },
    { DIS_EMITTER_NAME_13545, "DRBC 31A" },
    { DIS_EMITTER_NAME_13590, "DRBC 32A" },
    { DIS_EMITTER_NAME_13635, "DRBC 32D" },
    { DIS_EMITTER_NAME_13680, "DRBC 33A" },
    { DIS_EMITTER_NAME_13725, "DRBI 10" },
    { DIS_EMITTER_NAME_13770, "DRBI 23" },
    { DIS_EMITTER_NAME_13815, "DRBJ 11B" },
    { DIS_EMITTER_NAME_13860, "DRBN 30" },
    { DIS_EMITTER_NAME_13905, "DRBN 32" },
    { DIS_EMITTER_NAME_13950, "DRBR 51" },
    { DIS_EMITTER_NAME_13995, "DRBV 20B" },
    { DIS_EMITTER_NAME_14040, "DRBV 22" },
    { DIS_EMITTER_NAME_14085, "DRBV 26C" },
    { DIS_EMITTER_NAME_14130, "DRBV 30" },
    { DIS_EMITTER_NAME_14175, "DRBV 50" },
    { DIS_EMITTER_NAME_14220, "DRBV 51" },
    { DIS_EMITTER_NAME_14265, "DRBV 51A" },
    { DIS_EMITTER_NAME_14310, "DRBV 51B" },
    { DIS_EMITTER_NAME_14355, "DRBV 51C" },
    { DIS_EMITTER_NAME_14400, "Drop Kick" },
    { DIS_EMITTER_NAME_14445, "DRUA 31" },
    { DIS_EMITTER_NAME_14490, "Drum Tilt" },
    { DIS_EMITTER_NAME_14535, "Drum Tilt A" },
    { DIS_EMITTER_NAME_14545, "Drum Tilt B" },
    { DIS_EMITTER_NAME_14580, "Dumbo" },
    { DIS_EMITTER_NAME_14600, "ECR-90" },
    { DIS_EMITTER_NAME_14625, "Egg Cup A/B" },
    { DIS_EMITTER_NAME_14670, "EKCO 190" },
    { DIS_EMITTER_NAME_14715, "EL M 2001B" },
    { DIS_EMITTER_NAME_14760, "EL M 2207" },
    { DIS_EMITTER_NAME_14770, "EL/M 2216(V)" },
    { DIS_EMITTER_NAME_14805, "ELTA EL/M 2221 GM STGR" },
    { DIS_EMITTER_NAME_14810, "ELTA SIS" },
    { DIS_EMITTER_NAME_14850, "EMD 2900" },
    { DIS_EMITTER_NAME_14895, "End Tray" },
    { DIS_EMITTER_NAME_14940, "Exocet 1" },
    { DIS_EMITTER_NAME_14985, "Exocet 1 MH" },
    { DIS_EMITTER_NAME_15030, "Exocet 2" },
    { DIS_EMITTER_NAME_15075, "Eye Bowl" },
    { DIS_EMITTER_NAME_15120, "Eye Shield" },
    { DIS_EMITTER_NAME_15140, "F332Z" },
    { DIS_EMITTER_NAME_15160, "FALCON" },
    { DIS_EMITTER_NAME_15165, "Fan Song A" },
    { DIS_EMITTER_NAME_15200, "Fan Song B/F TA" },
    { DIS_EMITTER_NAME_15210, "Fan Song B/F TT" },
    { DIS_EMITTER_NAME_15220, "Fan Song C/E TA" },
    { DIS_EMITTER_NAME_15230, "Fan Song C/E TT" },
    { DIS_EMITTER_NAME_15240, "Fan Song C/E MG" },
    { DIS_EMITTER_NAME_15255, "Fan Song B/FF MG" },
    { DIS_EMITTER_NAME_15300, "Fan Tail" },
    { DIS_EMITTER_NAME_15310, "FCR-1401" },
    { DIS_EMITTER_NAME_15345, "Fin Curve" },
    { DIS_EMITTER_NAME_15390, "Fire Can" },
    { DIS_EMITTER_NAME_15435, "Fire Dish" },
    { DIS_EMITTER_NAME_15470, "Fire Dome TA" },
    { DIS_EMITTER_NAME_15475, "Fire Dome TT" },
    { DIS_EMITTER_NAME_15480, "Fire Dome TI" },
    { DIS_EMITTER_NAME_15525, "Fire Iron" },
    { DIS_EMITTER_NAME_15570, "Fire Wheel" },
    { DIS_EMITTER_NAME_15615, "Fish Bowl" },
    { DIS_EMITTER_NAME_15660, "Flap Lid" },
    { DIS_EMITTER_NAME_15705, "Flap Truck" },
    { DIS_EMITTER_NAME_15750, "Flap Wheel" },
    { DIS_EMITTER_NAME_15795, "Flash Dance" },
    { DIS_EMITTER_NAME_15840, "P-15, Flat Face A B C D" },
    { DIS_EMITTER_NAME_15885, "Flat Screen" },
    { DIS_EMITTER_NAME_15930, "Flat Spin" },
    { DIS_EMITTER_NAME_15975, "Flat Twin" },
    { DIS_EMITTER_NAME_16020, "Fledermaus" },
    { DIS_EMITTER_NAME_16030, "FLYCATCHER" },
    { DIS_EMITTER_NAME_16065, "Fly Screen" },
    { DIS_EMITTER_NAME_16110, "Fly Screen A&B" },
    { DIS_EMITTER_NAME_16155, "Fly Trap B" },
    { DIS_EMITTER_NAME_16200, "Fog Lamp MG" },
    { DIS_EMITTER_NAME_16245, "Fog Lamp TT" },
    { DIS_EMITTER_NAME_16290, "Foil Two" },
    { DIS_EMITTER_NAME_16335, "Fox Hunter" },
    { DIS_EMITTER_NAME_16380, "FOX FIREFox Fire AL" },
    { DIS_EMITTER_NAME_16390, "FOX FIRE ILL" },
    { DIS_EMITTER_NAME_16400, "FR-151A" },
    { DIS_EMITTER_NAME_16410, "FR-1505 DA" },
    { DIS_EMITTER_NAME_16420, "FR-2000" },
    { DIS_EMITTER_NAME_16421, "FR-2855W" },
    { DIS_EMITTER_NAME_16425, "Front Dome" },
    { DIS_EMITTER_NAME_16470, "Front Door" },
    { DIS_EMITTER_NAME_16515, "Front Piece" },
    { DIS_EMITTER_NAME_16560, "Furuno" },
    { DIS_EMITTER_NAME_16561, "Furuno 1721" },
    { DIS_EMITTER_NAME_16605, "Furuno 701" },
    { DIS_EMITTER_NAME_16650, "Furuno 711 2" },
    { DIS_EMITTER_NAME_16695, "Furuno 2400" },
    { DIS_EMITTER_NAME_16740, "GA 01 00" },
    { DIS_EMITTER_NAME_16785, "Gage" },
    { DIS_EMITTER_NAME_16830, "Garpin" },
    { DIS_EMITTER_NAME_16875, "GEM BX 132" },
    { DIS_EMITTER_NAME_16880, "MPDR-12, Gepard TA" },
    { DIS_EMITTER_NAME_16884, "Gepard TT" },
    { DIS_EMITTER_NAME_16888, "GERAN-F" },
    { DIS_EMITTER_NAME_16900, "GIRAFFE" },
    { DIS_EMITTER_NAME_16915, "Gin Sling TA" },
    { DIS_EMITTER_NAME_16920, "Gin Sling, Gin Sling TT" },
    { DIS_EMITTER_NAME_16925, "Gin Sling MG" },
    { DIS_EMITTER_NAME_16945, "GPN-22" },
    { DIS_EMITTER_NAME_16950, "GRN-9" },
    { DIS_EMITTER_NAME_16965, "Green Stain" },
    { DIS_EMITTER_NAME_17010, "Grid Bow" },
    { DIS_EMITTER_NAME_17025, "9S32, GRILL PAN TT" },
    { DIS_EMITTER_NAME_17055, "Guardsman" },
    { DIS_EMITTER_NAME_17070, "RPK-2, GUN DISH (ZSU-23/4)" },
    { DIS_EMITTER_NAME_17100, "Hair Net" },
    { DIS_EMITTER_NAME_17145, "Half Plate A" },
    { DIS_EMITTER_NAME_17190, "Half Plate B" },
    { DIS_EMITTER_NAME_17220, "HARD" },
    { DIS_EMITTER_NAME_17235, "Hawk Screech" },
    { DIS_EMITTER_NAME_17280, "Head Light A" },
    { DIS_EMITTER_NAME_17325, "Head Lights" },
    { DIS_EMITTER_NAME_17370, "Head Lights C" },
    { DIS_EMITTER_NAME_17415, "Head Lights MG A" },
    { DIS_EMITTER_NAME_17460, "Head Lights MG B" },
    { DIS_EMITTER_NAME_17505, "Head Lights TT" },
    { DIS_EMITTER_NAME_17550, "Head Net" },
    { DIS_EMITTER_NAME_17595, "Hen Egg" },
    { DIS_EMITTER_NAME_17640, "Hen House" },
    { DIS_EMITTER_NAME_17685, "Hen Nest" },
    { DIS_EMITTER_NAME_17730, "Hen Roost" },
    { DIS_EMITTER_NAME_17775, "High Brick" },
    { DIS_EMITTER_NAME_17820, "High Fix" },
    { DIS_EMITTER_NAME_17865, "High Lark TI" },
    { DIS_EMITTER_NAME_17910, "High Lark 1" },
    { DIS_EMITTER_NAME_17955, "High Lark 2" },
    { DIS_EMITTER_NAME_18000, "High Lark 4" },
    { DIS_EMITTER_NAME_18045, "High Lune" },
    { DIS_EMITTER_NAME_18090, "High Pole A&B" },
    { DIS_EMITTER_NAME_18135, "High Scoop" },
    { DIS_EMITTER_NAME_18150, "9S19MT, HIGH SCREEN" },
    { DIS_EMITTER_NAME_18180, "High Sieve" },
    { DIS_EMITTER_NAME_18200, "HN-503" },
    { DIS_EMITTER_NAME_18225, "Home Talk" },
    { DIS_EMITTER_NAME_18270, "Horn Spoon" },
    { DIS_EMITTER_NAME_18280, "HOT BRICK" },
    { DIS_EMITTER_NAME_18315, "Hot Flash" },
    { DIS_EMITTER_NAME_18320, "IRL144M, Hot Shot TA" },
    { DIS_EMITTER_NAME_18325, "IRL144M, Hot Shot TT" },
    { DIS_EMITTER_NAME_18330, "IRL144M, Hot Shot MG" },
    { DIS_EMITTER_NAME_18360, "IFF MK XII AIMS UPX 29" },
    { DIS_EMITTER_NAME_18405, "IFF MK XV" },
    { DIS_EMITTER_NAME_18410, "Javelin MG" },
    { DIS_EMITTER_NAME_18450, "Jay Bird" },
    { DIS_EMITTER_NAME_18460, "JRC-NMD-401" },
    { DIS_EMITTER_NAME_18495, "Jupiter" },
    { DIS_EMITTER_NAME_18540, "Jupiter II" },
    { DIS_EMITTER_NAME_18550, "JY-8" },
    { DIS_EMITTER_NAME_18555, "JY-9" },
    { DIS_EMITTER_NAME_18560, "JY-14" },
    { DIS_EMITTER_NAME_18585, "K376Z" },
    { DIS_EMITTER_NAME_18630, "Kelvin Hughes 2A" },
    { DIS_EMITTER_NAME_18675, "Kelvin Hughes 14/9" },
    { DIS_EMITTER_NAME_18720, "Kelvin Hughes type 1006" },
    { DIS_EMITTER_NAME_18765, "Kelvin Hughes type 1007" },
    { DIS_EMITTER_NAME_18785, "KH-902M" },
    { DIS_EMITTER_NAME_18810, "Kite Screech" },
    { DIS_EMITTER_NAME_18855, "Kite Screech A" },
    { DIS_EMITTER_NAME_18900, "Kite Screech B" },
    { DIS_EMITTER_NAME_18945, "Kivach" },
    { DIS_EMITTER_NAME_18990, "Knife Rest" },
    { DIS_EMITTER_NAME_19035, "P-10, Knife Rest B" },
    { DIS_EMITTER_NAME_19037, "KNIFE REST C" },
    { DIS_EMITTER_NAME_19050, "KR-75" },
    { DIS_EMITTER_NAME_19080, "KSA SRN" },
    { DIS_EMITTER_NAME_19125, "KSA TSR" },
    { DIS_EMITTER_NAME_19170, "Land Fall" },
    { DIS_EMITTER_NAME_19215, "Land Roll MG" },
    { DIS_EMITTER_NAME_19260, "Land Roll TA" },
    { DIS_EMITTER_NAME_19305, "Land Roll TT" },
    { DIS_EMITTER_NAME_19310, "LC-150" },
    { DIS_EMITTER_NAME_19350, "Leningraf" },
    { DIS_EMITTER_NAME_19395, "Light Bulb" },
    { DIS_EMITTER_NAME_19400, "LMT NRAI-6A" },
    { DIS_EMITTER_NAME_19440, "LN 55" },
    { DIS_EMITTER_NAME_19485, "Ln 66" },
    { DIS_EMITTER_NAME_19530, "Long Bow" },
    { DIS_EMITTER_NAME_19575, "Long Brick" },
    { DIS_EMITTER_NAME_19620, "Long Bull" },
    { DIS_EMITTER_NAME_19665, "Long Eye" },
    { DIS_EMITTER_NAME_19710, "Long Head" },
    { DIS_EMITTER_NAME_19755, "Long Talk" },
    { DIS_EMITTER_NAME_19800, "Long Track" },
    { DIS_EMITTER_NAME_19845, "Long Trough" },
    { DIS_EMITTER_NAME_19890, "Look Two" },
    { DIS_EMITTER_NAME_19935, "LORAN" },
    { DIS_EMITTER_NAME_19950, "Low Blow TA" },
    { DIS_EMITTER_NAME_19955, "Low Blow TT" },
    { DIS_EMITTER_NAME_19960, "Low Blow MG" },
    { DIS_EMITTER_NAME_19980, "Low Sieve" },
    { DIS_EMITTER_NAME_20025, "Low Trough" },
    { DIS_EMITTER_NAME_20040, "TRS-2050, LP-23" },
    { DIS_EMITTER_NAME_20070, "LW 08" },
    { DIS_EMITTER_NAME_20090, "M-1983 FCR" },
    { DIS_EMITTER_NAME_20115, "M22-40" },
    { DIS_EMITTER_NAME_20160, "M44" },
    { DIS_EMITTER_NAME_20205, "M401Z" },
    { DIS_EMITTER_NAME_20250, "M585Z" },
    { DIS_EMITTER_NAME_20295, "M588Z" },
    { DIS_EMITTER_NAME_20340, "MA 1 IFF Portion" },
    { DIS_EMITTER_NAME_20360, "MARELD" },
    { DIS_EMITTER_NAME_20385, "MA Type 909#" },
    { DIS_EMITTER_NAME_20430, "Marconi 1810" },
    { DIS_EMITTER_NAME_20475, "Marconi Canada HC 75" },
    { DIS_EMITTER_NAME_20495, "Marconi S 713" },
    { DIS_EMITTER_NAME_20520, "Marconi S 1802" },
    { DIS_EMITTER_NAME_20530, "Marconi S247" },
    { DIS_EMITTER_NAME_20565, "Marconi S 810" },
    { DIS_EMITTER_NAME_20585, "Marconi SA 10" },
    { DIS_EMITTER_NAME_20610, "Marconi type 967" },
    { DIS_EMITTER_NAME_20655, "Marconi type 968" },
    { DIS_EMITTER_NAME_20700, "Marconi type 992" },
    { DIS_EMITTER_NAME_20745, "Marconi/signaal type 1022" },
    { DIS_EMITTER_NAME_20790, "Marconi/signaal type 910" },
    { DIS_EMITTER_NAME_20835, "Marconi/signaal type 911" },
    { DIS_EMITTER_NAME_20880, "Marconi/signaal type 992R" },
    { DIS_EMITTER_NAME_20925, "Mesh Brick" },
    { DIS_EMITTER_NAME_20950, "Mirage ILL" },
    { DIS_EMITTER_NAME_20970, "MK 15 CIWS" },
    { DIS_EMITTER_NAME_21015, "MK-23" },
    { DIS_EMITTER_NAME_21060, "MK 23 TAS" },
    { DIS_EMITTER_NAME_21105, "MK 25" },
    { DIS_EMITTER_NAME_21150, "MK-35 M2" },
    { DIS_EMITTER_NAME_21195, "MK 92" },
    { DIS_EMITTER_NAME_21240, "MK-92 CAS" },
    { DIS_EMITTER_NAME_21285, "MK-92 STIR" },
    { DIS_EMITTER_NAME_21330, "MK 95" },
    { DIS_EMITTER_NAME_21340, "MLA-1" },
    { DIS_EMITTER_NAME_21375, "MM APS 705" },
    { DIS_EMITTER_NAME_21420, "MM SPG 74" },
    { DIS_EMITTER_NAME_21465, "MM SPG 75" },
    { DIS_EMITTER_NAME_21490, "MM SPN 703" },
    { DIS_EMITTER_NAME_21510, "MM SPS 702" },
    { DIS_EMITTER_NAME_21555, "MM SPS 768" },
    { DIS_EMITTER_NAME_21600, "MM SPS 774" },
    { DIS_EMITTER_NAME_21645, "Moon 4" },
    { DIS_EMITTER_NAME_21650, "MMRS" },
    { DIS_EMITTER_NAME_21690, "MPDR 18 X" },
    { DIS_EMITTER_NAME_21710, "MT-305X" },
    { DIS_EMITTER_NAME_21735, "Muff Cob" },
    { DIS_EMITTER_NAME_21780, "Mushroom" },
    { DIS_EMITTER_NAME_21825, "Mushroom 1" },
    { DIS_EMITTER_NAME_21870, "Mushroom 2" },
    { DIS_EMITTER_NAME_21880, "N920Z" },
    { DIS_EMITTER_NAME_21890, "Nanjing B" },
    { DIS_EMITTER_NAME_21895, "Nanjing C" },
    { DIS_EMITTER_NAME_21915, "Nayada" },
    { DIS_EMITTER_NAME_21960, "Neptun" },
    { DIS_EMITTER_NAME_21980, "NIKE TT" },
    { DIS_EMITTER_NAME_22005, "NRBA 50" },
    { DIS_EMITTER_NAME_22050, "NRBA 51" },
    { DIS_EMITTER_NAME_22095, "NRBF 20A" },
    { DIS_EMITTER_NAME_22140, "Nysa B" },
    { DIS_EMITTER_NAME_22185, "O524A" },
    { DIS_EMITTER_NAME_22230, "O580B" },
    { DIS_EMITTER_NAME_22275, "O625Z" },
    { DIS_EMITTER_NAME_22320, "O626Z" },
    { DIS_EMITTER_NAME_22345, "Odd Group" },
    { DIS_EMITTER_NAME_22365, "Odd Lot" },
    { DIS_EMITTER_NAME_22410, "Odd Pair" },
    { DIS_EMITTER_NAME_22455, "Oka" },
    { DIS_EMITTER_NAME_22500, "OKEAN" },
    { DIS_EMITTER_NAME_22545, "OKINXE 12C" },
    { DIS_EMITTER_NAME_22590, "OMEGA" },
    { DIS_EMITTER_NAME_22635, "Omera ORB32" },
    { DIS_EMITTER_NAME_22680, "One Eye" },
    { DIS_EMITTER_NAME_22690, "OP-28" },
    { DIS_EMITTER_NAME_22725, "OPS-16B" },
    { DIS_EMITTER_NAME_22730, "OPS-18" },
    { DIS_EMITTER_NAME_22740, "OPS-28" },
    { DIS_EMITTER_NAME_22770, "OR-2" },
    { DIS_EMITTER_NAME_22810, "ORB-31S" },
    { DIS_EMITTER_NAME_22815, "ORB 32" },
    { DIS_EMITTER_NAME_22860, "Orion Rtn 10X" },
    { DIS_EMITTER_NAME_22905, "Otomat MK II Teseo" },
    { DIS_EMITTER_NAME_22950, "Owl Screech" },
    { DIS_EMITTER_NAME_22955, "P360Z" },
    { DIS_EMITTER_NAME_22960, "PA-1660" },
    { DIS_EMITTER_NAME_22995, "Palm Frond" },
    { DIS_EMITTER_NAME_23040, "Palm Frond AB" },
    { DIS_EMITTER_NAME_23085, "Pat Hand TT" },
    { DIS_EMITTER_NAME_23095, "Pat Hand MG" },
    { DIS_EMITTER_NAME_23130, "Patty Cake" },
    { DIS_EMITTER_NAME_23175, "Pawn Cake" },
    { DIS_EMITTER_NAME_23220, "PBR 4 Rubin" },
    { DIS_EMITTER_NAME_23265, "Pea Sticks" },
    { DIS_EMITTER_NAME_23310, "Peel Cone" },
    { DIS_EMITTER_NAME_23355, "Peel Group" },
    { DIS_EMITTER_NAME_23400, "Peel Group A" },
    { DIS_EMITTER_NAME_23445, "Peel Group B" },
    { DIS_EMITTER_NAME_23490, "Peel Pair" },
    { DIS_EMITTER_NAME_23535, "Philips 9LV 200" },
    { DIS_EMITTER_NAME_23580, "Philips 9LV 331" },
    { DIS_EMITTER_NAME_23625, "Philips LV 223" },
    { DIS_EMITTER_NAME_23670, "Philips Sea Giraffe 50 HC" },
    { DIS_EMITTER_NAME_23690, "Pin Jib" },
    { DIS_EMITTER_NAME_23710, "Plank Shad" },
    { DIS_EMITTER_NAME_23715, "Plank Shave" },
    { DIS_EMITTER_NAME_23760, "Plank Shave A" },
    { DIS_EMITTER_NAME_23805, "Plank Shave B" },
    { DIS_EMITTER_NAME_23850, "Plate Steer" },
    { DIS_EMITTER_NAME_23895, "Plessey AWS 1" },
    { DIS_EMITTER_NAME_23940, "Plessey AWS 4" },
    { DIS_EMITTER_NAME_23985, "Plessey AWS 6" },
    { DIS_EMITTER_NAME_23990, "Plessey RJ" },
    { DIS_EMITTER_NAME_24030, "Plessey type 996" },
    { DIS_EMITTER_NAME_24075, "Plinth Net" },
    { DIS_EMITTER_NAME_24095, "Pluto" },
    { DIS_EMITTER_NAME_24100, "POHJANPALO" },
    { DIS_EMITTER_NAME_24120, "POLLUX" },
    { DIS_EMITTER_NAME_24165, "Pop Group" },
    { DIS_EMITTER_NAME_24210, "Pop Group MG" },
    { DIS_EMITTER_NAME_24255, "Pop Group TA" },
    { DIS_EMITTER_NAME_24300, "Pop Group TT" },
    { DIS_EMITTER_NAME_24345, "Pork Trough" },
    { DIS_EMITTER_NAME_24390, "Post Bow" },
    { DIS_EMITTER_NAME_24435, "Post Lamp" },
    { DIS_EMITTER_NAME_24480, "Pot Drum" },
    { DIS_EMITTER_NAME_24525, "Pot Head" },
    { DIS_EMITTER_NAME_24570, "PRIMUS 40 WXD" },
    { DIS_EMITTER_NAME_24615, "PRIMUS 300SL" },
    { DIS_EMITTER_NAME_24620, "Primus 3000" },
    { DIS_EMITTER_NAME_24650, "PS-05A" },
    { DIS_EMITTER_NAME_24660, "PS 46 A" },
    { DIS_EMITTER_NAME_24705, "PS 70 R" },
    { DIS_EMITTER_NAME_24710, "PS-890" },
    { DIS_EMITTER_NAME_24750, "Puff Ball" },
    { DIS_EMITTER_NAME_24770, "R-76" },
    { DIS_EMITTER_NAME_24780, "RAC-30" },
    { DIS_EMITTER_NAME_24795, "Racal 1229" },
    { DIS_EMITTER_NAME_24840, "Racal AC 2690 BT" },
    { DIS_EMITTER_NAME_24885, "Racal Decca 1216" },
    { DIS_EMITTER_NAME_24930, "Racal Decca 360" },
    { DIS_EMITTER_NAME_24975, "Racal Decca AC 1290" },
    { DIS_EMITTER_NAME_25020, "Racal Decca TM 1229" },
    { DIS_EMITTER_NAME_25065, "Racal Decca TM 1626" },
    { DIS_EMITTER_NAME_25110, "Racal DRBN 34A" },
    { DIS_EMITTER_NAME_25155, "Radar 24" },
    { DIS_EMITTER_NAME_25200, "RAN 7S" },
    { DIS_EMITTER_NAME_25205, "RAN 10S" },
    { DIS_EMITTER_NAME_25245, "RAN 11 LX" },
    { DIS_EMITTER_NAME_25260, "Rapier TA" },
    { DIS_EMITTER_NAME_25265, "Rapier 2000 TA, Dagger" },
    { DIS_EMITTER_NAME_25270, "Rapier MG" },
    { DIS_EMITTER_NAME_25280, "RAT-31S" },
    { DIS_EMITTER_NAME_25285, "RATAC (LCT)" },
    { DIS_EMITTER_NAME_25290, "Raytheon 1220" },
    { DIS_EMITTER_NAME_25300, "Raytheon 1302" },
    { DIS_EMITTER_NAME_25335, "Raytheon 1500" },
    { DIS_EMITTER_NAME_25380, "Raytheon 1645" },
    { DIS_EMITTER_NAME_25425, "Raytheon 1650" },
    { DIS_EMITTER_NAME_25470, "Raytheon 1900" },
    { DIS_EMITTER_NAME_25515, "Raytheon 2502" },
    { DIS_EMITTER_NAME_25560, "Raytheon TM 1650/6X" },
    { DIS_EMITTER_NAME_25605, "Raytheon TM 1660/12S" },
    { DIS_EMITTER_NAME_25630, "RAY-1220XR" },
    { DIS_EMITTER_NAME_25635, "RAY-1401" },
    { DIS_EMITTER_NAME_25650, "Ray 2900" },
    { DIS_EMITTER_NAME_25695, "Raypath" },
    { DIS_EMITTER_NAME_25735, "RBE2" },
    { DIS_EMITTER_NAME_25740, "RDM" },
    { DIS_EMITTER_NAME_25760, "RDY" },
    { DIS_EMITTER_NAME_25785, "RDN 72" },
    { DIS_EMITTER_NAME_25830, "RDR 1A" },
    { DIS_EMITTER_NAME_25835, "RDR 1E" },
    { DIS_EMITTER_NAME_25840, "RDR 4A" },
    { DIS_EMITTER_NAME_25875, "RDR 1200" },
    { DIS_EMITTER_NAME_25885, "RDR 1400" },
    { DIS_EMITTER_NAME_25890, "RDR 1400 C" },
    { DIS_EMITTER_NAME_25895, "RDR 1500" },
    { DIS_EMITTER_NAME_25920, "Rice Lamp" },
    { DIS_EMITTER_NAME_25965, "Rice Pad" },
    { DIS_EMITTER_NAME_26010, "Rice Screen" },
    { DIS_EMITTER_NAME_26055, "ROLAND BN" },
    { DIS_EMITTER_NAME_26100, "ROLAND MG" },
    { DIS_EMITTER_NAME_26145, "ROLAND TA" },
    { DIS_EMITTER_NAME_26190, "ROLAND TT" },
    { DIS_EMITTER_NAME_26235, "Round Ball" },
    { DIS_EMITTER_NAME_26280, "Round House" },
    { DIS_EMITTER_NAME_26325, "Round House B" },
    { DIS_EMITTER_NAME_26330, "RT-02/50" },
    { DIS_EMITTER_NAME_26350, "RTN-1A" },
    { DIS_EMITTER_NAME_26370, "RV2" },
    { DIS_EMITTER_NAME_26415, "RV3" },
    { DIS_EMITTER_NAME_26460, "RV5" },
    { DIS_EMITTER_NAME_26505, "RV10" },
    { DIS_EMITTER_NAME_26550, "RV17" },
    { DIS_EMITTER_NAME_26595, "RV18" },
    { DIS_EMITTER_NAME_26610, "RV-377" },
    { DIS_EMITTER_NAME_26640, "RV UM" },
    { DIS_EMITTER_NAME_26660, "RXN 2-60" },
    { DIS_EMITTER_NAME_26670, "S-1810CD" },
    { DIS_EMITTER_NAME_26685, "SA 2 Guideline" },
    { DIS_EMITTER_NAME_26730, "SA 3 Goa" },
    { DIS_EMITTER_NAME_26775, "SA 8 Gecko DT" },
    { DIS_EMITTER_NAME_26795, "SA-12 TELAR ILL" },
    { DIS_EMITTER_NAME_26820, "SA N 7 Gadfly TI" },
    { DIS_EMITTER_NAME_26865, "SA N 11 Cads 1 UN" },
    { DIS_EMITTER_NAME_26910, "Salt Pot A&B" },
    { DIS_EMITTER_NAME_26955, "SATURNE II" },
    { DIS_EMITTER_NAME_27000, "Scan Can" },
    { DIS_EMITTER_NAME_27045, "Scan Fix" },
    { DIS_EMITTER_NAME_27090, "Scan Odd" },
    { DIS_EMITTER_NAME_27135, "Scan Three" },
    { DIS_EMITTER_NAME_27140, "SCANTER (CSR)" },
    { DIS_EMITTER_NAME_27141, "SCORADS" },
    { DIS_EMITTER_NAME_27150, "SCOREBOARD" },
    { DIS_EMITTER_NAME_27180, "Scoup Plate" },
    { DIS_EMITTER_NAME_27190, "SCR-584" },
    { DIS_EMITTER_NAME_27225, "Sea Archer 2" },
    { DIS_EMITTER_NAME_27270, "Sea Hunter 4 MG" },
    { DIS_EMITTER_NAME_27315, "Sea Hunter 4 TA" },
    { DIS_EMITTER_NAME_27360, "Sea Hunter 4 TT" },
    { DIS_EMITTER_NAME_27405, "Sea Gull" },
    { DIS_EMITTER_NAME_27450, "Sea Net" },
    { DIS_EMITTER_NAME_27495, "Sea Spray" },
    { DIS_EMITTER_NAME_27540, "Sea Tiger" },
    { DIS_EMITTER_NAME_27570, "Searchwater" },
    { DIS_EMITTER_NAME_27585, "Selenia Orion 7" },
    { DIS_EMITTER_NAME_27630, "Selenia type 912" },
    { DIS_EMITTER_NAME_27675, "Selennia RAN 12 L/X" },
    { DIS_EMITTER_NAME_27720, "Selennia RTN 10X" },
    { DIS_EMITTER_NAME_27765, "Selinia ARP 1645" },
    { DIS_EMITTER_NAME_27810, "SGR 102 00" },
    { DIS_EMITTER_NAME_27855, "SGR 103/02" },
    { DIS_EMITTER_NAME_27870, "SGR-104" },
    { DIS_EMITTER_NAME_27900, "Sheet Bend" },
    { DIS_EMITTER_NAME_27945, "Sheet Curve" },
    { DIS_EMITTER_NAME_27990, "Ship Globe" },
    { DIS_EMITTER_NAME_28035, "Ship Wheel" },
    { DIS_EMITTER_NAME_28080, "SGR 114" },
    { DIS_EMITTER_NAME_28125, "Shore Walk A" },
    { DIS_EMITTER_NAME_28170, "Short Horn" },
    { DIS_EMITTER_NAME_28215, "Shot Dome" },
    { DIS_EMITTER_NAME_28260, "Side Globe JN" },
    { DIS_EMITTER_NAME_28280, "PRV-11, Side Net" },
    { DIS_EMITTER_NAME_28305, "Side Walk A" },
    { DIS_EMITTER_NAME_28350, "Signaal DA 02" },
    { DIS_EMITTER_NAME_28395, "Signaal DA 05" },
    { DIS_EMITTER_NAME_28440, "Signaal DA 08" },
    { DIS_EMITTER_NAME_28485, "Signaal LW 08" },
    { DIS_EMITTER_NAME_28530, "Signaal LWOR" },
    { DIS_EMITTER_NAME_28575, "Signaal M45" },
    { DIS_EMITTER_NAME_28620, "Signaal MW 08" },
    { DIS_EMITTER_NAME_28665, "Signaal SMART" },
    { DIS_EMITTER_NAME_28710, "Signaal STING" },
    { DIS_EMITTER_NAME_28755, "Signaal STIR" },
    { DIS_EMITTER_NAME_28800, "Signaal WM 20/2" },
    { DIS_EMITTER_NAME_28845, "Signaal WM 25" },
    { DIS_EMITTER_NAME_28890, "Signaal WM 27" },
    { DIS_EMITTER_NAME_28935, "Signaal WM 28" },
    { DIS_EMITTER_NAME_28980, "Signaal ZW 01" },
    { DIS_EMITTER_NAME_29025, "Signaal ZW 06" },
    { DIS_EMITTER_NAME_29070, "Ski Pole" },
    { DIS_EMITTER_NAME_29115, "Skin Head" },
    { DIS_EMITTER_NAME_29160, "Skip Spin" },
    { DIS_EMITTER_NAME_29185, "SKYGUARD TA, UAR-1021" },
    { DIS_EMITTER_NAME_29190, "SKYGUARD TT, UAR-1021" },
    { DIS_EMITTER_NAME_29205, "Sky Watch" },
    { DIS_EMITTER_NAME_29215, "SKYSHADOW" },
    { DIS_EMITTER_NAME_29220, "SKYSHIELD TA" },
    { DIS_EMITTER_NAME_29250, "SL" },
    { DIS_EMITTER_NAME_29270, "SL/ALQ-234" },
    { DIS_EMITTER_NAME_29295, "Slap Shot E" },
    { DIS_EMITTER_NAME_29340, "Slim Net" },
    { DIS_EMITTER_NAME_29385, "Slot Back A" },
    { DIS_EMITTER_NAME_29400, "Slot Back ILL" },
    { DIS_EMITTER_NAME_29430, "Slot Back B" },
    { DIS_EMITTER_NAME_29440, "Slot Rest" },
    { DIS_EMITTER_NAME_29475, "SMA 3 RM" },
    { DIS_EMITTER_NAME_29520, "SMA 3 RM 20" },
    { DIS_EMITTER_NAME_29565, "SMA 3RM 20A/SMG" },
    { DIS_EMITTER_NAME_29610, "SMA BPS 704" },
    { DIS_EMITTER_NAME_29655, "SMA SPIN 749 (V) 2" },
    { DIS_EMITTER_NAME_29700, "SMA SPN 703" },
    { DIS_EMITTER_NAME_29745, "SMA SPN 751" },
    { DIS_EMITTER_NAME_29790, "SMA SPOS 748" },
    { DIS_EMITTER_NAME_29835, "SMA SPQ 2" },
    { DIS_EMITTER_NAME_29880, "SMA SPQ 2D" },
    { DIS_EMITTER_NAME_29925, "SMA SPQ 701" },
    { DIS_EMITTER_NAME_29970, "SMA SPS 702 UPX" },
    { DIS_EMITTER_NAME_30015, "SMA ST 2 OTOMAT II MH" },
    { DIS_EMITTER_NAME_30060, "SMA 718 Beacon" },
    { DIS_EMITTER_NAME_30080, "SNAP SHOT" },
    { DIS_EMITTER_NAME_30105, "Snoop Drift" },
    { DIS_EMITTER_NAME_30150, "Snoop Head" },
    { DIS_EMITTER_NAME_30195, "Snoop Pair" },
    { DIS_EMITTER_NAME_30240, "Snoop Plate" },
    { DIS_EMITTER_NAME_30285, "Snoop Slab" },
    { DIS_EMITTER_NAME_30330, "Snoop Tray" },
    { DIS_EMITTER_NAME_30375, "Snoop Tray 1" },
    { DIS_EMITTER_NAME_30420, "Snoop Tray 2" },
    { DIS_EMITTER_NAME_30465, "Snoop Watch" },
    { DIS_EMITTER_NAME_30470, "9S18M1, Snow Drift" },
    { DIS_EMITTER_NAME_30510, "SO-1" },
    { DIS_EMITTER_NAME_30520, "SO-12" },
    { DIS_EMITTER_NAME_30555, "SO A Communist" },
    { DIS_EMITTER_NAME_30580, "SO-69" },
    { DIS_EMITTER_NAME_30600, "Sock Eye" },
    { DIS_EMITTER_NAME_30645, "SOM 64" },
    { DIS_EMITTER_NAME_30670, "SPADA TT" },
    { DIS_EMITTER_NAME_30690, "Sparrow (AIM/RIM-7) ILL" },
    { DIS_EMITTER_NAME_30700, "Sperry M-3" },
    { DIS_EMITTER_NAME_30735, "SPG 53F" },
    { DIS_EMITTER_NAME_30780, "SPG 70 (RTN 10X)" },
    { DIS_EMITTER_NAME_30825, "SPG 74 (RTN 20X)" },
    { DIS_EMITTER_NAME_30870, "SPG 75 (RTN 30X)" },
    { DIS_EMITTER_NAME_30915, "SPG 76 (RTN 30X)" },
    { DIS_EMITTER_NAME_30960, "Spin Scan A" },
    { DIS_EMITTER_NAME_31005, "Spin Scan B" },
    { DIS_EMITTER_NAME_31050, "Spin Trough" },
    { DIS_EMITTER_NAME_31095, "Splash Drop" },
    { DIS_EMITTER_NAME_31140, "SPN 35A" },
    { DIS_EMITTER_NAME_31185, "SPN 41" },
    { DIS_EMITTER_NAME_31230, "SPN 42" },
    { DIS_EMITTER_NAME_31275, "SPN 43A" },
    { DIS_EMITTER_NAME_31320, "SPN 43B" },
    { DIS_EMITTER_NAME_31365, "SPN 44" },
    { DIS_EMITTER_NAME_31410, "SPN 46" },
    { DIS_EMITTER_NAME_31455, "SPN 703" },
    { DIS_EMITTER_NAME_31500, "SPN 728 (V) 1" },
    { DIS_EMITTER_NAME_31545, "SPN 748" },
    { DIS_EMITTER_NAME_31590, "SPN 750" },
    { DIS_EMITTER_NAME_31635, "Sponge Cake" },
    { DIS_EMITTER_NAME_31680, "P-12, Spoon Rest" },
    { DIS_EMITTER_NAME_31681, "P-18, Spoon Rest A" },
    { DIS_EMITTER_NAME_31682, "P-18, Spoon Rest B" },
    { DIS_EMITTER_NAME_31684, "P-18, Spoon Rest D" },
    { DIS_EMITTER_NAME_31725, "SPQ 712 (RAN 12 L/X)" },
    { DIS_EMITTER_NAME_31770, "SPS 6C" },
    { DIS_EMITTER_NAME_31815, "SPS 10F" },
    { DIS_EMITTER_NAME_31860, "SPS 12" },
    { DIS_EMITTER_NAME_31905, "(deleted)SPS 58" },
    { DIS_EMITTER_NAME_31950, "(deleted)SPS 64" },
    { DIS_EMITTER_NAME_31995, "SPS 768 (RAN EL)" },
    { DIS_EMITTER_NAME_32040, "SPS 774 (RAN 10S)" },
    { DIS_EMITTER_NAME_32085, "SPY 790" },
    { DIS_EMITTER_NAME_32130, "Square Head" },
    { DIS_EMITTER_NAME_32175, "Square Pair" },
    { DIS_EMITTER_NAME_32220, "Square Slot" },
    { DIS_EMITTER_NAME_32265, "Square Tie" },
    { DIS_EMITTER_NAME_32310, "Squash Dome" },
    { DIS_EMITTER_NAME_32330, "P-15M, Squat Eye" },
    { DIS_EMITTER_NAME_32355, "Squint Eye" },
    { DIS_EMITTER_NAME_32400, "SRN 6" },
    { DIS_EMITTER_NAME_32445, "SRN 15" },
    { DIS_EMITTER_NAME_32490, "SRN 745" },
    { DIS_EMITTER_NAME_32535, "SRO 1" },
    { DIS_EMITTER_NAME_32580, "SRO 2" },
    { DIS_EMITTER_NAME_32625, "SS C 2B Samlet MG" },
    { DIS_EMITTER_NAME_32670, "SS N 2A B CSSC" },
    { DIS_EMITTER_NAME_32715, "SS N 2A B CSSC 2A 3A2 MH" },
    { DIS_EMITTER_NAME_32760, "SS N 2C Seeker" },
    { DIS_EMITTER_NAME_32805, "SS N 2C D Styx" },
    { DIS_EMITTER_NAME_32850, "SS N 2C D Styx C D MH" },
    { DIS_EMITTER_NAME_32895, "SS N 3 SSC SS C 18 BN" },
    { DIS_EMITTER_NAME_32940, "SS N 3B Sepal AL" },
    { DIS_EMITTER_NAME_32985, "SS N 3B Sepal MH" },
    { DIS_EMITTER_NAME_33030, "SS N 9 Siren" },
    { DIS_EMITTER_NAME_33075, "SS N 9 Siren AL" },
    { DIS_EMITTER_NAME_33120, "SS N 9 Siren MH" },
    { DIS_EMITTER_NAME_33165, "SS N 12 Sandbox AL" },
    { DIS_EMITTER_NAME_33210, "SS N 12 Sandbox MH" },
    { DIS_EMITTER_NAME_33255, "SS N 19 Shipwreck" },
    { DIS_EMITTER_NAME_33300, "SS N 19 Shipwreck AL" },
    { DIS_EMITTER_NAME_33345, "SS N 19 Shipwreck MH" },
    { DIS_EMITTER_NAME_33390, "SS N 21 AL" },
    { DIS_EMITTER_NAME_33435, "SS N 22 Sunburn" },
    { DIS_EMITTER_NAME_33480, "SS N 22 Sunburn MH" },
    { DIS_EMITTER_NAME_33525, "Stone Cake" },
    { DIS_EMITTER_NAME_33570, "STR 41" },
    { DIS_EMITTER_NAME_33590, "Straight Flush TA" },
    { DIS_EMITTER_NAME_33595, "Straight Flush TT" },
    { DIS_EMITTER_NAME_33600, "Straight Flush ILL" },
    { DIS_EMITTER_NAME_33615, "Strike Out" },
    { DIS_EMITTER_NAME_33660, "Strut Curve" },
    { DIS_EMITTER_NAME_33705, "Strut Pair" },
    { DIS_EMITTER_NAME_33750, "Strut Pair 1" },
    { DIS_EMITTER_NAME_33795, "Strut Pair 2" },
    { DIS_EMITTER_NAME_33840, "Sun Visor" },
    { DIS_EMITTER_NAME_33860, "Superfledermaus" },
    { DIS_EMITTER_NAME_33885, "Swift Rod 1" },
    { DIS_EMITTER_NAME_33930, "Swift Rod 2" },
    { DIS_EMITTER_NAME_33975, "T1166" },
    { DIS_EMITTER_NAME_34020, "T1171" },
    { DIS_EMITTER_NAME_34040, "T1202" },
    { DIS_EMITTER_NAME_34065, "T6004" },
    { DIS_EMITTER_NAME_34110, "T6031" },
    { DIS_EMITTER_NAME_34155, "T8067" },
    { DIS_EMITTER_NAME_34200, "T8068" },
    { DIS_EMITTER_NAME_34245, "T8124" },
    { DIS_EMITTER_NAME_34290, "T8408" },
    { DIS_EMITTER_NAME_34335, "T8911" },
    { DIS_EMITTER_NAME_34380, "T8937" },
    { DIS_EMITTER_NAME_34425, "T8944" },
    { DIS_EMITTER_NAME_34470, "T8987" },
    { DIS_EMITTER_NAME_34515, "P-14, Tall King" },
    { DIS_EMITTER_NAME_34560, "Tall Mike" },
    { DIS_EMITTER_NAME_34605, "Tall Path" },
    { DIS_EMITTER_NAME_34625, "Team Work" },
    { DIS_EMITTER_NAME_34640, "THAAD GBR" },
    { DIS_EMITTER_NAME_34650, "THD 225" },
    { DIS_EMITTER_NAME_34670, "THD 1940, Picador" },
    { DIS_EMITTER_NAME_34695, "THD 5500" },
    { DIS_EMITTER_NAME_34740, "Thin Path" },
    { DIS_EMITTER_NAME_34785, "PRV-9, Thin Skin" },
    { DIS_EMITTER_NAME_34795, "Thompson CSF TA-10" },
    { DIS_EMITTER_NAME_34830, "Thompson CSF TH D 1040 Neptune" },
    { DIS_EMITTER_NAME_34875, "Thompson CSF Calypso" },
    { DIS_EMITTER_NAME_34920, "Thompson CSF CASTOR" },
    { DIS_EMITTER_NAME_34965, "Thompson CSF Castor II" },
    { DIS_EMITTER_NAME_35010, "Thompson CSF DRBC 32A" },
    { DIS_EMITTER_NAME_35055, "Thompson CSF DRBJ 11 D/E" },
    { DIS_EMITTER_NAME_35100, "Thompson CSF DRBV 15A" },
    { DIS_EMITTER_NAME_35145, "Thompson CSF DRBV 15C" },
    { DIS_EMITTER_NAME_35190, "Thompson CSF DRBV 22D" },
    { DIS_EMITTER_NAME_35235, "Thompson CSF DRBV 23B" },
    { DIS_EMITTER_NAME_35280, "Thompson CSF DRUA 33" },
    { DIS_EMITTER_NAME_35325, "Thompson CSF Mars DRBV 21A" },
    { DIS_EMITTER_NAME_35370, "Thompson CSF Sea Tiger" },
    { DIS_EMITTER_NAME_35415, "Thompson CSF Triton" },
    { DIS_EMITTER_NAME_35460, "Thompson CSF Vega with DRBC 32E" },
    { DIS_EMITTER_NAME_35480, "TRS-2105, TIGER-G" },
    { DIS_EMITTER_NAME_35490, "TRS-2100, TIGER-S" },
    { DIS_EMITTER_NAME_35505, "Tie Rods" },
    { DIS_EMITTER_NAME_35550, "36D6, Tin Shield" },
    { DIS_EMITTER_NAME_35570, "Tin Trap" },
    { DIS_EMITTER_NAME_35580, "TIRSPONDER" },
    { DIS_EMITTER_NAME_35595, "Toad Stool 1" },
    { DIS_EMITTER_NAME_35640, "Toad Stool 2" },
    { DIS_EMITTER_NAME_35685, "Toad Stool 3" },
    { DIS_EMITTER_NAME_35730, "Toad Stool 4" },
    { DIS_EMITTER_NAME_35775, "Toad Stool 5" },
    { DIS_EMITTER_NAME_35800, "Tomb Stone" },
    { DIS_EMITTER_NAME_35820, "Top Bow" },
    { DIS_EMITTER_NAME_35865, "Top Dome" },
    { DIS_EMITTER_NAME_35910, "Top Knot" },
    { DIS_EMITTER_NAME_35955, "Top Mesh" },
    { DIS_EMITTER_NAME_36000, "Top Pair" },
    { DIS_EMITTER_NAME_36045, "Top Plate" },
    { DIS_EMITTER_NAME_36090, "Top Sail" },
    { DIS_EMITTER_NAME_36135, "Top Steer" },
    { DIS_EMITTER_NAME_36180, "Top Trough" },
    { DIS_EMITTER_NAME_36220, "Scrum Half TA" },
    { DIS_EMITTER_NAME_36225, "TorScrum Half TT, Tor" },
    { DIS_EMITTER_NAME_36230, "Scrum Half MG" },
    { DIS_EMITTER_NAME_36270, "Track Dish" },
    { DIS_EMITTER_NAME_36315, "TORSO M" },
    { DIS_EMITTER_NAME_36360, "Trap Door" },
    { DIS_EMITTER_NAME_36380, "TRISPONDE" },
    { DIS_EMITTER_NAME_36405, "TRS 3033" },
    { DIS_EMITTER_NAME_36420, "TRS 3405" },
    { DIS_EMITTER_NAME_36425, "TRS 3410" },
    { DIS_EMITTER_NAME_36430, "TRS 3415" },
    { DIS_EMITTER_NAME_36450, "TRS-N" },
    { DIS_EMITTER_NAME_36495, "TSE 5000" },
    { DIS_EMITTER_NAME_36540, "TSR 333" },
    { DIS_EMITTER_NAME_36585, "Tube Arm" },
    { DIS_EMITTER_NAME_36630, "Twin Eyes" },
    { DIS_EMITTER_NAME_36675, "Twin Pill" },
    { DIS_EMITTER_NAME_36720, "Twin Scan" },
    { DIS_EMITTER_NAME_36765, "Twin Scan Ro" },
    { DIS_EMITTER_NAME_36810, "Two Spot" },
    { DIS_EMITTER_NAME_36855, "TYPE 262" },
    { DIS_EMITTER_NAME_36900, "TYPE 275" },
    { DIS_EMITTER_NAME_36945, "TYPE 293" },
    { DIS_EMITTER_NAME_36990, "TYPE 343 SUN VISOR B" },
    { DIS_EMITTER_NAME_37035, "TYPE 347B" },
    { DIS_EMITTER_NAME_37050, "Type-404A(CH)" },
    { DIS_EMITTER_NAME_37080, "Type 756" },
    { DIS_EMITTER_NAME_37125, "TYPE 903" },
    { DIS_EMITTER_NAME_37170, "TYPE 909 TI" },
    { DIS_EMITTER_NAME_37215, "TYPE 909 TT" },
    { DIS_EMITTER_NAME_37260, "TYPE 910" },
    { DIS_EMITTER_NAME_37265, "TYPE-931(CH)" },
    { DIS_EMITTER_NAME_37305, "TYPE 965" },
    { DIS_EMITTER_NAME_37350, "TYPE 967" },
    { DIS_EMITTER_NAME_37395, "TYPE 968" },
    { DIS_EMITTER_NAME_37440, "TYPE 974" },
    { DIS_EMITTER_NAME_37485, "TYPE 975" },
    { DIS_EMITTER_NAME_37530, "TYPE 978" },
    { DIS_EMITTER_NAME_37575, "TYPE 992" },
    { DIS_EMITTER_NAME_37620, "TYPE 993" },
    { DIS_EMITTER_NAME_37665, "TYPE 994" },
    { DIS_EMITTER_NAME_37710, "TYPE 1006(1)" },
    { DIS_EMITTER_NAME_37755, "TYPE 1006(2)" },
    { DIS_EMITTER_NAME_37800, "TYPE 1022" },
    { DIS_EMITTER_NAME_37845, "UK MK 10" },
    { DIS_EMITTER_NAME_37850, "UPS-220C" },
    { DIS_EMITTER_NAME_37890, "UPX 1 10" },
    { DIS_EMITTER_NAME_37935, "UPX 27" },
    { DIS_EMITTER_NAME_37980, "URN 20" },
    { DIS_EMITTER_NAME_38025, "URN 25" },
    { DIS_EMITTER_NAME_38045, "VOLEX III/IV" },
    { DIS_EMITTER_NAME_38070, "W8818" },
    { DIS_EMITTER_NAME_38115, "W8838" },
    { DIS_EMITTER_NAME_38120, "W8852" },
    { DIS_EMITTER_NAME_38160, "WAS-74S" },
    { DIS_EMITTER_NAME_38205, "Wasp Head" },
    { DIS_EMITTER_NAME_38210, "WATCHDOG" },
    { DIS_EMITTER_NAME_38250, "Watch Guard" },
    { DIS_EMITTER_NAME_38260, "Watchman" },
    { DIS_EMITTER_NAME_38295, "Western Electric MK 10" },
    { DIS_EMITTER_NAME_38320, "Westinghouse ADR-4 LRSR" },
    { DIS_EMITTER_NAME_38340, "Westinghouse Electric SPG 50" },
    { DIS_EMITTER_NAME_38385, "Westinghouse Electric W 120" },
    { DIS_EMITTER_NAME_38430, "Westinghouse SPS 29C" },
    { DIS_EMITTER_NAME_38475, "Westinghouse SPS 37" },
    { DIS_EMITTER_NAME_38520, "Wet Eye" },
    { DIS_EMITTER_NAME_38565, "Wet Eye Mod" },
    { DIS_EMITTER_NAME_38570, "WGU-41/B" },
    { DIS_EMITTER_NAME_38572, "WGU-44/B" },
    { DIS_EMITTER_NAME_38610, "Whiff" },
    { DIS_EMITTER_NAME_38655, "Whiff Brick" },
    { DIS_EMITTER_NAME_38700, "Whiff Fire" },
    { DIS_EMITTER_NAME_38715, "WHITE HOUSE" },
    { DIS_EMITTER_NAME_38745, "Wild Card" },
    { DIS_EMITTER_NAME_38790, "Witch Eight" },
    { DIS_EMITTER_NAME_38835, "Witch Five" },
    { DIS_EMITTER_NAME_38880, "WM2X Series" },
    { DIS_EMITTER_NAME_38925, "WM2X Series CAS" },
    { DIS_EMITTER_NAME_38950, "WSR-74C" },
    { DIS_EMITTER_NAME_38955, "WSR-74S" },
    { DIS_EMITTER_NAME_38970, "Wood Gage" },
    { DIS_EMITTER_NAME_39015, "Yard Rake" },
    { DIS_EMITTER_NAME_39060, "Yew Loop" },
    { DIS_EMITTER_NAME_39105, "Yo-Yo" },
    { DIS_EMITTER_NAME_39150, "(deleted)" },
    { 0, NULL }
};
